//-< Htree.java >----------------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// GOODS Persistent Class Library                                    *  /  \  *
//                          Created:     15-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 18-Oct-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// H-Tree: combination of tree and hash table
//-------------------------------------------------------------------*--------*

package goodslib;
import  goodsjpi.*;

class HtreeItem extends Persistent {
    HtreeItem  next; // collision chain
    Persistent obj;
    Persistent key;
    int        hash;
    
    HtreeItem(Persistent obj, Persistent key, int hash, HtreeItem collisionChain)
    { 
	this.obj = obj;
	this.key = key;
	this.hash = hash;
	next = collisionChain;
    }
}

class HtreePage extends Persistent {
    Persistent table[];

    void reset(int height) { 
	if (--height != 0) { 
	    for (int i = table.length; --i >= 0;) { 
		HtreePage pg = (HtreePage)table[i];
		pg.reset(height);
	    }
	} else { 
	    Metaobject.modify();
	    Persistent tab[] = table;
	    for (int i = tab.length; --i >= 0; tab[i] = null);
	}
    }

    void setChild(int i, Persistent child) { 
	Metaobject.modify();
	table[i] = child;
    }

    HtreePage(int size) { 
	table = new Persistent[size];
    }
}


public class Htree extends Persistent { 
    /**
     * Constructs a new, empty Htree with the specified size and
     * logarithm of page size (number of bits from the key used as page index)
     *
     * @param      hashSize  hash key divider
     * @param      pageBits  logarithm of page size
     */
    public Htree(int hashSize, int pageBits) { 
	for (height = 1; (1L << height*pageBits) < hashSize; height++);
	this.hashSize = hashSize;
	this.pageBits = pageBits; 
	root = new HtreePage(1 << pageBits);
    }
   /**
     * Constructs a new, empty Htree with the specified hash size
     * and default page size (128)
     *
     * @param     hashSize     hash key divider
     */
    public Htree(int hashSize) { 
	this(hashSize, 7);
    }
   /**
     * Constructs a new, empty Htree with the default hash size
     * (1000003)  and default page size (128)
     */
    public Htree() { 
	this(1000003, 7);
    }

    /**
     * Returns the number of keys in this Htree.
     *
     * @return  the number of keys in this Htree.
     */
    public int size() {
	return count;
    }

    /**
     * Returns the value to which the specified key is mapped in this Htree
     *
     * @param   key   a key in the Htree.
     * @return  the value to which the key is mapped in this Htree;
     *          <code>null</code> if the key is not mapped to any value in
     *          this Htree.
     */
    public synchronized Persistent get(Object key) {
	int hashCode = HashTable.canonicalHashCode(key);
	int hashIndex = (hashCode & 0x7fffffff) % hashSize;
	HtreePage pg = root;
	int pageBits = this.pageBits;
	for (int i = height; --i > 0;) { 
	    int j = (hashIndex >>> i*pageBits) & ((1 << pageBits) - 1);
	    pg = (HtreePage)pg.table[j];
	    if (pg == null) { 
		return null;
	    }
	}
	HtreeItem item = (HtreeItem)pg.table[hashIndex & ((1 << pageBits)-1)];
	while (item != null) { 
	    if (item.hash == hashCode && item.key.equals(key)) { 
		return item.obj;
	    }
	    item = item.next;
	}
	return null;
    } 

    /**
     * Maps the specified <code>key</code> to the specified 
     * <code>value</code> in this Htree. Neither the key nor the 
     * value can be <code>null</code>. 
     * <p>
     * The value can be retrieved by calling the <code>get</code> method 
     * with a key that is equal to the original key. 
     *
     * @param      key     the Htree key.
     * @param      value   the value.
     * @return     the previous value of the specified key in this Htree,
     *             or <code>null</code> if it did not have one.
     */
    public synchronized Persistent put(Persistent key, Persistent value) {
	int hashCode = HashTable.canonicalHashCode(key);
	int hashIndex = (hashCode & 0x7fffffff) % hashSize;
	HtreePage pg = root;
	int pageBits = this.pageBits;
	int i;
	for (i = height; --i > 0;) { 
	    int j = (hashIndex >>> i*pageBits) & ((1 << pageBits) - 1);
	    HtreePage child = (HtreePage)pg.table[j];
	    if (child == null) {
		child = new HtreePage(1 << pageBits);
		pg.setChild(j, child);
	    }
	    pg = child;
	}
	i = hashIndex & ((1 << pageBits) - 1);
	HtreeItem item, collisionChain = (HtreeItem)pg.table[i];

	for (item = collisionChain; item != null; item = item.next) { 
	    if (item.hash == hashCode && item.key.equals(key)) {
		Persistent p = item.obj;
		item.obj = value;
		return p;
	    }
	}
	pg.setChild(i, new HtreeItem(value, key, hashCode, collisionChain));
	count += 1;
	return null;
    } 

    /** 
     * Associate object with string key. 
     */
    public Persistent put(String key, Persistent value) {
	return put(new ArrayOfChar(key), value);
    }
    /**
     * Removes the key (and its corresponding value) from this 
     * Htree. This method does nothing if the key is not in the Htree.
     *
     * @param   key   the key that needs to be removed.
     * @return  the value to which the key had been mapped in this Htree,
     *          or <code>null</code> if the key did not have a mapping.
     */
    public synchronized Persistent remove(Object key) {
	int hashCode = HashTable.canonicalHashCode(key);
	int hashIndex = (hashCode & 0x7fffffff) % hashSize;
	HtreePage pg = root;
	int pageBits = this.pageBits;
	int i;
	for (i = height; --i > 0;) { 
	    int j = (hashIndex >>> i*pageBits) & ((1 << pageBits) - 1);
	    pg = (HtreePage)pg.table[j];
	    if (pg == null) { 
		return null;
	    }
	}
	i = hashIndex & ((1 << pageBits) - 1);
	HtreeItem prev = null;
	HtreeItem item = (HtreeItem)pg.table[i];
	while (item != null) { 
	    if (item.hash == hashCode && item.key.equals(key)) { 
		if (prev == null) { 
		    pg.setChild(i, item.next);
		} else { 
		    prev.next = item.next;
		}
		count -= 1;
		return item.obj;
	    }
	    prev = item;
	    item = item.next;
	}
	return null;
    } 

    /**
     * Clears this Htree so that it contains no keys. 
     */
    public synchronized void clear() {
	root.reset(height);
	count = 0;
    }

    HtreePage root;
    int       count;
    int       height;
    int       hashSize;
    int       pageBits;
}
