//-< HashTable.java >------------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// GOODS Persistent Class Library                                    *  /  \  *
//                          Created:      9-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update:  9-Oct-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Hash table of persistent objects. This class was created from 
// JDK HashTable class implemented by Arthur van Hoff
//-------------------------------------------------------------------*--------*

package goodslib;
import  goodsjpi.*;

class HashTableEntry extends Persistent {
    int            hash;
    Persistent     key;
    Persistent     value;
    HashTableEntry next;

    static final Metaobject defaultMetaobject = new OptimisticMetaobject();
}

public class HashTable extends Persistent {
    protected HashTableEntry table[];

    protected int   count; // total number of entries
    
    protected int   threshold;
    
    protected float loadFactor;
   
    /**
     * Constructs a new, empty HashTable with the specified initial 
     * capacity and the specified load factor. 
     *
     * @param      initialCapacity   the initial capacity of the HashTable.
     * @param      loadFactor        a number between 0.0 and 1.0.
     * @exception  IllegalArgumentException  if the initial capacity is less
     *               than or equal to zero, or if the load factor is less than
     *               or equal to zero.
     */
    public HashTable(int initialCapacity, float loadFactor) {
	if (initialCapacity < 0 || loadFactor <= 0.0) {
	    throw new IllegalArgumentException();
	}
	table = new HashTableEntry[initialCapacity];
	this.loadFactor = loadFactor;
	threshold = (int)(initialCapacity * loadFactor);
    }
    
   /**
     * Constructs a new, empty HashTable with the specified initial capacity
     * and default load factor.
     *
     * @param   initialCapacity   the initial capacity of the HashTable.
     */
    public HashTable(int initialCapacity) {
	this(initialCapacity, 0.75f);
    }

    /**
     * Constructs a new, empty hashtable with a default capacity and load
     * factor, which is <tt>0.75</tt>. 
     */
    public HashTable() {
	this(101, 0.75f);
    }

    /**
     * Returns the number of keys in this HashTable.
     *
     * @return  the number of keys in this HashTable.
     */
    public int size() {
	return count;
    }

    /** 
     *  Different versions of JDK provides different implementations
     *  of hashCode() methods for the same classes. For example, 
     *  String.hashCode() will return different values in JDK 1.5 and JDK 1.2
     *  This method tries to solve this problem at least for String class.
     */
    public static int canonicalHashCode(Object obj) { 
	if (obj instanceof String) { 
	    String s = (String)obj;
	    int h = 0;
	    for (int i = 0, len = s.length(); i < len; i++) { 
		h = 31*h + s.charAt(i);
	    }
	    return h;
	} 
	return obj.hashCode();
    }


    /**
     * Returns the value to which the specified key is mapped in this HashTable
     *
     * @param   key   a key in the HashTable.
     * @return  the value to which the key is mapped in this HashTable;
     *          <code>null</code> if the key is not mapped to any value in
     *          this HashTable.
     */
    public synchronized Persistent get(Object key) {
	HashTableEntry tab[] = table;
	int hash = canonicalHashCode(key);
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (HashTableEntry e = tab[index] ; e != null ; e = e.next) {
	    if ((e.hash == hash) && e.key.equals(key)) {
		return e.value;
	    }
	}
	return null;
    }

    /**
     * Rehashes the contents of the HashTable into a HashTable with a 
     * larger capacity. This method is called automatically when the 
     * number of keys in the HashTable exceeds this HashTable's capacity 
     * and load factor. 
     *
     */
    protected void rehash() {
	int oldCapacity = table.length;
	HashTableEntry oldTable[] = table;

	int newCapacity = oldCapacity * 2 + 1;
	HashTableEntry newTable[] = new HashTableEntry[newCapacity];

	threshold = (int)(newCapacity * loadFactor);
	table = newTable;

	for (int i = oldCapacity ; --i >= 0 ;) {
	    for (HashTableEntry old = oldTable[i]; old != null;) {
		HashTableEntry e = old;
		old = old.next;

		int index = (e.hash & 0x7FFFFFFF) % newCapacity;
		e.next = newTable[index];
		newTable[index] = e;
	    }
	}
    }

    /**
     * Maps the specified <code>key</code> to the specified 
     * <code>value</code> in this HashTable. Neither the key nor the 
     * value can be <code>null</code>. 
     * <p>
     * The value can be retrieved by calling the <code>get</code> method 
     * with a key that is equal to the original key. 
     *
     * @param      key     the HashTable key.
     * @param      value   the value.
     * @return     the previous value of the specified key in this HashTable,
     *             or <code>null</code> if it did not have one.
     * @exception  NullPointerException  if the key or value is
     *               <code>null</code>.
     */
    public synchronized Persistent put(Persistent key, Persistent value) {
	// Make sure the value is not null
	if (value == null) {
	    throw new NullPointerException();
	}

	// Makes sure the key is not already in the HashTable.
	HashTableEntry tab[] = table;
	int hash = canonicalHashCode(key);
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (HashTableEntry e = tab[index] ; e != null ; e = e.next) {
	    if ((e.hash == hash) && e.key.equals(key)) {
		Persistent old = e.value;
		e.value = value;
		return old;
	    }
	}

	if (count >= threshold) {
	    // Rehash the table if the threshold is exceeded
	    rehash();
	    index = (hash & 0x7FFFFFFF) % tab.length;
	} 

	// Creates the new entry.
	HashTableEntry e = new HashTableEntry();
	e.hash = hash;
	e.key = key;
	e.value = value;
	e.next = tab[index];
	tab[index] = e;
	count++;
	return null;
    }

    /** 
     * Associate object with string key. 
     */
    public Persistent put(String key, Persistent value) {
	return put(new ArrayOfChar(key), value);
    }


    /**
     * Removes the key (and its corresponding value) from this 
     * HashTable. This method does nothing if the key is not in the HashTable.
     *
     * @param   key   the key that needs to be removed.
     * @return  the value to which the key had been mapped in this HashTable,
     *          or <code>null</code> if the key did not have a mapping.
     */
    public synchronized Persistent remove(Object key) {
	HashTableEntry tab[] = table;
	int hash = canonicalHashCode(key);
	int index = (hash & 0x7FFFFFFF) % tab.length;

	for (HashTableEntry e = tab[index], prev = null;
	     e != null; 
	     prev = e, e = e.next) 
	{
	    if ((e.hash == hash) && e.key.equals(key)) {
		if (prev != null) {
		    prev.next = e.next;
		} else {
		    tab[index] = e.next;
		}
		count--;
		return e.value;
	    }
	}
	return null;
    }

    /**
     * Clears this HashTable so that it contains no keys. 
     */
    public synchronized void clear() {
	HashTableEntry tab[] = table;
	for (int index = tab.length; --index >= 0; )
	    tab[index] = null;
	count = 0;
    }
}






