//-< Btree.java >----------------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// GOODS Persistent Class Library                                    *  /  \  *
//                          Created:     15-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 18-Oct-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// B-Tree
//-------------------------------------------------------------------*--------*

package goodslib;
import  goodsjpi.*;

public class Btree extends SetOwner { 
    protected Bpage root;
    protected int   height;
    protected int   pageSize;

    public synchronized void insert(SetMember mbr) { 
	Assert.that(mbr.owner == null);
	if (root == null) { 
	    super.insertFirst(mbr);
	    root = new Bpage(mbr, pageSize);
	    height = 1;
	} else { 
	    Bpage brother = root.insert(height, mbr, this);
	    if (brother != null) {
		root = new Bpage(root, brother, pageSize);
		height += 1;
	    }
	}
    }

    public synchronized void remove(SetMember mbr) { 
	super.remove(mbr);
	if (root.remove(height, mbr) && root.m == pageSize-1) { 
	    if (--height == 0) { 
		root = null;
	    } else { 
		root = (Bpage)root.e[pageSize-1];
	    }
	}
    }

    public synchronized SetMember find(Object key) { 
	if (root != null) { 
	    SetMember mbr = root.find(height, key);
	    if (mbr != null && mbr.compareKey(key) == 0) { 
		return mbr;
	    }
	}	
	return null;
    }

    public synchronized SetMember findGreaterOrEqual(Object key) {
	if (root == null) { 
	    return null;
	} else { 
	    return root.find(height, key);
	}	
    }

    public synchronized void insertAfter(SetMember after, SetMember mbr) { 
	Assert.that(after.owner == this);
	insert(mbr);
    }

    public synchronized void insertBefore(SetMember before, SetMember mbr) { 
	Assert.that(before.owner == this);
	insert(mbr);
    }

    public synchronized void insertFirst(SetMember mbr) { 
	insert(mbr);
    }
    
    public synchronized void insertLast(SetMember mbr) { 
	insert(mbr);
    }
    
    public synchronized void removeFirst() {
	remove(first);
    }

    public synchronized void removeLast() {
	remove(last);
    }
    
    public Btree(int pageSize) { 
	Assert.that((pageSize & 1) == 0);
	this.pageSize = pageSize;
    }

    //
    // Internal methods
    //
    final void linkAfter(SetMember after, SetMember mbr) { 
	super.insertAfter(after, mbr);
    }

    final void linkBefore(SetMember before, SetMember mbr) { 
	super.insertBefore(before, mbr);
    }

    final void linkLast(SetMember mbr) { 
	super.insertLast(mbr);
    }
}


class Bpage extends Ordered { 
    Ordered keymbr;
    Ordered e[]; 
    int     m;   // index of first used item at the page

    Bpage(Bpage root, Bpage brother, int pageSize) { 
	m = pageSize-2;
	e = new Ordered[pageSize];
	e[pageSize-2] = brother;
	e[pageSize-1] = root;
	keymbr = ((Bpage)root).keymbr;
    }

    Bpage(SetMember mbr, int pageSize) { 
	m = pageSize-2;
	e = new Ordered[pageSize];
	e[pageSize-2] = mbr;
	e[pageSize-1] = this;
	keymbr = null;
    }

    Bpage(int n) {
	e = new Ordered[n*2];
	m = n;
    }

    public int compare(Ordered mbr) { 
	return (keymbr != null) ? keymbr.compare(mbr) : 1;
    }

    public int compareKey(Object key) { 
	return (keymbr != null) ? keymbr.compareKey(key) : 1;
    }

    final SetMember find(int height, Object key) {
	int l = m, r = e.length-1;
	while (l < r)  {
	    int i = (l+r) >> 1;
	    if (e[i].compareKey(key) < 0) l = i+1; else r = i;
	}
	Assert.that(r == l && e[r].compareKey(key) >= 0); 
	if (--height != 0) { 
	    return ((Bpage)e[r]).find(height, key); 
	} else { 
	    return (SetMember)e[r]; 
	}
    }
    
    final static void arraynull(Object[] a, int index, int count) { 
	while (--count >= 0) { 
	    a[index++] = null;
	}
    }

    final Bpage insert(int level, SetMember mbr, Btree tree) {
	int  i, l = m, r = e.length-1;
	while (l < r)  {
	    i = (l+r) >> 1;
	    if (e[i].compare(mbr) < 0) l = i+1; else r = i;
	}
	Assert.that(r == l && e[r].compare(mbr) >= 0); 
	Ordered ins = mbr;
	if (--level == 0)  {
	    if (e[r].compare(mbr) == 0) { 
		tree.linkAfter((SetMember)e[r], mbr);
		return null;
	    } else { 
		if (e[r] == this) { 
		    tree.linkLast(mbr);
		} else { 
		    tree.linkBefore((SetMember)e[r], mbr);
		}
	    }		
	} else { 
	    ins = ((Bpage)e[r]).insert(level, mbr, tree);
	    if (ins == null) { 
		return null;
	    }
	}
	// insert before e[r]
	if (m > 0) {
	    System.arraycopy(e, m, e, m-1, r-m);
	    m -= 1;
	    e[r-1] = ins;
	    return null;
	} else { // page is full then divide page
	    int n = e.length >> 1;
	    Bpage b = new Bpage(n);
	    if (r < n) {
		System.arraycopy(e, 0, b.e, n, r);
		b.e[n+r] = ins;
		System.arraycopy(e, r, b.e, n+r+1, n-r-1);
	    } else {
		System.arraycopy(e, 0, b.e, n, n);
		System.arraycopy(e, n, e, n-1, r-n);
		e[r-1] = ins;
	    }
	    b.keymbr = b.e[2*n-1];
	    if (level != 0) {
		b.keymbr = ((Bpage)b.keymbr).keymbr;
	    }
	    m = n-1;
	    arraynull(e, 0, n-1);
	    return b;
	}
    }

    /**
     * Returns true if removing cause page underflow 
     */
    boolean remove(int level, SetMember mbr) { 
	int i, l = m, n = e.length >> 1, last = 2*n-1, r = last;

	while (l < r) {
	    i = (l+r) >> 1;
	    if (e[i].compare(mbr) < 0) l = i+1; else r = i;
	}
	Assert.that(r == l && e[r].compare(mbr) >= 0); 
	if (--level == 0) {
	    Assert.that(e[r].compare(mbr) == 0);  
	    if (mbr == e[r]) { 
		if (mbr.next != null && mbr.next.compare(mbr) == 0) { 
		    Metaobject.modify();
		    e[r] = mbr.next;
		} else {
		    System.arraycopy(e, m, e, m+1, r-m);
		    e[m++] = null;
		} 	
		if (r == last) { 
		    keymbr = e[last];
		}
	    }
	} else { 
	    if (((Bpage)e[r]).remove(level, mbr)) { 
		Bpage a, b;
		a = (Bpage)e[r];
		Assert.that(a.m == n+1);
		if (r < last) { // exists greater page
		    b = (Bpage)e[r+1];
		    int bm = b.m; 
		    if (bm < n) {// reallocation of nodes between pages a and b
			i = (n - bm + 1) >> 1;
			b.m += i;
			a.m -= i;
			System.arraycopy(a.e, n+1, a.e, n+1-i, n-1);
			System.arraycopy(b.e, bm, a.e, 2*n-i, i);
			a.keymbr = (level == 1) 
			    ? a.e[last] : ((Bpage)a.e[last]).keymbr;
			arraynull(b.e, bm, i);
		    } else { // merge page a to b  
			Assert.that(bm == n); 
			b.m = 1;
			System.arraycopy(a.e, n+1, b.e, 1, n-1);
			System.arraycopy(e, m, e, m+1, r-m);
			e[m++] = null;
			// dismiss page 'a'
		    }
		} else { // page b is before a
		    b = (Bpage)e[r-1];
		    int bm = b.m; 
		    if (bm < n) { // reallocation
			i = (n - bm + 1) >> 1;
			b.m += i;
			a.m -= i;
			System.arraycopy(b.e, 2*n-i, a.e, n+1-i, i);
			System.arraycopy(b.e, bm, b.e, bm+i, 2*n-bm-i);
			b.keymbr = (level == 1) 
			    ? b.e[last] : ((Bpage)b.e[last]).keymbr;
			arraynull(b.e, bm, i);
		    } else { // merge page b to a
			Assert.that(bm == n); 
			a.m = 1;
			System.arraycopy(b.e, n, a.e, 1, n);
			System.arraycopy(e, m, e, m+1, r-1-m);
		        e[m++] = null;
			// dismiss page 'b'
		    }
		}
	    }
	    if (r == last && ((Bpage)e[last]).keymbr != keymbr) { 
		keymbr = ((Bpage)e[last]).keymbr;
	    }
	}
	return m > n;
    } 

    static final Metaobject defaultMetaobject = new OptimisticMetaobject();
}
