//-< Storage.java >--------------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// Java Program Interface                                            *  /  \  *
//                          Created:      1-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update:  4-Nov-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Client interface with storage server
//-------------------------------------------------------------------*--------*

package goodsjpi;

import java.io.*;
import java.net.*;
import java.util.Dictionary;

public class Storage implements Runnable {
    protected int      id;
    protected Database database; 

    protected Socket       socket;
    protected InputStream  in;  // socket input stream
    protected OutputStream out; // socket output stream

    public static int LingerTime = 10; // linger paramter for the socket

    protected byte[] result;   // buffer for passing server reply to the client
    protected byte[] notification;// buffer for sending notifications to server
    protected byte[] reply;    // buffer for soring reply received from server
    protected byte[] request;  // buffer for sending single request to server
 
    //
    // Array to establish mapping between  persistent class indentifiers 
    // and application class descriptors
    //
    protected ClassDescriptor[] classDictionary;
    protected int               classDictionarySize;

    protected int[]             cpidTable;
    protected int               cpidTableSize;

    // 
    // The following variables are used to keep list of stubs for just loaded
    // objects. These stubs solve two problems: 
    // 1) avoiding load of the same object more than once
    // 2) handling invalidation message for the object during loading
    //
    protected Persistent loadList;
    protected Persistent freeList;

    protected Object cs; // Critiacal section for socket write operations

    protected Persistent root; // Root object in the storage

    //
    // These event objects are used to synchronize receiver thread with 
    // client application threads. Receiver will handle asynchronouse 
    // notifications from server itself and signal rcvEvent object when reply 
    // for client request arriives. Then receiver waits until client signal
    // repEvent object to continue receving loop.
    //
    protected CondEvent rcvEvent;
    protected CondEvent repEvent;

    //
    // Buffer for storing body of transaction. If size of transaction buffer 
    // exceeds maxTransBufferSize, then reference to it will be cleared
    // after the end oftransaction to allow GC to free space occupid by this
    // buffer. Otherwise reference to buffer is kept to reuse it in following
    // transactions.
    //
    protected byte[]  transBuffer;
    protected int     transBufferOffs;

    public static int initTransBufferSize = 64*1024;
    public static int maxTransBufferSize = 1024*1024;
    
    //
    // Weak cache of objects. This cache is used to locate loaded instances of 
    // persistent objects. Combination of object storage identifier and object
    // persistent identifier is used as key value for putting/searching
    // objects in cache. Weakness of the cache means that GC is able to
    // free object referenced only from this cache class.
    //
    protected static Dictionary objectCache;

    static { 
	//
	// Implementation of weak references depends on JDK version and vendor.
	// Weak references were standardized only in JDK 1.2. In JDK 1.1
	// class sun.misc.Cache is available which proides necessary 
	// functionality with Sun JVM. But it doesn't work with Microsoft
	// Jview, so we need to used MS specific com.ms.vm.WeakReference
	// class in this case. 
	// 
	try { 
	    // Solution for JDK 1.2 based on WeakReference class
	    Class.forName("java.lang.ref.WeakReference");
	    objectCache = (Dictionary)
		Class.forName("goodsjpi.weak.WeakHashTable").newInstance();
	} catch (Exception x1) { 
	    try { 
		// Solution for MS jview with com.ms.vm.WeakReference class
		Class.forName("com.ms.vm.WeakReference");
		objectCache = (Dictionary)
		  Class.forName("goodsjpi.weak.MsWeakHashTable").newInstance();
	    } catch (Exception x2) {
		try { 
		    objectCache = (Dictionary)
			Class.forName("sun.misc.Cache").newInstance();
		} catch (Exception x3) {}
	    }
	}
    }

    //
    // inform receiver about close initiation
    //
    protected volatile boolean closing; 

    protected Thread receiver; 

    //
    // Associate persistent class identifier with application class indentifier
    //
    final void associateCpidWithClass(ClassDescriptor desc, int cpid) {
	if (desc.cid >= cpidTableSize) { 
	    int newSize = desc.cid >= cpidTableSize*2 
		? desc.cid+1 : cpidTableSize*2;
	    int[] newTable = new int[newSize];
	    System.arraycopy(cpidTable, 0, newTable, 0, cpidTableSize);
	    cpidTable = newTable;
	    cpidTableSize = newSize;
	}
	cpidTable[desc.cid] = cpid;
    }

    // 
    // Extend size of transaction buffer 
    //
    final void appendTransBuffer(int size) { 
	if (transBufferOffs + size > transBuffer.length) { 
	    int newSize = transBufferOffs + size > transBuffer.length*2
		?  transBufferOffs + size : transBuffer.length*2;
	    byte[] newBuffer = new byte[newSize];
	    System.arraycopy(transBuffer, 0, newBuffer, 0, transBufferOffs);
	    transBuffer = newBuffer;
	}
	transBufferOffs += size;
    }
	
    //
    // Load object instances and initialize variables for detecting
    // deterioration of just loaded objects
    //
    final byte[] loadObjects(byte[] buf, int nObjects) { 
	Protocol.pack4(buf, Protocol.req_extra_offs, nObjects-1);
	return sendReceiveBody(buf, nObjects*Protocol.req_size,
			       Protocol.cmd_object);
    }

    //
    // Load all objects referenced from specified object "obj". 
    // When object is laoded rom database all its references are replaced 
    // with stub objects which type corresponds to th estatic type of field.
    // That stub object are used only for storing OPIDs of referenced objects.
    // Object is marked as RAW. Whan it is then accessed by its method 
    // invokation or component store/fetch operation, prepare() method
    // is called by metaobject to replace stub objects with real objects.
    //
    final void prepare(Persistent obj) {
	ClassDescriptor desc = obj.desc;
	byte[] data = obj.data;

	Assert.that((obj.state & (Persistent.DESTRUCTED|Persistent.RAW)) == 
		    Persistent.RAW && data != null && data.length > 0);

	int size = Protocol.unpack4(data, Protocol.hdr_size_offs);
	int varyingLength = 0;
	if (desc.varyingSize != 0) { 
	    varyingLength = (size - desc.fixedSize) / desc.varyingSize;
	} 
	int nRefs = desc.nFixedReferences
	          + varyingLength * desc.nVaryingReferences;
	Persistent refs[] = null; 
	if (nRefs > 0) { 
	    refs = new Persistent[nRefs];
	    byte[] reqbuf = new byte[nRefs*Protocol.req_size];
	    int[] permutation = new int[nRefs];
	    int[] sids = new int[nRefs];
	    int offs = Protocol.hdr_size;
	    int i, j, k;
	    for (i = 0; i < nRefs; i++) { 
		sids[i] = Protocol.unpack2(data, offs+Protocol.ref_sid_offs);
		offs += Protocol.ref_size;
	    }
	    for (i = database.storages.length; --i >= 0;) { 
		offs = Protocol.hdr_size;
		Storage storage = database.storages[i];
		synchronized (storage) { 
		    Persistent head = storage.loadList;
		    Persistent free = storage.freeList;
		    head.next = head.prev = head;
		    for (j = 0, k = 0; j < nRefs; j++) { 
			if (sids[j] == i) { 
			    int opid = 
				Protocol.unpack4(data, offs
						 + Protocol.ref_opid_offs);
			    if (opid != 0) { 
				ObjectId key = new ObjectId(i, opid);
				if ((refs[j] = (Persistent)
				     objectCache.get(key)) == null)
				{
				    int pos = k*Protocol.req_size;
				    reqbuf[pos + Protocol.req_cmd_offs] = 
					Protocol.cmd_load;
				    Protocol.pack2(reqbuf, pos +
						   Protocol.req_flags_offs, 0);
				    Protocol.pack4(reqbuf, pos + 
						   Protocol.req_opid_offs, 
						   opid);
				    permutation[k++] = j;
				    Persistent stub = free;
				    if (stub == null) { 
					stub=new Persistent((Metaobject)null);
				    } else { 
					free = free.next;
				    }
				    //
				    // Rememebr object index in "refs" array to
				    // make it possible to resolve other 
				    // references to this object at the end of 
				    // loading. 
				    //
				    stub.opid = j; 
				    stub.invalidated = false;
				    stub.next = head;
				    stub.prev = head.prev;
				    head.prev = stub;
				    stub.prev.next = stub;
				    objectCache.put(key, stub);
				}
			    }
			}
			offs += Protocol.ref_size;
		    }
		    if (k > 0) { 
			Persistent p;
 			byte[] body = storage.loadObjects(reqbuf, k);
			head = head.next;
			offs = 0;
			for (j = 0; j < k; j++) { 
			    p = storage.readObject(null, body, offs);
			    refs[permutation[j]] = p;
			    Persistent stub = head;
			    head = head.next;
			    if (stub.invalidated) { 
				p.state |= Persistent.DESTRUCTED;
				storage.throwObject(p);
				stub.invalidated = false;
			    }
			    stub.next = free;
			    free = stub;
			    offs += Protocol.hdr_size + Protocol.unpack4(body, 
				        offs+Protocol.hdr_size_offs);
			}
			for (j = 0; j < nRefs; j++) { 
			    p = refs[j];
			    if (p != null && p.storage == null) { // stub
				//
				// Stub object contains in "opid" field index
				// of real reference to the object in "refs"
				// array 
				//
				refs[j] = refs[p.opid];
			    }
			}
			storage.freeList = free;
		    }
		}
	    }
	}
	desc.unpack(obj, refs, varyingLength);
	obj.desc = desc.newClass;
    }

    // 
    // Load object instance from the server. After this method object
    // is in RAW state and prepare() should be called to resolve its references
    //
    protected synchronized Persistent load(int opid, Persistent obj) { 
	byte[] req = new byte[Protocol.req_size];
	req[Protocol.req_cmd_offs] = Protocol.cmd_load;
	Protocol.pack4(req, Protocol.req_opid_offs, opid);
	Protocol.pack2(req, Protocol.req_flags_offs, 0);
	byte[] body = loadObjects(req, 1);
	Persistent p = readObject(obj, body, 0);
	if (p != null) { 
	    Assert.that(obj == p || obj == null);
	    if (req[Protocol.req_cmd_offs] == Protocol.cmd_invalidate) { 
		p.invalidated = true;
	    }
	    prepare(p);
	}
	return p;
    }

    //
    // Append object or just object header(for unmodified objects) to 
    // the transaction buffer
    //
    protected void store(Persistent obj) { 
	ClassDescriptor desc = obj.desc;
	Assert.that(obj.opid != 0 && obj.storage == this);
	int cpid = 0;
	if (desc.cid < cpidTableSize) { 
	    cpid = cpidTable[desc.cid];
	}
	if (cpid == 0) { 
	    cpid = registerClass(desc.dbsClass);
	    Assert.that(cpid != 0);
	    associateCpidWithClass(desc, cpid);
	} 
	if (transBufferOffs == 0) { 
	    if (transBuffer == null) { 
		transBuffer = new byte[initTransBufferSize];
	    }
	    transBufferOffs = Protocol.req_size;
	}
	int offs = transBufferOffs;
	int flags = 0;
	int size = 0;

	if ((obj.state & Persistent.DIRTY) != 0) { 
	    size = desc.objectSize(obj);
	    flags = Protocol.tof_update;
	    appendTransBuffer(Protocol.hdr_size + size);
	    desc.pack(obj, transBuffer, offs+Protocol.hdr_size);
	} else { 
	    appendTransBuffer(Protocol.hdr_size);
	}
	if ((obj.state & Persistent.OPTIMISTIC) != 0) { 
	    flags |= Protocol.tof_validate;
	}
	if ((obj.state & (Persistent.XLOCKED|Persistent.SLOCKED)) != 0) { 
	    flags |= Protocol.tof_unlock;
	}
	Protocol.pack4(transBuffer, offs+Protocol.hdr_flags_offs,flags);
	Protocol.pack4(transBuffer, offs+Protocol.hdr_opid_offs, obj.opid);
	Protocol.pack2(transBuffer, offs+Protocol.hdr_sid_offs,  id);
	Protocol.pack2(transBuffer, offs+Protocol.hdr_cpid_offs, cpid);
	Protocol.pack4(transBuffer, offs+Protocol.hdr_size_offs, size);
    }
    
    //
    // Restore original state of transactio buffer. No request should be sent
    // to the server beacuse it knows nothing about this transaction.
    //
    protected void abortTransaction() { 
	if (transBuffer != null && transBuffer.length > maxTransBufferSize) {
	    // Buffer is too large, it is better to let GC to resue it space
	    transBuffer = null;
	}
	transBufferOffs = 0;
    }

    protected boolean isInvolvedInTransaction() {
	return transBufferOffs != 0;
    }

    //
    // Establish mapping between persistent and application classes for
    // loaded object and ask ClassDescritpor to unpack the object
    //
    protected Persistent readObject(Persistent obj, byte[] buffer, int offs) { 
	int cpid = Protocol.unpack2(buffer, offs+Protocol.hdr_cpid_offs);
	int opid = Protocol.unpack4(buffer, offs+Protocol.hdr_opid_offs);
	
	if (cpid == Protocol.RAW_CPID) { 
	    return null;
	}

	if (cpid >= classDictionarySize) { 
	    int newClassDictionarySize = (classDictionarySize*2 > cpid)
		? classDictionarySize*2 : cpid+1;
	    ClassDescriptor[] newClassDictionary = 
		new ClassDescriptor[newClassDictionarySize];
	    System.arraycopy(classDictionary, 0, newClassDictionary, 0,
			     classDictionarySize);
	    classDictionary = newClassDictionary;
	    classDictionarySize = newClassDictionarySize;
	}
	ClassDescriptor desc = classDictionary[cpid];
	
	if (desc == null) { 
	    byte[] dbsClassDesc = getClass(cpid);
	    Class cls = ClassDescriptor.getClass(dbsClassDesc);
	    desc = ClassDescriptor.lookup(cls);
	    if (!desc.equal(dbsClassDesc)) { 
		associateCpidWithClass(desc, registerClass(desc.dbsClass));
		desc = new ClassDescriptor(desc, dbsClassDesc);
	    } else { 
		associateCpidWithClass(desc, cpid);
	    }
	    classDictionary[cpid] = desc;
	}
	if (obj == null) { 
	    obj = desc.construct(this, opid, Persistent.RAW);
	} else { 
	    Assert.that(obj.desc == desc);
	    obj.state = (obj.state & ~Persistent.DESTRUCTED) | Persistent.RAW;
	}
	int size = Protocol.unpack4(buffer, offs+Protocol.hdr_size_offs);
	if (size + Protocol.hdr_size - offs != buffer.length) {
	    obj.data = new byte[size + Protocol.hdr_size];
	    System.arraycopy(buffer, offs, obj.data, 0, 
			     size + Protocol.hdr_size);
	} else { 
	    obj.data = buffer; 
	}
	return obj;
    }

    //
    // Retrieve reference to root object in the storage. Each GOODS
    // storage has preexested abstract root object. To prevent concurrent
    // initialization of storage by several clients, object with root PID
    // is locked and if this object is abstract root, null is returned to
    // client application allowing it to astablish it's own root object
    // as storage root (lcok prevent other clients from doing the same).
    // If root object is not abstract root, then it is unlocked and reference
    // to it is returnednto the client.
    // 
    protected synchronized Persistent getRoot() { 
	if (root == null) { 
	    root = new Persistent((Metaobject)null);
	    root.opid = Protocol.ROOT_OPID;
	    lock(root, Protocol.lck_exclusive, Protocol.lckattr_wait);
	    root.opid = 0;
	    root = load(Protocol.ROOT_OPID, null);
	    if (root != null) {
		unlock(root, Protocol.lck_none);
	    }
	}
	return root;
    }

    //
    // Set root for the storage. This method can only be called when previously
    // invoked getRoot() mrethod returns null. It means that storage root 
    // can be set only once.
    //
    protected synchronized void setRoot(Persistent root) { 
	Assert.that("only transient object can become a root", root.opid == 0);
	Assert.that("class was preprocessed by JavaMOP", root.accessCount==0);
	ObjectId key = new ObjectId(id, Protocol.ROOT_OPID);
	Persistent oldRoot = (Persistent)objectCache.get(key);
	Assert.that("root object can't be changed", oldRoot == null);
	
	root.opid = Protocol.ROOT_OPID;
	root.storage = this;
	root.state |= Persistent.DIRTY|Persistent.XLOCKED;
	objectCache.put(key, root);
	root.metaobject.preDaemon(root, Metaobject.MUTATOR);
	root.metaobject.postDaemon(root, Metaobject.MUTATOR, true);
	this.root = root;
    }

    //
    // Receiver thread
    //
    final void invalidate(int opid) {
	Persistent obj = (Persistent)objectCache.get(new ObjectId(id, opid));
	if (obj != null) { 
	    Metaobject metaobject = obj.metaobject;
	    if (metaobject != null) { 
		metaobject.invalidateObject(obj);
	    } else { 
		obj.invalidated = true; // stub 
	    }
	}
    }	    	

    public void run() { 
	while (true) { 
	    read(reply, Protocol.req_size);
	    switch (reply[Protocol.req_cmd_offs]) { 
	      case Protocol.cmd_invalidate:
		invalidate(Protocol.unpack4(reply, Protocol.req_opid_offs));
		int n = Protocol.unpack4(reply, Protocol.req_extra_offs);
		if (n != 0) { 
		    byte[] buf = new byte[n*Protocol.req_size];
		    read(buf, n*Protocol.req_size);
		    int offs = 0;
		    while (--n >= 0) { 
			invalidate(Protocol.unpack4(buf, offs + 
						    Protocol.req_opid_offs));
			offs += Protocol.req_size;
		    }
		}
		break;
	      case Protocol.cmd_bye:
		if (!closing) { 
		    database.disconnected(id);
		}
		repEvent.signal();
		return;
	      default:
		repEvent.signal();
		rcvEvent.waitSignal();
	    }
	}
    }
		
    //
    // Communication with server
    // 
    protected void write(byte[] buf, int size) { 
	synchronized(cs) {
	    try { 
		out.write(buf, 0, size);
	    } catch(IOException x) { 
		database.handleException(x);
	    }
	}
    }

    final void write(byte[] buf) { 
	write(buf, buf.length);
    }

    protected void read(byte[] buf, int size) { 
	try { 
	    int offs = 0;
	    while (size > 0) { 
		int len = in.read(buf, offs, size);
		if (len < 0) { 
		    database.handleError("Broken socket");
		}
		offs += len;
		size -= len;
	    }
	} catch(IOException x) { 
	    database.handleException(x);
	}
    }

    protected final void sendReceive(byte[] req, int expectedCmd) { 
	sendReceive(req, req.length, expectedCmd);
    }

    protected void sendReceive(byte[] req, int len, int expectedCmd) { 
	if (req != null) { 
	    write(req, len);
	}
	repEvent.waitSignal();
	Assert.that(reply[Protocol.req_cmd_offs] == expectedCmd);
	System.arraycopy(reply, 0, result, 0, Protocol.req_size);
	rcvEvent.signal();
    }

    protected final byte[] sendReceiveBody(byte[] req, int expectedCmd) { 
	return sendReceiveBody(req, req.length, expectedCmd);
    }

    protected byte[] sendReceiveBody(byte[] req, int len, int expectedCmd) { 
	if (req != null) { 
	    write(req, len);
	}
	repEvent.waitSignal();
	Assert.that(reply[Protocol.req_cmd_offs] == expectedCmd);
	System.arraycopy(reply, 0, result, 0, Protocol.req_size);

	int bodyLen = Protocol.unpack4(reply, Protocol.req_size_offs);
	Assert.that(bodyLen > 0);
	byte[] body = new byte[bodyLen];
	read(body, bodyLen);

	rcvEvent.signal();
	return body;
    }

    protected synchronized boolean lock(Persistent obj, 
					int lockType, int lockAttr) 
    {
	request[Protocol.req_cmd_offs] = Protocol.cmd_lock;
	request[Protocol.req_type_offs] = (byte)lockType;
	Protocol.pack2(request, Protocol.req_attr_offs, lockAttr);
	Protocol.pack4(request, Protocol.req_opid_offs, obj.opid);
	sendReceive(request, Protocol.cmd_lockresult);
	return result[Protocol.req_status_offs] != 0;
    }

    protected synchronized void unlock(Persistent obj, int lockType) { 
	request[Protocol.req_cmd_offs] = Protocol.cmd_unlock;
	request[Protocol.req_type_offs] = (byte)lockType;
	Protocol.pack4(request, Protocol.req_opid_offs, obj.opid);
	write(request);
    }

    protected synchronized void allocate(Persistent obj, int align) { 
	ClassDescriptor desc = obj.desc;
	int cpid = 0;
	if (desc.cid < cpidTableSize) { 
	    cpid = cpidTable[desc.cid];
	}
	if (cpid == 0) { 
	    cpid = registerClass(desc.dbsClass);
	    Assert.that(cpid != 0);
	    associateCpidWithClass(desc, cpid);
	} 
	request[Protocol.req_cmd_offs] = Protocol.cmd_alloc;
	request[Protocol.req_align_offs] = (byte)align;
	Protocol.pack2(request, Protocol.req_cpid_offs, cpid);
	Protocol.pack4(request, Protocol.req_size_offs, 
		       obj.desc.objectSize(obj));
	sendReceive(request, Protocol.cmd_location);
	obj.opid = Protocol.unpack4(result, Protocol.req_opid_offs);
	Assert.that(obj.opid != 0);
        objectCache.put(new ObjectId(id, obj.opid), obj);
    }

    protected synchronized void deallocate(Persistent obj) {
	request[Protocol.req_cmd_offs] = Protocol.cmd_alloc;
	Protocol.pack4(request, Protocol.req_opid_offs, obj.opid);
        write(request);
    }
	
    protected synchronized int registerClass(byte[] desc) { 
	byte[] req = new byte[Protocol.req_size + desc.length];
	System.arraycopy(desc, 0, req, Protocol.req_size, desc.length);
	req[Protocol.req_cmd_offs] = Protocol.cmd_putclass;
	Protocol.pack4(req, Protocol.req_size_offs, desc.length);
	sendReceive(req, Protocol.cmd_classid);
	return Protocol.unpack2(result, Protocol.req_cpid_offs);
    }

    protected synchronized byte[] getClass(int cpid) { 
	request[Protocol.req_cmd_offs] = Protocol.cmd_getclass;
	Protocol.pack2(request, Protocol.req_cpid_offs, cpid);
	return sendReceiveBody(request, Protocol.cmd_classdesc);
    }

    protected synchronized void changeClass(int cpid, byte[] desc) { 
	byte[] req = new byte[Protocol.req_size + desc.length];
	System.arraycopy(desc, 0, req, Protocol.req_size, desc.length);
	req[Protocol.req_cmd_offs] = Protocol.cmd_putclass;
	Protocol.pack2(request, Protocol.req_cpid_offs, cpid);
	Protocol.pack4(req, Protocol.req_size_offs, desc.length);
        write(req);
    }

    // Not synchronized to avoid deadlock
    protected void throwObject(Persistent obj) {
	synchronized (notification) { 
	    notification[Protocol.req_cmd_offs] = Protocol.cmd_throw;
	    Protocol.pack4(notification, Protocol.req_opid_offs, obj.opid);
	    Protocol.pack4(notification, Protocol.req_extra_offs, 0);
	    write(notification);
	}
    }

    // Not synchronized to avoid deadlock
    protected void forgetObject(Persistent obj) {
	synchronized (notification) { 
	    objectCache.remove(new ObjectId(id, obj.opid));
	    notification[Protocol.req_cmd_offs] = Protocol.cmd_forget;
	    Protocol.pack4(notification, Protocol.req_opid_offs, obj.opid);
	    Protocol.pack4(notification, Protocol.req_extra_offs, 0);
	    write(notification);
	}
    }

    protected synchronized int commitTransaction(int nTransServers, 
						 Storage[] servers)
    { 
	int transSize = transBufferOffs;
	if (nTransServers > 1) { 
	    appendTransBuffer(nTransServers*2);
	    for (int i = 0; i < nTransServers; i++) { 
		transSize = Protocol.pack2(transBuffer, transSize, 
					   servers[i].id);
	    }
	}
	transBuffer[Protocol.req_cmd_offs] = Protocol.cmd_transaction;
	transBuffer[Protocol.req_n_servers_offs] = (byte)nTransServers;
	Protocol.pack4(transBuffer, Protocol.req_size_offs, 
		       transSize - Protocol.req_size);
	sendReceive(transBuffer, transSize, Protocol.cmd_transresult);
	if (transSize > maxTransBufferSize) { 
	    transBuffer = null;
	}
	transBufferOffs = 0;
	return result[Protocol.req_status_offs] != 0
	    ? Protocol.unpack4(result, Protocol.req_tid_offs) : -1;
    }
	
    protected synchronized void commitSubtransaction(Storage coordinator, 
						     int nTransServers,
						     Storage[] servers, 
						     int tid) 
    { 
	int transSize = transBufferOffs;
	if (nTransServers > 1) { 
	    appendTransBuffer(nTransServers*2);
	    for (int i = 0; i < nTransServers; i++) { 
		transSize = Protocol.pack2(transBuffer, transSize, 
					   servers[i].id);
	    }
	}
	transBuffer[Protocol.req_cmd_offs] = Protocol.cmd_subtransact;
	transBuffer[Protocol.req_n_servers_offs] = (byte)nTransServers;
	Protocol.pack2(transBuffer, Protocol.req_coordinator_offs, 
		       coordinator.id);
	Protocol.pack4(transBuffer, Protocol.req_size_offs, 
		       transSize - Protocol.req_size);
	Protocol.pack4(transBuffer, Protocol.req_tid_offs, tid);
	write(transBuffer, transSize);
	if (transSize > maxTransBufferSize) { 
	    transBuffer = null;
	}
	transBufferOffs = 0;
    }

    protected synchronized boolean waitTransactionCompletion() {
	sendReceive(null, 0, Protocol.cmd_transresult);
	return result[Protocol.req_status_offs] != 0;
    }

    protected synchronized void open(String host, int port) throws IOException 
    { 
	socket = new Socket(host, port);
	socket.setTcpNoDelay(true);
	socket.setSoLinger(true, LingerTime);

	in = socket.getInputStream();
	out = socket.getOutputStream();

	String hostName = InetAddress.getLocalHost().getHostName();
	long time = System.currentTimeMillis();
	String clientId = hostName + "(" + time/(1000*60*60)%24 + ":" + 
	    time/(1000*60)%60 + ":" + time/1000%60 + "." + time%1000 + ")";
	int nameLen = clientId.length();
	byte[] req = new byte[Protocol.req_size + nameLen + 1];
	req[Protocol.req_cmd_offs] = Protocol.cmd_login;
	Protocol.pack4(req, Protocol.req_size_offs, nameLen + 1);
	Protocol.packAscii(req, Protocol.req_size, clientId);
	write(req);
	read(reply, Protocol.req_size);
	switch (reply[Protocol.req_cmd_offs]) { 
	  case Protocol.cmd_bye:
	    database.disconnected(id);
	  case Protocol.cmd_refused:
	    throw new ConnectionRefusedException();
	  default:
	    Assert.that(reply[Protocol.req_cmd_offs] == Protocol.cmd_ok);
	}

	closing = false;
	receiver = new Thread(this);
	receiver.start();	
    }

    protected synchronized void close() { 
	closing = true;
	request[Protocol.req_cmd_offs] = Protocol.cmd_logout;
	sendReceive(request, Protocol.cmd_bye);
	try { 
	    receiver.join();
	} catch(InterruptedException x) {}

	try { 
	    socket.close();
	} catch (IOException x) { 
	    database.handleException(x);
	}
    }

    protected Storage(Database db, int sid) {
	database = db;
	id = sid;
	classDictionarySize = 1024;
	classDictionary = new ClassDescriptor[classDictionarySize];
	cpidTableSize = 1024;
	cpidTable = new int[cpidTableSize];
	reply = new byte[Protocol.req_size];
	result = new byte[Protocol.req_size];
	request = new byte[Protocol.req_size];
	notification = new byte[Protocol.req_size];
	rcvEvent = new CondEvent();
	repEvent = new CondEvent();
	loadList = new Persistent((Metaobject)null);
	cs = new Object();
    }
}





