//-< IdentityHashtable.java >----------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// Java Program Interface                                            *  /  \  *
//                          Created:     18-Nov-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 18-Nov-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Hashtable based on objects identity (hash code returned by Object.hashCode)
//-------------------------------------------------------------------*--------*

package goodsjpi;

/**
 * IdentityHashtable collision list.
 */
class IdentityHashtableEntry {
    Object key;
    Object value;
    IdentityHashtableEntry next;
}

/**
 * This class implements a IdentityHashtable, which maps key object to some
 * other object.
 * An instance of <code>IdentityHashtable</code> has two parameters that 
 * affect its efficiency: its <i>capacity</i> and its <i>load 
 * factor</i>. The load factor should be between 0.0 and 1.0. When 
 * the number of entries in the IdentityHashtable exceeds the product of the 
 * load factor and the current capacity, the capacity is increased by 
 * calling the <code>rehash</code> method. Larger load factors use 
 * memory more efficiently, at the expense of larger expected time 
 * per lookup. 
 * <p>
 * If many entries are to be made into a <code>IdentityHashtable</code>, 
 * creating it with a sufficiently large capacity may allow the 
 * entries to be inserted more efficiently than letting it perform 
 * automatic rehashing as needed to grow the table. 
 */
public class IdentityHashtable {
    /**
     * The hash table data.
     */
    private transient IdentityHashtableEntry table[];

    /**
     * The total number of entries in the hash table.
     */
    private transient int count;

    /**
     * Rehashes the table when count exceeds this threshold.
     */
    private int threshold;

    /**
     * The load factor for the IdentityHashtable.
     */
    private float loadFactor;

    /**
     * Constructs a new, empty IdentityHashtable with the specified initial 
     * capacity and the specified load factor. 
     *
     * @param      initialCapacity   the initial capacity of the Hashtable.
     * @param      loadFactor        a number between 0.0 and 1.0.
     * @exception  IllegalArgumentException  if the initial capacity is less
     *               than or equal to zero, or if the load factor is less than
     *               or equal to zero.
     */
    public IdentityHashtable(int initialCapacity, float loadFactor) {
	if ((initialCapacity <= 0) || (loadFactor <= 0.0)) {
	    throw new IllegalArgumentException();
	}
	this.loadFactor = loadFactor;
	table = new IdentityHashtableEntry[initialCapacity];
	threshold = (int)(initialCapacity * loadFactor);
    }

    /**
     * Constructs a new, empty IdentityHashtable with the specified initial 
     * capacity and default load factor.
     *
     * @param   initialCapacity   the initial capacity of the IdentityHashtable.
     */
    public IdentityHashtable(int initialCapacity) {
	this(initialCapacity, 0.75f);
    }

    /**
     * Constructs a new, empty IdentityHashtable with a default capacity and load
     * factor. 
     */
    public IdentityHashtable() {
	this(101, 0.75f);
    }

    /**
     * Returns the number of keys in this IdentityHashtable.
     *
     * @return  the number of keys in this IdentityHashtable.
     */
    public int size() {
	return count;
    }

    /**
     * Tests if this IdentityHashtable maps no keys to values.
     *
     * @return  <code>true</code> if this Hashtable maps no keys to values;
     *          <code>false</code> otherwise.
     */
    public boolean isEmpty() {
	return count == 0;
    }

    /**
     * Tests if some key maps into the specified value in this IdentityHashtable.
     * This operation is more expensive than the <code>containsKey</code>
     * method.
     *
     * @param      value   a value to search for.
     * @return     <code>true</code> if some key maps to the
     *             <code>value</code> argument in this IdentityHashtable;
     *             <code>false</code> otherwise.
     */
    public synchronized boolean contains(Object value) {
	IdentityHashtableEntry tab[] = table;
	for (int i = tab.length ; i-- > 0 ;) {
	    for (IdentityHashtableEntry e = tab[i] ; e != null ; e = e.next) {
		if (e.value == value) {
		    return true;
		}
	    }
	}
	return false;
    }

    /**
     * Tests if the specified object is a key in this IdentityHashtable.
     * 
     * @param   key   possible key.
     * @return  <code>true</code> if the specified object is a key in this
     *          IdentityHashtable; <code>false</code> otherwise.
     */
    public synchronized boolean containsKey(Object key) {
	IdentityHashtableEntry tab[] = table;
	int hash = System.identityHashCode(key);
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (IdentityHashtableEntry e = tab[index] ; e != null ; e = e.next) {
	    if (e.key == key) {
		return true;
	    }
	}
	return false;
    }

    /**
     * Returns the value to which the specified key is mapped in this 
     * IdentityHashtable.
     *
     * @param   key   a key in the IdentityHashtable.
     * @return  the value to which the key is mapped in this IdentityHashtable;
     *          <code>null</code> if the key is not mapped to any value in
     *          this IdentityHashtable.
     */
    public synchronized Object get(Object key) {
	IdentityHashtableEntry tab[] = table;
	int hash = System.identityHashCode(key);
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (IdentityHashtableEntry e = tab[index] ; e != null ; e = e.next) {
	    if (e.key == key) {
		return e.value;
	    }
	}
	return null;
    }

    /**
     * Rehashes the contents of the IdentityHashtable into a IdentityHashtable 
     * with a larger capacity. This method is called automatically when the 
     * number of keys in the IdentityHashtable exceeds this IdentityHashtable's 
     * capacity and load factor. 
     *
     */
    protected void rehash() {
	int oldCapacity = table.length;
	IdentityHashtableEntry oldTable[] = table;

	int newCapacity = oldCapacity * 2 + 1;
	IdentityHashtableEntry newTable[] = 
	    new IdentityHashtableEntry[newCapacity];

	threshold = (int)(newCapacity * loadFactor);
	table = newTable;

	for (int i = oldCapacity ; i-- > 0 ;) {
	    for (IdentityHashtableEntry old = oldTable[i] ; old != null ; ) {
		IdentityHashtableEntry e = old;
		old = old.next;

		int index = (System.identityHashCode(e.key) & 0x7FFFFFFF) 
		    % newCapacity;
		e.next = newTable[index];
		newTable[index] = e;
	    }
	}
    }

    /**
     * Maps the specified <code>key</code> to the specified 
     * <code>value</code> in this IdentityHashtable. Neither the key nor the 
     * value can be <code>null</code>. 
     * <p>
     * The value can be retrieved by calling the <code>get</code> method 
     * with a key that is equal to the original key. 
     *
     * @param      key     the IdentityHashtable key.
     * @param      value   the value.
     * @return     the previous value of the specified key in this Hashtable,
     *             or <code>null</code> if it did not have one.
     */
    public synchronized Object put(Object key, Object value) {
	// Makes sure the key is not already in the IdentityHashtable.
	IdentityHashtableEntry tab[] = table;
	int hash = System.identityHashCode(key);
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (IdentityHashtableEntry e = tab[index] ; e != null ; e = e.next) {
	    if (e.key == key) {
		Object old = e.value;
		e.value = value;
		return old;
	    }
	}

	if (count >= threshold) {
	    // Rehash the table if the threshold is exceeded
	    rehash();
	    return put(key, value);
	} 

	// Creates the new entry.
	IdentityHashtableEntry e = new IdentityHashtableEntry();
	e.key = key;
	e.value = value;
	e.next = tab[index];
	tab[index] = e;
	count++;
	return null;
    }

    /**
     * Removes the key (and its corresponding value) from this 
     * IdentityHashtable. This method does nothing if the key is not in the 
     * IdentityHashtable.
     *
     * @param   key   the key that needs to be removed.
     * @return  the value to which the key had been mapped in this Hashtable,
     *          or <code>null</code> if the key did not have a mapping.
     */
    public synchronized Object remove(Object key) {
	IdentityHashtableEntry tab[] = table;
	int hash = System.identityHashCode(key);
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (IdentityHashtableEntry e = tab[index], prev = null; 
	     e != null; 
	     prev = e, e = e.next) 
	{
	    if (e.key == key) {
		if (prev != null) {
		    prev.next = e.next;
		} else {
		    tab[index] = e.next;
		}
		count--;
		return e.value;
	    }
	}
	return null;
    }

    /**
     * Clears this IdentityHashtable so that it contains no keys. 
     */
    public synchronized void clear() {
	IdentityHashtableEntry tab[] = table;
	for (int index = tab.length; --index >= 0; ) { 
	    tab[index] = null;
	}
	count = 0;
    }
}
