//-< FieldDescriptor.java >------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// Java Program Interface                                            *  /  \  *
//                          Created:      1-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 13-Jan-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Class reponsible for packing/unpacking fields of persistent object
//-------------------------------------------------------------------*--------*

package goodsjpi;

import java.lang.reflect.*;

class FieldDescriptor { 
    int     appType;
    int     dbsType;
    int     size;
    String  name;
    Field   field;    
    Class   cls;
    boolean isVarying;

    FieldDescriptor next;
    FieldDescriptor nextRef;
    FieldDescriptor nextStr;

    static final int t_byte         = 0;
    static final int t_short        = 1;
    static final int t_integer      = 2;
    static final int t_long         = 3;
    static final int t_float        = 4;
    static final int t_double       = 5;
    static final int t_boolean      = 6;
    static final int t_char         = 7;
    static final int t_reference    = 8;
    static final int t_string       = 9;

    static final int nullString     = 0xFFFF;
   
    static final Object[] bypassFlag = { new Boolean(true) };
    static Method setBypass;
    static Class persistentClass;
  
    static { 
        try { 
	    persistentClass = Class.forName("goodsjpi.Persistent");
	    Class c = Class.forName("java.lang.reflect.AccessibleObject");
	    Class[] param = { Boolean.TYPE };
	    // setAccessible() method is available only in JDK 1.2
	    setBypass = c.getMethod("setAccessible", param);
        } catch(Exception x) {}
    }      

    FieldDescriptor(Field f, Class cls) { 
	Class fieldClass = f.getType();
	this.name = f.getName();
	this.cls = cls;
	this.field = f;
	if (setBypass != null) { 
	    try { 
		setBypass.invoke(f, bypassFlag);
	    } catch(Exception x) {
		throw new ClassDefinitionError("Field '" + name 
					       + "' is not accessible");
	    }
	}
	if (persistentClass.isAssignableFrom(fieldClass)) { 
	    appType = t_reference;
	    dbsType = Protocol.fld_reference;
	    size = Protocol.ref_size;
	} else { 
	    String c = fieldClass.getName();
	    if (c.equals("byte") || c.equals("[B")) { 
		dbsType = Protocol.fld_signed_integer;
		appType = t_byte;
		size = 1;
	    } else if (c.equals("short") || c.equals("[S")) { 
		dbsType = Protocol.fld_signed_integer;
		appType = t_short;
		size = 2;
	    } else if (c.equals("int") || c.equals("[I")) { 
		dbsType = Protocol.fld_signed_integer;
		appType = t_integer;
		size = 4;
	    } else if (c.equals("long") || c.equals("[J")) { 
		dbsType = Protocol.fld_signed_integer;
		appType = t_long;
		size = 8;
	    } else if (c.equals("float") || c.equals("[F")) { 
		dbsType = Protocol.fld_real;
		appType = t_float;
		size = 4;
	    } else if (c.equals("double") || c.equals("[D")) { 
		dbsType = Protocol.fld_real;
		appType = t_double;
		size = 8;
	    } else if (c.equals("boolean") || c.equals("[Z")) { 
		dbsType = Protocol.fld_unsigned_integer;
		appType = t_boolean;
		size = 1;
	    } else if (c.equals("char") || c.equals("[C")) { 
		dbsType = Protocol.fld_unsigned_integer;
		appType = t_char;
		size = 2;
	    } else if (c.equals("java.lang.String")) { 
		dbsType = Protocol.fld_string;
		appType = t_string;
		size = 2;
	    } else { 
		Class componentType = fieldClass.getComponentType();
		if (componentType != null 
		    && persistentClass.isAssignableFrom(componentType))
		 {
		     dbsType = Protocol.fld_reference;
		     appType = t_reference;
		     size = Protocol.ref_size;
		 } else { 
		     throw new ClassDefinitionError("Type of non-transient field '" 
			 + name + "'should be derived from Persistent class");
		 }
	    }
	    isVarying = fieldClass.isArray();
	}
    }

    FieldDescriptor(String name, Class cls, int dbsType, int size) 
    {
	this.dbsType = dbsType;
	this.size = size;
	this.name = name;
	this.cls = cls;
	try { 
	    field = cls.getDeclaredField(name);
	} catch (Exception x) {}
	
	if (field == null) { 
	    return;
	}
	if (setBypass != null) { 
	    try { 
		setBypass.invoke(field, bypassFlag);
	    } catch(Exception x) {
		throw new ClassDefinitionError("Field '" + name 
					       + "' is not accessible");
	    }
	}
	if ((field.getModifiers() & (Modifier.TRANSIENT|Modifier.STATIC)) != 0)
	{
	    throw new ClassDefinitionError("Instance field of persistent object can not become transient or static");
	}
	Class fieldClass = field.getType();
	if (persistentClass.isAssignableFrom(fieldClass)) { 
	    if (dbsType != Protocol.fld_reference) { 
		throw new ClassDefinitionError("Field + '" + name
		    + "' can not be converted to reference type");
	    } 
	    appType = t_reference;
	} else { 
	    String c = fieldClass.getName();
	    if (c.equals("byte") || c.equals("[B")) { 
		appType = t_byte;
	    } else if (c.equals("short") || c.equals("[S")) { 
		appType = t_short;
	    } else if (c.equals("int") || c.equals("[I")) { 
		appType = t_integer;
	    } else if (c.equals("long") || c.equals("[J")) { 
		appType = t_long;
	    } else if (c.equals("float") || c.equals("[F")) { 
		appType = t_float;
	    } else if (c.equals("double") || c.equals("[D")) { 
		appType = t_double;
	    } else if (c.equals("boolean") || c.equals("[Z")) { 
		appType = t_boolean;
	    } else if (c.equals("char") || c.equals("[C")) { 
		appType = t_char;
	    } else if (c.equals("java.lang.String")) { 
		if (dbsType != Protocol.fld_string) { 
		    throw new ClassDefinitionError("Field + '" + name
			 + "' can not be converted to dtring type");
		} 
		appType = t_string;
	    } else { 
		Class componentType = fieldClass.getComponentType();
		if (componentType != null 
		    && persistentClass.isAssignableFrom(componentType))
		 {
		     if (dbsType != Protocol.fld_reference) { 
			 throw new ClassDefinitionError("Field + '" + name + 
			     "' can not be converted to array of reference type");
		     } 
		     appType = t_reference;
		 } else { 
		     throw new ClassDefinitionError("Type of non-transient field '" 
			 + name + "' should be derived from Persistent class");
		 }
	    }
	    isVarying = fieldClass.isArray();
	}
	if (appType != t_reference && dbsType == Protocol.fld_reference) { 
	    throw new ClassDefinitionError("Field + '" + name 
		+ "' can not be converted from reference type");
	} else if (appType != t_string && dbsType == Protocol.fld_string) { 
	    throw new ClassDefinitionError("Field + '" + name 
		+ "' can not be converted from string type");
	}
    }

    FieldDescriptor(FieldDescriptor copy) { // copy constructor
	appType = copy.appType;
	dbsType = copy.dbsType;
	size = copy.size;
	name = copy.name;
	field = copy.field;    
	cls = copy.cls;
	isVarying = copy.isVarying;
    }
	

    final int length(Persistent obj) { 
	if (field != null) {
	    try { 
		Object arr = field.get(obj);
		return arr != null ? Array.getLength(arr) : 0;
	    } catch (Exception x) {
		throw new Error(x.getMessage());
	    }
	}
	return 0;
    }

    final int pack(Persistent obj, byte[] data, int offs) 
    throws IllegalAccessException 
    {
	String s;
        switch (appType) { 
	  case t_byte:
	    data[offs] = field.getByte(obj);
	    break;
	  case t_short:
	    Protocol.pack2(data, offs, field.getShort(obj));
	    break;
	  case t_integer:
	    Protocol.pack4(data, offs, field.getInt(obj));
	    break;
	  case t_long:
	    Protocol.pack8(data, offs, field.getLong(obj));
	    break;
	  case t_float:
	    Protocol.pack4(data, offs, 
			   Float.floatToIntBits(field.getFloat(obj)));
	    break;
	  case t_double:
	    Protocol.pack8(data, offs, 
			   Double.doubleToLongBits(field.getDouble(obj)));
	    break;
	  case t_char:
	    Protocol.pack2(data, offs, field.getChar(obj));
	    break;
	  case t_boolean:
	    data[offs] = (byte)(field.getBoolean(obj) ? 1 : 0);
	    break;
	  case t_string:
	    s = (String)field.get(obj);
	    if (s == null) { 
		return Protocol.pack2(data, offs, nullString);
	    } else { 
		int len = s.length();
		offs = Protocol.pack2(data, offs, len);
		for (int i = 0; i < len; i++) { 
		    offs = Protocol.pack2(data, offs, s.charAt(i));
		}
		return offs;
	    }
	}
	return offs + size;
    }
    
    final int packRef(Persistent obj, byte[] data, int offs) 
    throws IllegalAccessException
    {
        Persistent p = (Persistent)field.get(obj);
	if (p != null) { 
	    if (p.opid == 0) { 
		p.metaobject.makePersistent(p, obj);
	    } else { 
		Assert.that(obj.storage.database == p.storage.database);
	    } 
	    Protocol.pack2(data, offs+Protocol.ref_sid_offs, 
			   p.storage.id);
	    Protocol.pack4(data, offs+Protocol.ref_opid_offs, p.opid);
	} else { 
	    for (int i = 0; i < Protocol.ref_size; i++) data[offs+i] = 0;
	}
	return offs + Protocol.ref_size;
    }

    final int packVarying(Persistent obj, byte[] data, int offs) 
    throws IllegalAccessException	
    { 

	int i;
	switch (appType) { 
	  case t_byte:
	    { 
		byte[] arr = (byte[])field.get(obj);
		System.arraycopy(arr, 0, data, offs, arr.length);
		offs += arr.length;
		break;
	    }
	  case t_short:
	    { 
		short[] arr = (short[])field.get(obj);
		for (i = 0; i < arr.length; i++) { 
		    offs = Protocol.pack2(data, offs, arr[i]);
		}
		break;
	    }
	  case t_integer:
	    { 
		int[] arr = (int[])field.get(obj);
		for (i = 0; i < arr.length; i++) { 
		    offs = Protocol.pack4(data, offs, arr[i]);
		}
		break;
	    }
	  case t_long:
	    { 
		long[] arr = (long[])field.get(obj);
		for (i = 0; i < arr.length; i++) { 
		    offs = Protocol.pack8(data, offs, arr[i]);
		}
		break;
	    }
	  case t_float:
	    { 
		float[] arr = (float[])field.get(obj);
		for (i = 0; i < arr.length; i++) { 
		    offs = Protocol.pack4(data, offs, 
					  Float.floatToIntBits(arr[i]));
		}
		break;
	    }
	  case t_double:
	    { 
		double[] arr = (double[])field.get(obj);
		for (i = 0; i < arr.length; i++) { 
		    offs = Protocol.pack8(data, offs, 
					  Double.doubleToLongBits(arr[i]));
		}
		break;
	    }
	  case t_char:
	    {  
		char[] arr = (char[])field.get(obj);
		for (i = 0; i < arr.length; i++) { 
		    offs = Protocol.pack2(data, offs, arr[i]);
		}
		break;
	    }
	  case t_boolean:
	    { 
		boolean[] arr = (boolean[])field.get(obj);
		for (i = 0; i < arr.length; i++) { 
		    data[offs++] = (byte)(arr[i] ? 1 : 0);
		}
		break;
	    }
	}
	return offs;
    }

    final int packObjectArray(Persistent obj, byte data[], int offs) 
    throws IllegalAccessException
    { 
	Persistent[] arr = null;
	try { 
	    arr = (Persistent[])field.get(obj);
	} catch (Exception x) { 
	    throw new Error(x.getMessage());
	}
	for (int i = 0; i < arr.length; i++) { 
	    Persistent p = arr[i];
	    if (p != null) { 
		if (p.opid == 0) { 
		    p.metaobject.makePersistent(p, obj);
		} else { 
		    Assert.that(obj.storage.database == p.storage.database);
		}
		Protocol.pack2(data, offs+Protocol.ref_sid_offs, p.storage.id);
		Protocol.pack4(data, offs+Protocol.ref_opid_offs, p.opid);
		offs += Protocol.ref_size;
	    } else { 
		for (int j = Protocol.ref_size; --j >= 0;) { 
		    data[offs++] = (byte)0;
		}
	    }
	}
	return offs;
    }
	
    final long unpackSignedInteger(byte[] data, int offs) 
    throws IllegalAccessException
    { 
	switch (size) { 
	  case 1:
	    return data[offs];
	  case 2:
	    return Protocol.unpack2(data, offs);
	  case 4:
	    return Protocol.unpack4(data, offs);
	  default:
	    return Protocol.unpack8(data, offs);
	}
    }

    final long unpackUnsignedInteger(byte[] data, int offs)
    throws IllegalAccessException
    { 
	switch (size) { 
	  case 1:
	    return data[offs] & 0xFF;
	  case 2:
	    return Protocol.unpack2(data, offs) & 0xFFFF;
	  case 4:
	    return (long)Protocol.unpack4(data, offs) & 0xFFFFFFFFL;
	  default:
	    return Protocol.unpack8(data, offs);
	}
    }

    final double unpackReal(byte[] data, int offs) {
	if (size == 4) { 
	    return Float.intBitsToFloat(Protocol.unpack4(data, offs));
	} else { 
	    return Double.longBitsToDouble(Protocol.unpack8(data, offs));
	}
    }	    
    
    final void storeInteger(Persistent obj, long val) 
    throws IllegalAccessException
    { 
	switch (appType) { 
	  case t_byte:
	    field.setByte(obj, (byte)val);
	    break;
	  case t_short:
	    field.setShort(obj, (short)val);
	    break;
	  case t_integer:
	    field.setInt(obj, (int)val);
	    break;
	  case t_long:
	    field.setLong(obj, val);
	    break;
	  case t_float:
	    field.setFloat(obj, (float)val);
	    break;
	  case t_double:
	    field.setDouble(obj, (double)val);
	    break;
	  case t_char:
	    field.setChar(obj, (char)val);
	    break;
	  case t_boolean:
	    field.setBoolean(obj, val != 0);
	    break;
	}
    }
		    
    final void storeReal(Persistent obj, double val) 
    throws IllegalAccessException
    { 
	switch (appType) { 
	  case t_byte:
	    field.setByte(obj, (byte)val);
	    break;
	  case t_short:
	    field.setShort(obj, (short)val);
	    break;
	  case t_integer:
	    field.setInt(obj, (int)val);
	    break;
	  case t_long:
	    field.setLong(obj, (long)val);
	    break;
	  case t_float:
	    field.setFloat(obj, (float)val);
	    break;
	  case t_double:
	    field.setDouble(obj, val);
	    break;
	  case t_char:
	    field.setChar(obj, (char)val);
	    break;
	  case t_boolean:
	    field.setBoolean(obj, val != 0.0);
	    break;
	}
    }

    final int storeString(Persistent obj, byte[] data, int offs)
    throws IllegalAccessException
    { 
	int len = Protocol.unpack2(data, offs) & 0xFFFF;
	offs += 2;
	if (len != nullString) { 
	    char[] arr = new char[len];
	    for (int i = 0; i < len; i++) { 
		arr[i] = (char)Protocol.unpack2(data, offs);
		offs += 2;
	    }
	    if (field != null) { 
		field.set(obj, new String(arr));
	    }
	} else if (field != null) { 
	    field.set(obj, null);
	}
	return offs;
    }


    final int unpack(byte[] data, int offs, Persistent obj) 
    throws IllegalAccessException
    { 
	switch (dbsType) { 
	  case Protocol.fld_signed_integer:
	    if (field != null) { 
		storeInteger(obj, unpackSignedInteger(data, offs));
	    }
	    break;
	  case Protocol.fld_unsigned_integer:
	    if (field != null) { 
		storeInteger(obj, unpackUnsignedInteger(data, offs));
	    }
	    break;
	  case Protocol.fld_real:
	    if (field != null) { 
		storeReal(obj, unpackReal(data, offs));
	    }
	    break;
	  case Protocol.fld_string:
	    return storeString(obj, data, offs);
	}
	return offs + size;
    }

    final int unpackVarying(byte[] data, int offs, Persistent obj, 
			    int varyingLength) throws IllegalAccessException
    { 
	int i;

	if (field == null) {
	    return offs + size*varyingLength;
	}
	switch (appType) { 
	  case t_byte:
	    { 
		byte[] arr = new byte[varyingLength];
		if (size == 1) { 
		    System.arraycopy(data, offs, arr, 0, varyingLength);
		    offs += varyingLength;
		} else { 
		    switch (dbsType) { 
		      case Protocol.fld_signed_integer:
			for (i = 0; i < varyingLength; i++) { 
			    arr[i] = (byte)unpackSignedInteger(data, offs);
			    offs += size;
			}
			break;
		      case Protocol.fld_unsigned_integer:
			for (i = 0; i < varyingLength; i++) { 
			    arr[i] = (byte)unpackUnsignedInteger(data, offs);
			    offs += size;
			}
			break;
		      case Protocol.fld_real:
			for (i = 0; i < varyingLength; i++) { 
			    arr[i] = (byte)unpackReal(data, offs);
			    offs += size;
			}
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	  case t_short:
	    { 
		short[] arr = new short[varyingLength];
		switch (dbsType) { 
		  case Protocol.fld_signed_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (short)unpackSignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_unsigned_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (short)unpackUnsignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_real:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (short)unpackReal(data, offs);
			offs += size;
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	  case t_integer:
	    { 
		int[] arr = new int[varyingLength];
		switch (dbsType) { 
		  case Protocol.fld_signed_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (int)unpackSignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_unsigned_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (int)unpackUnsignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_real:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (int)unpackReal(data, offs);
			offs += size;
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	  case t_long:
	    { 
		long[] arr = new long[varyingLength];
		switch (dbsType) { 
		  case Protocol.fld_signed_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = unpackSignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_unsigned_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = unpackUnsignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_real:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (long)unpackReal(data, offs);
			offs += size;
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	  case t_float:
	    { 
		float[] arr = new float[varyingLength];
		switch (dbsType) { 
		  case Protocol.fld_signed_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (float)unpackSignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_unsigned_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (float)unpackUnsignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_real:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (float)unpackReal(data, offs);
			offs += size;
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	  case t_double:
	    { 
		double[] arr = new double[varyingLength];
		switch (dbsType) { 
		  case Protocol.fld_signed_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (double)unpackSignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_unsigned_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (double)unpackUnsignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_real:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = unpackReal(data, offs);
			offs += size;
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	  case t_char:
	    { 
		char[] arr = new char[varyingLength];
		switch (dbsType) { 
		  case Protocol.fld_signed_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (char)unpackSignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_unsigned_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (char)unpackUnsignedInteger(data, offs);
			offs += size;
		    }
		    break;
		  case Protocol.fld_real:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = (char)unpackReal(data, offs);
			offs += size;
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	  case t_boolean:
	    { 
		boolean[] arr = new boolean[varyingLength];
		switch (dbsType) { 
		  case Protocol.fld_signed_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = unpackSignedInteger(data, offs) != 0;
			offs += size;
		    }
		    break;
		  case Protocol.fld_unsigned_integer:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = unpackUnsignedInteger(data, offs) != 0;
			offs += size;
		    }
		    break;
		  case Protocol.fld_real:
		    for (i = 0; i < varyingLength; i++) { 
			arr[i] = unpackReal(data, offs) != 0.0;
			offs += size;
		    }
		}
		field.set(obj, arr);
	    }
	    break;
	}
	return offs;
    }
}

