//-< Database.java >-------------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// Java Program Interface                                            *  /  \  *
//                          Created:      1-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 16-Nov-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Class representing database for client
//-------------------------------------------------------------------*--------*

package goodsjpi;

import java.io.*;

public class Database {
    protected Storage[] storages;
    protected IdentityHashtable notifications;

    public Database() { 
	notifications = new IdentityHashtable();
    }

    protected CondEvent onTransactionAbortEvent;

    public boolean open(String cfgFileName) { 
	try { 
	    FileInputStream in = new FileInputStream(cfgFileName);
	    StreamTokenizer scanner = new StreamTokenizer(in);	    
	    scanner.wordChars('0','9');
	    scanner.wordChars('_','_');
	    scanner.whitespaceChars(':',':');
	
	    if (scanner.nextToken() != StreamTokenizer.TT_NUMBER) { 
		in.close();
		handleError("Bad configuration file format");
		return false;
	    }
	    int nStorages = (int)scanner.nval;
	    storages = new Storage[nStorages];
	    while (--nStorages >= 0) { 
		if (scanner.nextToken() != StreamTokenizer.TT_NUMBER) { 
		    in.close();
		    handleError("Bad configuration file format");
		    return false;
		}
		int sid = (int)scanner.nval;
		if (sid < storages.length) { 
		    if (storages[sid] != null) { 
			in.close();
			handleError("Duplicated entry in configuration file");
			return false;
		    }
		    if (scanner.nextToken() != StreamTokenizer.TT_WORD) { 
			in.close();
			handleError("Host name not specified for storage " 
				    + sid);
			return false;
		    }
		    String hostname = scanner.sval;
		    if (scanner.nextToken() != StreamTokenizer.TT_NUMBER) { 
			in.close();
			handleError("Port not specified for storage " + sid);
			return false;
		    }
		    int port = (int)scanner.nval;
		    storages[sid] = createStorage(this, sid);
		    storages[sid].open(hostname, port);
		}
	    }
	    in.close();
	    return true;
	} catch (IOException x) { 
	    handleException(x);
	    return false;
	}

    }

    public void close() { 
	for (int i = storages.length; --i >= 0;) { 
	    storages[i].close();
	}
    }

    public int getNumberOfStorages() { 
	return storages.length; 
    }

    public Object getRoot() { 
	return storages[0].getRoot();
    }

    public Object getRoot(int sid) { 
	return storages[sid].getRoot();
    }

    public void setRoot(Object root) { 
	storages[0].setRoot((Persistent)root);
    }

    public void setRoot(int sid, Object root) { 
	storages[sid].setRoot((Persistent)root);
    }

    public void attach(Object obj, int sid) { 
	Persistent p = (Persistent)obj;
	Assert.that("persistent object can't change its location", p.opid == 0);
	p.storage = storages[sid];
    }

    public synchronized void notifyOnModification(Object obj, CondEvent event) 
    {
	Persistent p = (Persistent)obj;
	if (event == null) { 
	    p.metaobject.notifyOnModification(p, false);
	    notifications.remove(p);
	} else { 
	    p.metaobject.notifyOnModification(p, true);
	    notifications.put(p, event);
	}	
    }


    public synchronized void notifyOnTransactionAbort(CondEvent event) { 
	onTransactionAbortEvent = event;
    }


    protected void handleError(String text) { 
	System.err.println(text);
	throw new Error("Application terminated due to critical error");
    }

    protected void handleException(IOException x) { 
	x.printStackTrace();
	throw new Error("Application terminated due to critical error");
    }

    protected Storage createStorage(Database db, int id) { 
	return new Storage(db, id);
    }

    protected void disconnected(int sid) { 
	throw new SessionDisconnectedError("Server " + sid + 
					   " disconnect client session");
    }

    void abortTransaction() { 
	for (int i = storages.length; --i >= 0;) { 
	    storages[i].abortTransaction();
	}
	if (onTransactionAbortEvent != null) { 
	    onTransactionAbortEvent.signal();
	}
    }
}


