//-< ClassDescriptor.java >------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// Java Program Interface                                            *  /  \  *
//                          Created:      1-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 13-Jan-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Class reponsible for packing/unpacking persistent objects
//-------------------------------------------------------------------*--------*

package goodsjpi;

import java.lang.reflect.*;
import java.util.Hashtable;

		 
class ClassDescriptor {
    //
    // GOODS specific ClassDescriptor fields
    //

    static Hashtable classDictionary = new Hashtable();

    String name;
    int fixedSize;
    int varyingSize;
    int nFixedReferences;
    int nVaryingReferences;
    int nFields;
    int totalNamesSize;
    int cid;

    static int classCount;

    ClassDescriptor newClass;

    Class  appClass;
    byte[] dbsClass;

    boolean conversionNeeded; 

    ClassDescriptor superclass;
    
    FieldDescriptor firstField;
    FieldDescriptor lastField;
    FieldDescriptor refFields;
    FieldDescriptor varyingField;
    FieldDescriptor strFields;

    Constructor     constructor;
    Object[]        constructorParameters;
    Metaobject      defaultMetaobject;


    static Class metaobjectClass;
    static final Class[]  constructorProfile = new Class[1];
  
    static { 
        try { 
	    metaobjectClass = Class.forName("goodsjpi.Metaobject");
	    constructorProfile[0] = metaobjectClass;
        } catch(Exception x) {}
    }      

    static Class getClass(byte[] desc) { 
	int nFields = Protocol.unpack4(desc, Protocol.cds_n_fields_offs);
	String name = Protocol.unpackAscii(desc, Protocol.cds_header_size +
					   nFields*Protocol.fds_size);
	try { 
	    return Class.forName(name);
	} catch(ClassNotFoundException x) { 
	    throw new NoClassDefFoundError("Class " + name + 
					   " not found in application");
	}
    }

    final void initialize(Class cls) { 
	appClass = cls;
	name = cls.getName();
	try { 
	    constructor = appClass.getDeclaredConstructor(constructorProfile);
	    if (FieldDescriptor.setBypass != null) { 
		FieldDescriptor.setBypass.invoke(constructor, 
						 FieldDescriptor.bypassFlag);
	    }
	} catch (Exception x) {
	    throw new ClassDefinitionError("Failed to locate constructor " 
					   + name + "(Metaobject)");
	}
	constructorParameters = new Object[1];
	defaultMetaobject = Persistent.defaultMetaobject;
    }


    ClassDescriptor(ClassDescriptor newClass, byte[] desc) {
	initialize(newClass.appClass);
	this.newClass = newClass;
	fixedSize = Protocol.unpack4(desc, Protocol.cds_fixed_size_offs);
	varyingSize = Protocol.unpack4(desc, Protocol.cds_varying_size_offs);
	nFixedReferences = Protocol.unpack4(desc, 
					    Protocol.cds_n_fixed_refs_offs);
	nVaryingReferences = Protocol.unpack4(desc, 
					     Protocol.cds_n_varying_refs_offs);
	nFields = Protocol.unpack4(desc, Protocol.cds_n_fields_offs);
	totalNamesSize = Protocol.unpack4(desc, 
					  Protocol.cds_total_names_size_offs);
	dbsClass = desc;
	conversionNeeded = true;
	buildFieldsList(desc, Protocol.cds_header_size);
	constructorParameters[0] = defaultMetaobject;
    }

	    
    final int buildFieldsList(byte[] desc, int offs)
    {
        int dbsType, nameOffs, size, nItems, next;
	String fieldName;
	int n = 0;

	do { 
	    dbsType = Protocol.unpack2(desc, offs+Protocol.fds_type_offs);
	    nameOffs = Protocol.unpack2(desc, offs+Protocol.fds_name_offs)
                     + Protocol.cds_header_size;
	    size = Protocol.unpack4(desc, offs+Protocol.fds_size_offs);
	    nItems = Protocol.unpack4(desc, offs+Protocol.fds_nitems_offs);
	    next = Protocol.unpack4(desc, offs+Protocol.fds_next_offs);
	    fieldName = Protocol.unpackAscii(desc, nameOffs);

	    offs += Protocol.fds_size;

	    if (dbsType == Protocol.fld_structure) { 
		superclass = new ClassDescriptor(fieldName, desc, offs);
		n += superclass.nFields;
		offs += superclass.nFields*Protocol.fds_size;
		Assert.that(firstField == null);
		varyingField = superclass.varyingField;
		firstField = superclass.firstField;
		lastField = superclass.lastField;
		refFields = superclass.refFields;
		defaultMetaobject = superclass.defaultMetaobject;
	    } else { 
		FieldDescriptor fd = new FieldDescriptor(fieldName, appClass, 
							 dbsType, size);
		if (nItems == 0) { 
		    if (!fd.isVarying) { 
			throw new ClassDefinitionError("Field '" + fieldName +  
			   "' can not be converted from varying array type");
		    } 
		    if (varyingField != null) { 
			throw new ClassDefinitionError("More than one varying components in persistent capable class");
		    }
		    varyingField = fd;
		} else if (nItems != 1) { 
		    throw new ClassDefinitionError("Field '"  + fieldName + 
		        "' can not be converted from fixed array type");
		} else if (fd.isVarying) { 
		    throw new ClassDefinitionError("Field '" + fieldName +  
			"' can not be converted to varying array type");
		}
		if (firstField == null) { 
		    firstField = fd;
		} else { 
		    lastField.next = fd;
		}
		lastField = fd;
		if (dbsType == Protocol.fld_reference) { 
		    fd.nextRef = refFields;
		    refFields = fd;
		} else if (dbsType == Protocol.fld_string) { 
		    fd.nextStr = strFields;
		    strFields = fd;
		}   
	    }
	    n += 1;
	} while (next != 0);	    
	
	Field[] classFields = appClass.getDeclaredFields();
	for (int i = 0; i < classFields.length; i++) {
	    Field f = classFields[i];
	    if ((f.getModifiers() & Modifier.STATIC) != 0 &&
		f.getType() == metaobjectClass)
	    { 
		if ((f.getModifiers() & Modifier.FINAL) == 0) { 
		    throw new ClassDefinitionError("Default metaobject for persistent capable class should be specified by final static field");
		}
		try { 
		    defaultMetaobject = (Metaobject)f.get(null);
		} catch(Exception x) {}
		break;
	    } 
	}
	return n;
    }


    ClassDescriptor(String className, byte[] desc, int offs) {
	try { 
	    appClass = Class.forName(className);
	} catch(ClassNotFoundException x) { 
	    throw new NoClassDefFoundError("Class " + name + 
					   " not found in application");
	}
	name = className;
	nFields = buildFieldsList(desc, offs);
	defaultMetaobject = Persistent.defaultMetaobject;
    }


    static synchronized ClassDescriptor lookup(Class cls) { 
	ClassDescriptor desc = (ClassDescriptor)classDictionary.get(cls);
	return desc != null ? desc : new ClassDescriptor(cls);
    }

    boolean equal(byte[] desc) { 
	if (desc.length != dbsClass.length) { 
	    return false;
	}
	for (int i = desc.length; --i >= 0;) { 
	    if (desc[i] != dbsClass[i]) { 
		return false;
	    }
	}
	return true;
    }

    ClassDescriptor(Class cls) { 
	int i;
	FieldDescriptor fd;

	initialize(cls);
	cid = classCount++;
	newClass = this;

	Field[] classFields = cls.getDeclaredFields();
	Class base = cls.getSuperclass();
	if (base != FieldDescriptor.persistentClass) { 
	    superclass = lookup(base);
	    nFields = 1 + superclass.nFields;
	    fixedSize = superclass.fixedSize;
	    varyingSize = superclass.varyingSize;
	    nFixedReferences = superclass.nFixedReferences;
	    nVaryingReferences = superclass.nVaryingReferences;
	    totalNamesSize = superclass.totalNamesSize;
	    defaultMetaobject = superclass.defaultMetaobject;

	    for (fd = superclass.firstField; fd != null; fd = fd.next) { 
		FieldDescriptor copy = new FieldDescriptor(fd);
		if (lastField == null) { 
		    firstField = copy; 
		} else { 
		    lastField.next = copy;
		}
		lastField = copy;
		if (fd == superclass.varyingField) { 
		    varyingField = copy;
		}
		if (fd.dbsType == Protocol.fld_reference) { 
		    copy.nextRef = refFields;
		    refFields = copy;
		} else if (fd.dbsType == Protocol.fld_string) {
		    copy.nextStr = strFields;
		    strFields = copy;
		}
	    }		 
	}
	totalNamesSize += name.length() + 1;

	for (i = 0; i < classFields.length; i++) { 
	    Field f = classFields[i];

	    if ((f.getModifiers() & (Modifier.TRANSIENT|Modifier.STATIC)) != 0)
	    { 
		if (f.getType() == metaobjectClass) { 
		    if ((f.getModifiers() & Modifier.FINAL) == 0) { 
			throw new ClassDefinitionError("Default metaobject for persistent capable class should be specified by final static field");
		    }
		    try { 
			defaultMetaobject = (Metaobject)f.get(null);
		    } catch(Exception x) {}
		} 
		continue;
	    }
	    
	    fd = new FieldDescriptor(f, cls);
	    if (fd.dbsType == Protocol.fld_reference) { 
		if (fd.isVarying) {
		    nVaryingReferences = 1;
		} else { 
		    nFixedReferences += 1;
		} 
		fd.nextRef = refFields;
		refFields = fd;
	    } else if (fd.dbsType == Protocol.fld_string) { 
		fd.nextStr = strFields;
		strFields = fd;
	    }
	    if (fd.isVarying) { 
		if (varyingField != null) { 
		    throw new ClassDefinitionError("More than one varying components in persistent capable class");
		}
		varyingField = fd;
		varyingSize = fd.size;
	    } else { 	
		fixedSize += fd.size;
	    }
	    if (firstField == null) { 
		firstField = fd;
	    } else { 
		lastField.next = fd;
	    }
	    lastField = fd;
	    nFields += 1;
	    totalNamesSize += fd.name.length()+1;
	}
	if (varyingField != null && strFields != null) { 
	    throw new ClassDefinitionError("Persistent class with String componnents should not have array component");
	}
	constructorParameters[0] = defaultMetaobject;

        byte[] desc = new byte[totalNamesSize + Protocol.cds_header_size
			      + nFields * Protocol.fds_size];
	Protocol.pack4(desc, Protocol.cds_fixed_size_offs, fixedSize);
	Protocol.pack4(desc, Protocol.cds_varying_size_offs, varyingSize);
	Protocol.pack4(desc, Protocol.cds_n_fixed_refs_offs, nFixedReferences);
	Protocol.pack4(desc, Protocol.cds_n_varying_refs_offs, 
		       nVaryingReferences);
	Protocol.pack4(desc, Protocol.cds_n_fields_offs, nFields);
	Protocol.pack4(desc, Protocol.cds_total_names_size_offs, 
		       totalNamesSize);
	int nameOffs = Protocol.cds_header_size + nFields*Protocol.fds_size;
	int offs = Protocol.cds_header_size;
	int index = 0;
	int next = nFields;
	nameOffs = Protocol.packAscii(desc, nameOffs, name);

	for (ClassDescriptor sup = superclass; sup != null; sup=sup.superclass)
        { 
	    index += 1;
	    Protocol.pack2(desc, offs+Protocol.fds_type_offs, 
			   Protocol.fld_structure);
	    Protocol.pack2(desc, offs+Protocol.fds_name_offs, 
			   nameOffs - Protocol.cds_header_size);
	    Protocol.pack4(desc, offs+Protocol.fds_size_offs, sup.fixedSize);
	    int succ = index + sup.nFields;
	    Protocol.pack4(desc, offs+Protocol.fds_next_offs, 
			   succ == next ? 0 : succ);
	    Protocol.pack4(desc, offs+Protocol.fds_nitems_offs, 1);
	    nameOffs = Protocol.packAscii(desc, nameOffs, sup.name);
	    next = succ;
	    offs += Protocol.fds_size;
	}
	for (fd = firstField; fd != null; fd = fd.next) { 
	    index += 1;
	    Protocol.pack2(desc, offs+Protocol.fds_type_offs, fd.dbsType);
	    Protocol.pack2(desc, offs+Protocol.fds_name_offs, 
			   nameOffs - Protocol.cds_header_size);
	    Protocol.pack4(desc, offs+Protocol.fds_size_offs, fd.size);
	    Protocol.pack4(desc, offs+Protocol.fds_nitems_offs, 
			   fd == varyingField ? 0 : 1);
	    Protocol.pack4(desc, offs+Protocol.fds_next_offs, 
			   (fd.next == null || fd.next.cls != fd.cls) 
			   ? 0 : index);
	    nameOffs = Protocol.packAscii(desc, nameOffs, fd.name);
	    offs += Protocol.fds_size;
	}
	dbsClass = desc;
	classDictionary.put(cls, this); 
	conversionNeeded = false;
    }
    
    final Persistent construct(Storage storage, int opid, int state) { 
        try { 
	    Persistent obj = 
		(Persistent)constructor.newInstance(constructorParameters);
	    obj.desc = this;
	    obj.opid = opid;
	    obj.state = state;
	    obj.storage = storage;
	    Storage.objectCache.put(new ObjectId(storage.id, opid), obj);
	    return obj;
	} catch(Exception x) { 
	    throw new Error("Failed to create instance of class " + name);
	}
    }
    

    final void unpack(Persistent obj, Persistent refs[], int varyingLength) 
    {
	FieldDescriptor fd;
	int i, n = 0, offs = Protocol.hdr_size;
	byte[] data = obj.data;
	obj.data = null;

	if (refs != null) { 
	    offs += refs.length*Protocol.ref_size;
	}
    
	try {
	    if (!conversionNeeded) { 
		for (fd = firstField; fd != null; fd = fd.next) { 
		    if (fd == varyingField) {
			switch (fd.appType) { 
			  case FieldDescriptor.t_byte:
			    { 
			      byte[] arr = new byte[varyingLength];
			      System.arraycopy(data, offs, arr, 0,
					       varyingLength);
			      offs += varyingLength;
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_short:
			    { 
			      short[] arr = new short[varyingLength];
			      for (i = 0; i < varyingLength; i++) { 
				  arr[i]=(short)Protocol.unpack2(data, offs);
				  offs += 2;
			      }
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_integer:
			    { 
			      int[] arr = new int[varyingLength];
			      for (i = 0; i < varyingLength; i++) { 
				  arr[i] = Protocol.unpack4(data, offs);
				  offs += 4;
			      }
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_long:
			    { 
			      long[] arr = new long[varyingLength];
			      for (i = 0; i < varyingLength; i++) { 
				  arr[i] = Protocol.unpack8(data, offs);
				  offs += 8;
			      }
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_float:
			    { 
			      float[] arr = new float[varyingLength];
			      for (i = 0; i < varyingLength; i++) { 
				  arr[i] = 
				      Float.intBitsToFloat(Protocol.unpack4
							   (data, offs));
				  offs += 4;
			      }
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_double:
			    { 
			      double[] arr = new double[varyingLength];
			      for (i = 0; i < varyingLength; i++) { 
				  arr[i] = 
				      Double.longBitsToDouble(Protocol.unpack8
							      (data, offs));
				  offs += 8;
			      }
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_char:
			    { 
			      char[] arr = new char[varyingLength];
			      for (i = 0; i < varyingLength; i++) { 
				  arr[i] = (char)Protocol.unpack2(data, offs);
				  offs += 2;
			      }
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_boolean:
			    { 
			      boolean[] arr = new boolean[varyingLength];
			      for (i = 0; i < varyingLength; i++) { 
				  arr[i] = data[offs++] != 0;
			      }
			      fd.field.set(obj, arr);
			      break;
			    }
			  case FieldDescriptor.t_reference:
			    { 
			      Object arr = Array.newInstance
				  (fd.field.getType().getComponentType(), 
				   varyingLength);
			      System.arraycopy(refs,n, arr,0, varyingLength); 
			      n += varyingLength;
			      fd.field.set(obj, arr);
			      break;
			    }
			}
		    } else {
			switch (fd.appType) { 
			  case FieldDescriptor.t_byte:
			    fd.field.setByte(obj, data[offs++]);
			    break;
			  case FieldDescriptor.t_integer:
			    fd.field.setInt(obj,
					    Protocol.unpack4(data, offs));
			    offs += 4; 
			    break;
			  case FieldDescriptor.t_long:
			    fd.field.setLong(obj,
					     Protocol.unpack8(data, offs));
			    offs += 8; 
			    break;
			  case FieldDescriptor.t_short:
			    fd.field.setShort(obj, (short)Protocol.unpack2
					      (data, offs));
			    offs += 2; 
			    break;
			  case FieldDescriptor.t_char:
			    fd.field.setChar(obj, (char)Protocol.unpack2
					     (data, offs));
			    offs += 2; 
			    break;
			  case FieldDescriptor.t_float:
			    fd.field.setFloat(obj, Float.intBitsToFloat
					      (Protocol.unpack4(data, offs)));
			    offs += 4; 
			    break;
			  case FieldDescriptor.t_double:
			    fd.field.setDouble(obj, Double.longBitsToDouble
					      (Protocol.unpack8(data, offs)));
			    offs += 8; 
			    break;
			  case FieldDescriptor.t_boolean:
			    fd.field.setBoolean(obj, data[offs++] != 0);
			    break;
			  case FieldDescriptor.t_reference:
			    fd.field.set(obj, refs[n++]);
			    break;
			  case FieldDescriptor.t_string:
			    i = Protocol.unpack2(data, offs) & 0xFFFF;
			    offs += 2;
			    if (i == FieldDescriptor.nullString) { 
				fd.field.set(obj, null);
			    } else { 
				char[] arr = new char[i];
				for (int j = 0; j < i; j++) { 
				    arr[j] = (char)Protocol.unpack2(data,offs);
				    offs += 2;
				}
				fd.field.set(obj, new String(arr));
			    }
			}
		    }
		}
	    } else { 
		for (fd = firstField; fd != null; fd = fd.next) { 
		    if (fd == varyingField) {
			if (fd.dbsType == Protocol.fld_reference) { 
			    if (fd.field != null) { 
				Object arr = Array.newInstance
				    (fd.field.getType().getComponentType(), 
				     varyingLength);
				System.arraycopy(refs,n,arr,0,varyingLength);
				fd.field.set(obj, arr);
			    } 
			    n += varyingLength;
			} else { 
			    offs = varyingField.unpackVarying(data, offs, obj,
							      varyingLength);
			}
		    } else { 
			if (fd.dbsType == Protocol.fld_reference) { 
			    if (fd.field != null) { 
				fd.field.set(obj, refs[n]);
			    }
			    n += 1;
			} else { 
			    offs = fd.unpack(data, offs, obj);
			}
		    }
		}
	    }
	} catch (IllegalAccessException x) { 
	    throw new IllegalAccessError();
	}
	obj.state &= ~Persistent.RAW;
    }


    final int objectSize(Persistent obj) {
	int size = fixedSize;
	if (varyingField != null) { 
	    size += varyingField.length(obj)*varyingSize;
	}
	for (FieldDescriptor str = strFields; str != null; str = str.nextStr) {
	    try {
		if (str.field != null) { 
		    String s = (String)str.field.get(obj);
		    if (s != null) { 
			size += s.length()*2;
		    }
		}
	    } catch (Exception x) {
		throw new Error(x.getMessage());
	    }
	}
	return size;
    } 

    final void pack(Persistent obj, byte[] buffer, int offs) { 
	int varyingLength =
	    (varyingField != null) ? varyingField.length(obj) : 0;
	int scalarOffs = offs + Protocol.ref_size *
	    (nFixedReferences + varyingLength * nVaryingReferences);

	try { 
	    for (FieldDescriptor fd = firstField; fd != null; fd = fd.next) { 
		if (fd == varyingField) {
		    if (fd.dbsType == Protocol.fld_reference) { 
			offs = varyingField.packObjectArray(obj, buffer, offs);
		    } else { 
			scalarOffs = varyingField.packVarying(obj,buffer, 
							      scalarOffs); 
		    } 
		} else { 
		    if (fd.dbsType == Protocol.fld_reference) { 
			offs = fd.packRef(obj, buffer, offs);
		    } else { 
			scalarOffs = fd.pack(obj, buffer, scalarOffs);
		    } 
		}
	    }
	} catch (IllegalAccessException x) { 
	    throw new IllegalAccessError();
	}
    }
}
