//-< BasicMetaobject.java >------------------------------------------*--------*
// GOODS                      Version 2.02       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// Java Program Interface                                            *  /  \  *
//                          Created:      1-Oct-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 18-Oct-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Basic method methods implementation
//-------------------------------------------------------------------*--------*

package goodsjpi;

public abstract class BasicMetaobject extends Metaobject { 
    //
    // Critical section to synchronize access to static variables of metaobject
    //
    static protected Object globalCS = new Object();
    //
    // Critical section to synchronize access to object cache 
    //
    static protected Object cacheCS = new Object();
    //
    // Header of l2-list of objects involved in trabsaction. Modified objects 
    // are placed at the beginning of the list (transObjects.next...), while
    // not-modified objects involved in transaction are placed at the end of
    // the list (transObjects.prev...)
    // This is done to optimize sending of transaction to the storage server
    // (no extra sort or repacking of objects is needed).
    //
    // If object was first included in transaction list as "non-modified" and
    // later was modified, then it will be relinked to the beginning of the 
    // transaction list. 
    //
    // Object are kept in the transaction list until the end of transaction to 
    // prevent GC from deallocating them before end of tranasction.
    //
    static protected Persistent transObjects;
    //
    // Headers for two l2-lists used to implement enhanced LRU discipline
    // for replaing objects in object cache. Head of the list contains
    // most recently used object and tail - least recently used objects. 
    // To prevent object cache from replaing all object as a result of
    // database serach through large number of obejct (most of which will
    // not be used in future), standard LRU scheme was extended to more fit
    // specific of database applications. Object cache is divided into two 
    // part: primary cache and cache for frequently used objects. First time 
    // object is accessed, it is placed in primary cache. Object will be moved
    // to "frequently used objects" cache only when it is accessed more than
    // twice and it is not at the head of LRU list (case of several sucessive 
    // accesses to the object). Both cache parts are managed separately: when
    // total size of objects in the one part exceeds some limit value, least
    // recently used objects from this part are thrown away from clients memory
    // not affecting objects in another part of the cache.
    //
    static protected Persistent lru0;
    static protected Persistent lru1;
    //
    // Total sizes of objects in the part of the cache (see explanation above)
    // Stub objects are not placed in the cache so thier size in not included
    // into this values. The total size occpied by persistent objects in 
    // client memory therefore can be greater than cacheSize0+cacheSize1. 
    //
    static protected int cacheSize0;
    static protected int cacheSize1;
    //
    // Limits for total size of objects in both parts of the cache (see
    // explanation above). If size of dome object exceed this limit values, 
    // it will be placed in the cache in any case. 
    //
    static protected int cacheSizeLimit0 = 2*1024*1024;
    static protected int cacheSizeLimit1 = 2*1024*1024;
    //
    // Counter of nested transactions (or nested invocations of methods of
    // classes derived from Persistent class). BasicMetaobject implements 
    // implicit scheme of detection transaction boundaries. Transaction is 
    // committed when no more active methods exists for the classes controlled 
    // by this metaobject (or metaobjects derived from Basicmetaobject), 
    // i.e. when nestedTransactionCount is equal to zero. It is possbile
    // to manualy increment and decrement this count, so explictly
    // set transaction boundaries (see methods beginNestedTransaction() and
    // endNestedTransaction() below.
    //
    static protected int nestedTransactionCount;

    static { 
	lru0 = new Persistent((Metaobject)null);
	lru0.next = lru0.prev = lru0;
	lru1 = new Persistent((Metaobject)null);
	lru1.next = lru1.prev = lru1;
	transObjects = new Persistent((Metaobject)null);
	transObjects.next = transObjects.prev = transObjects;
    }
    
    //
    // Manipulations with L2 list of objects
    //
    protected static void linkAfter(Persistent after, Persistent obj) { 
	(obj.prev = after).next = (obj.next = after.next).prev = obj;
    }

    protected static void linkBefore(Persistent before, Persistent obj) { 
	(obj.next = before).prev = (obj.prev = before.prev).next = obj;
    }

    protected static void unlink(Persistent obj) { 
	Assert.that(obj.next.prev == obj && obj.prev.next == obj);
	obj.next.prev = obj.prev;
	obj.prev.next = obj.next;
	obj.next = obj.prev = null;
    }

    //
    // Exclude object from LRU list so making it not available for 
    // cached objects LRU relpacing algorithm. This method is called
    // in two situations: 
    //  1) when object is accessed to prevent it from throwing away until
    //     it is accessed by active method.
    //  2) when modification notification for this object was received from
    //     server and instance of deteriorated object instance is removed from
    //     clients memory.
    //
    protected static void removeFromCache(Persistent obj) { 
	synchronized (cacheCS) { 
	    if ((obj.state & Persistent.CACHED) != 0) { 
		Assert.that((obj.state
			     & (Persistent.DESTRUCTED|Persistent.RAW)) == 0);
		unlink(obj);
		if ((obj.state & Persistent.USEFUL) == 0) { 
		    if ((obj.state & Persistent.ACCESSED) != 0) { 
			if (obj != lru0.next) { 
			    obj.state |= Persistent.USEFUL;
			}
		    } else { 
			obj.state |= Persistent.ACCESSED;
		    }
		    cacheSize0 -= obj.desc.objectSize(obj);
		} else { 
		    cacheSize1 -= obj.desc.objectSize(obj);
		}
		obj.state &= ~Persistent.CACHED;
	    }
	}
    }

    //
    // Insert object in one of two cache LRU lists depending on its current
    // state. If total size of objects in this part of the cache exceeds limit 
    // value, least recently used objects will be removed from the cache
    // (and from client memory) until total size of objects becomes less or
    // equal than limit.
    //
    protected static void insertInCache(Persistent obj) { 
	synchronized (cacheCS) { 
	    Assert.that((obj.state 
			 & (Persistent.DESTRUCTED|Persistent.RAW)) == 0);
	    if ((obj.state & Persistent.USEFUL) == 0) { 
		cacheSize0 += obj.desc.objectSize(obj);
		while (cacheSize0 > cacheSizeLimit0 && lru0.prev != lru0) {
		    Persistent victim = lru0.prev;
		    cacheSize0 -= victim.desc.objectSize(victim);
		    victim.metaobject.destroyObject(victim);
		    victim.state &= ~Persistent.CACHED;
		    unlink(victim);
		}
		linkAfter(lru0, obj);
	    } else { 
		cacheSize1 += obj.desc.objectSize(obj);
		while (cacheSize1 > cacheSizeLimit1 && lru1.prev != lru1) {
		    Persistent victim = lru1.prev;
		    cacheSize1 -= victim.desc.objectSize(victim);
		    victim.metaobject.destroyObject(victim);
		    victim.state &= ~Persistent.CACHED;
		    unlink(victim);
		}
		linkAfter(lru1, obj);
	    }
	    obj.state |= Persistent.CACHED;
	}
    }

    //
    // Include object intransaction list. If object is modified it will be 
    // placed at the head of the list, otherwise at the tail of the list.
    // If object already was included in transaction as "read-only", and 
    // later was modified, it will relinked to the head of the list.
    // 
    protected static void addToTransaction(Persistent obj) { 
	synchronized (transObjects) { 
	    if ((obj.state & Persistent.DIRTY) != 0) { 
		if ((obj.state & Persistent.TRANSWRITE) == 0) { 
		    if ((obj.state & Persistent.TRANSREAD) != 0) { 
			obj.state &= ~Persistent.TRANSREAD;
			unlink(obj);
		    }
		    linkAfter(transObjects, obj);
		    obj.state |= Persistent.TRANSWRITE;
		}
	    } else { 
		if ((obj.state & Persistent.TRANSREAD) == 0) { 
		    linkBefore(transObjects, obj);
		    obj.state |= Persistent.TRANSREAD;
		}
	    }
	}
    }

    //
    // Invocation of this method is inserted by MOP geneator before each
    // method invocation or object component access.
    //
    public void preDaemon(Object o, int attr) {
	Persistent obj = (Persistent)o;
	synchronized(obj) {
	    if (obj.accessCount == 0) { 
		synchronized (this) { 
		    obj.accessCount = 1;
		    //
		    // Setting of access count to non-zero prevents from
		    // object destruction by receiving deterioration 
		    // notification from server. Instead of this "invalid"
		    // object attribute will set to true.
		    //
		}
		synchronized (globalCS) { 
		    nestedTransactionCount += 1;
		    if (obj.opid != 0) { // persistent object
			removeFromCache(obj);
		    }
		}
		if (obj.opid != 0) { // presistent objects
		    if ((attr & Metaobject.MUTATOR) != 0){
			beginWriteAccess(obj);
		    } else { 
			beginReadAccess(obj);
		    }
		    if (obj.invalidated) { 
			//
			// In case of using pessimistic scheme object is 
			// already locked at this moment, so it can not be 
			// more changed by another application before the end 
			// of transaction. 
			// And with optimistic scheme... hmmm. it nice to be 
			// optimist.
			//
			synchronized (this) { 
			    obj.invalidated = false;
			    destroyObject(obj);
			}
		    }
		    if ((obj.state & Persistent.DESTRUCTED) != 0) { 
			obj.storage.load(obj.opid, obj);
		    } else if ((obj.state & Persistent.RAW) != 0) {
			obj.storage.prepare(obj);
		    }
		}
	    } else { 
		obj.accessCount += 1;
	    }
	}
    }
    
    //
    // Invocation of this method is inserted by MOP geneator after each
    // method invocation or object component access.
    //
    public void postDaemon(Object o, int attr, boolean modified) {
	Persistent obj = (Persistent)o;
	synchronized(obj) {
	    if (modified) { 
		obj.state |= Persistent.DIRTY;
	    }
	    if (obj.accessCount == 1) { 
		if (obj.opid != 0) { 
		    endAccess(obj);
		}
		synchronized (globalCS) { 
		    if (obj.opid != 0) { // persistent object
			if ((obj.state & (Persistent.NOTIFY|
			     Persistent.TRANSREAD|Persistent.TRANSWRITE)) == 0)
			{
			    insertInCache(obj);
			}
		    }
		    if (--nestedTransactionCount == 0) { 
			commitTransaction();
		    }		
		}
		synchronized (this) { 
		    //
		    // We delay decrement of access counter till this moment
		    // to prevent object destruction on receiving deterioration
		    // notification from server. Synchronized construction 
		    // for metaobject object provides mutual exclusion with 
		    // objectInvalidate() method.
		    //
		    obj.accessCount = 0;
		    if (obj.invalidated) { 
			obj.invalidated = false;
			synchronized (cacheCS) { 
			    removeFromCache(obj);
			    destroyObject(obj);
			}
		    }
		}
	    } else { 
		obj.accessCount -= 1;
	    }
	}
    }		

    //
    // Assign persistent identifier to the transient object of persistent
    // capable class. The object is placed in the same storage as 
    // persistent object containing reference to this object unless
    // explicit storage attachment was specified for this object.
    //
    protected void makePersistent(Persistent obj, Persistent parent)
    {
	if (obj.storage == null) { 
	    //
	    // By default transient object is placed in the same storage as 
	    // persitent object referenced this object.
	    //
	    obj.storage = parent.storage;
	} else { 
	    // 
	    // Interdatabase references are prohibited
	    //
	    Assert.that(obj.storage.database == parent.storage.database);
	}
	obj.storage.allocate(obj, 0);
	//
	// Object is linked in transaction list after persistent object, 
	// from which it is referenced, so this object will handled by
	// commitTransaction() method soon after it complets with parent 
	// object.
	//
	linkAfter(parent, obj); 
	obj.state |= Persistent.TRANSWRITE|Persistent.DIRTY;
    }

    //
    // At this moment there are no more active methods for 
    // classes controlled by this metaobject (or metaobjects dericed from it).
    // This method will try to commit all transactions in all opened databases.
    // Until this method is completed no other invocation or access to
    // objects controled by this MOP is possoble (it is forced by declaring
    // this static method "synchronized" as well as static method fixObject(),
    // which is called before any access to the object).
    //
    protected static void commitTransaction() {	
	Persistent next;
      loop:
	while (transObjects.next != transObjects) { 
	    Persistent obj = transObjects.next;
	    if ((obj.state & Persistent.DIRTY) == 0) { 
		// 
		// As far as all modified objects are placed at the head of 
		// the list, and we are navigating through the list startig 
		// from the head, finding of non-modified object means that
		// there are no more other modified objects. So no more 
		// other transactions has to be send to the server. 
		// We only should release all locks hold by these transactions.
		//
		do { 
		    next = obj.next;
		    obj.metaobject.releaseObject(obj);
		} while((obj = next) != transObjects);
		break;
	    }
	    int nTransServers = 0;
	    Storage coordinator = obj.storage;
	    Database db = coordinator.database;
	    Storage[] servers = new Storage[db.storages.length];

	    do { // find all tranasction objects from the same database
		Storage storage = obj.storage;
		if (storage.database == db) { 
		    if (obj.invalidated) { 
			abortTransaction(db);
			continue loop;
		    }
		    if (!storage.isInvolvedInTransaction()) { 
			if (storage.id < coordinator.id) {
			    coordinator = storage;
			}
			servers[nTransServers++] = storage;
		    }
		    storage.store(obj);
		}
		obj = obj.next;
	    } while (obj != transObjects);

	    if (nTransServers != 0) { 
		//
		// commit local transaction or coordinator part of global one
		//
		int tid = coordinator.commitTransaction(nTransServers,servers);
		if (tid == -1) { 
		    abortTransaction(db);
		    continue;
		}
		if (nTransServers > 1) { // global transaction
		    for (int i = 0; i < nTransServers; i++) { 
			if (servers[i] != coordinator) { 
			    servers[i].commitSubtransaction(coordinator,
							    nTransServers, 
							    servers, tid);
			}
		    }
		    if (!coordinator.waitTransactionCompletion()) { 
			abortTransaction(db);
			continue;
		    }
		}
	    }
	    //
	    // Transaction was successfully commited
	    //
	    for (obj = transObjects.next; obj != transObjects; obj = next){
		next = obj.next;
		if (obj.storage.database == db) { 
		    obj.metaobject.commitObjectChanges(obj);
		}
	    }
	}
    }
    
    protected static void abortTransaction(Database db) { 
        Persistent obj, next;

	for (obj = transObjects.next; obj != transObjects; obj = next){
	    next = obj.next;
	    if (obj.storage.database == db) { 
	        obj.metaobject.undoObjectChanges(obj);
	    }
	}
	db.abortTransaction();
    }

    protected synchronized void commitObjectChanges(Persistent obj) { 
	Assert.that((obj.state & (Persistent.TRANSREAD|Persistent.TRANSWRITE))
		    != 0);
	unlink(obj); // remove object from transaction list
	obj.state &= ~(Persistent.TRANSREAD | Persistent.TRANSWRITE |
		       Persistent.XLOCKED | Persistent.SLOCKED | 
		       Persistent.DIRTY | Persistent.NEW);
	if (obj.invalidated) { 
	    obj.invalidated = false;
	    destroyObject(obj);
	} else if ((obj.state & Persistent.NOTIFY) == 0) { 
	    insertInCache(obj);
	}
    }
	
    protected synchronized void undoObjectChanges(Persistent obj) { 
	Assert.that((obj.state & (Persistent.TRANSREAD|Persistent.TRANSWRITE))
		    != 0);
	unlink(obj); // remove object from transaction list

	if ((obj.state & (Persistent.SLOCKED|Persistent.XLOCKED)) != 0) {
	    obj.storage.unlock(obj, Protocol.lck_none);
	}
	
	if ((obj.state & Persistent.DIRTY) != 0) {
	    if ((obj.state & Persistent.NEW) == 0) {
		//
		// All persistent objects should be reloaded from the server.
		//
		destroyObject(obj);
	    } else { 
		obj.storage.forgetObject(obj);
		obj.opid = 0; // object beacomes transsient once again
		obj.storage = null;
	    }
	} else if ((obj.state & Persistent.NOTIFY) == 0) {
	    insertInCache(obj);
	}
	obj.state &= ~(Persistent.TRANSREAD | Persistent.TRANSWRITE |
		       Persistent.XLOCKED | Persistent.SLOCKED | 
		       Persistent.DIRTY | Persistent.NEW);
    }
	
    
    protected synchronized void releaseObject(Persistent obj) { 
	Assert.that((obj.state & (Persistent.TRANSREAD|Persistent.TRANSWRITE))
		    != 0);
	unlink(obj); // remove object from transaction list

	if ((obj.state & (Persistent.SLOCKED|Persistent.XLOCKED)) != 0) {
	    obj.storage.unlock(obj, Protocol.lck_none);
	}	
	if ((obj.state & Persistent.NOTIFY) == 0) { 
	    insertInCache(obj);
	}
	obj.state &= ~(Persistent.TRANSREAD | Persistent.TRANSWRITE |
		       Persistent.XLOCKED | Persistent.SLOCKED | 
		       Persistent.DIRTY | Persistent.NEW);
    }
	
    //
    // Destoroy all object references to make it possible to provide
    // more unaccessible objects to GC
    //
    protected void destroyObject(Persistent obj) {
	if ((obj.state & Persistent.DESTRUCTED) == 0) { 
	    try { 
		FieldDescriptor fd;
		for (fd = obj.desc.refFields; fd != null; fd = fd.nextRef) {
		    fd.field.set(obj, null);
		}
	    } catch(Exception x) { 
		throw new Error(x.getMessage());
	    }
	    obj.state |= Persistent.DESTRUCTED|Persistent.RAW;
	    obj.storage.throwObject(obj);
	}
    }

    //
    // This method is asynchronously called by server storag agent when
    // deterioration notification from server is received. 
    //
    protected synchronized void invalidateObject(Persistent obj) { 
	if ((obj.state & Persistent.DESTRUCTED) == 0) {
	    if (obj.accessCount != 0 ||
		(obj.state & (Persistent.TRANSREAD|Persistent.TRANSWRITE))!=0)
	    { 
		//
		// It is not possible to detroy object immediatly, 
	        // just mark it as invalid and delay destruction till end
		// of access to the object or end of transaction.
		//
		obj.invalidated = true;
	    } else { 
		obj.invalidated = false;
		synchronized (cacheCS) { 
		    removeFromCache(obj);
		    destroyObject(obj);
		}
	    }
	}
	if ((obj.state & Persistent.NOTIFY) != 0) { 
	    CondEvent event = 
		(CondEvent)obj.storage.database.notifications.get(obj);
	    if (event != null) { 
		event.signal();
	    }
	}
    }

    //
    // Enable or disable notification on receving invalidation messages from
    // the server
    //
    protected synchronized void notifyOnModification(Persistent obj, 
						     boolean enabled)
    {
	synchronized(cacheCS) { 
	    if (enabled) { 
		obj.state |= Persistent.NOTIFY;
		if ((obj.state & Persistent.CACHED) == 0) { 
		    removeFromCache(obj);
		} 
	    } else { 
		obj.state &= ~Persistent.NOTIFY;
		if (obj.accessCount == 0 
		    && (obj.state 
			& (Persistent.TRANSREAD|Persistent.TRANSWRITE)) == 0)
	        {
		    insertInCache(obj);
		}
	    }
	}
    }



    //
    // Set new limits for object cache. See description of cache management
    // algorithm at the ebggining of this file.
    //
    public static void setCacheLimits(int l0, int l1) { 
	synchronized (globalCS) { 
	    cacheSizeLimit0 = l0;
	    cacheSizeLimit1 = l1;
	}
    }

    //
    // By incrementing nested transaction count we delay implicit transaction 
    // commit until the moment when counter is explicitly decremented  
    // endNXestedTransactionMethod()
    //
    public static void beginNestedTransaction() {
	synchronized (globalCS) { 
	    nestedTransactionCount += 1;
	}
    }

    public static void endNestedTransaction() { 
	synchronized (globalCS) { 
	    if (--nestedTransactionCount == 0) { 
		commitTransaction();
	    }
	}
    }
}



