import java.awt.*;
import goodsjpi.*;

public class GraphEditor extends Frame implements Runnable { 
    int      x, y;
    Figure   figure;
    Graphics graphics;
    Database db;
    Catalog  catalog;
    Drawing  drawing;
    Menu     addMenu;
    Thread   sync;
    boolean  selecting;

    volatile boolean running;
    CondEvent        event;

    public GraphEditor(Database database) { 
	super("Graphic Editor");
	db = database;
	running = true;
	event = new CondEvent();
	sync = new Thread(this);
	sync.start();

	catalog = (Catalog)db.getRoot();
	if (catalog == null) { 
	    catalog = new Catalog(101);
	    db.setRoot(catalog);
	}

	MenuBar mainMenu = new MenuBar();
	Menu fileMenu = new Menu("File");
	fileMenu.add(new MenuItem("New"));	
	fileMenu.add(new MenuItem("Open"));
	fileMenu.addSeparator();
	fileMenu.add(new MenuItem("Exit"));
	mainMenu.add(fileMenu);
	addMenu = new Menu("Add");
	addMenu.add(new MenuItem("Point"));
	addMenu.add(new MenuItem("Line"));
	addMenu.add(new MenuItem("Circle"));
	addMenu.add(new MenuItem("Rectangle"));
	addMenu.disable();
	mainMenu.add(addMenu);
	setMenuBar(mainMenu);
	setLayout(null);
	addNotify();
	resize(insets().left+insets().right + 500, 
	       insets().top +insets().bottom + 300);
	show();
	setCursor(CROSSHAIR_CURSOR);
	graphics = getGraphics();
    }

    public boolean handleEvent(Event event) { 
	switch (event.id) { 
	  case Event.WINDOW_DESTROY:
	    close();
	    return true;
	  case Event.MOUSE_DOWN:
	    x = event.x;
	    y = event.y;
	    if (y >= insets().top && figure != null) { 
		figure.set(x, y, x, y);

		if (figure instanceof Point) {
		    figure.paint(graphics);
		    drawing.add((Figure)figure.clone());
		} else if (figure != null) { 
		    if (!selecting) { 
			graphics.setXORMode(Color.red);
			figure.paint(graphics);
		    }
		    selecting = true;
		}
		return true;
	    }
	    break;
	  case Event.MOUSE_DRAG:
	    if (selecting && figure != null) { 
		graphics.setXORMode(Color.red);
		figure.paint(graphics);
		figure.set(x, y, event.x, event.y);
		figure.paint(graphics);
	    }
	    return true;
	  case Event.MOUSE_UP:
	    if (figure != null && selecting) { 
		figure.paint(graphics);
		graphics.setPaintMode();
		figure.set(x, y, event.x, event.y);
		figure.paint(graphics);
		drawing.add((Figure)figure.clone());
		selecting = false;
	    }
	    return true;
	}
	return super.handleEvent(event);
    }
    
    public void paint(Graphics g) { 
	super.paint(g);
	if (drawing != null) { 
	    graphics.setPaintMode();
	    drawing.paint(g);
	}
    }
    
    public boolean action(Event event, Object arg) { 
	if (event.target instanceof MenuItem) { 
	    String label = (String)arg;
	    if (label.equals("Exit")) { 
		handleEvent(new Event(this, Event.WINDOW_DESTROY, null));
	    } else if (label.equals("Open")) { 
		openGraphic();
	    } else if (label.equals("New")) { 
		newGraphic();
	    } else if (label.equals("Point")) {
		figure = new Point();
	    } else if (label.equals("Line")) { 
		figure = new Line();		
	    } else if (label.equals("Circle")) { 
		figure = new Circle();
	    } else if (label.equals("Rectangle")) { 
		figure = new Rectangle();
	    } else { 
		return super.action(event, arg);
	    }
	    return true;
	}				
	return super.action(event, arg);
    }
	      
    public static void main(String args[]) { 
	Database db = new Database();
	if (db.open("graphedt.cfg")) { 
	    new GraphEditor(db);
	} else { 
	    System.err.println("Failed to open database");
	    System.exit(1);
	}
    }

    public void openGraphic() {
	new OpenDialog(this, catalog.buildList());
    }
    
    public void newGraphic() { 
	new NewDialog(this);
    } 

    public synchronized void newDrawing(String name) { 
	if (drawing != null) { 
	    db.notifyOnModification(drawing, null);
	} 
	drawing = new Drawing(name);
	catalog.add(drawing);
	db.notifyOnModification(drawing, event);
	repaint();
    }

    public synchronized void openDrawing(String name) { 
	if (drawing != null) { 
	    db.notifyOnModification(drawing, null);
	} 
	drawing = catalog.find(name);
	db.notifyOnModification(drawing, event);
	repaint();
    }	

    public void run() { 
	while (true) { 
	    event.waitSignal();
	    if (running) { 
		repaint();
	    } else { 
		break;
	    }
	}
    }

    public synchronized void close() { 
	running = false;
	event.signal(); // stop update thread
	try { 
	    sync.join();
	} catch(Exception x) {}
	hide();
	dispose();
	System.out.println("Close database session");
	db.close();
	System.exit(0);
    }
}


class NewDialog extends Dialog { 
    GraphEditor editor;
    TextField   text;
    Button      okButton;
    
    public NewDialog(GraphEditor editor) { 
	super(editor, "New Graphic", true);
	this.editor = editor;
	setLayout(new FlowLayout(FlowLayout.CENTER, 1000, 40));
	addNotify();
	resize(insets().left+insets().right + 250, 
	       insets().top +insets().bottom + 150);
	text = new TextField(25);
	okButton = new Button("OK");
	add(text);
	add(okButton);
	show();
    }

    public void show() { 
	java.awt.Point origin = editor.getLocationOnScreen();
	java.awt.Rectangle parentBounds = editor.bounds();
	java.awt.Rectangle myBounds = bounds();
	move(origin.x + (parentBounds.width - myBounds.width) / 2,
	     origin.y + (parentBounds.height - myBounds.height) / 2);
	super.show();
    }
    
    public boolean handleEvent(Event event) { 
	if (event.id == Event.ACTION_EVENT &&
	    (event.target == okButton || event.target == text))
	{
	    String str = text.getText().trim();
	    if (str.length() == 0) { 
		System.err.print("\007"); // beep
	    } else { 
		editor.addMenu.enable();
		System.out.println("Create drawing '" + str + "'");
		handleEvent(new Event(this, Event.WINDOW_DESTROY, null));
		editor.newDrawing(str);
	    } 
	    return true;
	} else if (event.id == Event.WINDOW_DESTROY) { 
	    hide();
	    return true;
	}
	return super.handleEvent(event);
    }
}


class OpenDialog extends Dialog { 
    GraphEditor editor;
    List        list;
    Button      okButton;
    
    public OpenDialog(GraphEditor editor, List drawings) { 
	super(editor, "Open Graphic", true);
	this.editor = editor;
	setLayout(new FlowLayout(FlowLayout.CENTER, 1000, 20));
	addNotify();
	resize(insets().left+insets().right + 250, 
	       insets().top +insets().bottom + 150);
	list = drawings;

	okButton = new Button("OK");
	add(list);
	add(okButton);
	show();
    }
    
    public void show() { 
	java.awt.Point origin = editor.getLocationOnScreen();
	java.awt.Rectangle parentBounds = editor.bounds();
	java.awt.Rectangle myBounds = bounds();
	move(origin.x + (parentBounds.width - myBounds.width) / 2,
	     origin.y + (parentBounds.height - myBounds.height) / 2);
	super.show();
    }
    
    public boolean handleEvent(Event event) { 
	if (event.id == Event.ACTION_EVENT &&
	    (event.target == okButton || event.target == list))
	{
	    String str = list.getSelectedItem();
	    if (str == null) { 
		System.err.print("\007"); // beep
	    } else { 
		editor.addMenu.enable();
		System.out.println("Open drawing '" + str + "'");
		handleEvent(new Event(this, Event.WINDOW_DESTROY, null));
		editor.openDrawing(str);
	    } 
	    return true;
	} else if (event.id == Event.WINDOW_DESTROY) { 
	    hide();
	    return true;
	}
	return super.handleEvent(event);
    }
}
