//-< DATABASE.H >----------------------------------------------------*--------*
// GOODS                     Version 1.0         (c) 1997  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
//                                                                   *  /  \  *
//                          Created:      7-Jan-97    K.A. Knizhnik  * / [] \ *
//                          Last update: 16-Apr-97    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Application specific database interface
//-------------------------------------------------------------------*--------*

#ifndef __DATABASE_H__
#define __DATABASE_H__

#include "storage.h"

class class_descriptor; 
class object_reference; 

//
// This class obtain application dependent interface with database storage. 
// It is responsible for: 
//   1) synchronization of request to database storage
//   2) mapping between database and application class descriptors
//   3) convertion of loaded and stored objects
//   4) checking storage availablity 
//
class obj_storage : public dbs_application { 
    friend class database; 
    friend class object; 
    friend class object_handle; 

  protected: 
    dnm_array<class_descriptor*> descriptor_table;
    dnm_array<cpid_t>            cpid_table;
    dbs_storage*                 storage; 
    mutex                        cs; 

    //
    // Number of references to objects from this storage
    // The 'storage' object can be removed when there are no more references
    // to it from any object (n_references == 0)
    //
    long              n_references; 
    dnm_buffer        buf; 
    boolean           opened;
    
  public: 
    database* const   db; 
    const sid_t       id;
 
    void   add_reference() { 
	n_references += 1; 
    }

    void   remove_reference() { 
	if (--n_references == 0) { 
	    delete this; 
	}
    }

    //
    // Method called by server to invalidate object instance
    //
    virtual void invalidate(sid_t sid, opid_t opid);

    //
    // Method called by server to notify about server disconnection
    //
    virtual void disconnected(sid_t sid); 

    //
    // Authorization procedure fails at server sid
    // 
    virtual void login_refused(sid_t sid);

    //
    // Inform server that client no more has reference to specified object 
    //
    void    forget_object(opid_t opid) { storage->forget_object(opid); }

    //
    // Inform server that client no more has instance of specified object
    //
    void    throw_object(opid_t opid) { storage->throw_object(opid); }
    
    //
    // Get server class identifier for specified application class
    //
    cpid_t  get_cpid_by_descriptor(class_descriptor* desc);

    //
    // Allocate object at server
    //
    opid_t  allocate(cpid_t cpid, size_t size, boolean aligned); 

    //
    // Deallocate object at server
    //
    void    deallocate(opid_t opid) { storage->deallocate(opid); }

    //
    // Download object from server. This method is called with
    // global lock set, release lock while waiting for object and
    // set it again after receiving object from server
    //
    void    load(hnd_t hnd, int flags = lof_none); 

    //
    // Set new lock or upgrade an existed one (from shared to exclusive)
    //
    boolean lock(opid_t opid, lck_t lck, int attr);
    //
    // Remove or downgrade lock. 
    //
    void    unlock(opid_t opid, lck_t lck) { storage->unlock(opid, lck); } 
    
    //
    // Transaction protocol. 
    // No other storage methods (exept forget_object and throw_object
    // can be called until transaction completion. This syncronization should
    // be done by metaobject protocol. 
    //
    void    begin_transaction(); 
    
    void    include_object_in_transaction(hnd_t hnd, int flags);
    boolean commit_coordinator_transaction(int n_trans_servers,
					  sid_t* trans_servers,
					  tid_t& tid);
    void    commit_transaction(sid_t coordinator, 
			       int n_trans_servers,
			       sid_t* trans_servers,
			       tid_t tid);
    boolean wait_global_transaction_completion();

    boolean open(char const* connection_address);
    void    close();

    obj_storage(database* dbs, sid_t sid); 
    virtual~obj_storage();
};

//
// Class 'database' is collection of storages.
// It's main responsibility is transaction handling. 
//
class database { 
  protected:
    friend class object;
    friend class obj_storage;
    friend class field_descriptor; 

    obj_storage**  storages;
    int            n_storages;

    boolean        opened; 
    mutex          cs;
 
  protected: 
    virtual dbs_storage* create_dbs_storage(sid_t sid) const;

    //
    // Handle of one of storage servers disconnection
    //
    virtual void    disconnected(sid_t sid);

    //
    // Handle authorization error
    //
    virtual void    login_refused(sid_t sid);
 
    obj_storage*    get_storage(sid_t sid) {
	internal_assert(sid < n_storages);
	return storages[sid];
    }

  public:
    virtual boolean open(const char* database_configuration_file);
    virtual void    close();
    
    //
    // Get root object of specified storage
    //
    void get_root(object_reference& ref, sid_t sid = 0);

    int  get_number_of_storages() const { return n_storages; }

    database();
    virtual ~database();
};    

#endif
