//-< BUGDB.CXX  >----------------------------------------------------*--------*
// GOODS                     Version 1.0         (c) 1997  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
//                                                                   *  /  \  *
//                          Created:     27-Mar-99    K.A. Knizhnik  * / [] \ *
//                          Last update:  4-Apr-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Example of database Web publishing: Bug Tracking Database
//-------------------------------------------------------------------*--------*

#include "bugdb.h"

char const* const eCATEGORY_STRING[] = 
{
    "",
    "CRASH",
    "PROGRAM_HANGS",
    "UI_DISPLAY",
    "UI_BEHAVIOR",
    "CALCULATION",
    "ERROR_HANDLING",
    "PERFORMANCE",
    "LICENSING",
    "INSTALLATION", 
    "DOCUMENTATION",
    "ENHANCEMENT",
    "HOW_TO_QUESTION",
    NULL
};

char const* const eSTATUS_STRING[] = { 
    "",
    "OPENED",
    "FIXED",
    "CLOSED",
    "PENDING_ENGINEER",
    "PENDING_USER",
    "POSTPONED",
    "IRREPRODUCIBLE",
    "WITHDRAWN",
    "AS_DESIGNED",
    NULL
};

char const* const eSEVERITY_STRING[] = { 
    "",
    "FATAL",
    "SERIOUS",
    "MINOR",
    NULL
};

char const* const eFIXING_PRIORITY_STRING[] = { 
    "",
    "FIX_IMMEDIATELY",
    "FIX_BEFORE_NEXT_BUILD_RELEASE",
    "FIX_BEFORE_NEXT_MINOR_RELEASE",
    "FIX_BEFORE_NEXT_MAJOR_RELEASE",
    "FIX_IF_POSSIBLE",
    "OPTIONAL",
    NULL
};

database           BugDB;
ref<RootObject>    root;
ref<BugTrackingDB> db;


//---- SetMember ------------------------------------------

void SetMember::print(void const*) const {}

boolean SetMember::select(void const*) const 
{
    return False; 
}

void SetMember::selectTrampoline(ref<set_member> mbr, void const* arg)
{
    ref<SetMember> obj = mbr->obj;
    select_args* args = (select_args*)arg;
    if (obj->select(args->condition)) { 
	args->count += 1;
	obj->print(args->print_arg);
    }
}

void SetMember::printTrampoline(ref<set_member> mbr, void const* arg)
{
    ref<SetMember> obj = mbr->obj;
    obj->print(arg);
}

field_descriptor& SetMember::describe_components()
{
    return NO_FIELDS;
}

REGISTER(SetMember, object, pessimistic_scheme);


//- Person ------------------------------------------------

boolean Person::isUser() const 
{
    return True; 
}

void Person::changePassword(char const* password) 
{ 
    sPassword = String::create(password);
}

void Person::changeEmail(char const* e_mail) 
{ 
    sEmailAddress = String::create(e_mail);
}

void Person::print(void const* arg) const 
{ 
    CGIrequest& req = *(CGIrequest*)arg;
    req << TAG << "<OPTION VALUE=\"" << sName << "\">" << sName << "</OPTION>";
}
    
boolean Person::select(void const* pattern) const 
{ 
    SelectPattern* pat = (SelectPattern*)pattern;
    return 
	(pat->name == NULL || sName->index(pat->name) != -1) &&
	(pat->eMail == NULL || sEmailAddress->index(pat->eMail) != -1);
}

void Person::unregister(void const* software) const
{
    setSoftware.erase((char*)software);
}

void Person::exclude(void const* group) const
{
    setGroups.erase((char*)group);
}

field_descriptor& Person::describe_components()
{
    return FIELD(sName),FIELD(sEmailAddress),FIELD(sPassword),
	FIELD(setGroups),FIELD(setReportedBugs),FIELD(setSoftware);
}

Person::Person(char const* name, char const* eMail, char const* password,
	       class_descriptor& desc) 
: SetMember(desc) 
{
    sName = String::create(name);
    sEmailAddress = String::create(eMail);
    sPassword = String::create(password);
    setReportedBugs.initialize(this);
    setGroups.initialize(this);
    setSoftware.initialize(this);
}

REGISTER(Person, SetMember, pessimistic_scheme);

//---- Engineer ------------------------------------

Engineer::Engineer(char const* name, char const* eMail, char const* password)
: Person(name, eMail, password, self_class) 
{
    setAssignedBugs.initialize(this);
}

boolean Engineer::isUser() const 
{
    return False; 
}

void Engineer::unregister(void const* software) const
{
    Person::unregister(software);
}


field_descriptor& Engineer::describe_components()
{
    return FIELD(setAssignedBugs);
}

inline boolean isAdministrator(char const* name) { 
    return strcmp(name, "administrator") == 0;
}

REGISTER(Engineer, Person, pessimistic_scheme);

//----- UserGroup -------------------------------

void UserGroup::print(void const* arg) const 
{ 
    CGIrequest& req = *(CGIrequest*)arg;
    req << TAG << "<OPTION VALUE=\"" << sName << "\">" << sName << "</OPTION>";
}

boolean UserGroup::select(void const* pattern) const 
{ 
    SelectPattern* pat = (SelectPattern*)pattern;
    int nBugs = pat->minBugs != 0 || pat->maxBugs != INT_MAX
	? setBugs.size() : 0;
    return (pat->name == NULL || sName->index(pat->name) != -1) 
	&& nBugs >= pat->minBugs && nBugs <= pat->maxBugs;
}

void UserGroup::removeUser(void const* arg) const
{
    setUsers.erase((char*)arg);
}

void UserGroup::removeBug(void const* arg) const
{
    setBugs.erase((char*)arg);
}

boolean UserGroup::addUser(char const* group, char const* name,
			   ref<Person> person) const
{
    if (setUsers.insertUnique(name, person)) {
	person->setGroups.insert(group, this);
	return True;
    }
    return False;
}


void UserGroup::addBug(void const* arg) const
{
    char bugId[16];
    ref<Bug> bug = *(ref<Bug>*)arg;
    sprintf(bugId, "%d", bug->bugId);
    setBugs.insertUnique(bugId, bug);
}
    
UserGroup::UserGroup(char const* name) : SetMember(self_class) 
{
    sName = String::create(name);
    setBugs.initialize(this);
    setUsers.initialize(this);
}

field_descriptor& UserGroup::describe_components()
{
    return FIELD(sName),FIELD(setUsers),FIELD(setBugs);
}

REGISTER(UserGroup, SetMember, pessimistic_scheme);

//------- Version ---------------------------------------

void Version::print(void const* arg) const 
{ 
    CGIrequest& req = *(CGIrequest*)arg;
    req << TAG << "<OPTION VALUE=\"" << getVersionString() << 
	"\">Version " << getVersionString() << " " << sLabel << 
	" " << released.asString() << "</OPTION>";
}

int Version::getVersion() const 
{ 
    return majorVersionNumber*100 + minorVersionNumber; 
}

char* Version::getVersionString() const 
{
    static char buf[16];
    sprintf(buf, "%d.%02d", majorVersionNumber, minorVersionNumber);
    return buf;
}

boolean Version::select(void const* pattern) const 
{ 
    SelectPattern* pat = (SelectPattern*)pattern;
    return (pat->label == NULL || sLabel->index(pat->label) != -1)
	&& majorVersionNumber >= pat->minMajorVersionNumber
	&& majorVersionNumber <= pat->maxMajorVersionNumber
	&& minorVersionNumber >= pat->minMinorVersionNumber
	&& minorVersionNumber <= pat->maxMinorVersionNumber;
}


Version::Version(char const* label, int major,int minor, const char* comment)
: SetMember(self_class) 
{
    sLabel = String::create(label);
    majorVersionNumber = major;
    minorVersionNumber = minor;
    released.setCurrentDate();
    sComment = String::create(comment);
}

field_descriptor& Version::describe_components()
{
    return FIELD(sLabel),FIELD(majorVersionNumber),FIELD(minorVersionNumber),
	FIELD(released),FIELD(sComment);
}

REGISTER(Version, SetMember, pessimistic_scheme);

//----- Software -------------------------------------

int Software::getLastVersion() const 
{ 
    if (setVersions.empty()) { 
	return 0;
    }
    ref<Version> lastVersion = setVersions.members->last->obj;
    return lastVersion->getVersion();
}

char* Software::getLastVersionString() const 
{ 
    if (setVersions.empty()) { 
	return "";
    }
    ref<Version> lastVersion = setVersions.members->last->obj;
    return lastVersion->getVersionString();
}

boolean Software::attachEngineer(char const* name, ref<Engineer> engineer) 
const
{
    if (setEngineers.insertUnique(name, engineer)) { 
	char* software = sName->data();
	engineer->setSoftware.insert(software, this); 
	delete[] software;
	return True;
    }
    return False;
}

boolean Software::detachEngineer(char const* name) const
{
    ref<Engineer> engineer = setEngineers.erase(name);
    if (engineer != NULL) { 
	char* software = sName->data();
	engineer->setSoftware.erase(software);
	delete[] software;
	return True;
    }
    return False;
}

void Software::detachEngineerCallback(void const* arg) const 
{
    detachEngineer((char*)arg);
}

boolean Software::registerUser(char const* name, ref<Person> user) const
{
    if (setUsers.insertUnique(name, user)) { 
	char* software = sName->data();
	user->setSoftware.insert(software, this); 
	delete[] software;
	return True;
    }
    return False;
}

void Software::unregisterUserCallback(void const* arg) const
{
    unregisterUser((char*)arg);    
}

boolean Software::unregisterUser(char const* name) const
{
    ref<Person> user = setUsers.erase(name);
    if (user != NULL) { 
	char* software = sName->data();
	user->setSoftware.erase(software);
	delete[] software;
	return True;
    }
    return False;
}

void Software::print(void const* arg) const 
{ 
    CGIrequest& req = *(CGIrequest*)arg;
    req << TAG << "<OPTION VALUE=\"" << sName << "\">" << sName << "</OPTION>";
}

boolean Software::select(void const* pattern) const 
{ 
    SelectPattern* pat = (SelectPattern*)pattern;
    int nBugs = pat->minBugs != 0 || pat->maxBugs != INT_MAX
	? setBugs.size() : 0;
    int nUsers = pat->minUsers != 0 || pat->maxUsers != INT_MAX
	? setUsers.size() : 0;
    int nEngineers = pat->minEngineers != 0 || pat->maxEngineers != INT_MAX
	? setEngineers.size() : 0;
    return (pat->name == NULL || sName->index(pat->name) != -1) 
	&& nBugs >= pat->minBugs && nBugs <= pat->maxBugs
	&& nUsers >= pat->minUsers && nUsers <= pat->maxUsers 
	&& nEngineers >= pat->minEngineers 
	&& nEngineers <= pat->maxEngineers;
}

Software::Software(char const* name, ref<Version> version) 
: SetMember(self_class) 
{
    sName = String::create(name);
    pManager = NULL;
    setEngineers.initialize(this);
    setVersions.initialize(this);
    setUsers.initialize(this);
    setBugs.initialize(this);
    setVersions.insert(version->getVersionString(), version);
}

field_descriptor& Software::describe_components()
{
    return FIELD(sName),FIELD(pManager),FIELD(setEngineers),
	FIELD(setUsers),FIELD(setVersions),FIELD(setBugs);
}

REGISTER(Software, SetMember, pessimistic_scheme);

//----- Report -------------------------------------------

void Report::print(void const* arg) const 
{ 
    CGIrequest& req = *(CGIrequest*)arg;
    req << TAG << "<OPTION VALUE=" << index << ">" << pAuthor->sName << " " 
	<< creationDate.asString() << "</OPTION>";
}

void Report::update(char const* description, eSTATUS status)
{
    if (sDescription->compare(description) != 0) { 
	sDescription = String::create(description);
    }
    this->status = status;
}

boolean Report::select(void const* pattern) const 
{ 
    SelectPattern* pat = (SelectPattern*)pattern;
    return 
	(!pat->author || pAuthor->sName->index(pat->author) != -1)
	&& (!pat->createdBefore.isValid() 
	    || creationDate <= pat->createdBefore) 
	&& (!pat->createdAfter.isValid() 
	    || creationDate >= pat->createdAfter) 
	&& (pat->status == STATUS_NOT_SPECIFIED || pat->status == status);
}

Report::Report(int index,
	       char const* description, ref<Person> author, eSTATUS status) 
: SetMember(self_class)
{
    sDescription = String::create(description);
    this->index = index;
    this->status = status;
    pAuthor = author;
    creationDate.setCurrentDate();
}

field_descriptor& Report::describe_components()
{
    return FIELD(pAuthor),FIELD(creationDate),FIELD(status),
	FIELD(sDescription),FIELD(index);
}

REGISTER(Report, SetMember, pessimistic_scheme);

//--- Bug -----------------------------------------

void Bug::print(void const* arg) const 
{
    CGIrequest& req = *(CGIrequest*)arg;
    req << TAG << "<OPTION VALUE=" << bugId << ">" << sOneLineSummary 
	<< "</OPTION>";
}
    
boolean Bug::select(void const* pattern) const 
{ 
    SelectPattern* pat = (SelectPattern*)pattern;
    int version = pSoftware->getLastVersion();
    int majorVersionNumber = version / 100;
    int minorVersionNumber = version % 100;
    return 
	(pat->eCategory == CATEGORY_NOT_SPECIFIED 
	 || pat->eCategory == eCategory) &&
	(pat->eFixingPriority == PRIORITY_NOT_SPECIFIED 
	 || pat->eFixingPriority == eFixingPriority) &&
	(pat->eSeverity == SEVERITY_NOT_SPECIFIED 
	 || pat->eSeverity == eSeverity) &&
	(pat->platform==NULL || sHardwarePlatform->index(pat->platform)!=-1) &&
	(pat->os == NULL || sOperatingSystem->index(pat->os) != -1) &&
	(pat->software==NULL || pSoftware->sName->index(pat->software) != -1)&&
	(pat->summary==NULL || sOneLineSummary->index(pat->summary) != -1) &&
	(pat->assignedTo == NULL 
	 || pAssignedTo->sName->index(pat->assignedTo) != -1) &&
	(pat->reportedBy == NULL 
	 || pReportedBy->sName->index(pat->reportedBy) != -1)
	&& majorVersionNumber >= pat->minMajorVersionNumber
	&& majorVersionNumber <= pat->maxMajorVersionNumber
	&& minorVersionNumber >= pat->minMinorVersionNumber
	&& minorVersionNumber <= pat->maxMinorVersionNumber;
}

void Bug::update(char const* summary, 
		 eCATEGORY category,
		 eFIXING_PRIORITY fixingPriority,
		 eSEVERITY severity,
		 char const* os,
		 char const* platform,
		 char const* similarBugId,
		 ref<Bug> similarBug)
{
    if (sOneLineSummary->compare(summary) != 0) { 
	sOneLineSummary = String::create(summary);
    }
    if (sHardwarePlatform->compare(platform) != 0) { 
	sHardwarePlatform = String::create(platform);
    }
    if (sOperatingSystem->compare(os) != 0) {
	sOperatingSystem = String::create(os);
    }
    eCategory = category;
    eFixingPriority = fixingPriority;
    eSeverity = severity;
    if (similarBug != NULL) { 
	setSimilarBugs.insertUnique(similarBugId, similarBug);
    }
}

Bug::Bug(int4 id, 
	 char const* summary, 
	 eCATEGORY category,
	 eFIXING_PRIORITY fixingPriority,
	 eSEVERITY severity,
	 char const* os,
	 char const* platform,
	 ref<Person> reportedByPerson,
	 ref<Software> soft,
	 ref<Version> version)
: SetMember(self_class)
{
    bugId = id;
    sOneLineSummary = String::create(summary);
    eCategory = category;
    eFixingPriority = fixingPriority;
    eSeverity = severity;
    sHardwarePlatform = String::create(platform);
    sOperatingSystem = String::create(os);
    pAssignedTo = NULL;
    pReportedBy = reportedByPerson;
    pVersion = version;
    pSoftware = soft;
    setReportHistory.initialize(this);
    setWorkArounds.initialize(this);
    setSimilarBugs.initialize(this);
    nReports = 0;
}

field_descriptor& Bug::describe_components()
{
    return FIELD(sOneLineSummary),FIELD(bugId),FIELD(eCategory),
	FIELD(eFixingPriority),FIELD(eSeverity),FIELD(pReportedBy),
	FIELD(pAssignedTo),
	FIELD(sHardwarePlatform),FIELD(sOperatingSystem), FIELD(pSoftware),
	FIELD(pVersion), FIELD(setReportHistory), FIELD(setWorkArounds),
	FIELD(setSimilarBugs),FIELD(nReports);
}

REGISTER(Bug, SetMember, pessimistic_scheme);


//---- BugTRACKING -------------------------------------

BugTrackingDB::BugTrackingDB(int vers) 
: object(self_class)
{ 
    version = vers;
    setAllPeople.initialize(this);
    setAllUserGroups.initialize(this);
    setAllBugs.initialize(this);
    setAllSoftware.initialize(this);
    setAllPeople.insert("administrator", new Engineer("administrator","",""));
}

field_descriptor& BugTrackingDB::describe_components()
{
    return FIELD(version), FIELD(setAllPeople), FIELD(setAllUserGroups),
	FIELD(setAllBugs), FIELD(setAllSoftware);
}


REGISTER(BugTrackingDB, object, pessimistic_scheme);

//----- RootObject ------------------------------------

void RootObject::initialize() const 
{ 
    if (is_abstract_root()) { 
	ref<RootObject> root = this;
	modify(root)->become(new RootObject(BUG_TRACKING_DB_VERSION));
    } 
}

boolean RootObject::addBug(char* bugId, ref<Bug> bug, ref<Person> user) const
{ 
    if (db->setAllBugs.insertUnique(bugId, bug)) { 
	bug->pSoftware->setBugs.insert(bugId, bug);
	user->setReportedBugs.insert(bugId, bug);    
	user->setGroups.apply(&UserGroup::addBug, &bug);
	return True;
    }
    return False;
}

void RootObject::assignBug(char const* bugId, ref<Bug> bug, 
			   ref<Engineer> engineer) const 
{
    if (engineer != NULL) { 
	engineer->setAssignedBugs.insertUnique(bugId, bug);
    }
    if (bug->pAssignedTo != NULL) { 
	bug->pAssignedTo->setAssignedBugs.erase(bugId);
    }
    modify(bug)->pAssignedTo = engineer;
}

void RootObject::deassignBug(char const* bugId, ref<Bug> bug, 
			     ref<Engineer> engineer) const 
{
    engineer->setAssignedBugs.erase(bugId);
    modify(bug)->pAssignedTo = NULL;
}

boolean RootObject::removePerson(char const* name) const
{
    ref<Person> person = db->setAllPeople.erase(name);
    if (person == NULL) { 
	return False;
    }
    person->setGroups.apply(&UserGroup::removeUser, name); 
    if (!person->isUser()) { 
	ref<Engineer> engineer = person;
	engineer->setAssignedBugs.apply(&Bug::deassign);
	engineer->setSoftware.apply(&Software::detachEngineerCallback, name);
    } else { 
	person->setSoftware.apply(&Software::unregisterUserCallback, name);
    }
    return True;
}

boolean RootObject::removeGroup(char const* group) const
{
    ref<UserGroup> pGroup = db->setAllUserGroups.erase(group);
    if (pGroup == NULL) { 
	return False;
    }
    pGroup->setUsers.apply(&Person::exclude, group);
    return True;
}

boolean RootObject::removeSoftware(char const* software) const
{
    ref<Software> pSoftware = db->setAllSoftware.find(software);
    if (pSoftware == NULL) { 
	return False;
    }
    pSoftware->setEngineers.apply(&Engineer::unregister, software);
    pSoftware->setUsers.apply(&Person::unregister, software);
    return True;
}

boolean RootObject::removeBug(char const* bug) const
{
    ref<Bug> pBug = db->setAllBugs.erase(bug);
    if (pBug == NULL) { 
	return False;
    }
    pBug->pReportedBy->setReportedBugs.erase(bug);
    pBug->pReportedBy->setGroups.apply(&UserGroup::removeBug, bug);
    if (pBug->pAssignedTo != NULL) { 
	pBug->pAssignedTo->setAssignedBugs.erase(bug);
    }
    return True;
}


field_descriptor& RootObject::describe_components()
{
    return FIELD(db),FIELD(nBugs);
}

REGISTER(RootObject, object, pessimistic_exclusive_scheme);


//--- HTML specific part -------------------------------------------

#define HTML_HEAD "Content-type: text/html\n\n\
<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\"><HTML><HEAD>"

#define EMPTY_LIST "<OPTION>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</OPTION></SELECT>"

void mainMenuReference(CGIrequest& req)
{
    char* myself = req.get("myself");
    if (myself != NULL) { 
	req << TAG 
	    << "<P><HR><CENTER><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() << "&page=userForm&myself=" 
	    << URL << myself <<  "&name=" << URL << myself
	    << "\">Back to main menu</A></CENTER>";
    }
    req << TAG << "</BODY></HTML>";
}

void error(CGIrequest& req, char const* msg)
{
    req << TAG << 
	HTML_HEAD "<TITLE>BUGDB error</TITLE></HEAD><BODY>"
	"<H1><FONT COLOR=\"#FF0000\">"
	<< msg << "</FONT></H1></BODY></HTML>";
    mainMenuReference(req);
}


void message(CGIrequest& req, char const* msg)
{
    req << TAG << 
	HTML_HEAD "<TITLE>BUGDB message</TITLE></HEAD><BODY>"
	"<H1><FONT COLOR=\"#004000\">"
	<< msg << "</FONT></H1></BODY></HTML>";
    mainMenuReference(req);
}


boolean addUserForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Enter new user</TITLE></HEAD>"
	"<BODY>"
	"<H2>Add user</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"addUser\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>User name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"name\" SIZE=30</TD></TR>"
	"<TR><TH ALIGN=LEFT>E-mail:</TH>"
	"<TD><INPUT TYPE=text NAME=\"email\" SIZE=30 </TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return True;
}

boolean addEngineerForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Enter new engineer</TITLE></HEAD>"
	"<BODY>"
	"<H2>Add engineer</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"addEngineer\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>Engineer name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"name\" SIZE=30</TD></TR>"
	"<TR><TH ALIGN=LEFT>E-mail:</TH>"
	"<TD><INPUT TYPE=text NAME=\"email\" SIZE=30 </TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return True;
}


boolean addGroupForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Enter new user group</TITLE></HEAD>"
	"<BODY>"
	"<H2>Add user group</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"addGroup\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>Group name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"group\" SIZE=30 </TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return True;
}

boolean addSoftwareForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Enter new software product</TITLE></HEAD>"
	"<BODY>"
	"<H2>Add software product</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"addSoftware\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>Software name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"software\" SIZE=40</TD></TR>"
	"<TR><TH ALIGN=LEFT>Version number:</TH>"
	"<TD><INPUT TYPE=text NAME=\"version\" SIZE=8</TD></TR>"
	"<TR><TH ALIGN=LEFT>Version label:</TH>"
	"<TD><INPUT TYPE=text NAME=\"label\" SIZE=20</TD></TR>"
	"<TR><TH ALIGN=LEFT>Version comment:</TH>"
	"<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40</TD></TR>"
	"</TABLE><P><INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return True;
}

boolean selectSoftwareForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Select software product</TITLE></HEAD>"
	"<BODY>"
	"<H2>Select software product</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"softwareForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"software\">";
    if (db->setAllSoftware.print(&req) > 0) { 
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean removeSoftwareForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Remove software product</TITLE></HEAD>"
	"<BODY>"
	"<H2>Remove software product</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"removeSoftware\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"software\">";
    if (db->setAllSoftware.print(&req) > 0) { 
	req << TAG << "</SELECT><BR><INPUT TYPE=\"submit\" VALUE=\"Remove\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean selectGroupForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Select user group</TITLE></HEAD>"
	"<BODY>"
	"<H2>Select user group</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"userGroupForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"group\">";
    if (db->setAllUserGroups.print(&req) > 0) { 
	req << TAG << "</SELECT><BR><INPUT TYPE=submit VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean removeGroupForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Remove user group</TITLE></HEAD>"
	"<BODY>"
	"<H2>Remove user group</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"removeGroup\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"group\">";
    if (db->setAllUserGroups.print(&req) > 0) { 
	req << TAG << "</SELECT><BR><INPUT TYPE=\"submit\" VALUE=\"Remove\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean selectPersonForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Select a person</TITLE></HEAD>"
	"<BODY>"
	"<H2>Select a person</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"userForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=20 NAME=\"name\">";
    if (db->setAllPeople.print(&req) > 0) { 
	req << TAG << "</SELECT><BR><INPUT TYPE=submit VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean removePersonForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Remove a person</TITLE></HEAD>"
	"<BODY>"
	"<H2>Remove a person</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"removePerson\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=20 NAME=\"name\">";
    if (db->setAllPeople.print(&req) > 0) { 
	req << TAG << "</SELECT><BR><INPUT TYPE=submit VALUE=\"Remove\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean selectBugForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Select a bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Select a bug</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"bug\">";
    if (db->setAllBugs.print(&req) > 0) { 
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean removeBugForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Remove a bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Remove a bug</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"removeBug\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"bug\">";
    if (db->setAllBugs.print(&req) > 0) { 
	req << TAG << "</SELECT><BR><INPUT TYPE=submit VALUE=\"Remove\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean changePasswordForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Change password</TITLE></HEAD>"
	"<BODY>"
	"<H2>Change password</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"changePassword\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << req.get("name") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>New password:</TH>"
	"<TD><INPUT TYPE=password NAME=\"password\" SIZE=20</TD></TR>"
	"<TR><TH ALIGN=LEFT>Re-type password:</TH>"
	"<TD><INPUT TYPE=password NAME=\"password2\" SIZE=20</TD></TR>"
	"</TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Change\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\">"
	"</FORM>";
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}
	
boolean shutdown(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>BUGDB message</TITLE></HEAD><BODY>"
	"<H1>BUGDB server is terminated</H1></BODY></HTML>";
    return False;
}

boolean userForm(CGIrequest& req);
boolean userGroupForm(CGIrequest& req);
boolean softwareForm(CGIrequest& req);

boolean addUser(CGIrequest& req)
{
    char* name = req.get("name");   
    ref<Person> person = new Person(name, req.get("email"), "");
    if (!db->setAllPeople.insertUnique(name, person)) {
	error(req, "Person already exists");
	return True;
    }
    return userForm(req);
}

boolean addEngineer(CGIrequest& req)
{
    char* name = req.get("name");   
    ref<Person> person = new Engineer(name, req.get("email"), "");
    if (!db->setAllPeople.insertUnique(name, person)) {
	error(req, "Person already exists");
    }
    return userForm(req);
}

boolean removePerson(CGIrequest& req)
{
    char* name = req.get("name");
    if (name == NULL) { 
	error(req, "No person was selected");
	return True;
    }
    if (!root->removePerson(name)) { 
	error(req, "No such person");
    } else { 
	message(req, "Person is removed");
    }
    return True;
}
 
boolean addGroup(CGIrequest& req)
{
    char* group = req.get("group");
    if (!db->setAllUserGroups.insertUnique(group, new UserGroup(group))) { 
	error(req, "Group already exists");
    } else { 
	return userGroupForm(req);
    }
    return True;
}
		
boolean removeGroup(CGIrequest& req)
{
    char* group = req.get("group");
    if (group == NULL) { 
	error(req, "No user group was selected");
	return True;
    }
    if (!root->removeGroup(group)) { 
	error(req, "No such group");
    } else { 
	message(req, "User group is removed");
    }
    return True;
}

boolean addSoftware(CGIrequest& req)
{
    char* software = req.get("software");
    int major, minor;
    char* versionStr = req.get("version");
    if (sscanf(versionStr, "%d.%d", &major, &minor) != 2) { 
	error(req, "Bad version number (MAJOR.MINOR expected)");
	return True;
    }  
    ref<Version> version = new Version(req.get("label"),
				       major, minor, req.get("comment"));
    if (db->setAllSoftware.insertUnique(software, 
					new Software(software, version))) 
    {
	req.addPair("action", "Select");
	return softwareForm(req);
    }
    error(req, "Software product already exists");
    return True;
}
		
boolean removeSoftware(CGIrequest& req)
{
    if (!root->removeSoftware(req.get("software"))) { 
	error(req, "No such software product");
    } else { 
	message(req, "Software product is removed");
    }
    return True;
}

boolean removeBug(CGIrequest& req)
{
    char* bug = req.get("bug");
    if (bug == NULL) { 
	error(req, "No bug was selected");
    } else { 
	if (!root->removeBug(bug)) { 
	    error(req, "No such bug");
	} else { 
	    message(req, "Bug is removed");
	}
    }
    return True;
}

boolean changePassword(CGIrequest& req)
{
    char* password = req.get("password");
    char* password2 = req.get("password2");
    if (strcmp(password, password2) != 0) { 
	error(req, "Passwords are not equal");
    } else { 
	ref<Person> person = db->setAllPeople.find(req.get("name"));
	if (person == NULL) {
	    error(req, "No such person");
	} else { 
	    modify(person)->changePassword(password);
	    message(req, "Password changed");
	}
    }
    return True;
}

boolean changeEmail(CGIrequest& req)
{
    char* email = req.get("email");
    ref<Person> person = db->setAllPeople.find(req.get("name"));
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    } else { 
	if (person->sEmailAddress->compare(email) != 0) { 
	    modify(person)->changeEmail(email);
	}
    }
    return userForm(req);
}

boolean login(CGIrequest& req)
{
    char* name = req.get("name");
    ref<Person> person = db->setAllPeople.find(name);
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    } 
    if (!person->checkPassword(req.get("password"))) { 
	error(req, "Incorrect password");
	return True;
    } 
    req.addPair("myself", name);    
    return userForm(req);
}

boolean bugQueryForm(CGIrequest& req)
{
    int i;
    req << TAG << 
	HTML_HEAD "<TITLE>Query to locate bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug query</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugQuery\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>Description substring:</TH>"
	"<TD><INPUT TYPE=text NAME=\"summary\" SIZE=30</TD></TR>"
	"<TR><TH ALIGN=LEFT>Category:</TH>"
	"<TD><SELECT NAME=\"category\" SIZE=1>"
	"<OPTION VALUE=0 SELECTED></OPTION>";
    for (i = 1; eCATEGORY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eCATEGORY_STRING[i] << "</OPTION>";
    }
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Severity:</TH>"
	"<TD><SELECT NAME=\"severity\" SIZE=1>"
	"<OPTION VALUE=0 SELECTED></OPTION>";
    for (i = 1; eSEVERITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eSEVERITY_STRING[i] << "</OPTION>";
    }
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Fixing priority:</TH>"
	"<TD><SELECT NAME=\"priority\" SIZE=1>"
	"<OPTION VALUE=0 SELECTED></OPTION>";
    for (i = 1; eFIXING_PRIORITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eFIXING_PRIORITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Platform:</TH>"
	"<TD><INPUT TYPE=text NAME=\"platform\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>OS</TH>"
	"<TD><INPUT TYPE=text NAME=\"os\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Software:</TH>"
	"<TD><INPUT TYPE=text NAME=\"software\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Assigned to:</TH>"
	"<TD><INPUT TYPE=text NAME=\"engineer\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Reported by:</TH>"
	"<TD><INPUT TYPE=text NAME=\"user\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Major version number:</TH>"
	"<TD>from <INPUT TYPE=text NAME=\"minmajor\" SIZE=4>"
	" to <INPUT TYPE=text NAME=\"maxmajor\" SIZE=4</TD></TR>"
	"<TR><TH ALIGN=LEFT>Minor version number:</TH>"
	"<TD>from <INPUT TYPE=text NAME=\"minminor\" SIZE=4</TD>"
	" to <INPUT TYPE=text NAME=\"maxminor\" SIZE=4</TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Search\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\">"
	"</FORM></BODY></HTML>";
    return True;
}


boolean bugQuery(CGIrequest& req) 
{
    char* p;
    Bug::SelectPattern pattern;
    pattern.eCategory = (eCATEGORY)atoi(req.get("category"));
    pattern.eSeverity = (eSEVERITY)atoi(req.get("severity"));
    pattern.eFixingPriority = (eFIXING_PRIORITY)atoi(req.get("priority"));
    pattern.os = req.get("os");
    if (*pattern.os == '\0') pattern.os = NULL;
    pattern.platform = req.get("platform");
    if (*pattern.platform == '\0') pattern.platform = NULL;
    pattern.software = req.get("software");
    if (*pattern.software == '\0') pattern.software = NULL;
    pattern.assignedTo = req.get("engineer");
    if (*pattern.assignedTo == '\0') pattern.assignedTo = NULL;
    pattern.reportedBy = req.get("user");
    if (*pattern.reportedBy == '\0') pattern.reportedBy = NULL;
    pattern.summary = req.get("summary");
    if (*pattern.summary == '\0') pattern.summary = NULL;
    p = req.get("minmajor");
    pattern.minMajorVersionNumber = (*p == '\0') ? 0 : atoi(p);
    p = req.get("maxmajor");
    pattern.maxMajorVersionNumber = (*p == '\0') ? INT_MAX : atoi(p);
    p = req.get("minminor");
    pattern.minMinorVersionNumber = (*p == '\0') ? 0 : atoi(p);
    p = req.get("maxminor");
    pattern.maxMinorVersionNumber = (*p == '\0') ? INT_MAX : atoi(p);
    req << TAG << 
	HTML_HEAD "<TITLE>List of selected bugs</TITLE></HEAD>"
	"<BODY>"
	"<H2>Selected bugs</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT NAME=\"bug\" SIZE=20>";
    if (db->setAllBugs.select(&pattern, &req) != 0) { 
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}


boolean userForm(CGIrequest& req)
{
    char* name = req.get("name");
    if (name == NULL) { 
	error(req, "No person was selected");
	return True;
    }
    char* myself = req.get("myself");
    if (strcmp(name, "administrator") == 0) { 
	req << TAG << 
	    HTML_HEAD "<TITLE>BUGDB Administrator</TITLE>"
	    "</HEAD><BODY>"
	    "<H2>Administrator menu</H2><FONT SIZE=\"+1\">"
	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() 
	    << "&page=addUserForm&myself=administrator\">Add user</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=addEngineerForm&myself=administrator\">Add engineer</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=selectPersonForm&myself=administrator\">Select person"
	    "</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=removePersonForm&myself=administrator\">"
	    "Remove person</A></UL>"

	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress()
	    << "&page=addGroupForm&myself=administrator\">Add user group</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=selectGroupForm&myself=administrator\">"
	    "Select user group</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=removeGroupForm&myself=administrator\">"
	    "Remove user group</A></UL>"

	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress()
	    << "&page=addSoftwareForm&myself=administrator\">"
	    "Add software product</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=selectSoftwareForm&myself=administrator\">"
	    "Select software product</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=removeSoftwareForm&myself=administrator\">"
	    "Remove software product</A></UL>"

	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress()
	    << "&page=selectBugForm&myself=administrator\">Select bug</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress()
	    << "&page=removeBugForm&myself=administrator\">Remove bug</A></UL>"

	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() << "&page=changePasswordForm"
	    "&myself=administrator&name=administrator\">Change password</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=shutdown\">Shutdown server</A></UL>"
	    "</FONT></BODY></HTML>";
	return True;
    }
    ref<Person> person = db->setAllPeople.find(name);
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    }
    req <<
	HTML_HEAD "<TITLE>" << name << "</TITLE></HEAD>"
	"<BODY><H2>" << name << "</H2><FONT SIZE=\"+1\"><UL>";
    if (db->setAllSoftware.size() != 0) {
	    req << TAG << "<LI><A HREF=\"" << req.getStub() << "?socket="
		<< req.getAddress() << "&page=createBugReportForm&myself=" 
		<< URL << myself << "\">Create bug report</A>";
    }
    if (person->sEmailAddress->length() != 0) { 
	req << TAG << 
	    "<LI><A HREF=\"mailto:" << person->sEmailAddress 
	    << "\">Send e-mail</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=bugQueryForm&myself=" << URL << myself 
	    << "\">Find a bug</A>";
    }
    if (strcmp(myself, name) == 0) { 
	req << TAG << "<LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() << "&page=changePasswordForm&myself=" 
	    << URL << myself << "&name=" << URL << name << 
	    "\">Change password</A>";
    }
    req << TAG << 
	"</UL></FONT><P><TABLE><TR><TH ALIGN=LEFT>E-Mail:</TH>"
	"<TD><FORM METHOD=POST ACTION=\""
	<< req.getStub() << "\"><INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\""
	"changeEmail\"><INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself
	<< "\"><INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name 
	<< "\"><INPUT TYPE=text NAME=\"email\" SIZE=30 VALUE=\"" 
	<< person->sEmailAddress << "\">"
	"<INPUT TYPE=submit VALUE=\"Change\"></FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>User groups:</TH><TD>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	"VALUE=\"userGroupForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << 
	"\"><SELECT NAME=\"group\" SIZE=1>";
    if (person->setGroups.print(&req) != 0) { 
	req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << 
	"</FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>Join group:</TH><TD>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	"VALUE=\"joinGroup\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	"<SELECT NAME=\"group\" SIZE=1>";
    if (db->setAllUserGroups.print(&req) != 0) { 
	req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Join\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM></TD></TR>";
    if (!person->isUser()) { 
	req << TAG << "<TR><TH ALIGN=LEFT>Projects:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"softwareForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"software\" SIZE=1>";
	if (person->setSoftware.print(&req) != 0) { 
	    req << TAG << 
		"</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Detach\">";
	} else { 
	    req << TAG << EMPTY_LIST;
	}
	req << TAG << 
	    "</FORM></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Attach to project:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"attachToProject\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"software\" SIZE=1>";
	if (db->setAllSoftware.print(&req) != 0) { 
	    req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Attach\">";
	} else { 
	    req << TAG << EMPTY_LIST;
	}
	req << TAG << 
	    "</FORM></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Find a person:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" 
	    << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"findPerson\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=text NAME=\"name\" SIZE=30>"
	    "<INPUT TYPE=submit VALUE=\"Find\"></FORM></TD></TR>";
    } else { 
	req << TAG << "<TR><TH ALIGN=LEFT>Registered software:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"softwareForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"software\" SIZE=1>";
	if (person->setSoftware.print(&req) != 0) { 
	    req << TAG << 
		"</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Unregister\">";
	} else { 
	    req << TAG << EMPTY_LIST;
	}
	req << TAG << 
	    "</FORM></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Register for new software:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"registerSoftware\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"software\" SIZE=1>";
	if (db->setAllSoftware.print(&req) != 0) { 
	    req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Register\">";
	} else { 
	    req << TAG << EMPTY_LIST;
	}
	req << TAG << "</FORM></TD></TR>";
    }
    req << TAG << 
	"</TABLE><P><B>Reported bugs:</B><BR>"	
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	"VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	"<SELECT NAME=\"bug\" SIZE=5>";
    if (person->setReportedBugs.print(&req) != 0) {
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }	
    req << TAG << "</FORM><P>";
    if (!person->isUser()) { 
	req << TAG << 
	    "<P><B>Assigned bugs:</B><BR>"	
	    "<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden "
	    "NAME=\"page\" VALUE=\"bugForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"bug\" SIZE=5>";
	ref<Engineer> engineer = person;
	if (engineer->setAssignedBugs.print(&req) != 0) {
	    req << TAG << 
		"</SELECT><BR>"
		"<INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">&nbsp;"
		"<INPUT TYPE=submit NAME=\"action\" VALUE=\"Deassign\">";
	} else { 
	    req << TAG << EMPTY_LIST;
	}
	req << TAG << "</FORM>";
    }
    if (strcmp(name, myself) == 0) { 
	req << TAG << "</BODY></HTML>";
    } else { 
	mainMenuReference(req);
    }
    return True;
}


boolean createBugReportForm(CGIrequest& req)
{
    int i;
    req << TAG << 
	HTML_HEAD "<TITLE>Bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"createBugReport\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\""<<req.get("myself")<<"\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << ++modify(root)->nBugs<< ">"
	"<TABLE><TH ALIGN=LEFT>Summary:</TH>"
	"<TD><INPUT TYPE=text NAME=\"summary\" SIZE=40></TD></TR>"
	"<TR><TH ALIGN=LEFT>Category:</TH>"
	"<TD><SELECT NAME=\"category\" SIZE=1>";
    for (i = 1; eCATEGORY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION SELECTED VALUE=" << i << ">"
	    << eCATEGORY_STRING[i] << "</OPTION>";
    }
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Severity:</TH>"
	"<TD><SELECT NAME=\"severity\" SIZE=1>";
    for (i = 1; eSEVERITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION SELECTED VALUE=" << i << ">"
	    << eSEVERITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Priority:</TH>"
	"<TD><SELECT NAME=\"priority\" SIZE=1>";
    for (i = 1; eFIXING_PRIORITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION SELECTED VALUE=" << i << ">"
	    << eFIXING_PRIORITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Software:</TH>"
	"<TD><SELECT NAME=\"software\" SIZE=1>";
    db->setAllSoftware.print(&req);
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Version:</TH>"
	"<TD><INPUT TYPE=text NAME=\"version\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>Platform:</TH>"
	"<TD><INPUT TYPE=text NAME=\"platform\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>OS:</TH>"
	"<TD><INPUT TYPE=text NAME=\"os\"></TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Submit\">&nbsp;<INPUT TYPE=reset></FORM>";
    mainMenuReference(req);
    return True;
}

boolean bugForm(CGIrequest& req);

boolean createBugReport(CGIrequest& req)
{
    ref<Person> myself = db->setAllPeople.find(req.get("myself"));
    if (myself == NULL) { 
	error(req, "Author unknown");
	return True;
    }
    ref<Software> soft = db->setAllSoftware.find(req.get("software"));
    if (soft == NULL) { 
	error(req, "No such software product");
	return True;
    }
    ref<Version> version = soft->setVersions.find(req.get("version"));
    if (version == NULL) { 
	error(req, "No such software product version");
	return True;
    }
 
    char* bugId = req.get("bug");
    ref<Bug> bug = new Bug(atoi(bugId),
			   req.get("summary"), 
			   (eCATEGORY)atoi(req.get("category")),
			   (eFIXING_PRIORITY)atoi(req.get("priority")),
			   (eSEVERITY)atoi(req.get("severity")),
			   req.get("os"), req.get("platform"),
			   myself, soft, version);
    root->addBug(bugId, bug, myself);
    req.addPair("action", "Select");
    return bugForm(req);
}

boolean bugForm(CGIrequest& req)
{
    int i;
    char* bugId = req.get("bug");
    if (bugId == NULL) { 
	error(req, "No bug was selected");
	return True;
    }
    char* myself = req.get("myself");
    if (strcmp(req.get("action"), "Remove") == 0) { 
	ref<Bug> bug = db->setAllBugs.find(req.get("relatedbug"));
	if (bug == NULL) { 
	    error(req, "No such bug");
	    return True;
	}
	bug->setSimilarBugs.erase(bugId);
	req.addPair("bug", bugId);
	return bugForm(req);
    }
    ref<Bug> bug = db->setAllBugs.find(bugId);
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    ref<Person> person = db->setAllPeople.find(myself);
    if (person == NULL) { 
	error(req, "No such user");
	return True;
    }
    if (strcmp(req.get("action"), "Remove") == 0) { 
	root->deassignBug(bugId, bug, person);
	req.addPair("name", myself);
	return userForm(req);
    }

    boolean IamUser = person->isUser();
    req << TAG <<
	HTML_HEAD "<TITLE>Bug in " << bug->pSoftware->sName << " v. "
	<< bug->pVersion->getVersionString() << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug in " << bug->pSoftware->sName << " v. "
	<< bug->pVersion->getVersionString() << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateBug\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=\"" << bugId << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<TABLE><TH ALIGN=LEFT>Summary:</TH>"
	"<TD><INPUT TYPE=text NAME=\"summary\" SIZE=40 VALUE=\"" 
	<< bug->sOneLineSummary << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>Category:</TH>"
	"<TD><SELECT NAME=\"category\" SIZE=1>"
	"<OPTION SELECTED VALUE=" << bug->eCategory << ">"
	<< eCATEGORY_STRING[bug->eCategory] << "</OPTION>";
    for (i = 1; eCATEGORY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" 
	    << eCATEGORY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Severity:</TH>"
	"<TD><SELECT NAME=\"severity\" SIZE=1>"
	"<OPTION SELECTED VALUE=" << bug->eSeverity << ">"
	<< eSEVERITY_STRING[bug->eSeverity] << "</OPTION>";
    for (i = 1; eSEVERITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION  VALUE=" << i << ">"
	    << eSEVERITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Priority:</TH>"
	"<TD><SELECT NAME=\"priority\" SIZE=1>"
	"<OPTION SELECTED VALUE=" << bug->eFixingPriority << ">"
	<< eFIXING_PRIORITY_STRING[bug->eFixingPriority] << "</OPTION>";
    for (i = 1; eFIXING_PRIORITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eFIXING_PRIORITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Platform:</TH>"
	"<TD><INPUT TYPE=text NAME=\"platform\" VALUE=\""
	<< bug->sHardwarePlatform << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>OS:</TH>"
	"<TD><INPUT TYPE=text NAME=\"os\"VALUE=\""
	<< bug->sOperatingSystem << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>Assigned to:</TH>"
	"<TD><SELECT SIZE=1 NAME=\"name\">";
    if (bug->pAssignedTo != NULL) { 
	req << TAG << "<OPTION SELECTED VALUE=\"" << bug->pAssignedTo->sName
	    << "\">" <<  bug->pAssignedTo->sName << "</OPTION>";
    } else { 
	req << TAG << "<OPTION SELECTED VALUE=\"\"></OPTION>";
    }
    bug->pSoftware->setEngineers.print(&req); 
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Similar with:</TH>"
	"<TD><SELECT NAME=\"similar\" SIZE=1>"
	"<OPTION SELECTED VALUE=\"\"></OPTION>";
    db->setAllBugs.print(&req);
    req << TAG << "</SELECT></TD></TR></TABLE><BR>";
    if (!IamUser) { 
	req << TAG << 
	    "<INPUT TYPE=submit NAME=\"action\" VALUE=\"Update\">&nbsp;"
	    "<INPUT TYPE=reset VALUE=\"Reset\">";
    }
    req << TAG << "</FORM><P><FORM METHOD=POST ACTION=\"" << req.getStub()
	<< "\"><INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\">"
	"<INPUT TYPE=hidden NAME=\"page\" VALUE=\"updateReportForm\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=\"" << bugId << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<B>Report history:</B><BR><SELECT NAME=\"report\" SIZE=5>";
    if (bug->setReportHistory.print(&req) != 0) { 
	req << TAG << 
	    "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
	    "&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Add\">";
	if (isAdministrator(myself)) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
    } else { 
	req << TAG << EMPTY_LIST 
	    "<BR><INPUT TYPE=submit  NAME=\"action\" VALUE=\"Add\">";
    }
    req << TAG << "</FORM><P>";

    req << TAG << 
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateWorkAroundForm\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=\"" << bugId << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<B>Work arounds:</B><BR><SELECT NAME=\"workaround\" SIZE=5>";
    if (bug->setWorkArounds.print(&req) != 0) { 
	req << TAG << 
	    "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
	    "&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Add\">";
	if (isAdministrator(myself)) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
    } else { 
	req << TAG << EMPTY_LIST 
	    "<BR><INPUT TYPE=submit  NAME=\"action\" VALUE=\"Add\">";
    }
    req << TAG << "</FORM><P>";    

    if (bug->setSimilarBugs.size() != 0) { 
	req << TAG << 
	    "<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden "
	    "NAME=\"page\" VALUE=\"bugForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"relatedbug\" VALUE=\"" << bugId << "\">"
	    "<B>Similar bugs:</B><BR><SELECT NAME=\"bug\" SIZE=1>";
	bug->setSimilarBugs.print(&req);
	req << TAG << 
	    "</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
	if (isAdministrator(myself)) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
	req << TAG << "</FORM><P>";
    }
    req << TAG << 
	"</FORM><P>"
	"<FONT SIZE=\"+1\"><UL>";
    if (IamUser) { 
	if (bug->pAssignedTo != NULL) { 
	    req << TAG << "<LI>Assigned to <A HREF=\"mailto:"  
		<< bug->pAssignedTo->sEmailAddress << "\">" 
		<< bug->pAssignedTo->sName << "</A>";
	}
	req << TAG << "<LI>Reported by <A HREF=\"mailto:"
	    << bug->pReportedBy->sEmailAddress << "\">" 
	    << bug->pReportedBy->sName << "</A></OL></FONT>";
    } else { 
	if (bug->pAssignedTo != NULL) { 
	    req << TAG << "<LI>Assigned to <A HREF=\"" << req.getStub() 
		<< "?socket=" << req.getAddress() 
		<< "&page=userForm&myself=" << URL << myself 
		<< "&name=" << URL << bug->pAssignedTo->sName << "\">" 
		<< bug->pAssignedTo->sName << "</A>";
	}
	req << TAG 
	    << "<LI>Reported by <A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() 
	    << "&page=userForm&myself=" << URL << myself 
	    << "&name=" << URL << bug->pReportedBy->sName << "\">" 
	    << bug->pReportedBy->sName << "</A></OL></FONT>";
    }
    mainMenuReference(req);
    return True;
}

boolean updateBug(CGIrequest& req)
{
    char* bugId = req.get("bug");
    ref<Bug> bug = db->setAllBugs.find(bugId);
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    } 
    char* similar = req.get("similar");
    ref<Bug> similarBug;
    if (*similar != '\0') { 
	similarBug = db->setAllBugs.find(similar);
    }
    char* name = req.get("name");
    if ((bug->pAssignedTo == NULL && *name != 0) ||
	(bug->pAssignedTo != NULL 
	 && bug->pAssignedTo->sName->compare(name) != 0))
    {
	ref<Person> engineer = NULL;
	if (*name != '\0') { 
	    engineer = db->setAllPeople.find(name);
	    if (engineer == NULL || engineer->isUser()) { 
		error(req, "No such engineer");
		return True;
	    }
	}
	root->assignBug(bugId, bug, engineer);
    }
    modify(bug)->update(req.get("summary"), 
			(eCATEGORY)atoi(req.get("category")),
			(eFIXING_PRIORITY)atoi(req.get("priority")),
			(eSEVERITY)atoi(req.get("severity")),
			req.get("os"),
			req.get("platform"), 
			similar, similarBug);
    return bugForm(req);
}


boolean addReportForm(CGIrequest& req)
{
    char* bugId = req.get("bug");
    ref<Bug> bug = db->setAllBugs.find(bugId);
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>Bug report</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug report</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"addReport\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=\"" << bugId << "\">"
	"<INPUT TYPE=hidden NAME=\"index\" VALUE=" << ++modify(bug)->nReports
	<< "><INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself")
	<< "\"><B>Status: &nbsp;</B><SELECT SIZE=1 NAME=\"status\">";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG << 
	"</SELECT><P>"
	"<B>Bug description:</B><P>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\"></TEXTAREA><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return True;
}

boolean addReport(CGIrequest& req)
{
    ref<Bug> bug = db->setAllBugs.find(req.get("bug"));
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    ref<Person> author = db->setAllPeople.find(req.get("myself"));
    char* index = req.get("index");
    ref<Report> report = new Report(atoi(index),
				    req.get("description"), author, 
				    (eSTATUS)atoi(req.get("status"))); 
    bug->setReportHistory.insertUnique(index, report);
    req.addPair("action", "Select");
    return bugForm(req);
}

boolean addWorkAroundForm(CGIrequest& req)
{
    char* bugId = req.get("bug");
    ref<Bug> bug = db->setAllBugs.find(bugId);
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>Work around</TITLE></HEAD>"
	"<BODY>"
	"<H2>Work around</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"addWorkAround\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=\"" << bugId << "\">"
	"<INPUT TYPE=hidden NAME=\"index\" VALUE=" << ++modify(bug)->nReports
	<< "><INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself")
	<< "\"><B>Status: &nbsp;</B><SELECT SIZE=1 NAME=\"status\">";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG << 
	"</SELECT><P>"
	"<B>Description:</B><P>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\"></TEXTAREA><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return True;
}

boolean addWorkAround(CGIrequest& req)
{
    ref<Bug> bug = db->setAllBugs.find(req.get("bug"));
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    ref<Person> author = db->setAllPeople.find(req.get("myself"));
    char* index = req.get("index");
    ref<Report> report = new Report(atoi(index),
				    req.get("description"), author, 
				    (eSTATUS)atoi(req.get("status"))); 
    bug->setWorkArounds.insertUnique(index, report);
    req.addPair("action", "Select");
    return bugForm(req);
}

boolean updateReportForm(CGIrequest& req)
{
    if (strcmp(req.get("action"), "Add") == 0) { 
	return addReportForm(req);
    }
    char* bugId = req.get("bug");
    ref<Bug> bug = db->setAllBugs.find(bugId);
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    char* report = req.get("report");
    if (report == NULL) { 
	error(req, "No report was selected");
	return True;
    }
    if (strcmp(req.get("action"), "Remove") == 0) { 
	if (bug->setReportHistory.erase(report) == NULL) { 
	    error(req, "No such report");
	    return True;
	} 
	req.addPair("action", "Select");
	return bugForm(req);
    }
    ref<Report> pReport = bug->setReportHistory.find(report);
    if (pReport == NULL) { 
	error(req, "No such report");
	return True;
    } 
    char* date = pReport->creationDate.asString();
    char* myself = req.get("myself");
    ref<Person> person = db->setAllPeople.find(myself);
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>Bug report from " << date << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug report from " << date << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateReport\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=\"" << bugId << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><INPUT TYPE=hidden NAME=\"report\" VALUE=" << pReport->index << ">"
	"<B>Created by <A HREF=\"" << req.getStub() << "?socket=" 
	<< req.getAddress() 
	<< "&page=userForm&myself=" << URL << myself 
	<< "&name=" << URL << pReport->pAuthor->sName << "\">" 
	<< pReport->pAuthor->sName << "</A></B><P>"
	"<B>Status: </B><SELECT SIZE=1 NAME=\"status\">"
	"<OPTION SELECTED VALUE=" << pReport->status << ">" 
	<< eSTATUS_STRING[pReport->status] << "</OPTION>";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG <<
	"</SELECT><P>"
	"<B>Bug description:</B><BR>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\">"
	<< pReport->sDescription << "</TEXTAREA><P>";
    if (!person->isUser()) { 
	req << TAG << 
	    "<INPUT TYPE=submit VALUE=\"Update\">&nbsp;"
	    "<INPUT TYPE=reset VALUE=\"Reset\">";
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean updateWorkAroundForm(CGIrequest& req)
{
    if (strcmp(req.get("action"), "Add") == 0) { 
	return addWorkAroundForm(req);
    }
    char* bugId = req.get("bug");
    ref<Bug> bug = db->setAllBugs.find(bugId);
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    char* workaround = req.get("workaround");
    if (workaround == NULL) { 
	error(req, "No report was selected");
	return True;
    }
    if (strcmp(req.get("action"), "Remove") == 0) { 
	if (bug->setWorkArounds.erase(workaround) == NULL) { 
	    error(req, "No such work around");
	    return True;
	} 
	req.addPair("action", "Select");
	return bugForm(req);
    }
    ref<Report> report = bug->setWorkArounds.find(workaround);
    if (report == NULL) { 
	error(req, "No such work around");
	return True;
    } 
    char* date = report->creationDate.asString();
    char* myself = req.get("myself");
    ref<Person> person = db->setAllPeople.find(myself);
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>Work around " << date << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>Work around " << date << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateWorkAround\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=\"" << bugId << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><INPUT TYPE=hidden NAME=\"workaround\" VALUE=" << report->index <<
	"><B>Created by <A HREF=\"" << req.getStub() << "?socket=" 
	<< req.getAddress() 
	<< "&page=userForm&myself=" << URL << myself 
	<< "&name=" << URL << report->pAuthor->sName << "\">" 
	<< report->pAuthor->sName << "</A></B><P>"
	"<B>Status: </B><SELECT SIZE=1 NAME=\"status\">"
	"<OPTION SELECTED VALUE=" << report->status << ">" 
	<< eSTATUS_STRING[report->status] << "</OPTION>";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG << 
	"</SELECT><P>"
	"<B>Bug description:</B><BR>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\">"
	<< report->sDescription << "</TEXTAREA><P>";
    if (!person->isUser()) { 
	req << TAG << 
	    "<INPUT TYPE=submit VALUE=\"Update\">&nbsp;"
	    "<INPUT TYPE=reset VALUE=\"Reset\">";
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean updateReport(CGIrequest& req)
{
    ref<Bug> bug = db->setAllBugs.find(req.get("bug"));
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    ref<Report> report = bug->setReportHistory.find(req.get("report"));
    if (report == NULL) { 
	error(req, "No such report");
	return True;
    }
    modify(report)->update(req.get("description"), 
			   (eSTATUS)atoi(req.get("status")));
    req.addPair("action", "Select");
    return bugForm(req);
}

boolean updateWorkAround(CGIrequest& req)
{
    ref<Bug> bug = db->setAllBugs.find(req.get("bug"));
    if (bug == NULL) { 
	error(req, "No such bug");
	return True;
    }
    ref<Report> report = bug->setWorkArounds.find(req.get("workaround"));
    if (report == NULL) { 
	error(req, "No such workaround");
	return True;
    }
    modify(report)->update(req.get("description"), 
			   (eSTATUS)atoi(req.get("status")));
    req.addPair("action", "Select");
    return bugForm(req);
}


boolean attachToProject(CGIrequest& req)
{
    char* name = req.get("name");
    ref<Person> person = db->setAllPeople.find(name);
    if (person == NULL || person->isUser()) { 
	error(req, "No such engineer");
    } else { 
	ref<Software> soft = db->setAllSoftware.find(req.get("software"));
	if (soft == NULL) { 
	    error(req, "No such software product");
	} else { 
	    if (!soft->attachEngineer(name, person)) { 
		error(req, "Engineer already attached to the project");
	    } else { 
		return userForm(req);
	    }
	}
    }
    return True;
}


boolean registerSoftware(CGIrequest& req)
{
    char* name = req.get("name");
    ref<Person> person = db->setAllPeople.find(name);
    if (person == NULL) { 
	error(req, "No such person");
    } else { 
	ref<Software> soft = db->setAllSoftware.find(req.get("software"));
	if (soft == NULL) { 
	    error(req, "No such software product");
	} else { 
	    if (!soft->registerUser(name, person)) { 
		error(req, "User already registered");
	    } else { 
		return userForm(req);
	    }
	}
    }
    return True;
}


boolean joinGroup(CGIrequest& req)
{
    char* groupName = req.get("group");
    ref<UserGroup> group = db->setAllUserGroups.find(groupName);
    if (group == NULL) { 
	error(req, "No such group");
    } else { 
	char* name = req.get("name");
	ref<Person> person = db->setAllPeople.find(name);
	if (person == NULL) { 
	    error(req, "No such person");
	} else {
	    if (!group->addUser(groupName, name, person)) { 
		error(req, "Person is already in group");
	    } else { 
		return userForm(req);
	    }
	}
    }
    return True;
}

boolean findPerson(CGIrequest& req)
{
    ref<Person> person = db->setAllPeople.find(req.get("name"));
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    } 
    return userForm(req);
}

boolean softwareForm(CGIrequest& req)
{
    char* software = req.get("software");
    if (software == NULL) { 
	error(req, "No software product was selected");
	return True;
    }
    ref<Software> soft = db->setAllSoftware.find(software);
    if (soft == NULL) { 
	error(req, "No such software product");
	return True;
    }
    if (strcmp(req.get("action"), "Detach") == 0) { 
	if (!soft->detachEngineer(req.get("name"))) {  
	    error(req, "No such person");
	    return True;
	}
	return userForm(req);
    }
    if (strcmp(req.get("action"), "Unregister") == 0) { 
	if (!soft->unregisterUser(req.get("name"))) {  
	    error(req, "No such person");
	    return True;
	}
	return userForm(req);
    }
    char* myself = req.get("myself");
    ref<Person> person = db->setAllPeople.find(myself);
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    }
    boolean IamUser = person->isUser();
    req << TAG << 
	HTML_HEAD "<TITLE>" << software << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>" << software << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateSoftware\">"
	"<INPUT TYPE=hidden NAME=\"software\" VALUE=\"" << software << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><TABLE><TR><TH ALIGN=LEFT>Manager:</TH>"
	"<TD><INPUT TYPE=text NAME=\"manager\" SIZE=30 VALUE=\"";
    ref<Person> manager = soft->pManager;
    if (manager != NULL) { 
	req << manager;
    }
    req << TAG << "\"></TD></TR>";
    if (!soft->setVersions.empty()) { 
	ref<Version> lastVersion = soft->setVersions.members->last->obj;
	req << TAG << 
	    "<TR><TH ALIGN=LEFT>Current version:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"version\" SIZE=8 VALUE=\"" 
	    << lastVersion->getVersionString() << "\"></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version label:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"label\" SIZE=20 VALUE=\""
	    << lastVersion->sLabel << "\"></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version comment:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40 VALUE=\""
	    << lastVersion->sComment << "\"></TD></TR>";
    } else { 
	req << TAG << 
	    "<TR><TH ALIGN=LEFT>Current version:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"version\" SIZE=8></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version label:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"label\" SIZE=20></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version comment:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40></TD></TR>";
    }
    req << TAG << "</TABLE><BR>";
    if (!IamUser) { 
	req << TAG << 
	    "<INPUT TYPE=submit VALUE=\"Update\">&nbsp;<INPUT TYPE=reset>";
    }
    req << TAG << "</FORM><P>"
	"<TABLE><TR><TH ALIGN=LEFT>Engineers:</TH>"
	"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress()
	<< "\"><INPUT TYPE=HIDDEN NAME=\"page\" VALUE=\"userForm\">"
	"<INPUT TYPE=HIDDEN NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"name\" SIZE=1>";
    if (soft->setEngineers.print(&req) != 0) { 	
	req << TAG << "</SELECT>";
	if (!IamUser) { 
	    req << TAG << "<INPUT TYPE=submit VALUE=\"Select\">";
	}
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG <<
	"</FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>Users:</TH>"
	"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() 
	<< "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"userForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"name\" SIZE=1>";
    if (soft->setUsers.print(&req) != 0) { 
	req << TAG << "</SELECT>";
	if (!IamUser) { 
	    req << TAG << "<INPUT TYPE=submit VALUE=\"Select\">";
	}
    } else { 
	req << TAG << EMPTY_LIST;
    }    
    req << TAG << 
	"</FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>Bugs:</TH>"
	"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() 
	<< "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"bug\" SIZE=1>";
    if (soft->setBugs.print(&req) != 0) { 
	req << TAG << 
	    "</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << 
	"</FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>Versions:</TH><TD>"
	"<FORM METHOD=POST ACTION=\""<<req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() 
	<< "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"versionForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><INPUT TYPE=HIDDEN NAME=\"software\" VALUE=\"" << software << 
	"\"><SELECT NAME=\"version\" SIZE=1>";   
    if (soft->setVersions.print(&req) != 0) { 
	req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM></TD></TR></TABLE>";
    mainMenuReference(req);
    return True;
}

boolean updateSoftware(CGIrequest& req) 
{
    ref<Software> soft = db->setAllSoftware.find(req.get("software"));
    if (soft == NULL) { 
	error(req, "No such software product");
	return True;
    }
    char* manager = req.get("manager");
    if (soft->pManager == NULL || soft->pManager->sName->compare(manager)!=0) {
	if (*manager != '\0') { 
	    ref<Person> person = db->setAllPeople.find(manager);
	    if (person == NULL) { 
		error(req, "No such person");
		return True;
	    }
	    modify(soft)->pManager = person; 
	} else if (soft->pManager != NULL) { 
	    modify(soft)->pManager = NULL; 
	}
    }
    char* lastVersion = soft->getLastVersionString();
    char* currentVersion = req.get("version");
    char* label = req.get("label");
    char* comment = req.get("comment");
    if (strcmp(lastVersion, currentVersion) != 0) { 
	int major, minor;
	if (sscanf(currentVersion, "%d.%d", &major, &minor) != 2) { 
	    error(req, "Bad version number (MAJOR.MINOR expected)");
	    return True;
	}  
	if (!soft->setVersions.insertUnique(currentVersion, 
					    new Version(label, major, minor, 
							comment)))
	{
	    error(req, "Version already exist");
	    return True;
	}
    } else if (*currentVersion != '\0') { 
	ref<Version> pLastVersion = soft->setVersions.members->last->obj;
	if (pLastVersion->sComment->compare(comment) != 0) { 
	    modify(pLastVersion)->sComment = String::create(comment);
	}
	if (pLastVersion->sLabel->compare(label) != 0) { 
	    modify(pLastVersion)->sLabel = String::create(label);
	}
    }
    req.addPair("name", req.get("myself"));
    return userForm(req);
}

boolean userGroupForm(CGIrequest& req) 
{
    char* groupName = req.get("group");
    if (groupName == NULL) { 
	error(req, "No user group was selected");
	return True;
    }
    ref<UserGroup> group = db->setAllUserGroups.find(groupName);
    if (group == NULL) { 
	error(req, "No such user group");
	return True;
    }
    char* myself = req.get("myself");
    ref<Person> person = db->setAllPeople.find(myself);
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>User group " << groupName << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>User group " << groupName << "</H2>"
	"<B>Users:</B><BR>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"userForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"name\" SIZE=5>";
    if (group->setUsers.print(&req) != 0) { 
	req << TAG << "</SELECT>";
	if (!person->isUser()) { 
	    req << TAG << "<BR><INPUT TYPE=submit VALUE=\"Select\">";
	}
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << 
	"</FORM><P><B>Bugs:</B><BR>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"bug\" SIZE=5>";
    if (group->setBugs.print(&req) != 0) { 
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean versionForm(CGIrequest& req) 
{
    char* software = req.get("software");
    char* myself = req.get("myself");
    ref<Software> soft = db->setAllSoftware.find(software);
    if (soft == NULL) { 
	error(req, "No such software product");
	return True;
    }
    char* versionString = req.get("version");
    ref<Version> version = soft->setVersions.find(versionString);
    if (version == NULL) { 
	error(req, "No such version");
	return True;
    }
    ref<Person> person = db->setAllPeople.find(myself);
    if (person == NULL) { 
	error(req, "No such person");
	return True;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>" << software << " v. " << versionString << 
	"</TITLE></HEAD>"
	"<BODY>"
	"<H2>"  << software << " v. " << versionString << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateVersion\">"
	"<INPUT TYPE=hidden NAME=\"software\" VALUE=\"" << software << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<INPUT TYPE=hidden NAME=\"version\" VALUE=\"" << versionString <<"\">"
	"<TABLE><TR><TH ALIGN=LEFT>Released:</TH>"
	"<TD>" << version->released.asString() << "</TD></TR>"
	"<TR><TH ALIGN=LEFT>Label:</TH>"
	"<TD><INPUT TYPE=text NAME=\"label\" SIZE=20 VALUE=\""
	<< version->sLabel << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>Comment:</TH>"
	"<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40 VALUE=\"" 
	<< version->sComment << "\"></TD></TR>";
    if (!person->isUser()) { 
	req << TAG << 
	    "</TABLE><P><INPUT TYPE=submit NAME=\"action\" VALUE=\"Update\">";
	if (isAdministrator(myself)) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
	req << TAG << "&nbsp;<INPUT TYPE=reset>";
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return True;
}

boolean updateVersion(CGIrequest& req) 
{
    char* software = req.get("software");
    ref<Software> soft = db->setAllSoftware.find(software);
    if (soft == NULL) { 
	error(req, "No such software product");
	return True;
    }
    if (strcmp(req.get("action"), "Remove") == 0) { 
	if (soft->setVersions.erase(req.get("version")) == NULL) { 
	    error(req, "No such version");
	    return True;
	}
	req.addPair("action", "Select");
	return softwareForm(req);
    }    
    ref<Version> version = soft->setVersions.find(req.get("version"));
    if (version == NULL) { 
	error(req, "No such version");
	return True;
    }
    char* comment = req.get("comment");
    char* label = req.get("label");
    if (version->sComment->compare(comment) != 0) { 
	modify(version)->sComment = String::create(comment);
    }
    if (version->sLabel->compare(label) != 0) { 
	modify(version)->sLabel = String::create(label);
    }    
    return versionForm(req);
}

CGIapi::dispatcher dispatchTable[] = { 
    {"addUserForm", addUserForm},
    {"addEngineerForm", addEngineerForm},
    {"addGroupForm", addGroupForm},
    {"addSoftwareForm", addSoftwareForm},
    {"selectSoftwareForm", selectSoftwareForm},
    {"removeSoftwareForm", removeSoftwareForm},
    {"selectGroupForm", selectGroupForm},
    {"removeGroupForm", removeGroupForm},
    {"selectPersonForm", selectPersonForm},
    {"removePersonForm", removePersonForm},
    {"selectBugForm", selectBugForm},
    {"removeBugForm", removeBugForm},
    {"changePasswordForm", changePasswordForm},
    {"shutdown", shutdown},
    {"userForm", userForm},
    {"userGroupForm", userGroupForm},
    {"softwareForm", softwareForm},
    {"addUser", addUser},
    {"addEngineer", addEngineer},
    {"removePerson", removePerson},
    {"addGroup", addGroup},
    {"removeGroup", removeGroup},
    {"addSoftware", addSoftware},
    {"removeSoftware", removeSoftware},
    {"removeBug", removeBug},
    {"changePassword", changePassword},
    {"changeEmail", changeEmail},
    {"login", login},
    {"bugQueryForm", bugQueryForm},
    {"bugQuery", bugQuery},
    {"userForm", userForm},
    {"createBugReportForm", createBugReportForm},
    {"bugForm", bugForm},
    {"createBugReport", createBugReport},
    {"bugForm", bugForm},
    {"updateBug", updateBug},
    {"updateReportForm", updateReportForm},
    {"updateWorkAroundForm", updateWorkAroundForm},
    {"addReportForm", addReportForm},
    {"addReport", addReport},
    {"addWorkAroundForm", addWorkAroundForm},
    {"addWorkAround", addWorkAround},
    {"updateReport", updateReport},
    {"updateWorkAround", updateWorkAround},
    {"attachToProject", attachToProject},
    {"registerSoftware", registerSoftware},
    {"joinGroup", joinGroup},
    {"findPerson", findPerson},
    {"softwareForm", softwareForm},
    {"updateSoftware", updateSoftware},
    {"userGroupForm", userGroupForm},
    {"versionForm", versionForm},
    {"updateVersion", updateVersion}
}; 
    
CGIapi wwwServer(items(dispatchTable), dispatchTable);

int main(int argc, char* argv[])
{
    task::initialize(task::huge_stack);
    char* address = "localhost:6101";
    if (argc > 1) { 
	address = argv[1];
    }
    if (!wwwServer.open(address)) { 
	console::output("Failed to create socket for WWW server\n");
	return EXIT_FAILURE;
    }
    if (!BugDB.open("bugdb.cfg")) { 
	console::output("Failed to open database\n");
	return EXIT_FAILURE;
    }
    CGIrequest req;
    
    BugDB.get_root(root);
    root->initialize();
    db = root->db;
    
    while (wwwServer.get(req) && req.dispatch());
    BugDB.close();	
    console::output("End of session\n");
    return EXIT_SUCCESS;
}
