//-< BUGDB.CPP  >----------------------------------------------------*--------*
// GigaBASE                  Version 1.0         (c) 1999  GARRET    *     ?  *
// (Post Relational Database Management System)                      *   /\|  *
//                                                                   *  /  \  *
//                          Created:     27-Mar-99    K.A. Knizhnik  * / [] \ *
//                          Last update: 11-Apr-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Example of database Web publishing: Bug Tracking Database
//-------------------------------------------------------------------*--------*

#include "bugdb.h"

char const* const eCATEGORY_STRING[] = 
{
    "",
    "CRASH",
    "PROGRAM_HANGS",
    "UI_DISPLAY",
    "UI_BEHAVIOR",
    "CALCULATION",
    "ERROR_HANDLING",
    "PERFORMANCE",
    "LICENSING",
    "INSTALLATION", 
    "DOCUMENTATION",
    "ENHANCEMENT",
    "HOW_TO_QUESTION",
    NULL
};

char const* const eSTATUS_STRING[] = { 
    "",
    "OPENED",
    "FIXED",
    "CLOSED",
    "PENDING_ENGINEER",
    "PENDING_USER",
    "POSTPONED",
    "IRREPRODUCIBLE",
    "WITHDRAWN",
    "AS_DESIGNED",
    NULL
};

char const* const eSEVERITY_STRING[] = { 
    "",
    "FATAL",
    "SERIOUS",
    "MINOR",
    NULL
};

char const* const eFIXING_PRIORITY_STRING[] = { 
    "",
    "FIX_IMMEDIATELY",
    "FIX_BEFORE_NEXT_BUILD_RELEASE",
    "FIX_BEFORE_NEXT_MINOR_RELEASE",
    "FIX_BEFORE_NEXT_MAJOR_RELEASE",
    "FIX_IF_POSSIBLE",
    "OPTIONAL",
    NULL
};

dbDatabase db;

dbCursor<Bug>          allBugs;
dbCursor<Bug>          bugs(dbCursorForUpdate);
dbCursor<Report>       reports(dbCursorForUpdate);
dbCursor<Person>       persons(dbCursorForUpdate);
dbCursor<Software>     products(dbCursorForUpdate);
dbCursor<Version>      versions(dbCursorForUpdate);
dbCursor<BugSequencer> sequencer(dbCursorForUpdate);

dbQuery qBug;
dbQuery qReport;
dbQuery qAllReports;
dbQuery qVersion;
dbQuery qAllVersions;
dbQuery qPerson;
dbQuery qSoftware;

//
// Query paramters
//
char* key;
int   bugId; 
int   reportId;
dbReference<Report>  firstReport;
dbReference<Version> initialVersion;
int   majorVersion;
int   minorVersion;


//- Person ------------------------------------------------

void Person::print(CGIrequest& req) const 
{ 
    req << TAG << "<OPTION VALUE=\"" << sName << "\">" << sName << "</OPTION>";
}
    
REGISTER(Person);

//------- Version ---------------------------------------

void Version::print(CGIrequest& req) const 
{ 
    char buf[64];
    req << TAG << "<OPTION VALUE=\"" << getVersionString() << 
	"\">Version " << getVersionString() << " " << sLabel << 
	" " << released.toString(buf, sizeof buf) << "</OPTION>";
}

int Version::getVersion() const 
{ 
    return majorVersionNumber*100 + minorVersionNumber; 
}

char* Version::getVersionString() const 
{
    static char buf[16];
    sprintf(buf, "%d.%02d", majorVersionNumber, minorVersionNumber);
    return buf;
}

REGISTER(Version);

//----- Software -------------------------------------

int Software::getLastVersion() const 
{ 
    if (pVersions == null) { 
	return 0;
    }
    versions.at(pVersions);
    return versions->getVersion();
}

char* Software::getLastVersionString() const 
{ 
    if (pVersions == null) { 
	return "";
    }
    versions.at(pVersions);
    return versions->getVersionString();
}

void Software::print(CGIrequest& req) const 
{ 
    req << TAG << "<OPTION VALUE=\"" << sName << "\">" << sName << "</OPTION>";
}

REGISTER(Software);

//----- Report -------------------------------------------

void Report::print(CGIrequest& req) const 
{ 
    char buf[64];
    if (pAuthor != null) { 
	persons.at(pAuthor);
	req << TAG << "<OPTION VALUE=" << index << ">" << persons->sName << " "
	    << creationDate.toString(buf, sizeof buf) << "</OPTION>";
    } else { 
	req << TAG << "<OPTION VALUE=" << index << ">" << "Report from "
	    << creationDate.toString(buf, sizeof buf) << "</OPTION>";
    }
}

REGISTER(Report);

//--- Bug -----------------------------------------

void Bug::print(CGIrequest& req) const 
{
    req << TAG << "<OPTION VALUE=" << bugId << ">" << sOneLineSummary 
	<< "</OPTION>";
}
    
REGISTER(Bug);

//---- BugSequencer -------------------------------------

REGISTER(BugSequencer);

template<class T>
void print(CGIrequest& req, dbCursor<T>& cursor) {
    do { 
	cursor->print(req);
    } while(cursor.next());
}

template<class T>
void print(CGIrequest& req, dbArray<dbReference<T> > const& arr) {
    dbCursor<T> cursor;
    for (int i = 0, n = arr.length(); i < n; i++) { 
	cursor.at(arr[i])->print(req);
    } 
}

//--- HTML specific part -------------------------------------------

#define HTML_HEAD "Content-type: text/html\n\n\
<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\"><HTML><HEAD>"

#define EMPTY_LIST "<OPTION>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</OPTION></SELECT>"

void mainMenuReference(CGIrequest& req)
{
    char* myself = req.get("myself");
    if (myself != NULL) { 
	req << TAG 
	    << "<P><HR><CENTER><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() << "&page=userForm&myself=" 
	    << URL << myself <<  "&name=" << URL << myself
	    << "\">Back to main menu</A></CENTER>";
    }
    req << TAG << "</BODY></HTML>";
}

void error(CGIrequest& req, char const* msg)
{
    req << TAG << 
	HTML_HEAD "<TITLE>BUGDB error</TITLE></HEAD><BODY>"
	"<H1><FONT COLOR=\"#FF0000\">"
	<< msg << "</FONT></H1></BODY></HTML>";
    mainMenuReference(req);
}


void message(CGIrequest& req, char const* msg)
{
    req << TAG << 
	HTML_HEAD "<TITLE>BUGDB message</TITLE></HEAD><BODY>"
	"<H1><FONT COLOR=\"#004000\">"
	<< msg << "</FONT></H1></BODY></HTML>";
    mainMenuReference(req);
}


bool addUserForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Enter new user</TITLE></HEAD>"
	"<BODY>"
	"<H2>Add user</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"addUser\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>User name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"name\" SIZE=30</TD></TR>"
	"<TR><TH ALIGN=LEFT>E-mail:</TH>"
	"<TD><INPUT TYPE=text NAME=\"email\" SIZE=30 </TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return true;
}

bool addEngineerForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Enter new engineer</TITLE></HEAD>"
	"<BODY>"
	"<H2>Add engineer</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"addEngineer\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>Engineer name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"name\" SIZE=30</TD></TR>"
	"<TR><TH ALIGN=LEFT>E-mail:</TH>"
	"<TD><INPUT TYPE=text NAME=\"email\" SIZE=30 </TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return true;
}

bool addSoftwareForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Enter new software product</TITLE></HEAD>"
	"<BODY>"
	"<H2>Add software product</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"addSoftware\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>Software name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"software\" SIZE=40</TD></TR>"
	"<TR><TH ALIGN=LEFT>Version number:</TH>"
	"<TD><INPUT TYPE=text NAME=\"version\" SIZE=8</TD></TR>"
	"<TR><TH ALIGN=LEFT>Version label:</TH>"
	"<TD><INPUT TYPE=text NAME=\"label\" SIZE=20</TD></TR>"
	"<TR><TH ALIGN=LEFT>Version comment:</TH>"
	"<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40</TD></TR>"
	"</TABLE><P><INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    mainMenuReference(req);
    return true;
}

bool selectSoftwareForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Select software product</TITLE></HEAD>"
	"<BODY>"
	"<H2>Select software product</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"softwareForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"software\">";
    if (products.select() > 0) { 
	print(req, products);
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool removeSoftwareForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Remove software product</TITLE></HEAD>"
	"<BODY>"
	"<H2>Remove software product</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"removeSoftware\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"software\">";
    if (products.select() != 0) { 
	print(req, products);
	req << TAG << "</SELECT><BR><INPUT TYPE=\"submit\" VALUE=\"Remove\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool selectPersonForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Select a person</TITLE></HEAD>"
	"<BODY>"
	"<H2>Select a person</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"userForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=20 NAME=\"name\">";
    if (persons.select() != 0) { 
	print(req, persons);
	req << TAG << "</SELECT><BR><INPUT TYPE=submit VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool removePersonForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Remove a person</TITLE></HEAD>"
	"<BODY>"
	"<H2>Remove a person</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"removePerson\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=20 NAME=\"name\">";
    if (persons.select() != 0) { 
	print(req, persons);
	req << TAG << "</SELECT><BR><INPUT TYPE=submit VALUE=\"Remove\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool selectBugForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Select a bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Select a bug</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"bug\">";
    if (bugs.select() != 0) { 
	print(req, bugs);
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool removeBugForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Remove a bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Remove a bug</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"removeBug\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT SIZE=15 NAME=\"bug\">";
    if (bugs.select() != 0) { 
	print(req, bugs);
	req << TAG << "</SELECT><BR><INPUT TYPE=submit VALUE=\"Remove\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool changePasswordForm(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>Change password</TITLE></HEAD>"
	"<BODY>"
	"<H2>Change password</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"changePassword\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << req.get("name") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>New password:</TH>"
	"<TD><INPUT TYPE=password NAME=\"password\" SIZE=20</TD></TR>"
	"<TR><TH ALIGN=LEFT>Re-type password:</TH>"
	"<TD><INPUT TYPE=password NAME=\"password2\" SIZE=20</TD></TR>"
	"</TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Change\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\">"
	"</FORM>";
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}
	
bool shutdown(CGIrequest& req)
{
    req << TAG << 
	HTML_HEAD "<TITLE>BUGDB message</TITLE></HEAD><BODY>"
	"<H1>BUGDB server is terminated</H1></BODY></HTML>";
    return false;
}

bool userForm(CGIrequest& req);
bool userGroupForm(CGIrequest& req);
bool softwareForm(CGIrequest& req);

bool addUser(CGIrequest& req)
{
    Person person;
    person.sName = key = req.get("name");   
    person.sEmailAddress = req.get("email");
    person.sPassword = "";
    person.status = Person::isUser;
    person.nReports = 0;
    if (persons.select(qPerson) != 0) { 
	error(req, "Person already exists");
	return true;
    }
    insert(person);
    return userForm(req);
}

bool addEngineer(CGIrequest& req)
{
    Person person;
    person.sName = key = req.get("name");   
    person.sEmailAddress = req.get("email");
    person.sPassword = "";
    person.status = Person::isEngineer;
    person.nReports = 0;
    if (persons.select(qPerson) != 0) { 
	error(req, "Person already exists");
	return true;
    }
    insert(person);
    return userForm(req);
}

bool removePerson(CGIrequest& req)
{
    key = req.get("name");
    if (key == NULL) { 
	error(req, "No person was selected");
	return true;
    }
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
    } else if (persons->nReports > 0 
	       || persons->setReportedBugs.length() > 0) 
    {
	error(req, "It is not possible to delete person who is author "
	      "of existed bug reports");
    } else { 
	message(req, "Person is removed");
    }
    return true;
}

bool addSoftware(CGIrequest& req)
{
    Software software;
    Version  version;
    software.sName = key = req.get("software");
    if (products.select(qSoftware) != 0) { 
	error(req, "Software product already exists");
	return true;
    }
    char* versionStr = req.get("version");
    if (sscanf(versionStr, "%d.%d", &version.majorVersionNumber, 
	       &version.minorVersionNumber) != 2) 
    { 
	error(req, "Bad version number (MAJOR.MINOR expected)");
	return true;
    }  
    version.sComment = req.get("comment");
    version.sLabel = req.get("label");
    version.released = dbDateTime::current();
    software.pVersions = insert(version);
    insert(software);
    req.addPair("action", "Select");
    return softwareForm(req);
}
		
bool removeSoftware(CGIrequest& req)
{
    key = req.get("software");
    if (products.select(qSoftware) == 0) { 
	error(req, "No such software product");
	return true;
    }
    if (products->setBugs.length() != 0) { 
	error(req, "Can not remove software with non-empty reported bug list");
	return true;
    }
    products.remove();
    message(req, "Software product is removed");
    return true;
}

bool removeBug(CGIrequest& req)
{
    char* bug = req.get("bug");
    if (bug == NULL) { 
	error(req, "No bug was selected");
    } else { 
	bugId = atoi(bug);
	if (bugs.select(qBug) == 0) { 
	    error(req, "No such bug");
	} else { 
	    if (bugs->pReportHistory != null ||
		bugs->pWorkArounds != null)
	    {
		error(req, "Can not remove bug with existed reports");
		return true;
	    }
	    bugs.remove();
	    message(req, "Bug is removed");
	}
    }
    return true;
}

bool changePassword(CGIrequest& req)
{
    char* password = req.get("password");
    char* password2 = req.get("password2");
    if (strcmp(password, password2) != 0) { 
	error(req, "Passwords are not equal");
    } else { 
	key = req.get("name");
	if (persons.select(qPerson) == 0) { 
	    error(req, "No such person");
	} else { 
	    persons->sPassword = password;
	    persons.update();
	    message(req, "Password changed");
	}
    }
    return true;
}

bool updatePerson(CGIrequest& req)
{
    char* name = req.get("name");
    key = name;
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    } else { 
	char* newName = req.get("newname");
	char* eMail = req.get("email");
	if (eMail != NULL) { 
	    persons->sEmailAddress = eMail;
	}
	if (newName != NULL) { 
	    persons->sName = newName;
	    req.addPair("name", newName);
	    if (strcmp(name, req.get("myself")) == 0) { 
		req.addPair("myself", newName);
	    }
	}
	persons.update();
    }
    return userForm(req);
}

bool login(CGIrequest& req)
{
    char* name = req.get("name");
    key = req.get("name");
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    } 
    if (!persons->checkPassword(req.get("password"))) { 
	error(req, "Incorrect password");
	return true;
    } 
    req.addPair("myself", name);    
    return userForm(req);
}

bool bugQueryForm(CGIrequest& req)
{
    int i;
    req << TAG << 
	HTML_HEAD "<TITLE>Query to locate bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug query</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugQuery\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><TABLE>"
	"<TR><TH ALIGN=LEFT>Description substring:</TH>"
	"<TD><INPUT TYPE=text NAME=\"summary\" SIZE=30</TD></TR>"
	"<TR><TH ALIGN=LEFT>Category:</TH>"
	"<TD><SELECT NAME=\"category\" SIZE=1>"
	"<OPTION VALUE=0 SELECTED></OPTION>";
    for (i = 1; eCATEGORY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eCATEGORY_STRING[i] << "</OPTION>";
    }
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Severity:</TH>"
	"<TD><SELECT NAME=\"severity\" SIZE=1>"
	"<OPTION VALUE=0 SELECTED></OPTION>";
    for (i = 1; eSEVERITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eSEVERITY_STRING[i] << "</OPTION>";
    }
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Fixing priority:</TH>"
	"<TD><SELECT NAME=\"priority\" SIZE=1>"
	"<OPTION VALUE=0 SELECTED></OPTION>";
    for (i = 1; eFIXING_PRIORITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eFIXING_PRIORITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Platform:</TH>"
	"<TD><INPUT TYPE=text NAME=\"platform\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>OS</TH>"
	"<TD><INPUT TYPE=text NAME=\"os\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Software:</TH>"
	"<TD><INPUT TYPE=text NAME=\"software\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Assigned to:</TH>"
	"<TD><INPUT TYPE=text NAME=\"engineer\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Reported by:</TH>"
	"<TD><INPUT TYPE=text NAME=\"user\"</TD></TR>"
	"<TR><TH ALIGN=LEFT>Major version number:</TH>"
	"<TD>from <INPUT TYPE=text NAME=\"minmajor\" SIZE=4>"
	" to <INPUT TYPE=text NAME=\"maxmajor\" SIZE=4</TD></TR>"
	"<TR><TH ALIGN=LEFT>Minor version number:</TH>"
	"<TD>from <INPUT TYPE=text NAME=\"minminor\" SIZE=4</TD>"
	" to <INPUT TYPE=text NAME=\"maxminor\" SIZE=4</TD></TR></TABLE><P>"
	"<INPUT TYPE=submit VALUE=\"Search\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\">"
	"</FORM></BODY></HTML>";
    return true;
}


bool bugQuery(CGIrequest& req) 
{
    char* p;
    dbQuery query;
    query.reset();
    p = req.get("software");
    if (*p != '\0') { 
	query.add("pSoftware.sName like").add(p);
    }
    int4 category = atoi(req.get("category"));
    if (category != 0) { 
	query.and("eCategory=").add(category);
   }
    int4 severity = atoi(req.get("severity"));
    if (severity != 0) { 
	query.and("eSeverity=").add(severity);
    }
    int4 priority = atoi(req.get("priority"));
    if (priority != 0) { 
	query.and("eFixingPriority=").add(priority);
    }
    p = req.get("os");
    if (*p != '\0') { 
	query.and("sOperatingSystem like").add(p);
    }
    p = req.get("platform");
    if (*p != '\0') { 
	query.and("sHardwarePlatform like").add(p);
    }
    p = req.get("engineer");
    if (*p != '\0') { 
	query.and("pAssignedTo is not null and pAssignedTo.sName like").add(p);
    }
    p = req.get("user");
    if (*p != '\0') { 
	query.and("pReportedBy.sName like").add(p);
    }
    p = req.get("summary");
    if (*p != '\0') { 
	query.and("sOneLineSummary like").add(p);
    }
    p = req.get("minmajor");
    int minMajorVersionNumber = (*p == '\0') ? 0 : atoi(p);
    p = req.get("maxmajor");
    int maxMajorVersionNumber = (*p == '\0') ? INT_MAX : atoi(p);
    p = req.get("minminor");
    int minMinorVersionNumber = (*p == '\0') ? 0 : atoi(p);
    p = req.get("maxminor");
    int maxMinorVersionNumber = (*p == '\0') ? INT_MAX : atoi(p);
    if (minMajorVersionNumber != 0) { 
	if (maxMajorVersionNumber != INT_MAX) { 
	    query.and("pVersion.majorVersionNumber between")
		.add(minMajorVersionNumber)
		.add("and").add(maxMajorVersionNumber);
	} else { 
	    query.and("pVersion.majorVersionNumber>=")
		.add(minMajorVersionNumber);
	}
    } else if (maxMajorVersionNumber != INT_MAX) {	
	query.and("pVersion.majorVersionNumber<=").add(maxMajorVersionNumber);
    }
    if (minMinorVersionNumber != 0) { 
	if (maxMinorVersionNumber != INT_MAX) { 
	    query.and("pVersion.minorVersionNumber between")
		.add(minMinorVersionNumber)
		.add("and").add(maxMinorVersionNumber);
	} else { 
	    query.and("pVersion.minorVersionNumber>=")
		.add(minMinorVersionNumber);
	}
    } else if (maxMinorVersionNumber != INT_MAX) {	
	query.and("pVersion.minorVersionNumber<=").add(maxMinorVersionNumber);
    }
    req << TAG << 
	HTML_HEAD "<TITLE>List of selected bugs</TITLE></HEAD>"
	"<BODY>"
	"<H2>Selected bugs</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() << 
	"\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself") << 
	"\"><SELECT NAME=\"bug\" SIZE=20>";
    if (bugs.select(query) != 0) { 
	print(req, bugs);
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}


bool userForm(CGIrequest& req)
{
    char* name = req.get("name");
    if (name == NULL) { 
	error(req, "No person was selected");
	return true;
    }
    char* myself = req.get("myself");
    key = myself;
    if (persons.select(qPerson) == 0) { 
	error(req, "Person authorization failed");
	return true;
    }
    int selfStatus = persons->status;
    key = name;
    if (persons.select(qPerson) == 0) { 
	error(req, "Person not found");
	return true;
    }
    if (persons->status == Person::isAdministrator) { 
	req << TAG << 
	    HTML_HEAD "<TITLE>BUGDB Administrator</TITLE>"
	    "</HEAD><BODY>"
	    "<H2>Administrator menu</H2><FONT SIZE=\"+1\">"
	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() 
	    << "&page=addUserForm&myself=" << URL << myself 
	    << "\">Add user</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=addEngineerForm&myself=" << URL << myself 
	    << "\">Add engineer</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=selectPersonForm&myself=" << URL << myself 
	    << "\">Select person"
	    "</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=removePersonForm&myself=" << URL << myself 
	    << "\">Remove person</A></UL>"

	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress()
	    << "&page=addSoftwareForm&myself=" << URL << myself 
	    << "\">Add software product</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=selectSoftwareForm&myself=" << URL << myself 
	    << "\">Select software product</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=removeSoftwareForm&myself=" << URL << myself 
	    << "\">Remove software product</A></UL>"

	    "<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress()
	    << "&page=selectBugForm&myself=" << URL << myself 
	    << "\">Select bug</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress()
	    << "&page=removeBugForm&myself=" << URL << myself 
	    << "\">Remove bug</A></UL>";
	if (selfStatus == Person::isAdministrator) { 
	    req << TAG << 
		"<UL><LI><A HREF=\"" << req.getStub() << "?socket=" 
		<< req.getAddress() << "&page=changePasswordForm"
		"&myself=administrator&name=" << URL << myself 
		<< "\">Change password</A>"
		"<LI><A HREF=\"" << req.getStub() << "?socket=" 
		<< req.getAddress()
		<< "&page=shutdown\">Shutdown server</A></UL>";
	}
	req << TAG << "</FONT></BODY></HTML>";
	return true;
    }
    req <<
	HTML_HEAD "<TITLE>" << name << "</TITLE></HEAD>"
	"<BODY><H2>" << name << "</H2><FONT SIZE=\"+1\">"
	"<UL><LI><A HREF=\"" << req.getStub() << "?socket="
	<< req.getAddress() << "&page=createBugReportForm&myself=" 
	<< URL << myself << "\">Create bug report</A>";
    if (persons->sEmailAddress[0] != '\0') { 
	req << TAG << 
	    "<LI><A HREF=\"mailto:" << persons->sEmailAddress 
	    << "\">Send e-mail</A>"
	    "<LI><A HREF=\"" << req.getStub() << "?socket=" << req.getAddress()
	    << "&page=bugQueryForm&myself=" << URL << myself 
	    << "\">Find a bug</A>";
    }
    if (strcmp(myself, name) == 0 || selfStatus == Person::isAdministrator) { 
	req << TAG << "<LI><A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() << "&page=changePasswordForm&myself=" 
	    << URL << myself << "&name=" << URL << name << 
	    "\">Change password</A>";
    }
    req << TAG << 
	"</UL></FONT><P><TABLE><TR><TH ALIGN=LEFT>Person name:</TH>"
	"<TD><FORM METHOD=POST ACTION=\""
	<< req.getStub() << "\"><INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\""
	"updatePerson\"><INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself
	<< "\"><INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	"<INPUT TYPE=text NAME=\"newname\" SIZE=30 VALUE=\"" 
	<< name << "\"><INPUT TYPE=submit VALUE=\"Change\"></FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>E-Mail:</TH>"
	"<TD><FORM METHOD=POST ACTION=\""
	<< req.getStub() << "\"><INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\""
	"updatePerson\"><INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself
	<< "\"><INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	"<INPUT TYPE=text NAME=\"email\" SIZE=30 VALUE=\"" 
	<< persons->sEmailAddress << "\">"
	"<INPUT TYPE=submit VALUE=\"Change\"></FORM></TD></TR>";
    if (persons->status != Person::isUser) { 
	req << TAG << "<TR><TH ALIGN=LEFT>Projects:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"softwareForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"software\" SIZE=1>";
	if (persons->setProjects.length() != 0) { 
	    print(req, persons->setProjects);
	    req << TAG << 
		"</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Detach\">";
	} else { 
	    req << TAG << EMPTY_LIST;
	}
	if (products.select() != 0) { 
	    req << TAG << 
		"</FORM></TD></TR>"
		"<TR><TH ALIGN=LEFT>Attach to project:</TH>"
		"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
		"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
		<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
		"VALUE=\"attachToProject\">"
		"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<"\">"
		"<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
		"<SELECT NAME=\"software\" SIZE=1>";
	    print(req, products); 
	    req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Attach\">";
	} 
	req << TAG << 
	    "</FORM></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Find a person:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" 
	    << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"userForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=text NAME=\"name\" SIZE=30>"
	    "<INPUT TYPE=submit VALUE=\"Find\"></FORM></TD></TR>";
    } 
    req << TAG << "<TR><TH ALIGN=LEFT>Used software:</TH>"
	"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	"VALUE=\"softwareForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	"<SELECT NAME=\"software\" SIZE=1>";
    if (persons->setUsedSoftware.length() != 0) { 
	print(req, persons->setUsedSoftware);
	req << TAG << 
	    "</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
	    "&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Unregister\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    if (products.select() != 0) { 
	req << TAG << 
	    "</FORM></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Register as software user:</TH>"
	    "<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	    "VALUE=\"registerSoftware\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"software\" SIZE=1>";
	print(req, products);
	req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Register\">";
    }
    req << TAG << "</FORM></TD></TR></TABLE><P>"
	"<B>Reported bugs:</B><BR>"	
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	"VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	"<SELECT NAME=\"bug\" SIZE=5>";
    if (persons->setReportedBugs.length() != 0) {
	print(req, persons->setReportedBugs);
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }	
    req << TAG << "</FORM><P>";
    if (persons->status != Person::isUser) { 
	req << TAG << 
	    "<P><B>Assigned bugs:</B><BR>"	
	    "<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden "
	    "NAME=\"page\" VALUE=\"bugForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"name\" VALUE=\"" << name << "\">"
	    "<SELECT NAME=\"bug\" SIZE=5>";
	if (persons->setAssignedBugs.length() != 0) {
	    print(req, persons->setAssignedBugs);
	    req << TAG << 
		"</SELECT><BR>"
		"<INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">&nbsp;"
		"<INPUT TYPE=submit NAME=\"action\" VALUE=\"Deassign\">";
	} else { 
	    req << TAG << EMPTY_LIST;
	}
	req << TAG << "</FORM>";
    }
    if (strcmp(name, myself) == 0) { 
	req << TAG << "</BODY></HTML>";
    } else { 
	mainMenuReference(req);
    }
    return true;
}


bool createBugReportForm(CGIrequest& req)
{
    int i;
    sequencer.select();
    req << TAG << 
	HTML_HEAD "<TITLE>Bug</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"createBugReport\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\""<<req.get("myself")<<"\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << ++sequencer->nBugs << ">"
	"<TABLE><TH ALIGN=LEFT>Summary:</TH>"
	"<TD><INPUT TYPE=text NAME=\"summary\" SIZE=40></TD></TR>"
	"<TR><TH ALIGN=LEFT>Category:</TH>"
	"<TD><SELECT NAME=\"category\" SIZE=1>";
    for (i = 1; eCATEGORY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION SELECTED VALUE=" << i << ">"
	    << eCATEGORY_STRING[i] << "</OPTION>";
    }
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Severity:</TH>"
	"<TD><SELECT NAME=\"severity\" SIZE=1>";
    for (i = 1; eSEVERITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION SELECTED VALUE=" << i << ">"
	    << eSEVERITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Priority:</TH>"
	"<TD><SELECT NAME=\"priority\" SIZE=1>";
    for (i = 1; eFIXING_PRIORITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION SELECTED VALUE=" << i << ">"
	    << eFIXING_PRIORITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Software:</TH>"
	"<TD><SELECT NAME=\"software\" SIZE=1>";
    if (products.select() != 0) { 
	print(req, products);
    } 
    req << TAG <<
	    "</SELECT></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Version:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"version\"></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Platform:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"platform\"</TD></TR>"
	    "<TR><TH ALIGN=LEFT>OS:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"os\"></TD></TR></TABLE><P>"
	    "<INPUT TYPE=submit VALUE=\"Submit\">&nbsp;"
	    "<INPUT TYPE=reset></FORM>";
    mainMenuReference(req);
    sequencer.update();
    return true;
}

bool bugForm(CGIrequest& req);

bool createBugReport(CGIrequest& req)
{
    key = req.get("myself");
    if (persons.select(qPerson) == 0) { 
	error(req, "Author unknown");
	return true;
    }
    key = req.get("software");
    if (products.select(qSoftware) == 0) { 
	error(req, "No such software product");
	return true;
    }
    if (sscanf(req.get("version"), "%d.%d", &majorVersion, &minorVersion) != 2)
    {
	error(req, "Bad version format");
	return true;
    }
    initialVersion = products->pVersions;
    if (versions.select(qVersion) == 0) { 
	error(req, "No such software product version");
	return true;
    }
    
    Bug bug;
    bug.bugId = atoi(req.get("bug"));
    bug.sOneLineSummary = req.get("summary");
    bug.eCategory = atoi(req.get("category"));
    bug.eFixingPriority = atoi(req.get("priority"));
    bug.eSeverity = atoi(req.get("severity"));
    bug.sOperatingSystem = req.get("os");
    bug.sHardwarePlatform = req.get("platform");
    bug.pReportedBy = persons.currentId();
    bug.pAssignedTo = null;
    bug.pSoftware = products.currentId();
    bug.pVersion = versions.currentId();
    bug.nReports = 0;
    insert(bug);
    req.addPair("action", "Select");
    return bugForm(req);
}

bool bugForm(CGIrequest& req)
{
    int i;
    char* bugStr = req.get("bug"); 
    bugId = atoi(bugStr);
    if (bugs.select(qBug) == 0) { 
	error(req, "No bug was selected");
	return true;
    }
    char* myself = req.get("myself");
    if (strcmp(req.get("action"), "Remove") == 0) { 
	dbReference<Bug> pBug = bugs.currentId();
	bugId = atoi(req.get("relatedbug"));
	if (bugs.select(qBug) == 0) { 
	    error(req, "No such bug");
	    return true;
	}
	int i = rindex(bugs->setSimilarBugs, pBug);
	if (i < 0) { 
	    error(req, "No such related bug");
	    return true;
	}
	bugs->setSimilarBugs.remove(i);
	bugs.update();
	return bugForm(req);
    }
    key = myself;
    if (persons.select(qPerson) == 0) { 
	error(req, "No such user");
	return true;
    }
    if (strcmp(req.get("action"), "Deassign") == 0) { 
	int i = rindex(persons->setAssignedBugs, bugs.currentId());
	if (i < 0) { 
	    error(req, "Bug was not assigned");
	    return true;
	}
	persons->setAssignedBugs.remove(i);
	persons.update();
	req.addPair("name", myself);
	return userForm(req);
    }

    int personStatus = persons->status;
    products.at(bugs->pSoftware);
    versions.at(bugs->pVersion);
    req << TAG <<
	HTML_HEAD "<TITLE>Bug in " << products->sName << " v. " << 
	versions->getVersionString() << "</TITLE></HEAD><BODY>"
	"<H2>Bug in " << products->sName << " v. "
	<< versions->getVersionString() << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateBug\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << bugStr << ">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<TABLE><TH ALIGN=LEFT>Summary:</TH>"
	"<TD><INPUT TYPE=text NAME=\"summary\" SIZE=40 VALUE=\"" 
	<< bugs->sOneLineSummary << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>Category:</TH>"
	"<TD><SELECT NAME=\"category\" SIZE=1>"
	"<OPTION SELECTED VALUE=" << bugs->eCategory << ">"
	<< eCATEGORY_STRING[bugs->eCategory] << "</OPTION>";
    for (i = 1; eCATEGORY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" 
	    << eCATEGORY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Severity:</TH>"
	"<TD><SELECT NAME=\"severity\" SIZE=1>"
	"<OPTION SELECTED VALUE=" << bugs->eSeverity << ">"
	<< eSEVERITY_STRING[bugs->eSeverity] << "</OPTION>";
    for (i = 1; eSEVERITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION  VALUE=" << i << ">"
	    << eSEVERITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Priority:</TH>"
	"<TD><SELECT NAME=\"priority\" SIZE=1>"
	"<OPTION SELECTED VALUE=" << bugs->eFixingPriority << ">"
	<< eFIXING_PRIORITY_STRING[bugs->eFixingPriority] << "</OPTION>";
    for (i = 1; eFIXING_PRIORITY_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">"
	    << eFIXING_PRIORITY_STRING[i] << "</OPTION>";
    }
    req << TAG << 
	"</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Platform:</TH>"
	"<TD><INPUT TYPE=text NAME=\"platform\" VALUE=\""
	<< bugs->sHardwarePlatform << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>OS:</TH>"
	"<TD><INPUT TYPE=text NAME=\"os\"VALUE=\""
	<< bugs->sOperatingSystem << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>Assigned to:</TH>"
	"<TD><SELECT SIZE=1 NAME=\"name\">";
    if (bugs->pAssignedTo != null) { 
	persons.at(bugs->pAssignedTo);
	req << TAG << "<OPTION SELECTED VALUE=\"" << persons->sName
	    << "\">" <<  persons->sName << "</OPTION>";
    } else { 
	req << TAG << "<OPTION SELECTED VALUE=\"\"></OPTION>";
    }
    print(req, products->setEngineers); 
    req << TAG << "</SELECT></TD></TR>"
	"<TR><TH ALIGN=LEFT>Similar with:</TH>"
	"<TD><SELECT NAME=\"similar\" SIZE=1>"
	"<OPTION SELECTED VALUE=\"\"></OPTION>";
    allBugs.select();
    print(req, allBugs);
    req << TAG << "</SELECT></TD></TR></TABLE><BR>";

    if (personStatus != Person::isUser) { 
	req << TAG << 
	    "<INPUT TYPE=submit NAME=\"action\" VALUE=\"Update\">&nbsp;"
	    "<INPUT TYPE=reset VALUE=\"Reset\">";
    }
    req << TAG << "</FORM><P><FORM METHOD=POST ACTION=\"" << req.getStub()
	<< "\"><INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\">"
	"<INPUT TYPE=hidden NAME=\"page\" VALUE=\"updateReportForm\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << bugStr << ">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<B>Report history:</B><BR><SELECT NAME=\"report\" SIZE=5>";
    firstReport = bugs->pReportHistory;
    if (reports.select(qAllReports) != 0) { 
	print(req, reports);
	req << TAG << 
	    "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
	    "&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Add\">";
	if (personStatus == Person::isAdministrator) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
    } else { 
	req << TAG << EMPTY_LIST 
	    "<BR><INPUT TYPE=submit  NAME=\"action\" VALUE=\"Add\">";
    }
    req << TAG << "</FORM><P>";

    req << TAG << 
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateWorkAroundForm\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << bugStr << ">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<B>Work arounds:</B><BR><SELECT NAME=\"workaround\" SIZE=5>";
    firstReport = bugs->pWorkArounds;
    if (reports.select(qAllReports) != 0) { 
	print(req, reports);
	req << TAG << 
	    "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">"
	    "&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Add\">";
	if (personStatus == Person::isAdministrator) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
    } else { 
	req << TAG << EMPTY_LIST 
	    "<BR><INPUT TYPE=submit  NAME=\"action\" VALUE=\"Add\">";
    }
    req << TAG << "</FORM><P>";    

    if (bugs->setSimilarBugs.length() != 0) { 
	req << TAG << 
	    "<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	    "<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	    << req.getAddress() << "\"><INPUT TYPE=hidden "
	    "NAME=\"page\" VALUE=\"bugForm\">"
	    "<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	    "<INPUT TYPE=hidden NAME=\"relatedbug\" VALUE=" << bugStr << ">"
	    "<B>Similar bugs:</B><BR><SELECT NAME=\"bug\" SIZE=1>";
	print(req, bugs->setSimilarBugs);
	req << TAG << 
	    "</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
	if (personStatus == Person::isAdministrator) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
	req << TAG << "</FORM><P>";
    }
    req << TAG << 
	"</FORM><P>"
	"<FONT SIZE=\"+1\"><UL>";
    if (personStatus == Person::isUser) { 
	if (bugs->pAssignedTo != null) { 
	    persons.at(bugs->pAssignedTo);
	    req << TAG << "<LI>Assigned to <A HREF=\"mailto:"  
		<< persons->sEmailAddress << "\">" 
		<< persons->sName << "</A>";
	}
	persons.at(bugs->pReportedBy);
	req << TAG << "<LI>Reported by <A HREF=\"mailto:"
	    << persons->sEmailAddress << "\">" 
	    << persons->sName << "</A></OL></FONT>";
    } else { 
	if (bugs->pAssignedTo != null) { 
	    persons.at(bugs->pAssignedTo);
	    req << TAG << "<LI>Assigned to <A HREF=\"" << req.getStub() 
		<< "?socket=" << req.getAddress() 
		<< "&page=userForm&myself=" << URL << myself 
		<< "&name=" << URL << persons->sName << "\">" 
		<< persons->sName << "</A>";
	}
	persons.at(bugs->pReportedBy);
	req << TAG 
	    << "<LI>Reported by <A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() 
	    << "&page=userForm&myself=" << URL << myself 
	    << "&name=" << URL << persons->sName << "\">" 
	    << persons->sName << "</A></OL></FONT>";
    }
    mainMenuReference(req);
    return true;
}

bool updateBug(CGIrequest& req)
{
    char* bugStr = req.get("bug");
    bugId = atoi(bugStr);
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    } 
    char* similar = req.get("similar");
    if (*similar != '\0') { 
	int id = atoi(similar);
	if (id != bugId) { 
	    bugId = id;
	    if (allBugs.select(qBug) != 0) { 
		if (rindex(bugs->setSimilarBugs, allBugs.currentId()) < 0) { 
		    bugs->setSimilarBugs.append(allBugs.currentId());
		}
	    }
	}
    }
    key = req.get("name");
    if (*key != '\0') { 
	if (persons.select(qPerson) == 0 ||
	    persons->status == Person::isUser) 
	{ 
	    error(req, "No such engineer");
	    return true;
	}
	bugs->pAssignedTo = persons.currentId();
    }
    bugs.update();
    return bugForm(req);
}


bool addReportForm(CGIrequest& req)
{
    char* bugStr = req.get("bug");
    bugId = atoi(bugStr);
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>Bug report</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug report</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"addReport\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << bugStr << ">"
	"<INPUT TYPE=hidden NAME=\"index\" VALUE=" << ++bugs->nReports
	<< "><INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself")
	<< "\"><B>Status: &nbsp;</B><SELECT SIZE=1 NAME=\"status\">";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG << 
	"</SELECT><P>"
	"<B>Bug description:</B><P>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\"></TEXTAREA><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    bugs.update();
    mainMenuReference(req);
    return true;
}

bool addReport(CGIrequest& req)
{
    bugId = atoi(req.get("bug"));
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    key = req.get("myself");
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    }
    reportId = atoi(req.get("index"));
    firstReport = bugs->pReportHistory;
    if (reports.select(qReport) == 0) { 
	Report report;
	report.pAuthor = persons.currentId();
	persons->nReports += 1;
	report.sDescription = req.get("description");
	report.index = reportId;
	report.pNext = bugs->pReportHistory;
	report.status = atoi(req.get("status"));
	report.creationDate = dbDateTime::current();
	bugs->pReportHistory = insert(report);
	persons.update();
	bugs.update();
    }
    req.addPair("action", "Select");
    return bugForm(req);
}

bool addWorkAroundForm(CGIrequest& req)
{
    char* bugStr = req.get("bug");
    bugId = atoi(bugStr);
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>Work around</TITLE></HEAD>"
	"<BODY>"
	"<H2>Work around</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"addWorkAround\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << bugStr << ">"
	"<INPUT TYPE=hidden NAME=\"index\" VALUE=" << ++bugs->nReports
	<< "><INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << req.get("myself")
	<< "\"><B>Status: &nbsp;</B><SELECT SIZE=1 NAME=\"status\">";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG << 
	"</SELECT><P>"
	"<B>Description:</B><P>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\"></TEXTAREA><P>"
	"<INPUT TYPE=submit VALUE=\"Add\">&nbsp;"
	"<INPUT TYPE=reset VALUE=\"Reset\"></FORM>";
    bugs.update();
    mainMenuReference(req);
    return true;
}

bool addWorkAround(CGIrequest& req)
{
    bugId = atoi(req.get("bug"));
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    key = req.get("myself");
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    }
    reportId = atoi(req.get("index"));
    firstReport = bugs->pWorkArounds;
    if (reports.select(qReport) == 0) { 
	Report report;
	report.pAuthor = persons.currentId();
	persons->nReports += 1;
	report.sDescription = req.get("description");
	report.index = reportId;
	report.pNext = bugs->pWorkArounds;
	report.status = atoi(req.get("status"));
	report.creationDate = dbDateTime::current();
	bugs->pWorkArounds = insert(report);
	persons.update();
	bugs.update();
    }
    req.addPair("action", "Select");
    return bugForm(req);
}

bool updateReportForm(CGIrequest& req)
{
    if (strcmp(req.get("action"), "Add") == 0) { 
	return addReportForm(req);
    }
    char* bugStr = req.get("bug");
    bugId = atoi(bugStr);
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    char* report = req.get("report");
    if (report == NULL) { 
	error(req, "No report was selected");
	return true;
    }
    int index = atoi(report);
    dbReference<Report> prev, curr = null, next = bugs->pReportHistory;
    do  { 
	prev = curr;
	if (next == null) { 
	    error(req, "No such report");
	    return true;
	} 
	reports.at(next);
	curr = next;
	next = reports->pNext;
    } while (reports->index != index);

    if (strcmp(req.get("action"), "Remove") == 0) { 
	reports.remove();
	bugs->nReports -= 1;
	if (prev == null) { 
	    bugs->pReportHistory = next;
	} else { 
	    reports.at(prev);
	    reports->pNext = next;
	    reports.update();
	}
	bugs.update();
	req.addPair("action", "Select");
	return bugForm(req);
    }
    char date[64];
    reports->creationDate.toString(date, sizeof date);
    char* myself = req.get("myself");
    key = myself;
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    }
    int personStatus = persons->status;
    persons.at(reports->pAuthor);
    req << TAG << 
	HTML_HEAD "<TITLE>Bug report from " << date << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>Bug report from " << date << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateReport\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << bugStr << ">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><INPUT TYPE=hidden NAME=\"report\" VALUE=" << index << ">"
	"<B>Created by ";
    if (personStatus == Person::isUser) { 
	req << TAG << "<A HREF=\"mailto:"  
	    << persons->sEmailAddress << "\">" 
	    << persons->sName << "</A>";
    } else { 
	req << TAG << 
	    "<A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() 
	    << "&page=userForm&myself=" << URL << myself 
	    << "&name=" << URL << persons->sName << "\">" 
	    << persons->sName << "</A>";
    }
    req << TAG << "<P>Status: </B><SELECT SIZE=1 NAME=\"status\">"
	"<OPTION SELECTED VALUE=" << reports->status << ">" 
	<< eSTATUS_STRING[reports->status] << "</OPTION>";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG <<
	"</SELECT><P>"
	"<B>Bug description:</B><BR>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\">"
	<< reports->sDescription << "</TEXTAREA><P>";
    if (personStatus != Person::isUser) { 
	req << TAG << 
	    "<INPUT TYPE=submit VALUE=\"Update\">&nbsp;"
	    "<INPUT TYPE=reset VALUE=\"Reset\">";
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool updateWorkAroundForm(CGIrequest& req)
{
    if (strcmp(req.get("action"), "Add") == 0) { 
	return addWorkAroundForm(req);
    }
    char* bugStr = req.get("bug");
    bugId = atoi(bugStr);
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    char* workaround = req.get("workaround");
    int index = atoi(workaround);
    dbReference<Report> prev, curr = null, next = bugs->pWorkArounds;
    do  { 
	prev = curr;
	if (next == null) { 
	    error(req, "No such report");
	    return true;
	} 
	reports.at(next);
	curr = next;
	next = reports->pNext;
    } while (reports->index != index);

    if (strcmp(req.get("action"), "Remove") == 0) { 
	reports.remove();
	bugs->nReports -= 1;
	if (prev == null) { 
	    bugs->pWorkArounds = next;
	} else { 
	    reports.at(prev);
	    reports->pNext = next;
	    reports.update();
	}
	bugs.update();
	req.addPair("action", "Select");
	return bugForm(req);
    }
    char date[64];
    reports->creationDate.toString(date, sizeof date);
    char* myself = req.get("myself");
    key = myself;
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    }
    int personStatus = persons->status;
    persons.at(reports->pAuthor);
    req << TAG << 
	HTML_HEAD "<TITLE>Work around " << date << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>Work around " << date << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateWorkAround\">"
	"<INPUT TYPE=hidden NAME=\"bug\" VALUE=" << bugStr << ">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><INPUT TYPE=hidden NAME=\"workaround\" VALUE=" << index <<
	"><B>Created by ";
    if (personStatus == Person::isUser) { 
	req << TAG << "<A HREF=\"mailto:"  
	    << persons->sEmailAddress << "\">" 
	    << persons->sName << "</A>";
    } else { 
	req << TAG << 
	    "<A HREF=\"" << req.getStub() << "?socket=" 
	    << req.getAddress() 
	    << "&page=userForm&myself=" << URL << myself 
	    << "&name=" << URL << persons->sName << "\">" 
	    << persons->sName << "</A>";
    }
    req << TAG << "<P>Status: </B><SELECT SIZE=1 NAME=\"status\">"
	"<OPTION SELECTED VALUE=" << reports->status << ">" 
	<< eSTATUS_STRING[reports->status] << "</OPTION>";
    for (int i = 1; eSTATUS_STRING[i] != NULL; i++) { 
	req << TAG << "<OPTION VALUE=" << i << ">" << eSTATUS_STRING[i] 
	    << "</OPTION>";
    }
    req << TAG << 
	"</SELECT><P>"
	"<B>Bug description:</B><BR>"
	"<TEXTAREA COLS=40 ROWS=5 NAME=\"description\">"
	<< reports->sDescription << "</TEXTAREA><P>";
    if (personStatus != Person::isUser) { 
	req << TAG << 
	    "<INPUT TYPE=submit VALUE=\"Update\">&nbsp;"
	    "<INPUT TYPE=reset VALUE=\"Reset\">";
    }
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool updateReport(CGIrequest& req)
{
    bugId = atoi(req.get("bug"));
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    reportId = atoi(req.get("report"));
    firstReport = bugs->pReportHistory;
    if (reports.select(qReport) == 0) {  
	error(req, "No report was selected");
	return true;
    }
    reports->sDescription = req.get("description");
    reports->status = atoi(req.get("status"));
    reports.update();
    req.addPair("action", "Select");
    return bugForm(req);
}

bool updateWorkAround(CGIrequest& req)
{
    bugId = atoi(req.get("bug"));
    if (bugs.select(qBug) == 0) { 
	error(req, "No such bug");
	return true;
    }
    reportId = atoi(req.get("workaround"));
    firstReport = bugs->pWorkArounds;
    if (reports.select(qReport) == 0) {  
	error(req, "No report was selected");
	return true;
    }
    reports->sDescription = req.get("description");
    reports->status = atoi(req.get("status"));
    reports.update();
    req.addPair("action", "Select");
    return bugForm(req);
}


bool attachToProject(CGIrequest& req)
{
    key = req.get("name");
    if (persons.select(qPerson) == 0 || persons->status == Person::isUser) { 
	error(req, "No such engineer");
    } else { 
	key = req.get("software");
	if (products.select(qSoftware) == 0) { 
	    error(req, "No such software product");
	} else { 
	    if (rindex(products->setEngineers, persons.currentId()) >= 0) { 
		error(req, "Engineer already attached to the project");
	    } else { 
		products->setEngineers.append(persons.currentId());
		products.update();
		return userForm(req);
	    }
	}
    }
    return true;
}


bool registerSoftware(CGIrequest& req)
{
    key = req.get("name");
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
    } else { 
	key = req.get("software");
	if (products.select(qSoftware) == 0) { 
	    error(req, "No such software product");
	} else { 
	    if (rindex(products->setUsers, persons.currentId()) >= 0) { 
		error(req, "User already registered this software");
	    } else { 
		products->setUsers.append(persons.currentId());
		products.update();
		return userForm(req);
	    }
	}
    }
    return true;
}


bool softwareForm(CGIrequest& req)
{
    char* software = req.get("software");
    if (software == NULL) { 
	error(req, "No software product was selected");
	return true;
    }
    key = software;
    if (products.select(qSoftware) == 0) { 
	error(req, "No such software product");
	return true;
    }
    if (strcmp(req.get("action"), "Detach") == 0) { 
	key = req.get("name");
	if (persons.select(qPerson) == 0) { 
	    error(req, "No such person");
	    return true;
	}
	int i = rindex(persons->setProjects, products.currentId());
	if (i < 0) {
	    error(req, "Person was not attached to the project");
	    return true;
	}
	persons->setProjects.remove(i);
	persons.update();
	return userForm(req);
    }
    if (strcmp(req.get("action"), "Unregister") == 0) { 
	key = req.get("name");
	if (persons.select(qPerson) == 0) { 
	    error(req, "No such person");
	    return true;
	}
	int i = rindex(persons->setUsedSoftware, products.currentId());
	if (i < 0) {
	    error(req, "Person was not registered");
	    return true;
	}
	persons->setProjects.remove(i);
	persons.update();
	return userForm(req);
    }
    char* myself = req.get("myself");
    key = myself;
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    }
    int personStatus = persons->status;
    req << TAG << 
	HTML_HEAD "<TITLE>" << software << "</TITLE></HEAD>"
	"<BODY>"
	"<H2>" << software << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateSoftware\">"
	"<INPUT TYPE=hidden NAME=\"software\" VALUE=\"" << software << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
        "<TABLE><TR><TH ALIGN=LEFT>Product name:</TH>"
	"<TD><INPUT TYPE=text NAME=\"newname\" VALUE=\"" << software << "\">"
	"</TD></TR>";
    if (products->pVersions != null) { 
	versions.at(products->pVersions);
	req << TAG << 
	    "<TR><TH ALIGN=LEFT>Current version:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"version\" SIZE=8 VALUE=\"" 
	    << versions->getVersionString() << "\"></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version label:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"label\" SIZE=20 VALUE=\""
	    << versions->sLabel << "\"></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version comment:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40 VALUE=\""
	    << versions->sComment << "\"></TD></TR>";
    } else { 
	req << TAG << 
	    "<TR><TH ALIGN=LEFT>Current version:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"version\" SIZE=8></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version label:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"label\" SIZE=20></TD></TR>"
	    "<TR><TH ALIGN=LEFT>Current version comment:</TH>"
	    "<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40></TD></TR>";
    }
    req << TAG << "</TABLE><BR>";
    if (personStatus != Person::isUser) { 
	req << TAG << 
	    "<INPUT TYPE=submit VALUE=\"Update\">&nbsp;<INPUT TYPE=reset>";
    }
    req << TAG << "</FORM><P>"
	"<TABLE><TR><TH ALIGN=LEFT>Engineers:</TH>"
	"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress()
	<< "\"><INPUT TYPE=HIDDEN NAME=\"page\" VALUE=\"userForm\">"
	"<INPUT TYPE=HIDDEN NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"name\" SIZE=1>";
    if (products->setEngineers.length() != 0) { 
	print(req, products->setEngineers); 	
	req << TAG << "</SELECT>";
	if (personStatus != Person::isUser) { 
	    req << TAG << "<INPUT TYPE=submit VALUE=\"Select\">";
	}
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG <<
	"</FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>Users:</TH>"
	"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() 
	<< "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"userForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"name\" SIZE=1>";
    if (products->setUsers.length() != 0) { 
	print(req, products->setUsers); 	
	req << TAG << "</SELECT>";
	if (personStatus != Person::isUser) { 
	    req << TAG << "<INPUT TYPE=submit VALUE=\"Select\">";
	}
    } else { 
	req << TAG << EMPTY_LIST;
    }    
    req << TAG << 
	"</FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>Bugs:</TH>"
	"<TD><FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() 
	<< "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><SELECT NAME=\"bug\" SIZE=1>";
    if (products->setBugs.length() != 0) { 
	print(req, products->setBugs);
	req << TAG << 
	    "</SELECT><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << 
	"</FORM></TD></TR>"
	"<TR><TH ALIGN=LEFT>Versions:</TH><TD>"
	"<FORM METHOD=POST ACTION=\""<<req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" << req.getAddress() 
	<< "\"><INPUT TYPE=hidden NAME=\"page\" VALUE=\"versionForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself <<
	"\"><INPUT TYPE=HIDDEN NAME=\"software\" VALUE=\"" << software << 
	"\"><SELECT NAME=\"version\" SIZE=1>";   
    initialVersion = products->pVersions;
    if (versions.select(qAllVersions) != 0) { 
	print(req, versions);						
	req << TAG << "</SELECT><INPUT TYPE=submit VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }
    req << TAG << "</FORM></TD></TR></TABLE>";
    mainMenuReference(req);
    return true;
}


bool updateSoftware(CGIrequest& req) 
{
    char* software = req.get("software");
    key = software;
    if (products.select(qSoftware) == 0) { 
	error(req, "No such software product");
	return true;
    }
    Version version;
    char* currentVersion = req.get("version");
    version.sLabel = req.get("label");
    version.sComment = req.get("comment");
    if (sscanf(currentVersion, "%d.%d", &majorVersion, &minorVersion) != 2) 
    { 
	error(req, "Bad version number (MAJOR.MINOR expected)");
	return true;
    }  
    products->sName = req.get("newname");
    version.majorVersionNumber = majorVersion;
    version.minorVersionNumber = minorVersion;
    version.released = dbDateTime::current();
    if (products->pVersions != null) { 
	initialVersion = products->pVersions;
	if (versions.select(qVersion) != 0) { 
	    versions->sComment = version.sComment;
	    versions->sLabel = version.sLabel;
	} else { 
	    versions.at(products->pVersions);
	    if (versions->majorVersionNumber > majorVersion ||
		(versions->majorVersionNumber == majorVersion && 
		 versions->minorVersionNumber > minorVersion))
	    {
		error(req, "Version number less than of current version");
		return true;
	    }
	    version.pNext = products->pVersions;
	    products->pVersions = insert(version);
	} 
    } else { 
	version.pNext = null;
	products->pVersions = insert(version);
    }
    products.update();
    req.addPair("name", req.get("myself"));
    return userForm(req);
}


bool versionForm(CGIrequest& req) 
{
    char* software = req.get("software");
    char* myself = req.get("myself");
    char  buf[64];
    key = software;
    if (products.select(qSoftware) == 0) { 
	error(req, "No such software product");
	return true;
    }
    char* versionString = req.get("version");
    if (sscanf(versionString, "%d.%d", &majorVersion, &minorVersion) != 2) { 
	error(req, "Bad version format");
	return true;
    }
    initialVersion = products->pVersions;
    if (versions.select(qVersion) == 0) { 
	error(req, "No such version");
	return true;
    }
    key = myself;
    if (persons.select(qPerson) == 0) { 
	error(req, "No such person");
	return true;
    }
    req << TAG << 
	HTML_HEAD "<TITLE>" << software << " v. " << versionString << 
	"</TITLE></HEAD>"
	"<BODY>"
	"<H2>"  << software << " v. " << versionString << "</H2>"
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden "
	"NAME=\"page\" VALUE=\"updateVersion\">"
	"<INPUT TYPE=hidden NAME=\"software\" VALUE=\"" << software << "\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<INPUT TYPE=hidden NAME=\"version\" VALUE=\"" << versionString <<"\">"
	"<TABLE><TR><TH ALIGN=LEFT>Released:</TH>"
	"<TD>" << versions->released.toString(buf, sizeof buf) << "</TD></TR>"
	"<TR><TH ALIGN=LEFT>Label:</TH>"
	"<TD><INPUT TYPE=text NAME=\"label\" SIZE=20 VALUE=\""
	<< versions->sLabel << "\"></TD></TR>"
	"<TR><TH ALIGN=LEFT>Comment:</TH>"
	"<TD><INPUT TYPE=text NAME=\"comment\" SIZE=40 VALUE=\"" 
	<< versions->sComment << "\"></TD></TR></TABLE>";
    if (persons->status != Person::isUser) { 
	req << TAG << 
	    "<P><INPUT TYPE=submit NAME=\"action\" VALUE=\"Update\">";
	if (persons->status == Person::isAdministrator) { 
	    req << TAG << 
		"&nbsp;<INPUT TYPE=submit NAME=\"action\" VALUE=\"Remove\">";
	}
	req << TAG << "&nbsp;<INPUT TYPE=reset>";
    }
    req << TAG << "<P></FORM>"
	"<B>Bugs:</B><BR>"	
	"<FORM METHOD=POST ACTION=\"" << req.getStub() << "\">"
	"<INPUT TYPE=HIDDEN NAME=\"socket\" VALUE=\"" 
	<< req.getAddress() << "\"><INPUT TYPE=hidden NAME=\"page\" "
	"VALUE=\"bugForm\">"
	"<INPUT TYPE=hidden NAME=\"myself\" VALUE=\"" << myself << "\">"
	"<SELECT NAME=\"bug\" SIZE=5>";
    if (versions->setBugs.length() != 0) {
	print(req, versions->setBugs);
	req << TAG << 
	   "</SELECT><BR><INPUT TYPE=submit NAME=\"action\" VALUE=\"Select\">";
    } else { 
	req << TAG << EMPTY_LIST;
    }	
    req << TAG << "</FORM>";
    mainMenuReference(req);
    return true;
}

bool updateVersion(CGIrequest& req) 
{
    char* software = req.get("software");
    key = software;
    if (products.select(qSoftware) == 0) { 
	error(req, "No such software product");
	return true;
    }
    if (sscanf(req.get("version"), "%d.%d", &majorVersion, &minorVersion) != 2)
    { 
	error(req, "Bad version format");
	return true;
    }
    if (strcmp(req.get("action"), "Remove") == 0) { 
 	dbReference<Version> prev, curr = null, next = products->pVersions;
	do  { 
	    prev = curr;
	    if (next == null) { 
		error(req, "No such version");
		return true;
	    } 
	    versions.at(next);
	    curr = next;
	    next = versions->pNext;
	} while (versions->majorVersionNumber != majorVersion || 
		 versions->minorVersionNumber != minorVersion);
	if (versions->setBugs.length() != 0) { 
	    error(req, "Can not remove version with non-empty bugs list");
	    return true;
	}
	versions.remove();
	if (prev == null) { 
	    products->pVersions = next;
	    products.update();
	} else { 
	    versions.at(prev);
	    versions->pNext = next;
	    versions.update();
	}
	req.addPair("action", "Select");
	return softwareForm(req);
    }    
    initialVersion = products->pVersions;
    if (versions.select(qVersion) == 0) { 
	error(req, "No such version");
	return true;
    }
    versions->sComment = req.get("comment");
    versions->sLabel = req.get("label");
    versions.update();
    return versionForm(req);
}

CGIapi::dispatcher dispatchTable[] = { 
    {"addUserForm", addUserForm},
    {"addEngineerForm", addEngineerForm},
    {"addSoftwareForm", addSoftwareForm},
    {"selectSoftwareForm", selectSoftwareForm},
    {"removeSoftwareForm", removeSoftwareForm},
    {"selectPersonForm", selectPersonForm},
    {"removePersonForm", removePersonForm},
    {"selectBugForm", selectBugForm},
    {"removeBugForm", removeBugForm},
    {"changePasswordForm", changePasswordForm},
    {"shutdown", shutdown},
    {"userForm", userForm},
    {"softwareForm", softwareForm},
    {"addUser", addUser},
    {"addEngineer", addEngineer},
    {"removePerson", removePerson},
    {"addSoftware", addSoftware},
    {"removeSoftware", removeSoftware},
    {"removeBug", removeBug},
    {"changePassword", changePassword},
    {"updatePerson", updatePerson},
    {"login", login},
    {"bugQueryForm", bugQueryForm},
    {"bugQuery", bugQuery},
    {"userForm", userForm},
    {"createBugReportForm", createBugReportForm},
    {"bugForm", bugForm},
    {"createBugReport", createBugReport},
    {"bugForm", bugForm},
    {"updateBug", updateBug},
    {"updateReportForm", updateReportForm},
    {"updateWorkAroundForm", updateWorkAroundForm},
    {"addReportForm", addReportForm},
    {"addReport", addReport},
    {"addWorkAroundForm", addWorkAroundForm},
    {"addWorkAround", addWorkAround},
    {"updateReport", updateReport},
    {"updateWorkAround", updateWorkAround},
    {"attachToProject", attachToProject},
    {"registerSoftware", registerSoftware},
    {"softwareForm", softwareForm},
    {"updateSoftware", updateSoftware},
    {"versionForm", versionForm},
    {"updateVersion", updateVersion}
}; 
    
CGIapi wwwServer(items(dispatchTable), dispatchTable);

int main(int argc, char* argv[])
{
    char* address = "localhost:6101";
    if (argc > 1) { 
	address = argv[1];
    }
    if (!wwwServer.open(address)) { 
	fprintf(stderr, "Failed to create socket for WWW server\n");
	return EXIT_FAILURE;
    }
    if (!db.open("bugdb.dbs")) { 
	fprintf(stderr, "Failed to open database\n");
	return EXIT_FAILURE;
    }
    qBug = "bugId=",bugId; 
    qReport = "index=",reportId,"start from",firstReport,"follow by pNext";
    qAllReports = "start from",firstReport,"follow by pNext";
    qVersion = "majorVersionNumber=",majorVersion,"and minorVersionNumber=",
	minorVersion,"start from",initialVersion,"follow by pNext";
    qAllVersions = "start from",initialVersion,"follow by pNext";
    qPerson = "sName=",&key;
    qSoftware = "sName=",&key;

    if (sequencer.select() == 0) { 
	BugSequencer seq;
	seq.nBugs = 0;
	insert(seq);
    }
    if (persons.select() == 0) {
	Person administrator;
	administrator.sName = "administrator";
	administrator.sEmailAddress = "root";
	administrator.sPassword = "";
	administrator.status = Person::isAdministrator;
	administrator.nReports = 0;
	insert(administrator);
    }
    CGIrequest req;
    while (wwwServer.get(req) && req.dispatch()) { 
	db.commit();
    }
    db.close();	
    printf("End of session\n");
    return EXIT_SUCCESS;
}
