/* DB: Utility Record List View
 * Copyright (C) 1998,1999  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#include <UI/CharAttr.h>

#include "enum.h"
#include "db.h"

static FieldPtr
GetFocusedField(void)
{
    FormPtr form;
    Word focus;
    FormObjectKind objType;

    form = FrmGetActiveForm();
    focus = FrmGetFocus(form);
    if (focus == noFocus)
	return NULL;

    objType = FrmGetObjectType(form, focus);
    if (objType == frmFieldObj)
	return FrmGetObjectPtr(form, focus);
    else if (objType == frmTableObj)
	return TblGetCurrentField(FrmGetObjectPtr(form, focus));
    else
	return NULL;
}

Boolean
HandleCommonMenuEvent(Word menuitemID)
{
    FieldPtr fld;

    if (menuitemID == menuitemID_AppPrefs) {
	PopupAppPrefs();
	return true;
    } else  if (menuitemID == menuitemID_About) {
	FrmAlert(alertID_About);
	return true;
    } else if (menuitemID == menuitemID_KeyboardDialog) {
	SysKeyboardDialog(kbdDefault);
	return true;
    } else if (menuitemID == menuitemID_GraffitiHelp) {
	SysGraffitiReferenceDialog(referenceDefault);
	return true;
    } else {
	fld = GetFocusedField();
	if (!fld)
	    return false;

	switch (menuitemID) {
	case menuitemID_Undo:
	    FldUndo(fld);
	    break;
	case menuitemID_Cut:
	    FldCut(fld);
	    break;
	case menuitemID_Copy:
	    FldCopy(fld);
	    break;
	case menuitemID_Paste:
	    FldPaste(fld);
	    break;
	case menuitemID_SelectAll:
	    FldSetSelection(fld, 0, FldGetTextLength(fld));
	    break;
	default:
	    return false;
	}
	return true;
    }
}

Boolean
SeekRecord (UIntPtr indexP, Int offset, Int direction)
{
    DmSeekRecordInCategory (CurrentDB, indexP, offset,
			    direction, CurrentCategory);
    if (DmGetLastErr())
	return false;
   
    return true;
}

void
DbgPrintF(CharPtr fmt, ...)
{
    Char buf[256];

    StrVPrintF(buf, fmt, &fmt);
    FrmCustomAlert(alertID_Debug, buf, " ", " ");
}

CharPtr
GetStringPtr(Word stringID)
{
    VoidHand h;
    CharPtr s;

    h = DmGetResource('tSTR', stringID);
    if (h == 0)
	return NULL;

    s = MemHandleLock(h);
    return s;
}

Boolean
IsNumber(CharPtr s)
{
    WordPtr attr = GetCharAttr();
    int base, value;

    if (*s == '-')
	s++;

    base = 10;
    if (*s == '0') {
	base = 8;
	s++;
	if (*s == 'x' && IsHex(attr, s[1])) {
	    base = 16;
	    s++;
	}
    }

    while (*s && IsHex(attr, *s)) {
	if (IsDigit(attr, *s))
	    value = *s - '0';
	else if (IsLower(attr, *s))
	    value = *s - 'a' + 10;
	else if (IsUpper(attr, *s))
	    value = *s - 'A' + 10;
	else
	    value = base;

	if (value >= base) break;
	s++;
    }

    return (*s == '\0');
}

Long
String2Long(CharPtr cp)
{
    WordPtr attr = GetCharAttr();
    Long result = 0, value;
    int base;
    Boolean positive = true;

    if (*cp == '-') {
	positive = false;
	cp++;
    }

    base = 10;
    if (*cp == '0') {
	base = 8;
	cp++;
	if (*cp == 'x' && IsHex(attr, cp[1])) {
	    base = 16;
	    cp++;
	}
    }

    while (*cp && IsHex(attr, *cp)) {
	if (IsDigit(attr, *cp))
	    value = *cp - '0';
	else if (IsLower(attr, *cp))
	    value = *cp - 'a' + 10;
	else if (IsUpper(attr, *cp))
	    value = *cp - 'A' + 10;
	else
	    value = base;

	if (value >= base) break;

	result = result * base + value;
	cp++;
    }

    return positive ? result : (- result);
}

Boolean
MatchString(CharPtr str, CharPtr substr, Boolean casesens, Boolean wholeWord)
{
    UInt l_str, l_substr;
    CharPtr orig_str;
    WordPtr attr;

    orig_str = str;
    attr = GetCharAttr();

    l_substr = StrLen(substr);
    if (l_substr == 0)
	return true;

    l_str = StrLen(str);
    while (l_str >= l_substr) {
	l_str--;

	if (casesens) {
	    if (StrNCompare(str, substr, l_substr) == 0) {
		if (! wholeWord)
		    return true;
		else {
		    if ((str == orig_str || ! IsAlNum(attr, str[-1]))
			&& (str[l_substr] == '\0' || !IsAlNum(attr, str[l_substr]))) {
			return true;
		    }
		}
	    }
	} else {
	    if (StrNCaselessCompare(str, substr, l_substr) == 0) {
		if (! wholeWord)
		    return true;
		else {
		    if ((str == orig_str || ! IsAlNum(attr, str[-1]))
			&& (str[l_substr] == '\0' || !IsAlNum(attr, str[l_substr]))) {
			return true;
		    }
		}
	    }
	}

	str++;
    }

    return false;
}

Word
GetFlags(DmOpenRef db)
{
    UInt cardNo;
    LocalID dbID, appInfoID;
    DBAppInfoPtr appInfoPtr;
    Word flags;

    DmOpenDatabaseInfo(db, &dbID, 0, 0, &cardNo, 0);
    DmDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0);
    appInfoPtr = MemLocalIDToLockedPtr(appInfoID, cardNo);
    flags = appInfoPtr->flags;
    MemPtrUnlock(appInfoPtr);
    return flags;
}

void
SetFlags(DmOpenRef db, Word flags)
{
    UInt cardNo;
    LocalID dbID, appInfoID;
    DBAppInfoPtr appInfoPtr;

    DmOpenDatabaseInfo(db, &dbID, 0, 0, &cardNo, 0);
    DmDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0);
    appInfoPtr = MemLocalIDToLockedPtr(appInfoID, cardNo);
    DmWrite(appInfoPtr, offsetof(*appInfoPtr, flags), &flags, sizeof(Word)); 
    MemPtrUnlock(appInfoPtr);
}
