/*
 * DB: Database Program for Pilot
 * Copyright (C) 1998,1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "callback.h"
#include "db.h"
#include "linkaware.h"

DBPrefsType prefs;
Boolean HideSecretRecords;

static Boolean
AppHandleEvent(EventPtr event)
{
    Word formID;
    FormPtr form;

    if (event->eType == frmLoadEvent) {
	formID = event->data.frmLoad.formID;
	form = FrmInitForm(formID);
	FrmSetActiveForm(form);
	switch (formID) {
	case formID_Chooser:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) ChooserHandleEvent);
	    break;
	case formID_DesignView:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) DesignViewHandleEvent);
	    break;
	case formID_ListView:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) ListViewHandleEvent);
	    break;
	case formID_EditView:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) EditViewHandleEvent);
	    break;
	case formID_ListPropView:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) ListPropViewHandleEvent);
	    break;
	case formID_ChooserInfoDialog:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) ChooserInfoDialogHandleEvent);
	    break;
	default:
	    ErrDisplay("unknown form ID");
	    return false;
	}
	return true;
    }
    return false;
}

static void
EventLoop(void)
{
    Word err;
    EventType event;

    do {
	EvtGetEvent(&event, evtWaitForever);

	if (!SysHandleEvent(&event))
	    if (!MenuHandleEvent((void *)0, &event, &err))
		if (!AppHandleEvent(&event))
		    FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static void
InitPrefs(DBPrefsPtr prefs)
{
    MemSet(prefs, sizeof(*prefs), 0);

    prefs->dbWasOpen = false;
    prefs->flags = prefFlagEnableFind | prefFlagPageUpPage
	| prefFlagPageDownPage;
    prefs->ChooserCurrentCategory = dmAllCategories;
    prefs->ChooserShowAllCategories = true;
    prefs->ChooserSortOrder = chooserSortOrderManual;
}

static void
StartApplication(void)
{
    SystemPreferencesType sysPrefs;
    Word prefsSize = sizeof(prefs);
    SWord actualSize;

    /* Read the current application preferences. */
    actualSize = PrefGetAppPreferences(DBCreatorID, prefID_GlobalPrefs,
				       &prefs, &prefsSize, false);

    /* Reset the preferences if they don't exist or have the wrong size. */
    if (actualSize == noPreferenceFound || prefsSize != sizeof(prefs)) {
	InitPrefs(&prefs);
	PrefSetAppPreferences(DBCreatorID, prefID_GlobalPrefs, 0,
			      &prefs, sizeof(prefs), false);
    }

    /* Examine the system preferences for any info we need. */
    PrefGetPreferences(&sysPrefs);
    if (sysPrefs.hideSecretRecords)
	HideSecretRecords = true;
    else
	HideSecretRecords = false;
}

static void
ChooseInitialForm(void)
{
    ULong dbType;
    LocalID dbID;

    /* If a database was opened previously, then try to open it again. */
    if (prefs.dbWasOpen) {
	/* Make sure that database belongs to our application. */
	if ((dbID = DmFindDatabase(prefs.cardNo, prefs.dbName)) != 0
	    && 0 == DmDatabaseInfo(prefs.cardNo, dbID,
				   0, 0, 0, 0, 0, 0, 0, 0, 0, &dbType, 0)
	    && dbType == DBTypeID
	    && 0 == OpenDatabase(prefs.cardNo, dbID)) {
	    FrmGotoForm(formID_ListView);
	} else {
	    prefs.dbWasOpen = false;
	    FrmGotoForm(formID_Chooser);
	}
    } else {
	FrmGotoForm(formID_Chooser);
    }
}

static void
StopApplication(void)
{
    UInt cardNo;
    LocalID dbID;

    FrmSaveAllForms();

    FrmCloseAllForms();

    if (CurrentDB) {
	DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
	MemSet(prefs.dbName, sizeof(prefs.dbName), 0);
	DmDatabaseInfo(cardNo, dbID, prefs.dbName, 0,0,0,0,0,0,0,0,0,0);
	prefs.dbWasOpen = true;
	prefs.cardNo = cardNo;
	CloseDatabase();
    } else {
	prefs.dbWasOpen = false;
    }

    /* Save the application preferences. */
    PrefSetAppPreferences(DBCreatorID, prefID_GlobalPrefs, 0,
			  &prefs, sizeof(prefs), false);
}

static void
GoToItem(GoToParamsPtr goToParams, Boolean launchingApp)
{
    EventType event;
    UInt recordNum = goToParams->recordNum;
 
    /* If the current record is blank, then it will be deleted, so we'll
     * the record's unique id to find the record index again, after all
     * the forms are closed.
     */
    if (! launchingApp) {
	ULong uniqueID;
	UInt cardNo;
	LocalID dbID;

	if (CurrentDB) {
	    DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
	    if (cardNo == goToParams->dbCardNo && dbID == goToParams->dbID) {
		/* For same database, recordNum could change. */
		DmRecordInfo(CurrentDB, recordNum, NULL, &uniqueID, NULL);
		FrmCloseAllForms();
		DmFindRecordByID(CurrentDB, uniqueID, &recordNum);
	    } else {
		/* Switch to the other database. */
		CloseDatabase();
		FrmCloseAllForms();
		OpenDatabase(goToParams->dbCardNo, goToParams->dbID);
	    }
	} else {
	    OpenDatabase(goToParams->dbCardNo, goToParams->dbID);
	}
    }
 
    /* We go to the list view for now. */
    FrmGotoForm(formID_ListView);
 
    /* Send an event to select the matching text. */
    MemSet(&event, 0, sizeof(EventType));
 
    event.eType = frmGotoEvent;
    event.data.frmGoto.formID = formID_ListView;
    event.data.frmGoto.recordNum = goToParams->recordNum;
    event.data.frmGoto.matchPos = goToParams->matchPos;
    event.data.frmGoto.matchLen = goToParams->searchStrLen;
    event.data.frmGoto.matchFieldNum = goToParams->matchFieldNum;
    event.data.frmGoto.matchCustom = goToParams->matchCustom;
    EvtAddEventToQueue(&event);
}

DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    if (cmd == sysAppLaunchCmdNormalLaunch) {
	StartApplication();
	ChooseInitialForm();
	EventLoop();
	StopApplication();
    } else if (cmd == sysAppLaunchCmdSyncNotify) {
	DBPrefsType prefs;
	Word prefsSize = sizeof(prefs);

	if (PrefGetAppPreferences(DBCreatorID, prefID_GlobalPrefs, &prefs,
				  &prefsSize, false) == noPreferenceFound
	    || prefsSize != sizeof(prefs))
	    InitPrefs(&prefs);
	prefs.ChooserForceRescan = true;
	PrefSetAppPreferences(DBCreatorID, prefID_GlobalPrefs, 0,
			      &prefs, sizeof(prefs), false);
    } else if (cmd == linkAppLaunchRequestLink) {
	StartApplication();
	/* Go to index when a link is requested */
	FrmGotoForm(formID_Chooser);
	EventLoop();
	StopApplication();
    } else if (cmd == sysAppLaunchCmdFind) {
	Word prefsSize = sizeof(prefs);
	DBPrefsType prefs;

	if (PrefGetAppPreferences(DBCreatorID, prefID_GlobalPrefs, &prefs,
				  &prefsSize, false) != noPreferenceFound) {
	    if (prefs.flags & prefFlagEnableFind) {
		GlobalSearch((FindParamsPtr)cmdPBP);
	    }
	}
    } else if (cmd == sysAppLaunchCmdGoTo) {
	Boolean launched;
	GoToParamsPtr go;

	launched = launchFlags & sysAppLaunchFlagNewGlobals;
	go = (GoToParamsPtr) cmdPBP;

	if (launched) {
	    StartApplication();
	    if (OpenDatabase(go->dbCardNo, go->dbID) == 0) {
		GoToItem(go, launched);
		EventLoop();
		StopApplication();
	    }
	} else {
	    CALLBACK_PROLOGUE;
	    GoToItem(go, launched);
	    CALLBACK_EPILOGUE;
	}
    } else if (cmd == linkAppLaunchFollowLink
	       || cmd == linkAppLaunchFollowLinkSubCall) {
	GoToParamsPtr go;

	if (cmd == linkAppLaunchFollowLinkSubCall
	    && ! (launchFlags&sysAppLaunchFlagNewGlobals) ) {
	    DmSearchStateType state;
	    UInt cardNo;
	    LocalID dbID;

	    if (DmGetNextDatabaseByTypeCreator(true, &state,
					       sysFileTApplication,
					       DBCreatorID, false,
					       &cardNo, &dbID) == 0) {
		SysUIAppSwitch(cardNo, dbID, linkAppLaunchFollowLink, cmdPBP);
	    }
	    return 0;
	}
       
	go = LinkSimpleToGoToParams((LinkSimplePtr) cmdPBP);

	if (go) {
	    if (cmd == linkAppLaunchFollowLink) {
		StartApplication();
		if (OpenDatabase(go->dbCardNo, go->dbID) == 0) {
		    IsNewRecord = false;
		    CurrentRecord = go->recordNum;
		    FrmGotoForm(formID_EditView);
		    EventLoop();
		    StopApplication();
		}
	    } else {
		CALLBACK_PROLOGUE;
		GoToItem(go, false);
		CALLBACK_EPILOGUE;
	    }
	    MemPtrFree(go);
	} else {
	    FrmAlert(alertID_NoLinkRecord);
	}
    }

    return 0;
}
