/*
 * DB: List View Properties
 * Copyright (C) 1998,1999  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "db.h"
#include "callback.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define fieldNameColumn  0
#define fieldWidthColumn 1

static Word TopVisibleField;
static VoidHand * handles;

static void
UpdateScrollers(void)
{
    Word upIndex, downIndex, fieldNum;
    Boolean scrollUp, scrollDown;
    SWord row;
    FormPtr form;
    TablePtr table;

    scrollUp = TopVisibleField != 0;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_ListPropView_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1) {
	fieldNum = TblGetRowID(table, row);
	scrollDown = fieldNum < numFields - 1;
    }

    upIndex = FrmGetObjectIndex(form, ctlID_ListPropView_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_ListPropView_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static Err
LoadData(VoidPtr table, Word row, Word column,
                   Boolean editing, VoidHand * textH, WordPtr textOffset,
                   WordPtr textAllocSize, FieldPtr fld)
{
    Word fieldNum;
    CharPtr p;
    FieldAttrType attr;
    VoidHand handle;

    CALLBACK_PROLOGUE;
    fieldNum = (Word) TblGetRowID(table, row);

    *textH = handles[fieldNum];
    *textOffset = 0;
    *textAllocSize = MemHandleSize(handles[fieldNum]);

    if (fld) {
	FldGetAttributes(fld, &attr);
	attr.singleLine = 1;
	attr.dynamicSize = 0;
	attr.underlined = 1;
	FldSetAttributes(fld, &attr);
	
	FldSetMaxChars(fld, 16);
    }

    CALLBACK_EPILOGUE;

    return (0);
}           

static void
LoadTable(TablePtr table)
{
    Word numRows, row, fieldNum;
    ListPtr list;
    FormPtr form;

    numRows = TblGetNumberOfRows(table);
    fieldNum = TopVisibleField;
    for (row = 0; row < numRows; row++) {
	if (fieldNum < numFields) {
	    if (! TblRowUsable(table, row)
		|| TblGetRowID(table, row) != fieldNum) {
		TblSetRowUsable(table, row, true);
		TblMarkRowInvalid(table, row);
		TblSetItemStyle(table, row, fieldNameColumn, labelTableItem);
		TblSetItemPtr(table, row, fieldNameColumn,
			      fields[fieldNum].name);
		TblSetItemStyle(table, row, fieldWidthColumn, textTableItem);
		TblSetRowID(table, row, fieldNum);
	    }
	} else {
	    TblSetRowUsable(table, row, false);
	}
	fieldNum++;
    }

    UpdateScrollers();
}

static void
SetColumnWidths(TablePtr table)
{
    SWord width;
    SWord i, w;
    FontID oldFont;

    oldFont = FntSetFont(stdFont);
    width = FntCharsWidth(fields[0].name, StrLen(fields[0].name))
	+ FntCharsWidth(":", 1);
    for (i = 1; i < numFields; i++) {
	w = FntCharsWidth(fields[i].name, StrLen(fields[i].name))
	    + FntCharsWidth(":", 1);
	if (w > width)
	    width = w;
    }
    FntSetFont(oldFont);

    TblSetColumnWidth(table, fieldNameColumn, width);
    TblSetColumnWidth(table, fieldWidthColumn, 160 - width);
}

static void
InitTable(TablePtr table)
{
    Word row, numRows;

    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++)
	TblSetRowUsable(table, row, false);

    SetColumnWidths(table);

    TblSetColumnUsable(table, fieldNameColumn, true);
    TblSetColumnUsable(table, fieldWidthColumn, true);

    TblSetLoadDataProcedure(table, fieldWidthColumn, LoadData);

    LoadTable(table);
}

static void
Scroll(DirectionType direction)
{
    UInt newTopVisibleField;

    newTopVisibleField = TopVisibleField;

    if (direction == down) {
	newTopVisibleField++;
    } else {
	newTopVisibleField--;
    }

    if (TopVisibleField != newTopVisibleField) {
	FormPtr form;
	TablePtr table;

	TopVisibleField = newTopVisibleField;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListPropView_Table);
	LoadTable(table);
	TblRedrawTable(table);
    }
}

static Boolean
SaveFieldWidths(void)
{
    Word i;
    CharPtr w;
    Char buf[64];
    int colwidth[numFields];
    LocalID appInfoID, dbID;
    DBAppInfoPtr appInfo;
    UInt attr, cardNo;
    Long v;

    /* Make sure that each field width is numeric. */
    for (i = 0; i < numFields; i++) {
	w = MemHandleLock(handles[i]);
	if (! IsNumber(w)) {
	    FrmAlert(alertID_BadFieldWidth);
	    MemPtrUnlock(w);
	    return false;
	}
	v = String2Long(w);
	colwidth[i] = (int) v;
	MemPtrUnlock(w);
	if (colwidth[i] < 10 || colwidth[i] > 160) {
	    FrmAlert(alertID_BadFieldWidth);
	    return false;
	}
    }

    /* Update the field widths in the database information. */
    for (i = 0; i < numFields; i++) {
	fields[i].colwidth = colwidth[i];
    }

    DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);

    appInfoID = DmGetAppInfoID(CurrentDB);
    appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);
    DmWrite(appInfo, offsetof(*appInfo, fieldData),
	    fields, numFields * sizeof(DBFieldInfoType));
    MemPtrUnlock(appInfo);

    DmDatabaseInfo(cardNo, dbID, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    attr |= dmHdrAttrAppInfoDirty;
    DmSetDatabaseInfo(cardNo, dbID, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);

    return true;
}

Boolean
ListPropViewHandleEvent(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    Word i;

    switch (event->eType) {
    case frmOpenEvent:
	/* Allocate temporary memory space for field width strings. */
	handles = MemPtrNew(numFields * sizeof(VoidHand));
	for (i = 0; i < numFields; i++) {
	    Char buf[64];
	    CharPtr w;

	    handles[i] = MemHandleNew(16);
	    w = MemHandleLock(handles[i]);
	    StrIToA(buf, (Long) fields[i].colwidth);
	    StrCopy(w, buf);
	    MemPtrUnlock(w);
	}

	/* Initialize the table that holds the field widths. */
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListPropView_Table);
	InitTable(table);

	/* Draw the form with field #1 appearing at the top. */
	TopVisibleField = 0;
	FrmDrawForm(form);
	return true;

    case frmCloseEvent: {
	CALLBACK_PROLOGUE;

        /* Make sure table no longer has focus. */
        form = FrmGetActiveForm();
        table = GetObjectPtr(form, ctlID_ListPropView_Table);
        if (TblGetCurrentField(table)) {
            TblReleaseFocus(table);
        }

	for (i = 0; i < numFields; i++) {
	    MemHandleFree(handles[i]);
	}
	MemPtrFree(handles);
	CALLBACK_EPILOGUE;
	break;
    }

    case tblSelectEvent:
	if (event->data.tblSelect.column == fieldNameColumn) {
	    FieldPtr fld;

	    /* A tap on the label edits the field name. */
	    TblReleaseFocus(event->data.tblSelect.pTable);
	    TblUnhighlightSelection(event->data.tblSelect.pTable);
	    TblGrabFocus(event->data.tblSelect.pTable,
			 event->data.tblSelect.row, fieldWidthColumn);
	    fld = TblGetCurrentField(event->data.tblSelect.pTable);
	    if (fld) {
		FldGrabFocus(fld);
		FldMakeFullyVisible(fld);
	    }
	    return true;
	}
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_ListPropView_DoneButton:
	    if (! SaveFieldWidths())
		return true;

	case ctlID_ListPropView_CancelButton:
	    FrmGotoForm(formID_ListView);
	    return true;

	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_ListPropView_UpButton:
	    Scroll(up);
	    break;
	case ctlID_ListPropView_DownButton:
	    Scroll(down);
	    break;
	}
	break;

    case frmUpdateEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListPropView_Table);
	InitTable(table);
	TblMarkTableInvalid(table);
	TblDrawTable(table);
	return true;

    case menuEvent:
	return HandleCommonMenuEvent(event->data.menu.itemID);

    }

    return false;
}
