/* DB: Record List View
 * Copyright (C) 1998,1999  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "db.h"
#include "callback.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))
#define noRecord dmMaxRecordIndex

UInt TopVisibleRecord;
Word CurrentCategory = dmAllCategories;
static Word LeftmostField, numColumnsShown;
static Boolean LeftButtonActive, RightButtonActive;
static Boolean LastColClipped;

static Word LFindRecord;    /* record that was found */
static Word LFindField;     /* field to search, 0 for all fields */
static Boolean LFindCaseSens, LFindWholeWord;
static Char FindString[32]; /* previous search string */

static void
ListDrawTable(VoidPtr table, Word row, Word col, RectanglePtr bounds)
{
    Word recordNum, fieldNum;
    VoidHand recordHand;
    CharPtr p;
    SWord width,len;
    Boolean fitInWidth;
    Long value;
    Char buf[32];
    VoidPtr * ptrs;

    CALLBACK_PROLOGUE;

    if (row == 0) {
	FontID oldFont = FntSetFont(boldFont);
	WinDrawChars(fields[LeftmostField+col].name,
		     StrLen(fields[LeftmostField+col].name),
		     bounds->topLeft.x, bounds->topLeft.y);
	FntSetFont(oldFont);
	CALLBACK_EPILOGUE;
	return;
    }

    /* Retrieve the record. XXX cache lookup */
    recordNum = (Word) TblGetRowID(table, row);
    fieldNum = LeftmostField + col;
    ptrs = MemPtrNew(numFields * sizeof(VoidPtr));
    GetRecord(recordNum, ptrs, &recordHand);

    /* Figure out what string we want to display. */
    switch (fields[fieldNum].type) {
    case dbFieldTypeString:
	p = ptrs[fieldNum];
	break;
    case dbFieldTypeInteger:
	MemMove(&value, ptrs[fieldNum], 4);
	StrIToA(buf, value);
	p = buf;
	break;
    }

    /* Determine if string will fit within the display width. */
    width = bounds->extent.x;
    len = StrLen(p);
    fitInWidth = false;
    FntCharsInWidth(p, &width, &len, &fitInWidth);

    if (fitInWidth) {
	WinDrawChars(p, len, bounds->topLeft.x, bounds->topLeft.y);
    } else {
	Char ellipsis = horizEllipsisChr;

	/* Figure out how many chars we can show with the ellipsis. */
	width = bounds->extent.x - FntCharWidth(horizEllipsisChr);
	len = StrLen(p);
	fitInWidth = false;
	FntCharsInWidth(p, &width, &len, &fitInWidth);

	/* Draw the string and the ellipsis. */
	WinDrawChars(p, len, bounds->topLeft.x, bounds->topLeft.y);
	WinDrawChars(&ellipsis, 1, bounds->topLeft.x + width,
		     bounds->topLeft.y);
    }

    MemHandleUnlock(recordHand);
    MemPtrFree(ptrs);

    /* Invert the bounds if this is a found record. */
    if (recordNum == LFindRecord) {
	WinInvertRectangle(bounds, 0);
    }

    CALLBACK_EPILOGUE;
}

static void
UpdateFieldButtons(void)
{
    FormPtr form;
    Word leftIndex, rightIndex;

    form = FrmGetActiveForm();
    leftIndex = FrmGetObjectIndex(form, ctlID_ListView_LeftButton);
    rightIndex = FrmGetObjectIndex(form, ctlID_ListView_RightButton);

    if (LeftmostField > 0) {
	if (!LeftButtonActive)
	    FrmShowObject(form, leftIndex);
	LeftButtonActive = true;
    } else {
	if (LeftButtonActive)
	    FrmHideObject(form, leftIndex);
	LeftButtonActive = false;
    }

    if (LastColClipped
	|| LeftmostField + numColumnsShown - 1 < numFields - 1) {
	if (!RightButtonActive)
	    FrmShowObject(form, rightIndex);
	RightButtonActive = true;
    } else {
	if (RightButtonActive)
	    FrmHideObject(form, rightIndex);
	RightButtonActive = false;
    }
}

static void
ListUpdateScrollButtons(void)
{
    Word upIndex, downIndex, recordNum;
    SWord row;
    FormPtr form;
    TablePtr table;

    Boolean scrollUp, scrollDown;

    /* We can scroll up if the top record in the list view is not the first
     * record in the current category.
     */
    recordNum = TopVisibleRecord;
    scrollUp = SeekRecord(&recordNum, 1, dmSeekBackward);

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_ListView_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1)
	recordNum = TblGetRowID(table, row);

    scrollDown = SeekRecord(&recordNum, 1, dmSeekForward);

    upIndex = FrmGetObjectIndex(form, ctlID_ListView_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_ListView_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static void
ListLoadTable(TablePtr table)
{
    Word numRows, row, col;
    UInt recordNum;
    VoidHand recordH;
    VoidPtr ptrs[numFields];

    numRows = TblGetNumberOfRows(table);

    /* Adjust the top visible record so it is in the current category. */
    SeekRecord(&TopVisibleRecord, 0, dmSeekForward);

    /* Make sure that the entire display area is used. */
    recordNum = TopVisibleRecord;
    if (! SeekRecord(&recordNum, numRows - 2, dmSeekForward)) {
	TopVisibleRecord = dmMaxRecordIndex;
	if (!SeekRecord (&TopVisibleRecord, numRows - 2, dmSeekBackward)) {
	    /* There are not enough records to fill one page so make
	     * the first record visible.
	     */
	    TopVisibleRecord = 0;
	    SeekRecord(&TopVisibleRecord, 0, dmSeekForward);
	}
    }

    recordNum = TopVisibleRecord;
    for (row = 1; row < numRows; row++) {
	/* Stop the loop if no more records are available. */
	if (! SeekRecord(&recordNum, 0, dmSeekForward))
	    break;

	TblSetRowUsable(table, row, true);
	TblMarkRowInvalid(table, row);
	TblSetRowID(table, row, recordNum);

	for (col = 0; col < numColumnsShown; col++) {
	    switch (fields[LeftmostField + col].type) {
	    case dbFieldTypeString:
	    case dbFieldTypeInteger:
		TblSetItemStyle(table, row, col, customTableItem);
		break;

	    case dbFieldTypeBoolean:
		TblSetItemStyle(table, row, col, checkboxTableItem);
		GetRecord(recordNum, ptrs, &recordH);
		if ( *((BytePtr) ptrs[LeftmostField+col]) )
		    TblSetItemInt(table, row, col, 1);
		else
		    TblSetItemInt(table, row, col, 0);
		MemHandleUnlock(recordH);
		break;
	    }
	}

	recordNum++;
    }

    /* Mark the remainder of the rows as unusable. */
    while (row < numRows) {
	TblSetRowUsable(table, row, false);
	row++;
    }

    /* Make the scroll buttons reflect the table. */
    ListUpdateScrollButtons();
}

static void
ListInitTable(TablePtr table)
{
    RectangleType r;
    Word x, colnum, width, row;
    TblGetBounds(table, &r);

    /* Initialize the table. */
    TblSetRowUsable(table, 0, true);
    TblSetRowID(table, 0, (Word) dmMaxRecordIndex);
    TblMarkRowInvalid(table, 0);
    x = colnum = 0;
    LastColClipped = false;
    while (x < r.extent.x) {
	/* Stop processing if we reached the last column or field. */
	if (LeftmostField+colnum >= numFields || colnum >= 16)
	    break;

	width = fields[LeftmostField+colnum].colwidth;
	if (width > r.extent.x - x)
	    width = r.extent.x - x;
	if (LeftmostField+colnum == numFields-1
	    && width < r.extent.x - x)
	    width = r.extent.x - x;

	TblSetItemStyle(table, 0, colnum, customTableItem);
	TblSetColumnUsable(table, colnum, true);
	TblSetColumnWidth(table, colnum, width);
	TblSetCustomDrawProcedure(table, colnum, ListDrawTable);

	x += width;
	colnum++;
    }
    numColumnsShown = colnum;
    LastColClipped = (width < fields[colnum-1].colwidth);

    /* Make sure that the remainder of the columns are not usable. */
    while (colnum < 16) {
	TblSetColumnUsable(table, colnum, false);
	colnum++;
    }

    /* Mark all of the data rows unusable so they are reloaded. */
    for (row = 1; row < TblGetNumberOfRows(table); row++) {
	TblSetRowUsable(table, row, false);
    }

    ListLoadTable(table);
    UpdateFieldButtons();
}

static void
ListScroll(DirectionType direction, Boolean byLine)
{
    FormPtr form;
    TablePtr table;
    UInt newTopVisibleRecord;
    Word numRows;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_ListView_Table);
    numRows = TblGetNumberOfRows(table) - 1;

    newTopVisibleRecord = TopVisibleRecord;

    if (direction == down) {
	if (byLine) {
	    if (! SeekRecord(&newTopVisibleRecord, 1, dmSeekForward)) {
		newTopVisibleRecord = dmMaxRecordIndex;
		/* no need to seek. ListLoadTable will take care of it */
	    }
	} else {
	    if (! SeekRecord(&newTopVisibleRecord, numRows, dmSeekForward)) {
		newTopVisibleRecord = dmMaxRecordIndex;
		/* no need to seek. ListLoadTable will take care of it */
	    }
	}
    } else {
	if (byLine) {
	    if (! SeekRecord(&newTopVisibleRecord, 1, dmSeekBackward)) {
		newTopVisibleRecord = 0;
		SeekRecord(&newTopVisibleRecord, 0, dmSeekForward);
	    }
	} else {
	    if (! SeekRecord(&newTopVisibleRecord, numRows, dmSeekBackward)) {
		newTopVisibleRecord = 0;
		SeekRecord(&newTopVisibleRecord, 0, dmSeekForward);
	    }
	}
    }

    if (TopVisibleRecord != newTopVisibleRecord) {
	TopVisibleRecord = newTopVisibleRecord;

	ListLoadTable(table);
	TblEraseTable(table);
	TblDrawTable(table);
    }
}

static void
AddNewRecord(void)
{
    VoidHand recordH;
    UInt recordNum, i;
    CharPtr rec;
    ULong offset = 0;
    ULong alloced;

    /* Figure out how much space to allocate. */
    alloced = 0;
    for (i = 0; i < numFields; i++) {
	switch (fields[i].type) {
	case dbFieldTypeString:
	case dbFieldTypeBoolean:
	    alloced += 1;
	    break;
	case dbFieldTypeInteger:
	    alloced += 4;
	    break;
	}
    }

    /* Create a new record in the database. */
    recordNum = dmMaxRecordIndex;
    recordH = DmNewRecord(CurrentDB, &recordNum, alloced);
    if (recordH == 0) {
	/* If we couldn't make the record, alert the user and bail out. */
	FrmAlert(DeviceFullAlert);
	return;
    }

    /* Null out the new record. */
    rec = MemHandleLock(recordH);
    DmSet(rec, 0, alloced, 0);
    MemPtrUnlock(rec);

    /* Release the record. */
    DmReleaseRecord(CurrentDB, recordNum, true);

    /* Activate the record editor form. */
    CurrentRecord = recordNum;
    IsNewRecord = true;
    FrmGotoForm(formID_EditView);
}

static CharPtr
StrCaselessStr(CharPtr src, CharPtr sub)
{
    CharPtr dst, result1, result2;

    dst = MemPtrNew(StrLen(src) + 1);
    StrToLower(dst, src);

    result1 = StrStr(dst, sub);
    if (result1)
	result2 = src + (result1 - dst);
    else
	result2 = 0;

    MemPtrFree(dst);

    return result2;
}

static void
LocalSearch(CharPtr str)
{
    Word recNum, foundRecNum, position;
    VoidPtr ptrs[numFields], packed;
    VoidHand recordH;
    Boolean match;
    FormPtr form;
    TablePtr table;

    recNum = (LFindRecord == noRecord) ? 0 : (LFindRecord + 1);
    while (1) {
	recordH = DmQueryNextInCategory(CurrentDB, &recNum, dmAllCategories);
	if (!recordH) {
	    foundRecNum = noRecord;
	    break;
	}

	/* Lock down the record and unpack it. */
	packed = MemHandleLock(recordH);
	UnpackRecord(packed, ptrs, numFields, fields);

	match = false;
	if (LFindField == 0) {
	    Word i;

	    /* Search all string fields for the string. */
	    for (i = 0; i < numFields; i++) {
		if (fields[i].type == dbFieldTypeString) {
		    match = MatchString(ptrs[i], str,
					LFindCaseSens, LFindWholeWord);
		    if (match)
			break;
		}
	    }
	} else {
	    /* Search the user-specified field. */
	    match = MatchString(ptrs[LFindField - 1], str,
				LFindCaseSens, LFindWholeWord);
	}

	/* Unlock the unpacked record. */
	MemHandleUnlock(recordH);

	/* If a match was found, then stop the search. */
	if (match) {
	    foundRecNum = recNum;
	    break;
	}

	/* Increment the record number we are searching. */
	recNum++;
    }

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_ListView_Table);

    /* Invalidate the current highlighted row, if any. */
    if (LFindRecord != noRecord) {
	Word row;

	if (TblFindRowID(table, LFindRecord, &row)) {
	    TblMarkRowInvalid(table, row);
	}
    }

    /* Highlight the found record or scroll table so record is visible. */
    if (foundRecNum != noRecord) {
	Word row;

	if (TblFindRowID(table, foundRecNum, &row)) {
	    /* Row is on screen so just invalidate it. */
	    TblMarkRowInvalid(table, row);
	} else {
	    /* Row is off-screen. Scroll to it. */
	    TopVisibleRecord = foundRecNum;
	    ListLoadTable(table);
	}
    } else {
	FrmAlert(alertID_MatchNotFound);
    }

    LFindRecord = foundRecNum;
    TblRedrawTable(table);
}

static void
PopupFindDialog(void)
{
    FormPtr form, oldForm;
    VoidHand strH;
    FieldPtr fld;
    ControlPtr ctl;
    ListPtr lst;
    CharPtr str, names[numFields+1];
    Boolean doSearch;
    Word i, listIndex, selectIndex, flags;
    SWord width, w;

    oldForm = FrmGetActiveForm();
    form = FrmInitForm(formID_FindDialog);
    FrmSetActiveForm(form);

    ctl = GetObjectPtr(form, ctlID_FindDialog_CaseSensCheckbox);
    CtlSetValue(ctl, LFindCaseSens ? 1 : 0);

    ctl = GetObjectPtr(form, ctlID_FindDialog_WholeWordCheckbox);
    CtlSetValue(ctl, LFindWholeWord ? 1 : 0);

    names[0] = GetStringPtr(stringID_AllFields);
    width = FntCharsWidth(names[0], StrLen(names[0]));
    selectIndex = 0;
    listIndex = 1;
    for (i = 0; i < numFields; i++) {
	if (LFindField != 0 && i == LFindField - 1) {
	    selectIndex = listIndex;
	}
	if (fields[i].type == dbFieldTypeString) {
	    names[listIndex] = fields[i].name;
	    w = FntCharsWidth(names[listIndex], StrLen(names[listIndex]));
	    if (w > width)
		width = w;
	    listIndex++;
	}
    }

    width += 10;

    lst = GetObjectPtr(form, ctlID_FindDialog_FieldList);
    lst->bounds.extent.x = width; /* HACK!! PalmOS has no equivalent. */
    LstSetListChoices(lst, names, listIndex);
    LstSetSelection(lst, selectIndex);
    LstSetHeight(lst, listIndex);

    ctl = GetObjectPtr(form, ctlID_FindDialog_FieldTrigger);
    CtlSetLabel(ctl, names[selectIndex]);

    strH = MemHandleNew(32);
    str = MemHandleLock(strH);
    StrCopy(str, FindString);
    MemPtrUnlock(str);
    
    fld = GetObjectPtr(form, ctlID_FindDialog_Field);
    FldSetTextHandle(fld, (Handle) strH);
    FrmSetFocus(form, FrmGetObjectIndex(form, ctlID_FindDialog_Field));

    doSearch = false;
    if (FrmDoDialog(form) == ctlID_FindDialog_FindButton) {
	doSearch = true;

	str = MemHandleLock(strH);
	if (LFindCaseSens)
	    StrCopy(FindString, str);
	else
	    StrToLower(FindString, str);
	MemPtrUnlock(str);

	ctl = GetObjectPtr(form, ctlID_FindDialog_CaseSensCheckbox);
	LFindCaseSens = (CtlGetValue(ctl)) ? true : false;

	ctl = GetObjectPtr(form, ctlID_FindDialog_WholeWordCheckbox);
	LFindWholeWord = (CtlGetValue(ctl)) ? true : false;

	lst = GetObjectPtr(form, ctlID_FindDialog_FieldList);
	selectIndex = LstGetSelection(lst);
	if (selectIndex == 0) {
	    /* Search all fields. */
	    LFindField = 0;
	} else {
	    /* Search a single field. */
	    listIndex = 1;
	    for (i = 0; i < numFields; i++) {
		if (listIndex == selectIndex)  LFindField = i + 1;
		if (fields[i].type == dbFieldTypeString) listIndex++;
	    }
	}

	/* Update the database flags with the current checkbox settings. */
	flags = GetFlags(CurrentDB);
	if (LFindCaseSens)
	    flags |= dbFlagLFindCaseSens;
	else
	    flags &= ~(dbFlagLFindCaseSens);
	if (LFindWholeWord)
	    flags |= dbFlagLFindWholeWord;
	else
	    flags &= ~(dbFlagLFindWholeWord);
	SetFlags(CurrentDB, flags);
    }

    MemPtrUnlock(names[0]);

    FrmDeleteForm(form);
    FrmSetActiveForm(oldForm);

    if (doSearch && FindString[0] != '\0')
	LocalSearch(FindString);
}

static Boolean
DoCommand(Word itemID)
{
    FormPtr form;
    TablePtr table;

    if (HandleCommonMenuEvent(itemID))
	return true;

    switch (itemID) {
    case menuitemID_DatabaseDesign:
	DesignNewDB = false;
	FrmGotoForm(formID_DesignView);
	return true;

    case menuitemID_EditFieldWidths:
	FrmGotoForm(formID_ListPropView);
	return true;

    case menuitemID_NewRecord:
	AddNewRecord();
	return true;

    case menuitemID_GotoTop:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListView_Table);
	TopVisibleRecord = 0;
	ListLoadTable(table);
	TblEraseTable(table);
	TblDrawTable(table);
	return true;

    case menuitemID_GotoBottom:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListView_Table);
	TopVisibleRecord = dmMaxRecordIndex;
	ListLoadTable(table);
	TblEraseTable(table);
	TblDrawTable(table);
	return true;

    case menuitemID_DatabasePrefs:
	MenuEraseStatus(0);
	PopupDBPrefs();
	return true;
    }
    return false;
}

Boolean
ListViewHandleEvent(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    UInt cardNo;
    LocalID dbID;
    Char name[dmDBNameLength+1], title[4+dmDBNameLength+1];
    Word flags;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	LeftmostField = 0;
	LeftButtonActive = RightButtonActive = true;
	LFindRecord = noRecord;
	LFindField = 0;
	flags = GetFlags(CurrentDB);
	LFindCaseSens = ((flags & dbFlagLFindCaseSens) != 0);
	LFindWholeWord = ((flags & dbFlagLFindWholeWord) != 0);
	FindString[0] = '\0';
	table = GetObjectPtr(form, ctlID_ListView_Table);

	/* Make the form title reflect the database name. */
	DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
	DmDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	StrCopy(title, "DB: ");
	StrCat(title, name);
	FrmCopyTitle(form, title);

	ListInitTable(table);

	FrmDrawForm(form);
	return true;

    case frmGotoEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListView_Table);
	TopVisibleRecord = event->data.frmGoto.recordNum;
	ListLoadTable(table);
	TblEraseTable(table);
	TblDrawTable(table);
	return true;

    case tblSelectEvent:
	table = event->data.tblSelect.pTable;
	if (event->data.tblSelect.row > 0) {
	    switch (fields[LeftmostField+event->data.tblSelect.column].type){
	    case dbFieldTypeBoolean: {
		VoidHand recordH;
		VoidPtr ptrs[numFields];
		Word recordNum;
		VoidPtr packed;
		Byte b;

		b = TblGetItemInt(table, event->data.tblSelect.row,
				  event->data.tblSelect.column) ? 1 : 0;

		recordNum = TblGetRowID(table, event->data.tblSelect.row);
		recordH = DmGetRecord(CurrentDB, recordNum);
		packed = MemHandleLock(recordH);
		UnpackRecord(packed, ptrs, numFields, fields);
		DmWrite(packed, ptrs[LeftmostField + event->data.tblSelect.column] - packed,
			&b, 1);
		MemHandleUnlock(recordH);
		DmReleaseRecord(CurrentDB, recordNum, true);
		break;
	    }

	    case dbFieldTypeString:
	    default:
		CurrentRecord = TblGetRowID(table, event->data.tblSelect.row);
		IsNewRecord = false;
		FrmGotoForm(formID_EditView);
		break;
	    }
	}
	return true;

    case menuEvent:
	return DoCommand(event->data.menu.itemID);

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_ListView_DoneButton:
	    CloseDatabase();
	    FrmGotoForm(formID_Chooser);
	    return true;

	case ctlID_ListView_NewButton:
	    AddNewRecord();
	    return true;

	case ctlID_ListView_FindButton:
	    PopupFindDialog();
	    return true;

	case ctlID_ListView_RepeatButton:
	    if (FindString[0])
		LocalSearch(FindString);
	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_ListView_UpButton:
	    ListScroll(up, (prefs.flags&prefFlagUpArrowPage)?false:true);
	    break;

	case ctlID_ListView_DownButton:
	    ListScroll(down, (prefs.flags&prefFlagDownArrowPage)?false:true);
	    break;

	case ctlID_ListView_LeftButton:
	    form = FrmGetActiveForm();
	    table = GetObjectPtr(form, ctlID_ListView_Table);
	    LeftmostField--;
	    ListInitTable(table);
	    TblMarkTableInvalid(table);
	    TblRedrawTable(table);
	    break;

	case ctlID_ListView_RightButton:
	    form = FrmGetActiveForm();
	    table = GetObjectPtr(form, ctlID_ListView_Table);
	    LeftmostField++;
	    ListInitTable(table);
	    TblMarkTableInvalid(table);
	    TblRedrawTable(table);
	    break;
	}
	break;

    case keyDownEvent:
	switch (event->data.keyDown.chr) {
	case pageUpChr:
	    ListScroll(up, (prefs.flags&prefFlagPageUpPage)?false:true);
	    return true;
	case pageDownChr:
	    ListScroll(down, (prefs.flags&prefFlagPageDownPage)?false:true);
	    return true;
	}
	break;
    }

    return false;
}
