/*
 * DB: Database I/O code
 * Copyright (C) 1998,1999  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "db.h"

DmOpenRef CurrentDB = 0;
Word numFields;
DBFieldInfoPtr fields;

Err
OpenDatabase(UInt cardNo, LocalID dbID)
{
    LocalID appInfoID;
    DBAppInfoPtr appInfoPtr;

    ErrFatalDisplayIf(CurrentDB != 0, "DB already open");

    CurrentDB = DmOpenDatabase(cardNo, dbID, dmModeReadWrite);
    if (CurrentDB == 0) {
	return DmGetLastErr();
    }

    /* Retrieve the app info block and copy to local storage. */
    appInfoID = DmGetAppInfoID(CurrentDB);
    appInfoPtr = MemLocalIDToLockedPtr(appInfoID, cardNo);
    TopVisibleRecord = appInfoPtr->topVisibleRecord;
    numFields = appInfoPtr->numFields;
    fields = MemPtrNew(numFields * sizeof(DBFieldInfoType));
    MemMove(fields, &(appInfoPtr->fieldData),
	    numFields * sizeof(DBFieldInfoType));
    MemPtrUnlock(appInfoPtr);

    return 0;
}

void
CloseDatabase(void)
{
    LocalID appInfoID;
    DBAppInfoPtr appInfoPtr;
    UInt cardNo;

    ErrFatalDisplayIf(CurrentDB == 0, "DB already closed");

    DmOpenDatabaseInfo(CurrentDB, 0, 0, 0, &cardNo, 0);

    appInfoID = DmGetAppInfoID(CurrentDB);
    appInfoPtr = MemLocalIDToLockedPtr(appInfoID, cardNo);
    DmWrite(appInfoPtr, offsetof(*appInfoPtr, topVisibleRecord),
	    &TopVisibleRecord, sizeof(TopVisibleRecord));
    MemPtrUnlock(appInfoPtr);

    DmCloseDatabase(CurrentDB);
    CurrentDB = 0;
}

Err
CreateDatabase(CharPtr name, Word numFields, DBFieldInfoPtr fields)
{
    Err err;
    VoidHand appInfoHand;
    LocalID dbID, appInfoID;
    DmOpenRef db;
    DBAppInfoPtr appInfo;

    err = DmCreateDatabase(0, name, DBCreatorID, DBTypeID, false);
    if (err)
	return err;

    dbID = DmFindDatabase(0, name);

    db = DmOpenDatabase(0, dbID, dmModeReadWrite);
    if (db == 0)
	return DmGetLastErr();

    /* Create an application information block for this database. */
    appInfoHand = DmNewHandle(db, 3 * sizeof(Word) + numFields *
			      sizeof(DBFieldInfoType));
    appInfoID = MemHandleToLocalID(appInfoHand);
    appInfo = MemLocalIDToLockedPtr(appInfoID, 0);
    DmSet(appInfo, 0, 3 * sizeof(Word) + numFields * sizeof(DBFieldInfoType), 0);
    DmWrite(appInfo, offsetof(*appInfo, numFields), &numFields,
	    sizeof(numFields));
    DmWrite(appInfo, offsetof(*appInfo, fieldData), fields,
	    numFields * sizeof(DBFieldInfoType));
    MemPtrUnlock(appInfo);

    DmSetDatabaseInfo(0, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0);
    DmCloseDatabase(db);

    /* Add this database into the chooser screen. */
    Chooser_AddDatabase(0, name);

    return 0;
}

void
UnpackRecord(VoidPtr packed, VoidPtr unpacked[],
	     int numFields, DBFieldInfoPtr fields)
{
    BytePtr rec = packed;
    Word i;

    for (i = 0; i < numFields; i++) {
	switch (fields[i].type) {
	case dbFieldTypeString:
	    unpacked[i] = rec;
	    rec += StrLen(rec) + 1;
	    break;

	case dbFieldTypeBoolean:
	    unpacked[i] = rec;
	    rec += 1;
	    break;

	case dbFieldTypeInteger:
	    unpacked[i] = rec;
	    rec += 4;
	    break;

	default:
	    ErrDisplay("unknown field type");
	    break;
	}
    }
}

ULong
PackedRecordSize(VoidPtr unpacked[])
{
    ULong size = 0;
    Word i;

    for (i = 0; i < numFields; i++) {
	switch (fields[i].type) {
	case dbFieldTypeString:
	    size += StrLen(unpacked[i]) + 1;
	    break;

	case dbFieldTypeBoolean:
	    size += 1;
	    break;

	case dbFieldTypeInteger:
	    size += 4;
	    break;

	default:
	    ErrDisplay("unknown field type");
	    break;
	}
    }

    return size;
}

void
PackRecord(VoidPtr unpacked[], VoidPtr packed)
{
    ULong offset = 0;
    Word i;
    Byte b;

    for (i = 0; i < numFields; i++) {
	switch (fields[i].type) {
	case dbFieldTypeString:
	    DmWrite(packed, offset, unpacked[i], StrLen(unpacked[i]) + 1);
	    offset += StrLen(unpacked[i]) + 1;
	    break;

	case dbFieldTypeBoolean:
	    DmWrite(packed, offset, unpacked[i], 1);
	    offset += 1;
	    break;

	case dbFieldTypeInteger:
	    DmWrite(packed, offset, unpacked[i], 4);
	    offset += 4;
	    break;

	default:
	    ErrDisplay("unknown field type");
	    break;
	}
    }
}

Err
GetRecord(Word index, VoidPtr unpacked[], VoidHand * handleP)
{
    VoidHand handle;
    VoidPtr packed;

    handle = DmQueryRecord(CurrentDB, index);
    ErrFatalDisplayIf(handle == 0, "handle is NULL");
    packed = MemHandleLock(handle);
    UnpackRecord(packed, unpacked, numFields, fields);
    if (handleP) {
	*handleP = handle;
    }
    return 0;
}
