/*
 * DB: Global Find Support
 * Copyright (C) 1998,1999  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "callback.h"
#include "db.h"

typedef struct {
    DmSearchStateType searchState;
    UInt cardNo;
    LocalID dbID;
} DBFindPrefsType, *DBFindPrefsPtr;

void
GlobalSearch(FindParamsPtr params)
{
    Err error;
    RectangleType bounds;
    DBFindPrefsType data;
    DmOpenRef db;
    Char dbname[dmDBNameLength+1], title[dmDBNameLength+1+5];
    DBAppInfoPtr appInfoPtr;
    LocalID appInfoID;
    Word numFields;
    DBFieldInfoPtr fields;
    VoidHand recordH;
    VoidPtr packed;
    Boolean match;
    UInt recordNum;
    Word position;
    int i;

    /* By default, we assume there are more things to search. */
    params->more = true;

    /* If we are not continuing, then try to find the first database. */
    if (! params->continuation) {
	error = DmGetNextDatabaseByTypeCreator(true, &data.searchState,
					       DBTypeID, NULL, false,
					       &data.cardNo, &data.dbID);
	if (error != 0) {
	    /* No databases were found so we are done. */
	    params->more = false;
	    return;
	}
    } else {
	Word prefsSize = sizeof(DBFindPrefsType);

	/* Retrieve data from the last run of the find. */
	PrefGetAppPreferences(DBCreatorID, prefID_FindPrefs, &data,
			      &prefsSize, false);
    }

    do {
	/* Open the current search database. */
	db = DmOpenDatabase(data.cardNo, data.dbID, params->dbAccesMode);
	if (!db) {
	    /* On error, don't continue at all. */
	    params->more = false;
	    return;
	}

	/* Build the title we will display in the list. */
	DmDatabaseInfo(data.cardNo, data.dbID, dbname, 0, 0, 0, 0, 0, 0,
		       &appInfoID, 0, 0, 0);
	StrCopy(title, "DB: ");
	StrCat(title, dbname);

	/* Get the number of fields in the database. */
	appInfoPtr = MemLocalIDToLockedPtr(appInfoID, data.cardNo);
	if (appInfoPtr->flags & dbFlagDisableFind) {
	    /* unlock and close done by code after label */
	    goto next_db;
	}
	numFields = appInfoPtr->numFields;
	fields = (DBFieldInfoPtr) &appInfoPtr->fieldData;

	/* Draw the header. */
	if (FindDrawHeader(params, title)) {
	    MemPtrUnlock(appInfoPtr);
	    DmCloseDatabase(db);
	    PrefSetAppPreferences(DBCreatorID, prefID_FindPrefs, 0,
				  &data, sizeof(data), false);
	    return;
	}
  
	/* Iterate over all the records looking for a match */
	recordNum = params->recordNum;  

	while (1) {
	    VoidPtr ptrs[numFields];

	    recordH = DmQueryNextInCategory(db, &recordNum, dmAllCategories);
	    if (! recordH) {
		/* No more records in this database. */
		params->recordNum = 0;
		break;
	    }

	    /* Try to find a match in this record. */
	    packed = MemHandleLock(recordH);
	    UnpackRecord(packed, ptrs, numFields, fields);
	    match = false;
	    for (i = 0; i < numFields; i++) {
		if (fields[i].type == dbFieldTypeString) {
		    match = FindStrInStr(ptrs[i], params->strToFind,
					 &position);
		    if (match) break;
		}
	    }

	    /* If a match was found, save it. */
	    if (match) {
		if (FindSaveMatch(params, recordNum, position, i, 0,
				  data.cardNo, data.dbID)) {
		    /* A true result means we have to stop the search here. */
		    MemHandleUnlock(recordH);
		    MemPtrUnlock(appInfoPtr);
		    DmCloseDatabase(db);
		    PrefSetAppPreferences(DBCreatorID, prefID_FindPrefs, 0,
					  &data, sizeof(data), false);
		    return;
		}

		/* Draw the record. */
		FindGetLineBounds(params, &bounds);
		WinDrawChars(ptrs[i], StrLen(ptrs[i]),
			     bounds.topLeft.x, bounds.topLeft.y);
		params->lineNumber++;
	    }

	    /* Unlock the record so we can search again. */
	    MemHandleUnlock(recordH);    

	    /* Increment the record number. */
	    recordNum++;
	}

    next_db:
	MemPtrUnlock(appInfoPtr);
	DmCloseDatabase(db);
  
	/* Get the next database. */
	error = DmGetNextDatabaseByTypeCreator(false, &data.searchState,
					       DBTypeID, 0, false,
					       &data.cardNo, &data.dbID);
    } while (! error);

    /* Finished search. */
    params->more = false;
}
