/*
 * DB: Record Edit Form
 * Copyright (C) 1998,1999  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#include <UI/CharAttr.h>

#include "enum.h"
#include "db.h"
#include "linkaware.h"
#include "callback.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))
#define noFieldIndex 32

UInt CurrentRecord;
Boolean IsNewRecord;

static UInt TopVisibleField, CurrentField;
static VoidHand * handles;
static Boolean IsDirty;

static Err
EditViewLoadRecord(VoidPtr table, Word row, Word column, 
		   Boolean editing, VoidHand * textH, WordPtr textOffset,
		   WordPtr textAllocSize, FieldPtr fld)
{
    Word fieldNum;
    CharPtr fieldP;
    FieldAttrType attr;

    CALLBACK_PROLOGUE;
    fieldNum = (Word) TblGetRowID(table, row);
    ErrFatalDisplayIf(fieldNum >= numFields, "out of range field #");
    fieldP = MemHandleLock(handles[fieldNum]);
    *textH = handles[fieldNum];
    *textOffset = 0;
    *textAllocSize = StrLen(fieldP) + 1;
    MemPtrUnlock(fieldP);

    if (fields[fieldNum].type == dbFieldTypeInteger && fld) {
	FldSetMaxChars(fld, 32);
	FldGetAttributes(fld, &attr);
	attr.singleLine = true;
	attr.dynamicSize = false;
	FldSetAttributes(fld, &attr);
    }

    CALLBACK_EPILOGUE;

    return (0);
}

static Boolean
EditViewSaveRecord(VoidPtr table, Word row, Word column)
{
    FieldPtr fld;

    CALLBACK_PROLOGUE;

    fld = TblGetCurrentField(table);
    if (fld && FldDirty(fld))
	IsDirty = true;

    CALLBACK_EPILOGUE;
}

static void
EditViewUpdateScrollers(FormPtr frm, Word bottomField, Boolean lastItemClipped)
{
    Word upIndex;
    Word downIndex;
    Boolean scrollableUp;
    Boolean scrollableDown;
      
    /* If the first field displayed is not the fist field in the
     * record, enable the up scroller.
     */
    scrollableUp = TopVisibleField > 0;

    /* If the last field displayed is not the last field in the
     * record, enable the down scroller.
     */
    scrollableDown = (lastItemClipped ||
		      (bottomField < numFields - 1));
    
    /* Update the scroll button. */
    upIndex = FrmGetObjectIndex(frm, ctlID_EditView_UpButton);
    downIndex = FrmGetObjectIndex(frm, ctlID_EditView_DownButton);
    FrmUpdateScrollers (frm, upIndex, downIndex, scrollableUp, scrollableDown);
}

static Word
EditViewGetFieldHeight(TablePtr table, Word fieldNum, Word columnWidth,
		       Word maxHeight)
{
    Word height, lineHeight;
    CharPtr str;
    VoidHand handle;

    switch (fields[fieldNum].type) {
    case dbFieldTypeString:
	handle = handles[fieldNum];
	str = MemHandleLock(handle);
	height = FldCalcFieldHeight(str, columnWidth);
	lineHeight = FntLineHeight ();
	height = min (height, (maxHeight / lineHeight));
	height *= lineHeight;
	MemHandleUnlock(handle);
	break;

    case dbFieldTypeBoolean:
    case dbFieldTypeInteger:
	height = FntLineHeight();
	break;

    }
    return (height);
}

static void
EditViewInitTableRow(TablePtr table, Word row, Word fieldNum, short rowHeight)
{
    BytePtr b;

    /* Make the row usable. */
    TblSetRowUsable(table, row, true);

    /* Set the height of the row to the height of the desc. */
    TblSetRowHeight(table, row, rowHeight);

    /* Store the record number as the row id. */
    TblSetRowID(table, row, (Word) fieldNum);
    ErrFatalDisplayIf(fieldNum >= numFields, "out-of-range field #");

    /* Mark the row invalid so that it will draw when we call the 
     * draw routine.
     */
    TblMarkRowInvalid(table, row);

    /* Setup the field label column. */
    TblSetItemStyle(table, row, 0, labelTableItem);
    TblSetItemPtr(table, row, 0, fields[fieldNum].name);

    /* Setup the field data column. */
    switch (fields[fieldNum].type) {
    case dbFieldTypeString:
    case dbFieldTypeInteger:
	TblSetItemStyle(table, row, 1, textTableItem);
	break;

    case dbFieldTypeBoolean:
	TblSetItemStyle(table, row, 1, checkboxTableItem);
	b = MemHandleLock(handles[fieldNum]);
	TblSetItemInt(table, row, 1, (Word) *b);
	MemPtrUnlock(b);
	break;
    }
}

static void
EditViewLoadTable(TablePtr table)
{
    Word row;
    Word numRows;
    Word fieldIndex, lastFieldIndex;
    Word lineHeight,dataHeight,tableHeight;
    Word columnWidth;
    Word pos, oldPos;
    Word height, oldHeight;
    FontID fontID, currFont;
    FormPtr frm;
    Boolean rowUsable;
    Boolean rowsInserted = false;
    Boolean lastItemClipped;
    RectangleType r;
    Handle recordH;
	
    TblGetBounds (table, &r);
    tableHeight = r.extent.y;
    columnWidth = TblGetColumnWidth(table, 1);

    /* If we currently have a selected record, make sure that it is not
     * above the first visible record.
     */
    if (CurrentField != noFieldIndex) {
	if (CurrentField < TopVisibleField)
	    TopVisibleField = CurrentField;
    }

    row = 0;
    dataHeight = 0;
    oldPos = pos = 0;
    fieldIndex = TopVisibleField;
    lastFieldIndex = fieldIndex;

    /* Load records into the table. */
    while (fieldIndex < numFields) {		
	/* Compute the height of the field's text string. */
	height = EditViewGetFieldHeight(table, fieldIndex, columnWidth,
					tableHeight);
	
	/* Is there enough room for at least one line of the the field. */
	lineHeight = FntLineHeight();
	if (tableHeight >= dataHeight + lineHeight) {
	    rowUsable = TblRowUsable(table, row);

	    /* Get the height of the current row. */
	    if (rowUsable)
		oldHeight = TblGetRowHeight(table, row);
	    else
		oldHeight = 0;
	    
	    /* If the field is not already being displayed in the current 
	     * row, load the field into the table.
	     */
	    if ((! rowUsable) ||
		(TblGetRowID (table, row) != fieldIndex)) {
		EditViewInitTableRow(table, row, fieldIndex, height);
	    }
			
	    /* If the height or the position of the item has changed
	     * draw the item.
	     */
	    else if (height != oldHeight) {
		TblSetRowHeight(table, row, height);
		TblMarkRowInvalid (table, row);
	    } else if (pos != oldPos) {
		TblMarkRowInvalid (table, row);
	    }

	    pos += height;
	    oldPos += oldHeight;
	    lastFieldIndex = fieldIndex;
	    fieldIndex++;
	    row++;
	}

	dataHeight += height;

	/* Is the table full? */
	if (dataHeight >= tableHeight) {
	    /* If we have a currently selected field, make sure that it is
	     * not below the last visible field.  If the currently selected 
	     * field is the last visible record, make sure the whole field 
	     * is visible.
	     */
	    if (CurrentField == noFieldIndex)
		break;
	    
	    /* Above last visible? */
	    else if (CurrentField < fieldIndex)
		break;
	    
	    /* Last visible? */
	    else if (fieldIndex == lastFieldIndex) {
		if ((fieldIndex == TopVisibleField) || (dataHeight == tableHeight))
		    break;
	    }	

	    /* Remove the top item from the table and reload the table
	     * again.
	     */
	    TopVisibleField++;
	    fieldIndex = TopVisibleField;


	    // Below last visible.
	    //			else
	    //				TopVisibleField = CurrentField;
				
	    row = 0;
	    dataHeight = 0;
	    oldPos = pos = 0;
	}
    }


    /* Hide the item that don't have any data. */
    numRows = TblGetNumberOfRows (table);
    while (row < numRows) {		
	TblSetRowUsable (table, row, false);
	row++;
    }
		
    /* If the table is not full and the first visible field is not the
     * first field in the record, displays enough fields to fill out
     * the table by adding fields to the top of the table.
     */
    while (dataHeight < tableHeight) {
	fieldIndex = TopVisibleField;
	if (fieldIndex == 0) break;
	fieldIndex--;
	
	/* Compute the height of the field. */
	height = EditViewGetFieldHeight(table, fieldIndex,
					columnWidth, tableHeight);
			
	/* If adding the item to the table will overflow the height of
	 * the table, don't add the item.
	 */
	if (dataHeight + height > tableHeight)
	    break;
		
	/* Insert a row before the first row. */
	TblInsertRow(table, 0);

	EditViewInitTableRow(table, 0, fieldIndex, height);
		
	TopVisibleField = fieldIndex;
	
	rowsInserted = true;
	
	dataHeight += height;
    }
    
    /* If rows were inserted to full out the page, invalidate the
     * whole table, it all needs to be redrawn.
     */
    if (rowsInserted)
	TblMarkTableInvalid(table);

    /* If the height of the data in the table is greater than the
     * height of the table, then the bottom of the last row is clip
     * and the table is scrollable.
     */
    lastItemClipped = (dataHeight > tableHeight);

    /* Update the scroll arrows. */
    EditViewUpdateScrollers(FrmGetActiveForm(),
			    lastFieldIndex, lastItemClipped);
}

static void
EditViewInitTable(TablePtr table)
{
    Word labelWidth, width, numRows, i;

    /* Initialize the table rows to a none default. */
    numRows = TblGetNumberOfRows(table);
    for (i = 0; i < numRows; i++) {
	TblSetItemStyle(table, i, 0, labelTableItem);
	TblSetRowUsable(table, i, false);
    }

    /* Enable the label and data columns. */
    TblSetColumnUsable(table, 0, true);
    TblSetColumnUsable(table, 1, true);

    /* Figure out the width of the label column. */
    labelWidth = FntCharsWidth(fields[0].name,
			       StrLen(fields[0].name)) + FntCharWidth(':');
    for (i = 1; i < numFields; i++) {
	width = FntCharsWidth(fields[i].name,
			      StrLen(fields[i].name)) + FntCharWidth(':');
	if (width > labelWidth)
	    labelWidth = width;
    }
    if (labelWidth > 80)
	labelWidth = 80;
    TblSetColumnWidth(table, 0, labelWidth);
    TblSetColumnWidth(table, 1, 160 - labelWidth);

    TblSetLoadDataProcedure(table, 1, EditViewLoadRecord);
    TblSetSaveDataProcedure(table, 1, EditViewSaveRecord);

    EditViewLoadTable(table);
}

static void
EditViewResizeDescription(EventPtr event)
{
    Word pos;
    Word row;
    Word column;
    Word lastRow;
    Word fieldIndex;
    Word lastFieldIndex;
    Word topFieldIndex;
    FieldPtr fld;
    TablePtr table;
    Boolean restoreFocus = false;
    Boolean lastItemClipped;
    RectangleType itemR;
    RectangleType tableR;
    RectangleType fieldR;


    // Get the current height of the field;
    fld = event->data.fldHeightChanged.pField;
    FldGetBounds (fld, &fieldR);

    // Have the table object resize the field and move the items below
    // the field up or down.
    table = GetObjectPtr(FrmGetActiveForm(), ctlID_EditView_Table);
    TblHandleEvent (table, event);

    // If the field's height has expanded , we're done.
    if (event->data.fldHeightChanged.newHeight >= fieldR.extent.y) {
	topFieldIndex = TblGetRowID (table, 0);
	if (topFieldIndex != TopVisibleField)
	    TopVisibleField = topFieldIndex;
	else {
	    // Since the table has expanded we may be able to scroll
	    // when before we might not have.
	    lastRow = TblGetLastUsableRow (table);
	    TblGetBounds (table, &tableR);
	    TblGetItemBounds (table, lastRow, 1, &itemR);
	    lastItemClipped = (itemR.topLeft.y + itemR.extent.y > 
			       tableR.topLeft.y + tableR.extent.y);
	    lastFieldIndex = TblGetRowID (table, lastRow);
		
	    EditViewUpdateScrollers (FrmGetActiveForm (), lastFieldIndex, 
				     lastItemClipped);
	    
	    return;
	}
    }

    // If the field's height has contracted and the field edit field
    // is not visible then the table may be scrolled.  Release the 
    // focus,  which will force the saving of the field we are editing.
    else if (TblGetRowID (table, 0) != 0) {
	TblGetSelection (table, &row, &column);
	fieldIndex = TblGetRowID (table, row);
	
	fld = TblGetCurrentField (table);
	pos = FldGetInsPtPosition (fld);
	TblReleaseFocus (table);
	
	restoreFocus = true;
    }

    // Add items to the table to fill in the space made available by the 
    // shorting the field.
    EditViewLoadTable(table);
    TblRedrawTable (table);
    
    // Restore the insertion point position.
    if (restoreFocus) {
	TblFindRowID (table, fieldIndex, &row);
	TblGrabFocus (table, row, column);
	FldSetInsPtPosition (fld, pos);
	FldGrabFocus (fld);
    }
}

static void
EditViewHandleSelectField(Word row, const Byte column)
{
    TablePtr table;
    FieldPtr fld;
    Word fieldNum;
    Char buf[16];

    StrIToA(buf, column);
    FrmCustomAlert(alertID_Debug, "row = ", buf, " ");

    table = GetObjectPtr(FrmGetActiveForm(), ctlID_EditView_Table);
    fieldNum = TblGetRowID(table, row);

    if (fieldNum != CurrentField) {
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	    FrmCustomAlert(alertID_Debug, "focus released", " ", " ");
	}
	TblUnhighlightSelection(table);
	TblGrabFocus(table, row, 1);
	fld = TblGetCurrentField(table);
	if (fld) {
	    FldGrabFocus(fld);
	    FldMakeFullyVisible(fld);
	}
	FrmCustomAlert(alertID_Debug, "focus grabbed", " ", " ");
	CurrentField = fieldNum;
    }
}

static void
EditViewScroll(DirectionType direction)
{
    Word row, height, fieldIndex, columnWidth, tableHeight;
    TablePtr       table;
    RectangleType   r;

    table = GetObjectPtr(FrmGetActiveForm(), ctlID_EditView_Table);
    TblReleaseFocus(table);

    /* Get the height of the table and the width of the description
     * column.
     */
    TblGetBounds (table, &r);
    tableHeight = r.extent.y;
    height = 0;
    columnWidth = TblGetColumnWidth (table, 1);

    /* Scroll the table down. */
    if (direction == down) {
	// Get the index of the last visible field, this will become 
	// the index of the top visible field, unless it occupies the 
	// whole screeen, in which case the next field will be the
	// top field.
      
	row = TblGetLastUsableRow (table);
	fieldIndex = TblGetRowID (table, row);
      
	// If the last visible field is also the first visible field
	// then it occupies the whole screeen.
	if (row == 0)
	    fieldIndex = min (numFields-1, fieldIndex+1);
    }
    // Scroll the table up.
    else {
	// Scan the fields before the first visible field to determine 
	// how many fields we need to scroll.  Since the heights of the 
	// fields vary,  we sum the height of the records until we get
	// a screen full.

	fieldIndex = TblGetRowID (table, 0);
	ErrFatalDisplayIf(fieldIndex >= numFields, "Invalid field Index");
	if (fieldIndex == 0)
	    return;
         
	height = TblGetRowHeight (table, 0);
	if (height >= tableHeight)
	    height = 0;                     

	while (height < tableHeight && fieldIndex > 0) {
	    CharPtr str = MemHandleLock(handles[fieldIndex]);
	    switch (fields[fieldIndex].type) {
	    case dbFieldTypeString:
		height += FldCalcFieldHeight(str,columnWidth) * FntLineHeight();
		break;
	    case dbFieldTypeBoolean:
	    case dbFieldTypeInteger:
		height += FntLineHeight();
		break;
	    }
	    if ((height <= tableHeight)
		|| (fieldIndex == TblGetRowID (table, 0)))
		fieldIndex--;
	    MemPtrUnlock(str);
	}
    }

    TblMarkTableInvalid (table);
    CurrentField = noFieldIndex;
    TopVisibleField = fieldIndex;

    TblUnhighlightSelection (table);
    EditViewLoadTable(table);

    TblRedrawTable (table);
}

/* Returns true if the data in the record is valid. */
static Boolean
ValidateData(void)
{
    CharPtr s;
    Word i;

    for (i = 0; i < numFields; i++) {
	switch (fields[i].type) {
	case dbFieldTypeInteger:
	    s = MemHandleLock(handles[i]);
	    if (! IsNumber(s)) {
		MemPtrUnlock(s);
		return false;
	    }
	    MemPtrUnlock(s);
	    break;
	}
    }

    return true;
}

static void
SaveRecord(void)
{
    VoidPtr ptrs[numFields];
    Long values[numFields];
    VoidHand handle, oldH;
    VoidPtr packed;
    UInt recordNum;
    Word i;
    ULong size;

    /* Only save the record if the record is dirty. */
    if (!IsDirty)
	return;

    /* Place pointers to each field into structure. */
    for (i = 0; i < numFields; i++) {
	VoidPtr s;

	switch (fields[i].type) {
	case dbFieldTypeString:
	case dbFieldTypeBoolean:
	    s = MemHandleLock(handles[i]);
	    ptrs[i] = s;
	    break;

	case dbFieldTypeInteger:
	    s = MemHandleLock(handles[i]);
	    values[i] = String2Long(s);
	    ptrs[i] = &values[i];
	    break;
	}
    }

    /* Determine the new size of the record. */
    size = PackedRecordSize(ptrs);

    /* Resize the record to fit new size. */
    DmResizeRecord(CurrentDB, CurrentRecord, size);

    /* Retrieve a copy of the record handle. */
    handle = DmGetRecord(CurrentDB, CurrentRecord);
    ErrFatalDisplayIf(handle == 0, "NULL handle");

    /* Pack the new data into the record. */
    packed = MemHandleLock(handle);
    PackRecord(ptrs, packed);
    MemPtrUnlock(packed);

    /* Release the record. */
    DmReleaseRecord(CurrentDB, CurrentRecord, true);

    /* Unlock all of the field handles. */
    for (i = 0; i < numFields; i++) {
	MemHandleUnlock(handles[i]);
    }
}

static Boolean
DoCommand(Word menuitemID)
{
    if (menuitemID == menuitemID_PublishLink) {
	Char name[dmDBNameLength];
	LocalID dbID;
	UInt cardNo;
	LinkInfoPtr link;

	DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
	DmDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	link = (LinkInfoPtr) LinkSimpleNew(CurrentDB, CurrentRecord, name);
	if (! LinkPublish(link)) {
	  FrmAlert(alertID_NoLinkMaster);
	}

	return true;
    } else
	return HandleCommonMenuEvent(menuitemID);
}

static void
SetupRecordData(void)
{
    VoidPtr ptrs[numFields];
    VoidHand recordH;
    Long value;
    CharPtr s;
    Word i;

    /* Retrieve a copy of the record. */
    GetRecord(CurrentRecord, ptrs, &recordH);

    /* Allocate space for the memory handles. */
    handles = MemPtrNew(numFields * sizeof(VoidHand));

    /* Copy the record data into place. */
    for (i = 0; i < numFields; i++) {
	switch (fields[i].type) {
	case dbFieldTypeString:
	    handles[i] = MemHandleNew(StrLen(ptrs[i]) + 1);
	    s = MemHandleLock(handles[i]);
	    StrCopy(s, ptrs[i]);
	    MemPtrUnlock(s);
	    break;

	case dbFieldTypeBoolean:
	    handles[i] = MemHandleNew(sizeof(Byte));
	    s = MemHandleLock(handles[i]);
	    *((BytePtr) s) = *((BytePtr) ptrs[i]);
	    MemPtrUnlock(s);
	    break;

	case dbFieldTypeInteger:
	    MemMove(&value, ptrs[i], 4);
	    handles[i] = MemHandleNew(32);
	    s = MemHandleLock(handles[i]);
	    MemSet(s, 32, 0);
	    StrIToA(s, value);
	    MemPtrUnlock(s);
	    break;
	}
    }

    /* Unlock the record. */
    MemHandleUnlock(recordH);
}

Boolean
EditViewHandleEvent(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    Word fieldNum;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_EditView_Table);

	/* Initialize the globals we use. */
	IsDirty = false;   /* Data is not dirty initially. */
	TopVisibleField = 0;
	CurrentField = noFieldIndex;

	SetupRecordData();
	EditViewInitTable(table);

	FrmDrawForm(form);
	return true;

    case frmCloseEvent: {
	Word i;

	CALLBACK_PROLOGUE;

	/* Make sure table no longer has focus. */
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_EditView_Table);
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	}

	/* Erase the record if it was new and not modified. */
	if (IsNewRecord && !IsDirty) {
	    DmRemoveRecord(CurrentDB, CurrentRecord);
	}

	/* Free the memory we allocated. */
	for (i = 0; i < numFields; i++) {
	    if (handles[i]) MemHandleFree(handles[i]);
	}
	MemPtrFree(handles);

	CALLBACK_EPILOGUE;
	return false;
    }

    case frmSaveEvent: {
	CALLBACK_PROLOGUE;
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_EditView_Table);
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	}
	if (ValidateData())
	    SaveRecord();
	CALLBACK_EPILOGUE;
	return true;
    }
	
    case tblSelectEvent:
	table = event->data.tblSelect.pTable;
	fieldNum = TblGetRowID(table, event->data.tblSelect.row);
	if (event->data.tblSelect.column == 0) {
	    if (fields[fieldNum].type == dbFieldTypeString
		|| fields[fieldNum].type == dbFieldTypeInteger) {
		FieldPtr fld;

		/* A tap on the label edits the field. */
		if (TblGetCurrentField(event->data.tblSelect.pTable)) {
		    TblReleaseFocus(event->data.tblSelect.pTable);
		}
		TblUnhighlightSelection(event->data.tblSelect.pTable);
		TblGrabFocus(event->data.tblSelect.pTable,
			     event->data.tblSelect.row, 1);
		fld = TblGetCurrentField(event->data.tblSelect.pTable);
		if (fld) {
		    FldGrabFocus(fld);
		    FldMakeFullyVisible(fld);
		}
		return true;
	    }
	} else if (event->data.tblSelect.column == 1) {
	    if (fields[fieldNum].type == dbFieldTypeBoolean) {
		Word v = TblGetItemInt(table, event->data.tblSelect.row,
				       event->data.tblSelect.column);
		BytePtr b = MemHandleLock(handles[fieldNum]);
		if (v)
		    *b = 1;
		else
		    *b = 0;
		MemPtrUnlock(b);
		IsDirty = true;
	    }
	}
	return false;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_EditView_DoneButton:
	    if (!ValidateData()) {
		FrmAlert(alertID_ValidationFailure);
		return true;
	    }
	    SaveRecord();
	    /* fall-through */

	case ctlID_EditView_CancelButton:
	    FrmGotoForm(formID_ListView);
	    return true;

	case ctlID_EditView_DeleteButton:
	    if (FrmAlert(alertID_ConfirmDelete) == 0) {
		DmRemoveRecord(CurrentDB, CurrentRecord);
		FrmGotoForm(formID_ListView);
	    }
	    return true;
	}
	break;

    case fldHeightChangedEvent:
	EditViewResizeDescription(event);
	return true;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_EditView_UpButton:
	    EditViewScroll(up);
	    break;
	case ctlID_EditView_DownButton:
	    EditViewScroll(down);
	    break;
	}

    case menuEvent:
	return DoCommand(event->data.menu.itemID);
    }

    return false;
}
