/*
 * DB: Database Design Form
 * Copyright (C) 1998,1999  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "db.h"
#include "callback.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define oldNameColumn    0
#define fieldNameColumn  1
#define fieldTypeColumn  2

typedef struct
{
    VoidHand handle;
    Word num;
    Char name[32];
} OldFieldInfoType, *OldFieldInfoPtr;
    
static SWord TopVisibleField;
static SWord allocedNumFields;
SWord designNumFields;
Boolean DesignNewDB;

DBFieldInfoPtr designFields;
OldFieldInfoPtr oldFieldInfo;


static void
UpdateScrollers(void)
{
    Word upIndex, downIndex, fieldNum;
    Boolean scrollUp, scrollDown;
    SWord row;
    FormPtr form;
    TablePtr table;

    scrollUp = TopVisibleField != 0 && designNumFields > 0;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_DesignView_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1 && row != 0) {
	fieldNum = TblGetRowID(table, row);
	scrollDown = fieldNum < designNumFields - 1;
    } else {
	scrollDown = false;
    }

    upIndex = FrmGetObjectIndex(form, ctlID_DesignView_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_DesignView_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);

    if (! DesignNewDB) {
	Word addIndex, delIndex;

	addIndex = FrmGetObjectIndex(form, ctlID_DesignView_AddButton);
	delIndex = FrmGetObjectIndex(form, ctlID_DesignView_DelButton);

	if (designNumFields > 0)
	    FrmShowObject(form, delIndex);
	else
	    FrmHideObject(form, delIndex);

	FrmShowObject(form, addIndex);
    }
}

static Err
LoadData(VoidPtr table, Word row, Word column,
	 Boolean editing, VoidHand * textH, WordPtr textOffset,
	 WordPtr textAllocSize, FieldPtr fld)
{
    Word fieldNum;
    CharPtr p;
    FieldAttrType attr;
    VoidHand handle;

    CALLBACK_PROLOGUE;

    fieldNum = TblGetRowID(table, row);

    handle = oldFieldInfo[fieldNum].handle;

    if (MemHandleSize(handle) < 32)
	MemHandleResize(handle, 32);

    p = MemHandleLock(handle);
    StrCopy(p, designFields[fieldNum].name);
    MemPtrUnlock(p);

    *textH = handle;
    *textOffset = 0;
    *textAllocSize = 32;

    if (editing && fld) {
	FldGetAttributes(fld, &attr);
	attr.singleLine = 1;
	attr.dynamicSize = 0;
	attr.underlined = 1;
	FldSetAttributes(fld, &attr);
	
	FldSetMaxChars(fld, 31);
    }

    CALLBACK_EPILOGUE;

    return (0);
}           

static Boolean
SaveData(VoidPtr table, Word row, Word col)
{
    Word fieldNum;
    CharPtr p;

    CALLBACK_PROLOGUE;

    fieldNum = TblGetRowID(table, row);
    p = MemHandleLock(oldFieldInfo[fieldNum].handle);
    StrCopy(designFields[fieldNum].name, p);
    MemPtrUnlock(p);

    CALLBACK_EPILOGUE;

    return false;
}

static void
LoadTable(TablePtr table)
{
    Word numRows, row, fieldNum;
    ListPtr list;
    FormPtr form;

    form = FrmGetActiveForm();
    list = GetObjectPtr(form, ctlID_DesignView_PopupList);

    numRows = TblGetNumberOfRows(table);
    fieldNum = TopVisibleField;
    for (row = 0; row < numRows; row++) {
	if (fieldNum < designNumFields) {
	    if (! TblRowUsable(table, row)
		|| TblGetRowID(table, row) != fieldNum) {
		TblSetRowUsable(table, row, true);
		TblMarkRowInvalid(table, row);
		TblSetItemStyle(table, row, oldNameColumn, labelTableItem);
		TblSetItemPtr(table, row, oldNameColumn, oldFieldInfo[fieldNum].name);
		TblSetItemStyle(table, row, fieldNameColumn, textTableItem);
		TblSetItemStyle(table, row, fieldTypeColumn, popupTriggerTableItem);
		TblSetItemPtr(table, row, fieldTypeColumn, list);
		TblSetItemInt(table, row, fieldTypeColumn,
			      designFields[fieldNum].type);
		TblSetRowID(table, row, fieldNum);
	    }
	} else {
	    TblSetRowUsable(table, row, false);
	}
	fieldNum++;
    }

    UpdateScrollers();
}

static void
SetColumnWidths(TablePtr table)
{
    SWord width;
    SWord i, w, m;
    FontID oldFont;

    oldFont = FntSetFont(stdFont);
    width = FntCharsWidth(oldFieldInfo[0].name, StrLen(oldFieldInfo[0].name))
	+ FntCharsWidth(":", 1);
    m = designNumFields;
    for (i = 1; i < m; i++) {
	w = FntCharsWidth(oldFieldInfo[i].name, StrLen(oldFieldInfo[i].name))
	    + FntCharsWidth(":", 1);
	if (w > width)
	    width = w;
    }
    FntSetFont(oldFont);

    TblSetColumnWidth(table, oldNameColumn, width);
    TblSetColumnWidth(table, fieldNameColumn, 110 - width);
    TblSetColumnWidth(table, fieldTypeColumn, 50);
}

static void
InitTable(TablePtr table)
{
    Word row, numRows;

    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++)
	TblSetRowUsable(table, row, false);

    SetColumnWidths(table);

    TblSetColumnUsable(table, oldNameColumn, true);
    TblSetColumnUsable(table, fieldNameColumn, true);
    TblSetColumnUsable(table, fieldTypeColumn, true);

    TblSetLoadDataProcedure(table, fieldNameColumn, LoadData);
    TblSetSaveDataProcedure(table, fieldNameColumn, SaveData);

    LoadTable(table);
}

static void
Scroll(DirectionType direction)
{
    UInt newTopVisibleField;

    newTopVisibleField = TopVisibleField;

    if (direction == down) {
	newTopVisibleField++;
    } else {
	newTopVisibleField--;
    }

    if (TopVisibleField != newTopVisibleField) {
	FormPtr form;
	TablePtr table;

	TopVisibleField = newTopVisibleField;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	LoadTable(table);
	TblRedrawTable(table);
    }
}

static Boolean
UpdateNewDesign(void)
{
    FormPtr form;
    FieldPtr fld;
    VoidHand name_handle;
    CharPtr name;
    int i, numFields;

    /* Update the number of fields by finding the last nonempty field. */
    for (i = designNumFields - 1; i >= 0; i--) {
	if (designFields[i].name[0] != '\0')
	    break;
    }
    numFields = i + 1;
    if (numFields == 0) {
	FrmAlert(alertID_NoFields);
	return false;
    }

    /* Verify that we can use the database name. */
    form = FrmGetActiveForm();
    fld = GetObjectPtr(form, ctlID_DesignView_NameField);
    name_handle = (VoidHand) FldGetTextHandle(fld);
    name = MemHandleLock(name_handle);
    if (name[0] == '\0' || DmFindDatabase(0, name) != 0) {
	FrmAlert(name[0] == '\0' ? alertID_NameBlank : alertID_NameConflict);
	MemPtrUnlock(name);
	return false;
    }

    /* Create the database. Creation routine will display any errors. */
    CreateDatabase(name, numFields, designFields);
    MemPtrUnlock(name);

    return true;
}

static Boolean
UpdateExistingDesign(void)
{
    FormPtr form;
    FieldPtr fld;
    VoidHand name_handle;
    CharPtr name;
    Char dbname[dmDBNameLength+1];
    UInt cardNo;
    LocalID dbID;
    Boolean updateDesign = false;

    /* Validate any changes to the database name. */
    form = FrmGetActiveForm();
    fld = GetObjectPtr(form, ctlID_DesignView_NameField);
    name_handle = (VoidHand) FldGetTextHandle(fld);
    name = MemHandleLock(name_handle);
    if (name[0] == '\0') {
	FrmAlert(alertID_NameBlank);
	MemPtrUnlock(name);
	return false;
    }

    DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
    DmDatabaseInfo(cardNo, dbID, dbname, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    if (StrCompare(dbname, name) != 0) {
	if (DmFindDatabase(cardNo, name) != 0) {
	    FrmAlert(alertID_NameConflict);
	    MemPtrUnlock(name);
	    return false;
	}
    }

    /* The database design is not allowed to be changed at this time. */
    if (designNumFields != numFields) {
	FrmAlert(alertID_DesignChange);
	MemPtrUnlock(name);
	return false;
    } else {
	Word i;

	for (i = 0; i < designNumFields; i++) {
	    /* Reordering fields or changing field type is a no-no. */
	    if (oldFieldInfo[i].num != i
		|| designFields[i].type != fields[i].type) {
		FrmAlert(alertID_DesignChange);
		MemPtrUnlock(name);
		return false;
	    }

	    /* If the field name changed, then we update the design. */
	    if (StrCompare(designFields[i].name,
			   fields[i].name) != 0) {
		updateDesign = true;
	    }
	}
    }

    /* Make the database name change. */
    if (StrCompare(dbname, name) != 0) {
	DmSetDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	Chooser_RenameDatabase(cardNo, dbname, name);
    }

    /* Make any database design changes. */
    if (updateDesign) {
	LocalID appInfoID;
	DBAppInfoPtr appInfo;
	UInt attr;

	appInfoID = DmGetAppInfoID(CurrentDB);
	appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);
	DmWrite(appInfo, offsetof(*appInfo, fieldData), designFields,
		numFields * sizeof(DBFieldInfoType));
	MemPtrUnlock(appInfo);

	DmDatabaseInfo(cardNo, dbID, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	attr |= dmHdrAttrAppInfoDirty;
	DmSetDatabaseInfo(cardNo, dbID, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);

	MemMove(fields, designFields, numFields * sizeof(DBFieldInfoType));
    }

    MemPtrUnlock(name);
    return true;
}

static void
InsertField(SWord WhichField)
{
    CharPtr str1, str2;
    SWord i;

    /* Allocate more space if we are out of room. */
    if (designNumFields + 1 > allocedNumFields) {
	VoidHand h;
	Err err;

	allocedNumFields++;

	h = MemPtrRecoverHandle(designFields);
	ErrFatalDisplayIf(h == 0, "h is NULL");
	MemHandleUnlock(h);
	err = MemHandleResize(h, allocedNumFields * sizeof(DBFieldInfoType));
	ErrFatalDisplayIf(err != 0, "could not resize handle");
	designFields = MemHandleLock(h);

	h = MemPtrRecoverHandle(oldFieldInfo);
	ErrFatalDisplayIf(h == 0, "h is NULL");
	MemHandleUnlock(h);
	err = MemHandleResize(h, allocedNumFields * sizeof(OldFieldInfoType));
	ErrFatalDisplayIf(err != 0, "could not resize handle");
	oldFieldInfo = MemHandleLock(h);

	oldFieldInfo[allocedNumFields-1].handle = MemHandleNew(32);
    }

    /* Move the other fields up. */
    for (i = designNumFields - 1; i >= WhichField; i--) {
	MemMove(&(designFields[i+1]), &(designFields[i]),
		sizeof(DBFieldInfoType));
	StrCopy(oldFieldInfo[i+1].name, oldFieldInfo[i].name);
	oldFieldInfo[i+1].num = oldFieldInfo[i].num;
    }
    designNumFields++;

    str1 = GetStringPtr(stringID_NewField);
    ErrFatalDisplayIf(str1 == 0, "str1 is NULL");
    str2 = GetStringPtr(stringID_New);
    ErrFatalDisplayIf(str2 == 0, "str2 is NULL");
	
    StrCopy(oldFieldInfo[WhichField].name, str2);
    oldFieldInfo[WhichField].num = 100;
    StrCopy(designFields[WhichField].name, str1);
    designFields[WhichField].type = 0;
    designFields[WhichField].colwidth = 80;

    MemPtrUnlock(str1);
    MemPtrUnlock(str2);
}

static void
DeleteField(Word WhichField)
{
    SWord i;

    /* Move the other fields down. */
    for (i = WhichField; i < designNumFields - 1; i++) {
	MemMove(&(designFields[i]), &(designFields[i+1]),
		sizeof(designFields[i]));
	StrCopy(oldFieldInfo[i].name, oldFieldInfo[i+1].name);
	oldFieldInfo[i].num = oldFieldInfo[i+1].num;
    }

    designNumFields--;
}

static void
PopupFieldSelectDialog(Boolean doInsert)
{
    FormPtr form, oldForm;
    CharPtr title, label, str1, names[numFields+1];
    ControlPtr ctl;
    ListPtr list;
    SWord i;
    Boolean didAction;

    oldForm = FrmGetActiveForm();
    form = FrmInitForm(formID_FieldSelectDialog);
    FrmSetActiveForm(form);

    /* Determine the title and label to use. */
    str1 = GetStringPtr(stringID_Append);
    if (doInsert) {
	title = GetStringPtr(stringID_InsertField);
	label = GetStringPtr(stringID_InsertFieldLabel);
    } else {
	title = GetStringPtr(stringID_DeleteField);
	label = GetStringPtr(stringID_DeleteFieldLabel);
    }

    /* Set the form title and the label. */
    FrmSetTitle(form, title);
    FrmCopyLabel(form, ctlID_FieldSelectDialog_Label, label);

    for (i = 0; i < designNumFields; i++)
	names[i] = designFields[i].name;
    if (doInsert)
	names[designNumFields] = str1;

    list = GetObjectPtr(form, ctlID_FieldSelectDialog_List);
    i = doInsert ? (designNumFields + 1) : designNumFields;
    LstSetListChoices(list, names, i);
    LstSetSelection(list, 0);
    LstSetHeight(list, i);

    ctl = GetObjectPtr(form, ctlID_FieldSelectDialog_Trigger);
    CtlSetLabel(ctl, names[0]);

    didAction = false;
    if (FrmDoDialog(form) == ctlID_FieldSelectDialog_OkayButton) {
	i = (SWord) LstGetSelection(list);
	if (doInsert)
	    InsertField(i);
	else
	    DeleteField(i);
	didAction = true;
    }

    FrmDeleteForm(form);
    FrmSetActiveForm(oldForm);

    MemPtrUnlock(str1);
    MemPtrUnlock(title);
    MemPtrUnlock(label);

    if (didAction) {
	TablePtr table;

	table = GetObjectPtr(oldForm, ctlID_DesignView_Table);
	if (doInsert)
	    InitTable(table);
	else
	    LoadTable(table);
	TblMarkTableInvalid(table);
	TblRedrawTable(table);
    }
}

Boolean
DesignViewHandleEvent(EventPtr event)
{
    FormPtr form;
    FieldPtr fld;
    TablePtr table;
    VoidHand name_handle, h;
    CharPtr name;
    Word i;

    switch (event->eType) {
    case frmOpenEvent:
	/* Determine the number of fields that we will display. */
	if (DesignNewDB)
	    designNumFields = MAX_NUM_FIELDS_FOR_NEW_DB;
	else
	    designNumFields = numFields;

	/* Allocate temporary memory space. */
	allocedNumFields = designNumFields;
	h = MemHandleNew(allocedNumFields * sizeof(DBFieldInfoType));
	designFields = MemHandleLock(h);
	h = MemHandleNew(allocedNumFields * sizeof(OldFieldInfoType));
	oldFieldInfo = MemHandleLock(h);

	for (i = 0; i < allocedNumFields; i++) {
	    oldFieldInfo[i].handle = MemHandleNew(32);
	    oldFieldInfo[i].num = i;
	}

	form = FrmGetActiveForm();
	if (DesignNewDB) {
	    for (i = 0; i < designNumFields; i++) {
		designFields[i].name[0] = '\0';
		designFields[i].type = 0;
		designFields[i].colwidth = 80;
		StrPrintF(oldFieldInfo[i].name, "Fld %d", i + 1);
	    }
	    name = GetStringPtr(stringID_NewDatabase);
	    FrmCopyTitle(form, name);
	    MemPtrUnlock(name);
	    FrmHideObject(form, FrmGetObjectIndex(form, ctlID_DesignView_AddButton));
	    FrmHideObject(form, FrmGetObjectIndex(form, ctlID_DesignView_DelButton));

	} else {
	    MemMove(designFields, fields,
		    designNumFields * sizeof(DBFieldInfoType));
	    for (i = 0; i < designNumFields; i++) {
		StrNCopy(oldFieldInfo[i].name, designFields[i].name,
			 sizeof(oldFieldInfo[i].name) - 1);
		oldFieldInfo[i].name[sizeof(oldFieldInfo[i].name)-1] = '\0';
	    }

	    name = GetStringPtr(stringID_DatabaseDesign);
	    FrmCopyTitle(form, name);
	    MemPtrUnlock(name);
	}

	/* Update the name field. */
	fld = GetObjectPtr(form, ctlID_DesignView_NameField);
	name_handle = MemHandleNew(dmDBNameLength+1);
	name = MemHandleLock(name_handle);
	name[0] = '\0';
	if (! DesignNewDB) {
	    Char dbname[dmDBNameLength+1];
	    UInt cardNo;
	    LocalID dbID;

	    DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
	    DmDatabaseInfo(cardNo, dbID, dbname, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	    StrCopy(name, dbname);
	}
	MemPtrUnlock(name);

	FldSetTextHandle(fld, (Handle) name_handle);
	FldSetMaxChars(fld, dmDBNameLength);

	/* Initialize the table that holds the design information. */
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	InitTable(table);

	/* Draw the form with field #1 appearing at the top. */
	TopVisibleField = 0;
	FrmDrawForm(form);
	return true;

    case frmCloseEvent: {
	CALLBACK_PROLOGUE;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	}

	/* Free the handles allocated for the table. */
	for (i = 0; i < allocedNumFields; i++) {
	    MemHandleFree(oldFieldInfo[i].handle);
	}

	/* Now free the temporary space. */
	h = MemPtrRecoverHandle(designFields);
	MemHandleUnlock(h);
	MemHandleFree(h);
	h = MemPtrRecoverHandle(oldFieldInfo);
	MemHandleUnlock(h);
	MemHandleFree(h);

	CALLBACK_EPILOGUE;
	break;
    }

    case tblSelectEvent:
	if (event->data.tblSelect.column == fieldTypeColumn) {
	    Word fieldNum;

	    fieldNum = TblGetRowID(event->data.tblSelect.pTable,
				   event->data.tblSelect.row);
	    designFields[fieldNum].type = TblGetItemInt(event->data.tblSelect.pTable,event->data.tblSelect.row,event->data.tblSelect.column);
	    return true;
	} else if (event->data.tblSelect.column == oldNameColumn) {
	    /* A tap on the label edits the field name. */
	    TblReleaseFocus(event->data.tblSelect.pTable);
	    TblUnhighlightSelection(event->data.tblSelect.pTable);
	    TblGrabFocus(event->data.tblSelect.pTable,
			 event->data.tblSelect.row, fieldNameColumn);
	    fld = TblGetCurrentField(event->data.tblSelect.pTable);
	    if (fld) {
		FldGrabFocus(fld);
		FldMakeFullyVisible(fld);
	    }
	    return true;
	}
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_DesignView_DoneButton:
	    if (DesignNewDB) {
		if (! UpdateNewDesign())
		    return true;
	    } else {
		if (! UpdateExistingDesign())
		    return true;
	    }

	case ctlID_DesignView_CancelButton:
	    if (DesignNewDB)
		FrmGotoForm(formID_Chooser);
	    else
		FrmGotoForm(formID_ListView);

	    return true;

	case ctlID_DesignView_AddButton:
	    PopupFieldSelectDialog(true);
	    return true;

	case ctlID_DesignView_DelButton:
	    PopupFieldSelectDialog(false);
	    return true;

	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_DesignView_UpButton:
	    Scroll(up);
	    break;
	case ctlID_DesignView_DownButton:
	    Scroll(down);
	    break;
	}
	break;

    case menuEvent:
	return HandleCommonMenuEvent(event->data.menu.itemID);

    }

    return false;
}
