/*
 * DB: Database Selection Form
 * Copyright (C) 1998,1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#include <ExgMgr.h>

#include "enum.h"
#include "callback.h"
#include "db.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define updateCategoryChanged   0x0001
#define updateFontChanged       0x0002
#define updateSortOrderChanged  0x0004

typedef struct
{
    UInt cardNo;
    Char name[dmDBNameLength];
} ChooserRecord;

typedef struct {
    UInt renamedCategories; /* bitfield of categories with a different name */
    char    categoryLabels[dmRecNumCategories][dmCategoryLength];
    Byte    categoryUniqIDs[dmRecNumCategories];
    Byte    lastUniqID;     /* Uniq IDs generated by the device are between */
                            /* 0 - 127.  Those from the PC are 128 - 255.   */
    Byte    reserved1;      /* from the compiler word aligning things       */
    Word    reserved2;
    Word    reserved3;
} ChooserAppInfoType;

static Word TopVisibleDatabase, CurrentCategory = dmAllCategories;
static DmOpenRef ChooserDB = 0;
static Char CategoryName[dmCategoryLength];
static Boolean ShowAllCategories = true;
static Word InfoDatabase = 0;

static void
RescanDatabases(void)
{
    DmSearchStateType state;
    UInt cardNo, recNum, numRecords;
    LocalID dbID;
    Boolean flag;
    VoidHand h;
    ChooserRecord record, *p;
    Boolean found;
    Char name[dmDBNameLength];

    /* First, we make sure all DB databases on the device are in the
     * chooser database.
     */
    flag = true;
    while (DmGetNextDatabaseByTypeCreator(flag, &state, DBTypeID, 0,
					  false, &cardNo, &dbID) == 0) {

	/* Retrieve the database name. */
	DmDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

	/* Scan the chooser database for this database. */
	found = false;
	numRecords = DmNumRecords(ChooserDB);
	for (recNum = 0; recNum < numRecords; ++recNum) {
	    h = DmQueryRecord(ChooserDB, recNum);
	    p = MemHandleLock(h);
	    if (p->cardNo == cardNo && StrCompare(p->name, name) == 0)
		found = true;
	    MemPtrUnlock(p);
	}

	if (!found) {
	    /* Construct the record that we will store. */
	    record.cardNo = cardNo;
	    StrCopy(record.name, name);

	    /* Store the record in the database. */
	    recNum = dmMaxRecordIndex;
	    h = DmNewRecord(ChooserDB, &recNum, sizeof(record));
	    p = MemHandleLock(h);
	    DmWrite(p, 0, &record, sizeof(record));
	    MemPtrUnlock(p);
	    DmReleaseRecord(ChooserDB, recNum, true);
	}

	/* Tell routine to continue search. */
	flag = false;
    }

    /* Second, we proceed through the chooser database and make sure
     * that each database is present on the device.
     */
    numRecords = DmNumRecords(ChooserDB);
    for (recNum = 0; recNum < numRecords; ) {
	h = DmQueryRecord(ChooserDB, recNum);
	p = MemHandleLock(h);
	if (DmFindDatabase(p->cardNo, p->name) == 0) {
	    MemPtrUnlock(p);
	    DmRemoveRecord(ChooserDB, recNum);
	    numRecords = DmNumRecords(ChooserDB);
	} else {
	    MemPtrUnlock(p);
	    ++recNum;
	}
    }
}

static Err
AppInfoInit(DmOpenRef db)
{
    UInt cardNo;
    LocalID dbID, appInfoID;
    VoidHand h;
    ChooserAppInfoType * appInfoPtr;

    if (DmOpenDatabaseInfo(db, &dbID, NULL, NULL, &cardNo, NULL))
	return dmErrInvalidParam;

    if (DmDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0))
	return dmErrInvalidParam;

    if (appInfoID == 0) {
	h = DmNewHandle(db, sizeof(ChooserAppInfoType));
	if (!h)
	    return dmErrMemError;
                
	appInfoID = MemHandleToLocalID(h);
	DmSetDatabaseInfo(cardNo, dbID, 0,0,0,0,0,0,0, &appInfoID, 0,0,0);
    }
    appInfoPtr = MemLocalIDToLockedPtr(appInfoID, cardNo);

    /* Clear the app info block. */
    DmSet(appInfoPtr, 0, sizeof(ChooserAppInfoType), 0);

    /* Initialize the categories. */
    CategoryInitialize((AppInfoPtr) appInfoPtr, 100);

    MemPtrUnlock(appInfoPtr);

    return 0;
}

static void
CreateChooserDB(void)
{
    UInt mode;

    if (HideSecretRecords)
	mode = dmModeReadWrite;
    else
	mode = dmModeReadWrite | dmModeShowSecret;

    /* Create the chooser database. */
    DmCreateDatabase(0, "DBDatabases", DBCreatorID, 'LIST', false);
    ChooserDB = DmOpenDatabaseByTypeCreator('LIST', DBCreatorID, mode);

    /* Initialize the app info block. */
    AppInfoInit(ChooserDB);

    /* Force all databases on the device into the chooser database. */
    RescanDatabases();
}

static void
OpenChooserDB(void)
{
    UInt mode;

    if (HideSecretRecords)
	mode = dmModeReadWrite;
    else
	mode = dmModeReadWrite | dmModeShowSecret;

    ChooserDB = DmOpenDatabaseByTypeCreator('LIST', DBCreatorID, mode);
    if (! ChooserDB) {
	/* Database not found. Recreate it. */
	CreateChooserDB();
    }
}

static void
CloseChooserDB(void)
{
    DmCloseDatabase(ChooserDB);
    ChooserDB = 0;
}

static Int
ChooserCompareRecords(ChooserRecord * r1, ChooserRecord * r2, Int sortOrder,
		      SortRecordInfoPtr info1, 
		      SortRecordInfoPtr info2, VoidHand appInfoH)
{
    return StrCompare(r1->name, r2->name);
}

static void
SortAlphabetic(void)
{
    DmInsertionSort(ChooserDB, (DmComparF *) &ChooserCompareRecords,
		    (Int) prefs.ChooserSortOrder);
}

void
Chooser_AddDatabase(UInt dbCardNo, CharPtr name)
{
    ChooserRecord record;
    Word recNum;
    VoidHand h;
    VoidPtr p;
    UInt cardNo, attr;
    LocalID dbID;

    OpenChooserDB();
    DmOpenDatabaseInfo(ChooserDB, &dbID, 0, 0, &cardNo, 0);

    /* Construct the record that we will store. */
    record.cardNo = dbCardNo;
    StrCopy(record.name, name);

    /* Store the record in the database. */
    recNum = dmMaxRecordIndex;
    h = DmNewRecord(ChooserDB, &recNum, sizeof(record));
    p = MemHandleLock(h);
    DmWrite(p, 0, &record, sizeof(record));
    MemPtrUnlock(p);
    DmReleaseRecord(ChooserDB, recNum, true);

    /* Set the category to whatever was viewed last in chooser. */
    DmRecordInfo(ChooserDB, recNum, &attr, 0, 0);
    attr &= ~dmRecAttrCategoryMask;
    if (prefs.ChooserCurrentCategory == dmAllCategories
	|| prefs.ChooserShowAllCategories)
	attr |= dmUnfiledCategory;
    else
	attr |= prefs.ChooserCurrentCategory;
    DmSetRecordInfo(ChooserDB, recNum, &attr, 0);

    /* If the sort order is alphabetic, then sort the database. We
     * assume there won't be a lot of chooser records so we do a full
     * sort.
     */
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	SortAlphabetic();

    CloseChooserDB();
}

void
Chooser_RenameDatabase(UInt cardNo, CharPtr oldName, CharPtr newName)
{
    UInt numRecords, recNum;

    OpenChooserDB();

    numRecords = DmNumRecords(ChooserDB);
    for (recNum = 0; recNum < numRecords; ++recNum) {
	VoidHand h;
	ChooserRecord *record, tmp_record;

	h = DmGetRecord(ChooserDB, recNum);
	record = MemHandleLock(h);
	if (record->cardNo == cardNo
	    && StrCompare(record->name, oldName) == 0) {
	    /* Construct the new entry. */
	    tmp_record.cardNo = record->cardNo;
	    StrCopy(tmp_record.name, newName);

	    /* Write the new entry into the database. */
	    DmWrite(record, 0, &tmp_record, sizeof(tmp_record));

	    /* Unlock the pointer and record. */
	    MemPtrUnlock(record);
	    DmReleaseRecord(ChooserDB, recNum, true);
	} else {
	    /* No match. Just unlock the pointer and record. */
	    MemPtrUnlock(record);
	    DmReleaseRecord(ChooserDB, recNum, false);
	}
    }

    /* If the sort order is alphabetic, then sort the database. We
     * assume there won't be a lot of chooser records so we do a full
     * sort.
     */
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	SortAlphabetic();

    CloseChooserDB();
}

static Boolean
Seek(UIntPtr indexP, Int offset, Int direction)
{
    DmSeekRecordInCategory(ChooserDB, indexP, offset, direction,
			   CurrentCategory);
    if (DmGetLastErr())
	return false;

    return true;
}

static void
UpdateScrollers(void)
{
    Word upIndex, downIndex, recNum;
    Boolean scrollUp, scrollDown;
    SWord row;
    FormPtr form;
    TablePtr table;

    recNum = TopVisibleDatabase;
    scrollUp = Seek(&recNum, 1, dmSeekBackward);

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_Chooser_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1)
	recNum = (Word) TblGetRowID(table, row);

    scrollDown = Seek(&recNum, 1, dmSeekForward);

    upIndex = FrmGetObjectIndex(form, ctlID_Chooser_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_Chooser_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static void
DrawChooserItem(VoidPtr table, Word row, Word col, RectanglePtr bounds)
{
    Word recNum;
    VoidHand h;
    ChooserRecord * record;
    FontID oldFont;

    CALLBACK_PROLOGUE;

    oldFont = FntSetFont(stdFont);

    recNum = (Word) TblGetRowID(table, row);
    h = DmQueryRecord(ChooserDB, recNum);
    record = MemHandleLock(h);
    WinDrawChars(record->name, StrLen(record->name),
		 bounds->topLeft.x, bounds->topLeft.y);
    MemPtrUnlock(record);

    FntSetFont(oldFont);

    CALLBACK_EPILOGUE;
}

static void
LoadTable(TablePtr table)
{
    Word numRows, row, lineHeight, dataHeight, tableHeight;
    RectangleType r;
    UInt recordNum;
    FontID current_font;
    ULong uniqueID;

    /* Retrieve the height of the table. */
    TblGetBounds(table, &r);
    tableHeight = r.extent.y;

    /* Determine the height of the font that we draw with. */
    current_font = FntSetFont(stdFont);
    lineHeight = FntLineHeight();
    FntSetFont(current_font);

    /* Make sure that CurrentCategory is set correctly. */
    if (ShowAllCategories) {
	CurrentCategory = dmAllCategories;
    }

    numRows = TblGetNumberOfRows(table);
    recordNum = TopVisibleDatabase;
    dataHeight = 0;

    for (row = 0; row < numRows; row++) {
	/* Stop the loop if no more records are available. */
	if (! Seek(&recordNum, 0, dmSeekForward))
	    break;

	/* Stop the loop if there is no more space. */
	if (! (tableHeight >= dataHeight + lineHeight))
	    break;

	TblSetRowID(table, row, (Word) recordNum);
	TblSetItemStyle(table, row, 0, customTableItem);
	TblSetRowHeight(table, row, lineHeight);

	DmRecordInfo(ChooserDB, recordNum, 0, &uniqueID, 0);
	if ( (!TblRowUsable(table, row))
	     || (TblGetRowData(table, row) != uniqueID)) {
	    TblSetRowUsable(table, row, true);
	    TblSetRowData(table, row, uniqueID);
	    TblMarkRowInvalid(table, row);
	}

	/* Increment the record number and the data height. */
	recordNum++;
	dataHeight += lineHeight;
    }

    /* Mark the remainder of the rows as unusable. */
    while (row < numRows) {
        TblSetRowUsable(table, row, false);
        row++;
    }

    UpdateScrollers();
}

static void
InitTable(TablePtr table)
{
    Word row, numRows;

    TblSetCustomDrawProcedure(table, 0, DrawChooserItem);
    TblSetColumnUsable(table, 0, true);

    /* Mark all of the data rows unusable so they are reloaded. */
    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++) {
        TblSetRowUsable(table, row, false);
    }

    LoadTable(table);
}

static void
Scroll(DirectionType direction)
{
    UInt newTopVisibleDatabase;

    newTopVisibleDatabase = TopVisibleDatabase;

    if (direction == down) {
	Seek(&newTopVisibleDatabase, 1, dmSeekForward);
    } else {
	Seek(&newTopVisibleDatabase, 1, dmSeekBackward);
    }

    if (TopVisibleDatabase != newTopVisibleDatabase) {
	TopVisibleDatabase = newTopVisibleDatabase;
	FrmUpdateForm(FrmGetActiveFormID(), 0);
    }
}

static Err
WriteDBData(const void * dataP, ULong * sizeP, void * userDataP)
{
    Err err;

    /* Try to send as many bytes as were requested by the caller. */
    *sizeP = ExgSend((ExgSocketPtr)userDataP, (void *)dataP, *sizeP, &err);
    return err;
}

static Err
SendDatabase(Word cardNo, LocalID dbID)
{
    Char name[dmDBNameLength+4];
    ExgSocketType exgSocket;
    Err err;

    /* Form the file name that we will send. */
    DmDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    StrCat(name, ".pdb");

    /* Create exgSocket structure. */
    MemSet(&exgSocket, sizeof(exgSocket), 0);
    exgSocket.description = name;
    exgSocket.name = name;

    /* Initiate the exchange. */
    err = ExgPut(&exgSocket);
    if (!err) {
	err = ExgDBWrite(WriteDBData, &exgSocket, NULL, dbID, cardNo);
	err = ExgDisconnect(&exgSocket, err);
    }

    return err;
}

static Word
SelectCategory(void)
{
    FormPtr form;
    TablePtr table;
    Word category;
    Boolean categoryEdited;

    /* Process the category popup list. */
    category = CurrentCategory;

    form = FrmGetActiveForm();
    categoryEdited = CategorySelect(ChooserDB, form,
				    ctlID_Chooser_CategoryTrigger,
				    ctlID_Chooser_CategoryList, true,
				    &category, CategoryName, 1, 0);
        
    if (category == dmAllCategories)
	ShowAllCategories = true;
    else
	ShowAllCategories = false;

    if (categoryEdited || (category != CurrentCategory)) {
	CurrentCategory = category;
	TopVisibleDatabase = 0;
	FrmUpdateForm(FrmGetActiveFormID(), updateCategoryChanged);
    }

    return category;
}

static Boolean
UpdateForm(FormPtr form, Word updateCode)
{
    TablePtr table;
    ControlPtr ctl;

    if (updateCode & frmRedrawUpdateCode)
	return false;

    table = GetObjectPtr(form, ctlID_Chooser_Table);
    ctl = GetObjectPtr(form, ctlID_Chooser_CategoryTrigger);

    if (updateCode & updateFontChanged) {
	TblEraseTable(table);
	InitTable(table);
	TblDrawTable(table);
	return true;
    }

    if (updateCode & updateCategoryChanged) {
	/* Set the category trigger to the new category. */
	CategoryGetName(ChooserDB, CurrentCategory, CategoryName);
	CategorySetTriggerLabel(ctl, CategoryName);
    }

    /* Force an update of the table. */
    LoadTable(table);
    TblRedrawTable(table);

    return true;
}

static WinHandle InvertMoveIndicator(RectanglePtr itemR, WinHandle savedBits, 
				     Boolean draw)
{
    UInt i;
    Word err;
    WinHandle winH = 0;
    RectangleType indictatorR;
    CustomPatternType pattern;
    CustomPatternType savedPattern;

    indictatorR.topLeft.x = itemR->topLeft.x;
    indictatorR.topLeft.y = itemR->topLeft.y + itemR->extent.y - 2;
    indictatorR.extent.x = itemR->extent.x;
    indictatorR.extent.y = 2;
                                
    if (draw) {
	WinGetPattern (savedPattern);
        
	for (i = 0; i < sizeof (CustomPatternType) / sizeof (*pattern); i++)
	    pattern[i]= 0xAA55;
        
	WinSetPattern(pattern);
        
	winH = WinSaveBits(&indictatorR, &err);
        
	WinFillRectangle(&indictatorR, 0);
        
	WinSetPattern(savedPattern);
    } else {
	WinRestoreBits(savedBits,
		       indictatorR.topLeft.x, indictatorR.topLeft.y);
    }
        
    return (winH);
}

static void
SelectItem(EventPtr event)
{
    Short row;
    Word selectedRow;
    Word column;
    Word recordNum;
    Word selectedRecord;
    Short x, y;
    Boolean penDown = true;
    Boolean moving = false;
    Boolean selected = true;
    TablePtr table;
    WinHandle savedBits;
    RectangleType r;

    row = event->data.tblEnter.row;
    column = event->data.tblEnter.column;
    table = event->data.tblEnter.pTable;

    // Highlight the item the pen when down on.
    selectedRecord = TblGetRowID(table, row);
    TblGetItemBounds(table, row, column, &r);
    WinInvertRectangle(&r, 0);

    while (true) {
	PenGetPoint (&x, &y, &penDown);
	if (! penDown)
	    break;
		
	if (! moving) {
	    if (prefs.ChooserSortOrder == chooserSortOrderManual) {
		if (! RctPtInRectangle(x, y, &r)) {
		    moving = true;
		    TblGetItemBounds(table, row, column, &r);
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
	    } else {
		selected = RctPtInRectangle(x, y, &r);
	    }
	}

	else if (! RctPtInRectangle(x, y, &r)) {
	    // Above the first item ?
	    if (row < 0) {
		if (y >= r.topLeft.y) {
		    row++;
		    InvertMoveIndicator(&r, savedBits, false);
		    r.topLeft.y -= r.extent.y;
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}					
	    }

	    // Move up.
	    else if (y < r.topLeft.y) {
		recordNum = TblGetRowID(table, row);
		if (Seek(&recordNum, 1, dmSeekBackward)) {
		    InvertMoveIndicator(&r, savedBits, false);
		    if (row)
			row--;
		    else {
			Scroll(up);
			if (TblFindRowID(table, selectedRecord, &selectedRow))
			    {
				TblGetItemBounds(table, selectedRow,
						 column, &r);
				WinInvertRectangle(&r, 0);
			    }
		    }
		    TblGetItemBounds(table, row, column, &r);
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
		else if (row == 0) {
		    row--;
		    InvertMoveIndicator(&r, savedBits, false);
		    r.topLeft.y -= r.extent.y;
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
	    }

	    // Move down
	    else {
		recordNum = TblGetRowID(table, row);
		if (Seek(&recordNum, 1, dmSeekForward)) {
		    InvertMoveIndicator(&r, savedBits, false);
		    if (row < TblGetLastUsableRow(table))
			row++;
		    else {
			Scroll(down);
			if (TblFindRowID(table, selectedRecord, &selectedRow))
			    {
				TblGetItemBounds(table,
						 selectedRow, column, &r);
				WinInvertRectangle (&r, 0);
			    }
		    }
		    TblGetItemBounds(table, row, column, &r);
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
	    }
	}
    }
		

    // Turn off the move indicator, if it is on.
    if (moving) {
	savedBits = InvertMoveIndicator(&r, savedBits, false);
    }
	
    // If the highlighted item is visible, unhighlight it.
    if (TblFindRowID(table, selectedRecord, &selectedRow)) {
	TblGetItemBounds(table, selectedRow, column, &r);
	WinInvertRectangle(&r, 0);
    }

    if (moving) {
	if (row >= 0) {
	    recordNum = TblGetRowID(table, row);
	    if (selectedRecord == recordNum)
		return;
			
	    recordNum++;
	} else {
	    recordNum = TblGetRowID(table, 0);;
	}

	DmMoveRecord(ChooserDB, selectedRecord, recordNum);
		
	if (selectedRecord < TopVisibleDatabase)
	    TopVisibleDatabase--;

	LoadTable(table);
	TblRedrawTable(table);
    }
		
    // If we didn't move the item, then it's been selected so we will
    // just force a tblSelectEvent to trigger the normal handler.
    else if (prefs.ChooserSortOrder == chooserSortOrderManual || selected) {
	EventType e;

	EvtCopyEvent(event, &e);
	e.eType = tblSelectEvent;
	EvtAddEventToQueue(&e);
    }
}

static void PopupOptionsDialog(void);

Boolean
ChooserHandleEvent_1(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    ControlPtr ctl;
    Word index;
    DWord romVersion;
    ChooserRecord * record;
    VoidHand h;
    UInt cardNo;
    LocalID dbID;
    Char name[dmDBNameLength];

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_Chooser_Table);
	TopVisibleDatabase = 0;
	CurrentCategory = prefs.ChooserCurrentCategory;
	ShowAllCategories = prefs.ChooserShowAllCategories;

	/* Open the chooser database and do a rescan if need be. */
	OpenChooserDB();
	if (prefs.ChooserForceRescan) {
	    RescanDatabases();
	    prefs.ChooserForceRescan = false;
	}

	InitTable(table);
	FrmSetControlGroupSelection(form, 1, ctlID_Chooser_OpenButton);

	/* Figure out if we should allow beaming.  */
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < sysMakeROMVersion(3,0,0,0,0)) {
	    FrmHideObject(form,
			  FrmGetObjectIndex(form, ctlID_Chooser_BeamButton));
	}

	/* Set the category name in the trigger. */
	ctl = GetObjectPtr(form, ctlID_Chooser_CategoryTrigger);
	CategoryGetName(ChooserDB, CurrentCategory, CategoryName);
	CategorySetTriggerLabel(ctl, CategoryName);

	FrmDrawForm(form);

	return true;

    case frmUpdateEvent:
	form = FrmGetActiveForm();
	return UpdateForm(form, event->data.frmUpdate.updateCode);

    case frmCloseEvent:
	CloseChooserDB();
	prefs.ChooserCurrentCategory = CurrentCategory;
	prefs.ChooserShowAllCategories = ShowAllCategories;
	break;

    case tblEnterEvent:
	SelectItem(event);
	return true;

    case tblSelectEvent:
	index = TblGetRowID(event->data.tblSelect.pTable,
			    event->data.tblSelect.row);
	h = DmQueryRecord(ChooserDB, index);
	record = MemHandleLock(h);
	cardNo = record->cardNo;
	StrCopy(name, record->name);
	dbID = DmFindDatabase(cardNo, name);
	MemPtrUnlock(record);

	form = FrmGetActiveForm();
	if (CtlGetValue(GetObjectPtr(form, ctlID_Chooser_DelButton))) {
	    TblUnhighlightSelection(event->data.tblSelect.pTable);
	    if (FrmCustomAlert(alertID_ConfirmDatabaseDelete,
			       name, " ", " ") == 0) {

		DmDeleteDatabase(cardNo, dbID);
		DmRemoveRecord(ChooserDB, index);
		FrmUpdateForm(FrmGetActiveFormID(), 0);
	    }
	} else if (CtlGetValue(GetObjectPtr(form, ctlID_Chooser_InfoButton))) {
	    TblUnhighlightSelection(event->data.tblSelect.pTable);
	    InfoDatabase = index;
	    FrmPopupForm(formID_ChooserInfoDialog);
	} else if (CtlGetValue(GetObjectPtr(form, ctlID_Chooser_BeamButton))) {
	    TblUnhighlightSelection(event->data.tblSelect.pTable);
	    SendDatabase(cardNo, dbID);
	} else {
	    if (! OpenDatabase(cardNo, dbID)) {
		FrmGotoForm(formID_ListView);
	    } else {
		FrmAlert(alertID_CannotOpenDB);
	    }
	}
	return true;
					      
    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_Chooser_NewButton:
	    DesignNewDB = true;
	    FrmGotoForm(formID_DesignView);
	    return true;

	case ctlID_Chooser_CategoryTrigger:
	    SelectCategory();
	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_Chooser_UpButton:
	    Scroll(up);
	    break;
	case ctlID_Chooser_DownButton:
	    Scroll(down);
	    break;
	}
	break;

    case keyDownEvent:
        switch (event->data.keyDown.chr) {
        case pageUpChr:
            Scroll(up);
            return true;

        case pageDownChr:
            Scroll(down);
            return true;
        }
        break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_RescanDatabases:
	    RescanDatabases();
	    FrmUpdateForm(FrmGetActiveFormID(), 0);
	    return true;

	case menuitemID_NewDatabase:
	    DesignNewDB = true;
	    FrmGotoForm(formID_DesignView);
	    return true;

	case menuitemID_ChooserDisplayOpts:
	    PopupOptionsDialog();
	    return true;

	default:
	    return HandleCommonMenuEvent(event->data.menu.itemID);
	}
	break;
    }

    return false;
}

Boolean
ChooserHandleEvent(EventPtr event)
{
    Boolean result;

    CALLBACK_PROLOGUE;
    result = ChooserHandleEvent_1(event);
    CALLBACK_EPILOGUE;

    return result;
}

static void
PopupOptionsDialog(void)
{
    FormPtr form, oldForm;
    CharPtr labels[3] = {0, 0, 0};
    ControlPtr ctl;
    ListPtr lst;
    Byte OldChooserSortOrder = prefs.ChooserSortOrder;

    oldForm = FrmGetActiveForm();
    form = FrmInitForm(formID_ChooserOptsDialog);
    FrmSetActiveForm(form);

    labels[0] = GetStringPtr(stringID_Alphabetic);
    labels[1] = GetStringPtr(stringID_Manual);

    ctl = GetObjectPtr(form, ctlID_ChooserOptsDialog_SortOrderTrigger);
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	CtlSetLabel(ctl, labels[0]);
    else
	CtlSetLabel(ctl, labels[1]);

    lst = GetObjectPtr(form, ctlID_ChooserOptsDialog_SortOrderList);
    LstSetListChoices(lst, labels, 2);
    LstSetHeight(lst, 2);
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	LstSetSelection(lst, 0);
    else
	LstSetSelection(lst, 1);

    if (FrmDoDialog(form) == ctlID_ChooserOptsDialog_OkayButton) {
	lst = GetObjectPtr(form, ctlID_ChooserOptsDialog_SortOrderList);
	if (LstGetSelection(lst) == 0)
	    prefs.ChooserSortOrder = chooserSortOrderAlphabetic;
	else
	    prefs.ChooserSortOrder = chooserSortOrderManual;
    }

    FrmDeleteForm(form);
    FrmSetActiveForm(oldForm);

    MemPtrUnlock(labels[0]);
    MemPtrUnlock(labels[1]);

    if (OldChooserSortOrder != prefs.ChooserSortOrder) {
	FrmUpdateForm(formID_Chooser, updateSortOrderChanged);
	if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	    SortAlphabetic();
    }
}

static Word
ApplyChanges(Word recNum, Boolean CategoryEdited,
	     UInt category, Boolean secret)
{
    ControlPtr ctl;
    Word updateCode = 0;
    Boolean dirty = false;
    UInt attr;

    /* Retrieve the existing status of the record. */
    DmRecordInfo(ChooserDB, recNum, &attr, 0, 0);

    /* If the category changed, store the new category. */
    if (CategoryEdited || (attr & dmRecAttrCategoryMask) != category) {
	attr &= ~(dmRecAttrCategoryMask);
	attr |= category;

	dirty = true;
	updateCode |= updateCategoryChanged;
    }

    if (((attr & dmRecAttrSecret) == dmRecAttrSecret) != secret) {
	if (secret)
	    attr |= dmRecAttrSecret;
	else
	    attr &= ~(dmRecAttrSecret);
	dirty = true;
    }

    /* Save the new category and/or secret status. */
    if (dirty) {
	attr |= dmRecAttrDirty;
	DmSetRecordInfo(ChooserDB, recNum, &attr, 0);
    }

    return updateCode;
}

static Boolean
ChooserInfoDialogHandleEvent_1(EventPtr event)
{
    static Char InfoName[dmDBNameLength];
    static Boolean CategoryEdited;
    static UInt InfoCategory;

    FormPtr form;
    ControlPtr ctl;
    ChooserRecord * record;
    VoidHand h;
    CharPtr label;
    UInt attr;
    Word updateCode;

    form = FrmGetActiveForm();

    switch (event->eType) {
    case frmOpenEvent:
	/* Initialze the state variables. */
	CategoryEdited = false;

	/* Set the title to the database name. */
	h = DmQueryRecord(ChooserDB, InfoDatabase);
	record = MemHandleLock(h);
	StrCopy(InfoName, record->name);
	MemPtrUnlock(record);
	FrmSetTitle(form, InfoName);

	/* Set the category trigger label. */
	DmRecordInfo(ChooserDB, InfoDatabase, &attr, 0, 0);
	InfoCategory = attr & dmRecAttrCategoryMask;
	ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_CategoryTrigger);
	label = CtlGetLabel(ctl);
        CategoryGetName(ChooserDB, InfoCategory, label);
        CategorySetTriggerLabel(ctl, label);

	/* Set the secret checkbox to current status. */
	ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_SecretCheckbox);
	if (attr & dmRecAttrSecret)
	    CtlSetValue(ctl, 1);
	else
	    CtlSetValue(ctl, 0);

	/* Finally, draw the form. */
	FrmDrawForm(form);

	return true;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_ChooserInfoDialog_OkayButton:
	    ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_SecretCheckbox);
	    updateCode = ApplyChanges(InfoDatabase, CategoryEdited,
				      InfoCategory, CtlGetValue(ctl));
	    FrmReturnToForm(0);
	    FrmUpdateForm(formID_Chooser, updateCode);
	    return true;

	case ctlID_ChooserInfoDialog_CancelButton:
	    FrmReturnToForm(0);
	    if (CategoryEdited)
		FrmUpdateForm(formID_Chooser, updateCategoryChanged);
	    return true;

	case ctlID_ChooserInfoDialog_CategoryTrigger:
	    ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_CategoryTrigger);
	    label = CtlGetLabel(ctl);
	    CategoryEdited = CategorySelect(ChooserDB, form,
					    ctlID_ChooserInfoDialog_CategoryTrigger,
					    ctlID_ChooserInfoDialog_CategoryList,
					    false, &InfoCategory, label, 1, 0);
	    return true;

	}
	return false;

    }

    return false;
}

Boolean
ChooserInfoDialogHandleEvent(EventPtr event)
{
    Boolean result;

    CALLBACK_PROLOGUE;
    result = ChooserInfoDialogHandleEvent_1(event);
    CALLBACK_EPILOGUE;

    return result;
}
