/*
 * DB: Sort Editor (UI for sorting a data source)
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include <TxtGlue.h>
#include <WinGlue.h>

#include "enum.h"
#include "db.h"
#include "grid.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define col_popup     0
#define col_order     1
#define col_sensitive 2
#define col_field     3

typedef struct {
    UInt16 fieldNum;
    Boolean descending;
    Boolean case_sensitive;
    UInt32 uniqueID;
} SortEntry;

static UInt16 numEntries = 0;
static UInt32 NextUniqueID;
static SortEntry * entries = 0;
static MemHandle entries_handle = 0;
static char ** field_names = 0;

static UInt16 TopVisibleEntry;

static void DrawPopupIndicator(void *, Int16, Int16, RectangleType *) SECT_UI;
static void UpdateScrollers(void) SECT_UI;
static void LoadTable(TablePtr table) SECT_UI;
static void InitTable(TablePtr table) SECT_UI;
static void DrawTableLabels(TableType* table) SECT_UI;
static void Scroll(WinDirectionType direction) SECT_UI;
static void InsertEntry(UInt16) SECT_UI;
static void DeleteEntry(UInt16) SECT_UI;
static void FreeData(void) SECT_UI;
static void DoSort(void) SECT_UI;
static void HandleContextMenu(FormType *, TableType *, Int16) SECT_UI;


static void
DrawPopupIndicator(void * table, Int16 row, Int16 col, RectangleType *b)
{
    Coord x, y, width;

    width = 8;
    x = b->topLeft.x + 1;
    y = b->topLeft.y + (TblGetRowHeight(table, row) - (width / 2)) / 2;

    while (width > 0) {
	WinDrawLine(x, y, x + width - 1, y);
	x += 1;
	y += 1;
	width -= 2;
    }
}

static void UpdateScrollers(void)
{
    UInt16 upIndex, downIndex, fieldNum;
    Boolean scrollUp, scrollDown;
    Int16 row;
    FormPtr form;
    TablePtr table;

    scrollUp = TopVisibleEntry != 0;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_SortEditor_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1) {
	fieldNum = TblGetRowID(table, row);
	scrollDown = fieldNum < numEntries - 1;
    } else {
	scrollDown = false;
    }

    upIndex = FrmGetObjectIndex(form, ctlID_SortEditor_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_SortEditor_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static void LoadTable(TablePtr table)
{
    UInt16 numRows, row, entry;
    ListPtr list;
    FormPtr form;

    form = FrmGetActiveForm();
    list = GetObjectPtr(form, ctlID_SortEditor_PopupList);

    numRows = TblGetNumberOfRows(table);
    entry = TopVisibleEntry;
    for (row = 0; row < numRows; row++) {
	/* Stop the loop if we reached the end of the entries. */
	if (entry >= numEntries)
	    break;

	if (! TblRowUsable(table, row)
	    || TblGetRowData(table, row) != entries[entry].uniqueID) {

	    TblSetRowUsable(table, row, true);
	    TblSetRowSelectable(table, row, true);
	    TblSetRowHeight(table, row, FntLineHeight());
	    TblSetRowID(table, row, entry);
	    TblSetRowData(table, row, entries[entry].uniqueID);
	    TblMarkRowInvalid(table, row);

	    TblSetItemStyle(table, row, col_popup, customTableItem);

	    TblSetItemStyle(table, row, col_order, checkboxTableItem);
	    if (entries[entry].descending)
		TblSetItemInt(table, row, col_order, 1);
	    else
		TblSetItemInt(table, row, col_order, 0);

	    TblSetItemStyle(table, row, col_sensitive, checkboxTableItem);
	    if (entries[entry].case_sensitive)
		TblSetItemInt(table, row, col_sensitive, 1);
	    else
		TblSetItemInt(table, row, col_sensitive, 0);

	    TblSetItemStyle(table, row, col_field, popupTriggerTableItem);
	    TblSetItemPtr(table, row, col_field, list);
	    TblSetItemInt(table, row, col_field, entries[entry].fieldNum);

	}

	/* Advance to the next entry. */
	entry++;
    }

    /* Disable the remainder of the fields. */
    while (row < numRows) {
	TblSetRowUsable(table, row, false);
	row++;
    }

    /* Enable/disable the scroll arrows according to the current positions. */
    UpdateScrollers();
}

static void InitTable(TablePtr table)
{
    UInt16 row, numRows;
    RectangleType bounds;

    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; ++row) {
	TblSetRowUsable(table, row, false);
    }

    TblSetColumnUsable(table, col_popup, true);
    TblSetColumnWidth(table, col_popup, 10);
    TblSetCustomDrawProcedure(table, col_popup, DrawPopupIndicator);

    TblSetColumnUsable(table, col_order, true);
    TblSetColumnWidth(table, col_order, 14);

    TblSetColumnUsable(table, col_sensitive, true);
    TblSetColumnWidth(table, col_sensitive, 14);

    TblGetBounds(table, &bounds);
    TblSetColumnUsable(table, col_field, true);
    TblSetColumnWidth(table, col_field, bounds.extent.x - 10 - 14 - 14);

    LoadTable(table);
}

static void DrawTableLabels(TableType* table)
{
    RectangleType bounds, r;
    Coord x;
    FontID oldFont;
    char * str;

    TblGetBounds(table, &bounds);

    oldFont = FntSetFont(boldFont);

    r.topLeft.x = bounds.topLeft.x;
    r.topLeft.y = bounds.topLeft.y - FntLineHeight();
    r.extent.x = bounds.extent.x;
    r.extent.y = FntLineHeight();
    WinEraseRectangle(&r, 0);

    str = GetStringResource(stringID_Sort_Label_D);
    x = r.topLeft.x + TblGetColumnWidth(table, 0);
    x += (TblGetColumnWidth(table, 0) - FntCharsWidth(str, StrLen(str))) / 2;
    WinDrawChars(str, StrLen(str), x, r.topLeft.y);
    PutStringResource(str);

    str = GetStringResource(stringID_Sort_Label_CS);
    x = r.topLeft.x + TblGetColumnWidth(table, 0);
    x += TblGetColumnWidth(table, 1);
    x += (TblGetColumnWidth(table, 2) - FntCharsWidth(str, StrLen(str))) / 2;
    WinDrawChars(str, StrLen(str), x, r.topLeft.y);
    PutStringResource(str);

    FntSetFont(oldFont);
}

static void Scroll(WinDirectionType direction)
{
    UInt16 newTopVisibleEntry;

    newTopVisibleEntry = TopVisibleEntry;

    if (direction == winDown) {
	newTopVisibleEntry++;
    } else {
	newTopVisibleEntry--;
    }

    if (TopVisibleEntry != newTopVisibleEntry) {
	FormPtr form;
	TablePtr table;

	TopVisibleEntry = newTopVisibleEntry;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_SortEditor_Table);
	LoadTable(table);
	TblRedrawTable(table);
    }
}

static void
InsertEntry(UInt16 pos)
{
    UInt16 i;

    /* Reallocate the entries_handle to fit new column. */
    MemHandleUnlock(entries_handle);
    MemHandleResize(entries_handle, (numEntries + 1) * sizeof(SortEntry));
    entries = MemHandleLock(entries_handle);

    /* Move the entries after the insert position to the right. */
    if (pos < numEntries) {
        for (i = numEntries; i != pos; --i) {
            MemMove(&entries[i], &entries[i - 1], sizeof(entries[i]));
        }
    }

    /* Fill in the new entry. */
    entries[pos].fieldNum = 0;
    entries[pos].descending = false;
    entries[pos].case_sensitive = false;
    entries[pos].uniqueID = NextUniqueID++;

    /* Increment the number of entries. */
    numEntries++;
}

static void
DeleteEntry(UInt16 entry)
{
    UInt16 i;

    /* Move every width handle and column info above this point down. */
    for (i = entry + 1; i < numEntries; ++i) {
	entries[i-1] = entries[i];
    }

    /* Reallocate the entries_handle to fit. */
    MemHandleUnlock(entries_handle);
    MemHandleResize(entries_handle, (numEntries - 1) * sizeof(SortEntry));
    entries = MemHandleLock(entries_handle);

    /* Decrement the number of columns. */
    numEntries--;
}

static void
FreeData(void)
{
    if (entries) {
	MemPtrUnlock(entries);
	entries = 0;
	MemHandleFree(entries_handle);
	entries_handle = 0;
    }

    if (field_names) {
	MemPtrFree(field_names);
	field_names = 0;
    }
}

static void
DoSort(void)
{
    DataSourceSortInfo info;
    DataSourceSortField fields[numEntries];
    void * indicator_context;
    UInt16 i;

    /* Build the information used by the standard sorting callback. */
    info.numFields = numEntries;
    info.fields = &fields[0];
    for (i = 0; i < info.numFields; ++i) {
	info.fields[i].fieldNum = entries[i].fieldNum;
	if (entries[i].descending)
	    info.fields[i].direction = SORT_DIRECTION_DESCENDING;
	else
	    info.fields[i].direction = SORT_DIRECTION_ASCENDING;
	info.fields[i].case_sensitive = entries[i].case_sensitive;
    }

    /* Tell the data source to sort itself. */
    indicator_context = DrawWorkingIndicator();
    CurrentSource->ops.Sort(CurrentSource, DS_StandardSortCallback, &info);
    EraseWorkingIndicator(indicator_context);
}

static void
HandleContextMenu(FormType* form, TableType* table, Int16 row)
{
    ListType* list;
    RectangleType r;
    char * choices[3];

    /* Remove any selection UI that the table control does. */
    TblUnhighlightSelection(table);

    /* Make copies of strings to be used for the list choices. */
    choices[0] = CopyStringResource(stringID_InsertBefore);
    choices[1] = CopyStringResource(stringID_InsertAfter);
    choices[2] = CopyStringResource(stringID_Remove);

    /* Retrieve the popup choices list. */
    list = GetObjectPtr(form, ctlID_SortEditor_InserterList);

    /* Set the location for the popup list. */
    TblGetItemBounds(table, row, col_popup, &r);
    LstSetPosition(list, r.topLeft.x, r.topLeft.y);

    /* Set the choices which will be presented to the user. */
    LstSetSelection(list, noListSelection);
    if (numEntries == 1) {
	LstSetListChoices(list, choices, 2);
	LstSetHeight(list, 2);
    } else {
	LstSetListChoices(list, choices, 3);
	LstSetHeight(list, 3);
    }

    /* Let the user choose an action. */
    LstPopupList(list);

    /* Insert or delete a column as necessary. */
    if (LstGetSelection(list) != noListSelection) {
	UInt16 entry = TblGetRowID(table, row);

	switch (LstGetSelection(list)) {
	case 0:
	    InsertEntry(entry);
	    break;

	case 1:
	    InsertEntry(entry + 1);
	    break;

	case 2:
	    /* Remove the row. */
	    DeleteEntry(entry);
	    break;
	}

	/* Redraw the table now that we have changed the ordering. */
	LoadTable(table);
	TblRedrawTable(table);
    }

    /* Free the list choices that we allocated earlier. */
    MemPtrFree(choices[0]);
    MemPtrFree(choices[1]);
    MemPtrFree(choices[2]);
}

Boolean
SortEditor_HandleEvent(EventPtr event)
{
    FormType* form;
    TableType* table;
    ListType* list;
    FieldType* field;
    MemHandle h;
    UInt16 i;

    switch (event->eType) {
    case frmOpenEvent:
	/* Retrieve a pointer to the current form. */
	form = FrmGetActiveForm();

	/* Load the definition that will be edited. */
	numEntries = 1;
	entries_handle = MemHandleNew(numEntries * sizeof(SortEntry));
	entries = MemHandleLock(entries_handle);
	entries[0].fieldNum = 0;
	entries[0].descending = false;
	entries[0].case_sensitive = false;

	/* Allocate and fill the field names for the popup list. */
	field_names = MemPtrNew(CurrentSource->schema.numFields
				* sizeof(char *));
	for (i = 0; i < CurrentSource->schema.numFields; ++i) {
	    field_names[i] = CurrentSource->schema.fields[i].name;
	}

	/* Make the popup list use the field names data. */
	list = GetObjectPtr(form, ctlID_SortEditor_PopupList);
	LstSetListChoices(list, field_names, CurrentSource->schema.numFields);
	LstSetHeight(list, CurrentSource->schema.numFields);

	/* Allocate a handle for the name field. (will support in 0.3.2) */
	h = MemHandleNew(32);
	MemSet(MemHandleLock(h), 32, 0);
	MemHandleUnlock(h);
	field = GetObjectPtr(form, ctlID_SortEditor_NameField);
	FldSetTextHandle(field, h);

	/* Initialize the table. */
	TopVisibleEntry = 0;
	table = GetObjectPtr(form, ctlID_SortEditor_Table);
	InitTable(table);

	/* Remove the name label and field from the form. */
	FrmHideObject(form, FrmGetObjectIndex(form, ctlID_SortEditor_NameLabel));
	FrmHideObject(form, FrmGetObjectIndex(form, ctlID_SortEditor_NameField));

	/* Draw the form. */
	FrmDrawForm(form);
	DrawTableLabels(table);

	return true;

    case frmCloseEvent:
	FreeData();
	break;

    case frmUpdateEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_SortEditor_Table);
	FrmDrawForm(form);
	DrawTableLabels(table);
	return true;

    case tblSelectEvent:
	if (event->data.tblSelect.column == col_popup) {
	    HandleContextMenu(FrmGetActiveForm(),
			      event->data.tblSelect.pTable,
			      event->data.tblSelect.row);
	    return true;
	} else {
	    UInt16 entry = TblGetRowID(event->data.tblSelect.pTable,
				       event->data.tblSelect.row);
	    UInt16 entry_int = TblGetItemInt(event->data.tblSelect.pTable,
					    event->data.tblSelect.row,
					    event->data.tblSelect.column);

	    switch (event->data.tblSelect.column) {
	    case col_field:
		entries[entry].fieldNum = entry_int;
		break;

	    case col_order:
		entries[entry].descending = (entry_int) ? true : false;
		break;

	    case col_sensitive:
		entries[entry].case_sensitive = (entry_int) ? true : false;
		break;
	    }

	    return true;
	}
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_SortEditor_DoneButton:
	    /* If the data source is in read-only mode, discard any edits. */
	    if (CurrentSource->readonly) {
		FrmGotoForm(formID_ListView);
		return true;
	    }
	    DoSort();
	    FrmGotoForm(formID_ListView);
	    return true;

	case ctlID_SortEditor_CancelButton:
	    FrmGotoForm(formID_ListView);
	    return true;

	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_SortEditor_UpButton:
	    Scroll(winUp);
	    break;

	case ctlID_SortEditor_DownButton:
	    Scroll(winDown);
	    break;
	}
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_Help:
	    FrmHelp(stringID_Help_SortEditor);
	    return true;

	default:
	    return HandleCommonMenuEvent(event->data.menu.itemID);
	}
	break;

    default:
	break;

    }

    return false;
}
