/*
 * DB: List View Names Editor
 * Copyright (C) 2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file contains the GUI code which allows the editing of which
 * list views appear in a database. Actual editing of a list view is
 * left to the list view editor.
 */

#include <PalmOS.h>

#include "enum.h"
#include "db.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

static void GetListViewNames(char*** names_ptr, UInt16* count_ptr) SECT_UI;
static void PutListViewNames(char ** names) SECT_UI;
static void AppendNewListView(DataSource * source) SECT_UI;
static void UpdateButtons(FormType* form) SECT_UI;

/* Return an array of strings containing the names of all the list views. */
static void
GetListViewNames(char*** names_ptr, UInt16* count_ptr)
{
    char ** names;
    UInt16 i, count;

    count = CurrentSource->ops.GetNumOfViews(CurrentSource);
    names = MemPtrNew(count * sizeof(char *));
    for (i = 0; i < count; ++i) {
	DataSourceView* view = CurrentSource->ops.GetView(CurrentSource, i);
	names[i] = MemPtrNew(1 + StrLen(view->name));
	StrCopy(names[i], view->name);
	CurrentSource->ops.PutView(CurrentSource, i, view);
    }

    *names_ptr = names;
    *count_ptr = count;
}

/* Free the memory associated with the list view names list. */
static void
PutListViewNames(char ** names)
{
    UInt16 i, count;

    count = CurrentSource->ops.GetNumOfViews(CurrentSource);
    for (i = 0; i < count; ++i) {
	MemPtrFree(names[i]);
    }
    MemPtrFree(names);
}

/* Append a simple list view to the data source. */
static void
AppendNewListView(DataSource * source)
{
    DataSourceView view;
    DataSourceViewColumn cols[source->schema.numFields];
    UInt16 i, num_views;

    MemSet(&view, sizeof(view), 0);
    StrCopy(view.name, "All Fields");
    view.numCols = source->schema.numFields;
    view.cols = &cols[0];
    for (i = 0; i < view.numCols; ++i) {
	view.cols[i].fieldNum = i;
	view.cols[i].width = 80;
    }

    num_views = source->ops.GetNumOfViews(source);
    source->ops.StoreView(source, num_views, &view);
}

/* Show/hide actions buttons based on the data source's capabilities. */
static void
UpdateButtons(FormType* form)
{
#define ShowObject(f,id) FrmShowObject((f),FrmGetObjectIndex((f),(id)))
#define HideObject(f,id) FrmHideObject((f),FrmGetObjectIndex((f),(id)))

    if (CurrentSource->ops.Capable(CurrentSource, DS_CAPABLE_LISTVIEW_INSERT))
	ShowObject(form, ctlID_NamesEditor_NewButton);
    else
	HideObject(form, ctlID_NamesEditor_NewButton);

    if (CurrentSource->ops.Capable(CurrentSource, DS_CAPABLE_LISTVIEW_DELETE))
	ShowObject(form, ctlID_NamesEditor_DeleteButton);
    else
	HideObject(form, ctlID_NamesEditor_DeleteButton);

#undef ShowObject
#undef HideObject
}

Boolean
NamesEditorHandleEvent(EventType* event)
{
    static char ** names;
    static char* label;

    FormType* form;
    ControlType* ctl;
    ListType* list;
    UInt16 count;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	GetListViewNames(&names, &count);
	list = GetObjectPtr(form, ctlID_NamesEditor_List);
	LstSetListChoices(list, names, count);
	LstSetSelection(list, noListSelection);

	if (CurrentSource->ops.Capable(CurrentSource,
				       DS_CAPABLE_LISTVIEW_EDIT)) {
	    label = CopyStringResource(stringID_Edit);
	} else {
	    label = CopyStringResource(stringID_View);
	}
	ctl = GetObjectPtr(form, ctlID_NamesEditor_EditButton);
	CtlSetLabel(ctl, label);

	UpdateButtons(form);
	FrmDrawForm(form);
	return true;

    case frmCloseEvent:
	PutListViewNames(names);
	MemPtrFree(label);
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_NamesEditor_DoneButton:
	    FrmGotoForm(formID_ListView);
	    return true;

	case ctlID_NamesEditor_EditButton:
	    form = FrmGetActiveForm();
	    list = GetObjectPtr(form, ctlID_NamesEditor_List);
	    count = LstGetSelection(list);
	    if (count != noListSelection) {
		ListViewEditor_ViewNum = count;
		FrmGotoForm(formID_ListViewEditor);
	    }
	    return true;

	case ctlID_NamesEditor_NewButton:
	    PutListViewNames(names);
	    AppendNewListView(CurrentSource);
	    GetListViewNames(&names, &count);
	    form = FrmGetActiveForm();
	    list = GetObjectPtr(form, ctlID_NamesEditor_List);
	    LstSetListChoices(list, names, count);
	    LstSetSelection(list, noListSelection);
	    LstEraseList(list);
	    LstDrawList(list);
	    UpdateButtons(form);
	    return true;

	case ctlID_NamesEditor_DeleteButton:
	    form = FrmGetActiveForm();
	    list = GetObjectPtr(form, ctlID_NamesEditor_List);
	    count = LstGetSelection(list);
	    if (count != noListSelection) {
		PutListViewNames(names);
		CurrentSource->ops.DeleteView(CurrentSource, count);
		GetListViewNames(&names, &count);
		form = FrmGetActiveForm();
		list = GetObjectPtr(form, ctlID_NamesEditor_List);
		LstSetListChoices(list, names, count);
		LstSetSelection(list, noListSelection);
		LstEraseList(list);
		LstDrawList(list);
		UpdateButtons(form);
	    }
	    return true;

	}
	break;

    default:
	break;
    }

    return false;
}
