/*
 * DB: View Editor (edits the views of a DataSource)
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>

#include "enum.h"
#include "db.h"
#include "grid.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define col_popup 0
#define col_field  1
#define col_width 2

typedef struct {
    UInt16 fieldNum;
    UInt16 width;
    UInt32 uniqueID;
} ColInfo;

/* Set by the names editor to tell this form what view to edit. */
UInt16 ListViewEditor_ViewNum;

static UInt16 numCols = 0;
static UInt32 NextUniqueID;
static ColInfo * cols = 0;
static MemHandle cols_handle = 0;
static MemHandle * width_handles = 0;
static MemHandle width_handles_handle = 0;
static char ** field_names = 0;

static UInt16 TopVisibleField;
static MemHandle name_handle;

static void DrawPopupIndicator(void *, Int16, Int16, RectangleType *) SECT_UI;
static void DrawFieldName(void *, Int16, Int16, RectangleType*) SECT_UI;
static void DrawWidth(void *, Int16, Int16, RectangleType*) SECT_UI;
static void UpdateScrollers(void) SECT_UI;
static Err LoadData(void *, Int16, Int16, Boolean, MemHandle *, Int16 *,
		    Int16 *, FieldPtr) SECT_UI;
static void LoadTable(TablePtr table) SECT_UI;
static void InitTable(TablePtr table) SECT_UI;
static void Scroll(WinDirectionType direction) SECT_UI;
static Boolean SaveFieldWidths(void) SECT_UI;
static void GetDefinition(FormType*, UInt16 defNum) SECT_UI;
static void PutDefinition(UInt16 defNum, Boolean doStore) SECT_UI;
static void InsertEntry(UInt16 colNum) SECT_UI;
static void DeleteEntry(UInt16 colNum) SECT_UI;
static void HandleContextMenu(FormType*, TableType*, Int16) SECT_UI;

static void
DrawPopupIndicator(void * table, Int16 row, Int16 col, RectangleType *b)
{
    Coord x, y, width;

    width = 8;
    x = b->topLeft.x + 1;
    y = b->topLeft.y + (TblGetRowHeight(table, row) - (width / 2)) / 2;

    while (width > 0) {
        WinDrawLine(x, y, x + width - 1, y);
        x += 1;
        y += 1;
        width -= 2;
    }
}

static void
DrawFieldName(void * table, Int16 row, Int16 col, RectangleType *b)
{
    UInt16 colNum;
    FontID oldFont;
    char * name;

    colNum = TblGetRowID(table, row);
    name = CurrentSource->schema.fields[cols[colNum].fieldNum].name;

    oldFont = FntSetFont(stdFont);
    MyWinDrawTruncChars(name, StrLen(name),
			b->topLeft.x, b->topLeft.y, b->extent.x);
    FntSetFont(oldFont);
}

static void
DrawWidth(void * table, Int16 row, Int16 col, RectangleType *b)
{
    UInt16 colNum;
    FontID oldFont;
    char buf[64];

    colNum = (UInt16) TblGetRowID(table, row);
    StrPrintF(buf, "%u", cols[colNum].width);

    oldFont = FntSetFont(stdFont);
    MyWinDrawTruncChars(buf, StrLen(buf),
			b->topLeft.x, b->topLeft.y, b->extent.x);
    FntSetFont(oldFont);
}

static void 
UpdateScrollers(void)
{
    UInt16 upIndex, downIndex, fieldNum;
    Boolean scrollUp, scrollDown;
    Int16 row;
    FormPtr form;
    TablePtr table;

    scrollUp = TopVisibleField != 0;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_ListViewEditor_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1) {
	fieldNum = TblGetRowID(table, row);
	scrollDown = fieldNum < numCols - 1;
    } else {
	scrollDown = false;
    }

    upIndex = FrmGetObjectIndex(form, ctlID_ListViewEditor_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_ListViewEditor_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static Err LoadData(void * table, Int16 row, Int16 column,
		    Boolean editing, MemHandle * textH, Int16 * textOffset,
		    Int16 * textAllocSize, FieldPtr fld)
{
    UInt16 colNum;
    FieldAttrType attr;

    colNum = (UInt16) TblGetRowID(table, row);

    *textH = width_handles[colNum];
    *textOffset = 0;
    *textAllocSize = MemHandleSize(width_handles[colNum]);

    if (fld) {
	FldGetAttributes(fld, &attr);
	attr.singleLine = 1;
	attr.dynamicSize = 0;
	attr.underlined = 1;
	FldSetAttributes(fld, &attr);
	
	FldSetMaxChars(fld, 16);
    }

    return (0);
}           

static void LoadTable(TablePtr table)
{
    UInt16 numRows, row, colNum;
    ListType* list;
    FormType* form;
    Boolean cap_field;
    Boolean cap_width;

    cap_field = CurrentSource->ops.Capable(CurrentSource, 
					   DS_CAPABLE_LISTVIEW_FIELD_EDIT);
    cap_width = CurrentSource->ops.Capable(CurrentSource,
					   DS_CAPABLE_LISTVIEW_WIDTH_EDIT);

    form = FrmGetActiveForm();
    list = GetObjectPtr(form, ctlID_ListViewEditor_PopupList);

    numRows = TblGetNumberOfRows(table);
    colNum = TopVisibleField;
    for (row = 0; row < numRows; row++) {
	/* Bail out of the loop if there are no more entries. */
	if (colNum >= numCols)
	    break;

	if (! TblRowUsable(table, row)
	    || TblGetRowData(table, row) != cols[colNum].uniqueID) {
	    TblSetRowUsable(table, row, true);
	    TblSetRowSelectable(table, row, true);
	    TblSetRowHeight(table, row, FntLineHeight());
	    TblSetRowID(table, row, colNum);
	    TblSetRowData(table, row, cols[colNum].uniqueID);
	    TblMarkRowInvalid(table, row);

	    TblSetItemStyle(table, row, col_popup, customTableItem);

	    if (cap_field) {
		TblSetItemStyle(table, row, col_field,
				popupTriggerTableItem);
		TblSetItemPtr(table, row, col_field, list);
		TblSetItemInt(table, row,
			      col_field, cols[colNum].fieldNum);
	    } else {
		TblSetItemStyle(table, row, col_field, customTableItem);
	    }
	    
	    if (cap_width)
		TblSetItemStyle(table, row, col_width, textTableItem);
	    else
		TblSetItemStyle(table, row, col_width, customTableItem);
	    
	    if (cap_field || cap_width)
		TblSetRowSelectable(table, row, true);
	    else
		TblSetRowSelectable(table, row, false);
	}

	/* Increment to the next list view column number. */
	colNum++;
    }

    /* Disable the remainder of the rows. */
    while (row < numRows) {
        TblSetRowUsable(table, row, false);
        row++;
    }

    UpdateScrollers();
}

static void InitTable(TablePtr table)
{
    UInt16 row, numRows, width;
    RectangleType bounds;
    Boolean cap_insert, cap_remove;

    /* Query the data source for insert and remove capabilities. */
    cap_insert = CurrentSource->ops.Capable(CurrentSource, 
					    DS_CAPABLE_LISTVIEW_FIELD_INSERT);
    cap_remove = CurrentSource->ops.Capable(CurrentSource,
					    DS_CAPABLE_LISTVIEW_FIELD_DELETE);

    TblGetBounds(table, &bounds);

    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++) {
	TblSetRowUsable(table, row, false);
    }

    width = 0;

    if (cap_insert || cap_remove) {
	TblSetColumnUsable(table, col_popup, true);
	TblSetColumnWidth(table, col_popup, 10);
	TblSetCustomDrawProcedure(table, col_popup, DrawPopupIndicator);
	width += 10;
    } else {
	TblSetColumnUsable(table, col_popup, false);
    }

    TblSetColumnUsable(table, col_width, true);
    TblSetColumnWidth(table, col_width, 40);
    TblSetCustomDrawProcedure(table, col_width, DrawWidth);
    TblSetLoadDataProcedure(table, col_width, LoadData);
    width += 40;

    TblSetColumnUsable(table, col_field, true);
    TblSetColumnWidth(table, col_field, bounds.extent.x - width);
    TblSetCustomDrawProcedure(table, col_field, DrawFieldName);

    LoadTable(table);
}

static void Scroll(WinDirectionType direction)
{
    UInt16 newTopVisibleField;

    newTopVisibleField = TopVisibleField;

    if (direction == winDown) {
	newTopVisibleField++;
    } else {
	newTopVisibleField--;
    }

    if (TopVisibleField != newTopVisibleField) {
	FormPtr form;
	TablePtr table;

	TopVisibleField = newTopVisibleField;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListViewEditor_Table);
	LoadTable(table);
	TblRedrawTable(table);
    }
}

static Boolean SaveFieldWidths(void)
{
    UInt16 i;
    UInt16 widths[numCols];
    char *p;

    /* Make sure that each field width is numeric and in range. */
    for (i = 0; i < numCols; ++i) {
	p = MemHandleLock(width_handles[i]);
	if (! IsNumber(p)) {
	    FrmAlert(alertID_BadFieldWidth);
	    MemPtrUnlock(p);
	    return false;
	}
	widths[i] = (UInt16) String2Long(p);
	MemPtrUnlock(p);

	if (widths[i] < 10 || widths[i] > 160) {
	    FrmAlert(alertID_BadFieldWidth);
	    return false;
	}
    }

    /* Move the column widths into the list view definition. */
    for (i = 0; i < numCols; ++i) cols[i].width = widths[i];

    return true;
}

/* Initialize the state associated with editing a list view. */
static void
GetDefinition(FormType* form, UInt16 defNum)
{
    DataSourceView * view;
    FieldPtr fld;
    FieldAttrType attr;
    char * p;
    UInt16 i;

    /* Load the data source view. */
    view = CurrentSource->ops.GetView(CurrentSource, defNum);

    /* Copy the view information into the holding places for editing. */
    numCols = view->numCols;
    cols_handle = MemHandleNew(numCols * sizeof(ColInfo));
    width_handles_handle = MemHandleNew(numCols * sizeof(MemHandle));
    cols = MemHandleLock(cols_handle);
    width_handles = MemHandleLock(width_handles_handle);
    NextUniqueID = 0;
    for (i = 0; i < numCols; ++i) {
	cols[i].fieldNum = view->cols[i].fieldNum;
	cols[i].width = view->cols[i].width;
	cols[i].uniqueID = NextUniqueID++;
	width_handles[i] = MemHandleNew(16);
	p = MemHandleLock(width_handles[i]);
	StrPrintF(p, "%u", cols[i].width);
	MemPtrUnlock(p);
    }

    /* Allocate and fill a handle for the view name. */
    name_handle = MemHandleNew(1 + StrLen(view->name));
    p = MemHandleLock(name_handle);
    StrCopy(p, view->name);
    MemPtrUnlock(p);

    /* Connect the name field to the handle. */
    fld = GetObjectPtr(form, ctlID_ListViewEditor_NameField);
    FldSetTextHandle(fld, name_handle);
    FldGetAttributes(fld, &attr);
    if (CurrentSource->ops.Capable(CurrentSource,
				   DS_CAPABLE_LISTVIEW_NAME_EDIT)) {
	attr.editable = true;
	attr.underlined = true;
    } else {
	attr.editable = false;
	attr.underlined = false;
    }
    FldSetAttributes(fld, &attr);

    /* Free the data source view since we have copied it to local storage. */
    CurrentSource->ops.PutView(CurrentSource, defNum, view);
}

static void
PutDefinition(UInt16 defNum, Boolean doStore)
{
    DataSourceView view;
    UInt16 i;
    char * p;

    if (doStore) {
	/* Copy the name field back into the view. */
	p = MemHandleLock(name_handle);
	StrNCopy(view.name, p, sizeof(view.name) - 1);
	view.name[sizeof(view.name)-1] = '\0';
	MemPtrUnlock(p);

	/* Construct the column information. */
	view.numCols = numCols;
	view.cols = MemPtrNew(numCols * sizeof(DataSourceViewColumn));
	for (i = 0; i < numCols; ++i) {
	    view.cols[i].fieldNum = cols[i].fieldNum;
	    view.cols[i].width = cols[i].width;
	    MemHandleFree(width_handles[i]);
	}

	/* Tell the data source to update this view. */
	CurrentSource->ops.StoreView(CurrentSource, defNum, &view);

	/* Free the pointer we allocated before. */
	MemPtrFree(view.cols);
    }

    /* Free the global structures used for this form's state. */
    MemPtrUnlock(cols); cols = 0;
    MemHandleFree(cols_handle); cols_handle = 0;
    MemPtrUnlock(width_handles); width_handles = 0;
    MemHandleFree(width_handles_handle); width_handles_handle = 0;
}

static void
InsertEntry(UInt16 colNum)
{
    UInt16 i;
    char * p;

    /* Reallocate the cols_handle to fit new column. */
    MemHandleUnlock(cols_handle);
    MemHandleResize(cols_handle, (numCols + 1) * sizeof(ColInfo));
    cols = MemHandleLock(cols_handle);

    /* Move the entries after the insert position to the right. */
    if (colNum < numCols) {
        for (i = numCols; i != colNum; --i) {
            MemMove(&cols[i], &cols[i - 1], sizeof(cols[i]));
        }
    }

    /* Fill in the new column. */
    cols[colNum].fieldNum = 0;
    cols[colNum].width = 80;
    cols[colNum].uniqueID = NextUniqueID++;

    /* Reallocate the width_handles_handle to fit new column. */
    MemHandleUnlock(width_handles_handle);
    MemHandleResize(width_handles_handle, (numCols + 1) * sizeof(MemHandle));
    width_handles = MemHandleLock(width_handles_handle);

    /* Move the entries after the insert position to the right. */
    if (colNum < numCols) {
        for (i = numCols; i != colNum; --i) {
            width_handles[i] = width_handles[i - 1];
        }
    }

    /* Fill in the new column. */
    width_handles[colNum] = MemHandleNew(16);
    p = MemHandleLock(width_handles[colNum]);
    StrPrintF(p, "%u", cols[colNum].width);
    MemPtrUnlock(p);

    /* Increment the number of columns. */
    numCols++;
}

static void
DeleteEntry(UInt16 colNum)
{
    UInt16 i;

    /* Free the width handle associated with the named column. */
    MemHandleFree(width_handles[colNum]);

    /* Move every width handle and column info above this point down. */
    for (i = colNum + 1; i < numCols; ++i) {
	cols[i-1] = cols[i];
	width_handles[i-1] = width_handles[i];
    }

    /* Reallocate the cols_handle to fit. */
    MemHandleUnlock(cols_handle);
    MemHandleResize(cols_handle, (numCols - 1) * sizeof(ColInfo));
    cols = MemHandleLock(cols_handle);

    /* Reallocate the width_handles_handle to fit. */
    MemHandleUnlock(width_handles_handle);
    MemHandleResize(width_handles_handle, (numCols - 1) * sizeof(MemHandle));
    width_handles = MemHandleLock(width_handles_handle);

    /* Decrement the number of columns. */
    numCols--;
}

static void
HandleContextMenu(FormType* form, TableType* table, Int16 row)
{
    ListType* list;
    RectangleType r;
    char * choices[3];
    Boolean cap_insert, cap_remove;
    UInt16 i, choices_count;

    /* Remove any selection UI that the table control does. */
    TblUnhighlightSelection(table);

    /* Query the data source for its capabilities. */
    cap_insert = CurrentSource->ops.Capable(CurrentSource, 
					    DS_CAPABLE_LISTVIEW_FIELD_INSERT);
    cap_remove = CurrentSource->ops.Capable(CurrentSource,
					    DS_CAPABLE_LISTVIEW_FIELD_DELETE);

    /* If we cannot present any menu choices, don't do anything. */
    if (!cap_insert && !cap_remove)
	return;

    /* Make copies of strings to be used for the list choices. */
    i = 0;
    if (cap_insert) {
	choices[i++] = CopyStringResource(stringID_InsertBefore);
	choices[i++] = CopyStringResource(stringID_InsertAfter);
    }
    if (cap_remove && numCols > 1) {
	choices[i++] = CopyStringResource(stringID_Remove);
    }
    choices_count = i;

    /* Retrieve the popup choices list. */
    list = GetObjectPtr(form, ctlID_ListViewEditor_InserterList);

    /* Set the location for the popup list. */
    TblGetItemBounds(table, row, col_popup, &r);
    LstSetPosition(list, r.topLeft.x, r.topLeft.y);

    /* Set the choices which will be presented to the user. */
    LstSetSelection(list, noListSelection);
    LstSetListChoices(list, choices, choices_count);
    LstSetHeight(list, choices_count);

    /* Let the user choose an action. */
    LstPopupList(list);

    /* Insert or delete a column as necessary. */
    if (LstGetSelection(list) != noListSelection) {
        UInt16 entry = TblGetRowID(table, row);
	UInt16 choice = LstGetSelection(list);
	enum { NONE, INSERT_BEFORE, INSERT_AFTER, REMOVE } action = NONE;

	/* Determine what action was called for. */
	i = 0;
	if (cap_insert) {
	    if (i == choice) action = INSERT_BEFORE;
	    i++;
	    if (i == choice) action = INSERT_AFTER;
	    i++;
	}
	if (cap_remove && numCols > 1) {
	    if (i == choice) action = REMOVE;
	    i++;
	}

        switch (action) {
        case INSERT_BEFORE:
            InsertEntry(entry);
            break;

        case INSERT_AFTER:
            InsertEntry(entry + 1);
            break;

        case REMOVE:
            /* Remove the row. */
            DeleteEntry(entry);
            break;

	default:
	case NONE:
	    break;
        }

        /* Redraw the table now that we have changed the ordering. */
        LoadTable(table);
        TblRedrawTable(table);
    }

    /* Free the list choices that we allocated earlier. */
    for (i = 0; i < choices_count; ++i) {
	MemPtrFree(choices[i]);
    }
}

Boolean
ListViewEditor_HandleEvent(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    ListPtr list;
    UInt16 i;

    switch (event->eType) {
    case frmOpenEvent:
	/* Retrieve the current form. */
	form = FrmGetActiveForm();

	/* Load the list view to be edited into the form. */
	GetDefinition(form, ListViewEditor_ViewNum);

	/* Allocate and fill the field names for the popup list. */
	field_names = MemPtrNew(CurrentSource->schema.numFields
				* sizeof(char *));
	for (i = 0; i < CurrentSource->schema.numFields; ++i) {
	    field_names[i] = CurrentSource->schema.fields[i].name;
	}

	/* Make the popup list use the field names data. */
	list = GetObjectPtr(form, ctlID_ListViewEditor_PopupList);
	LstSetListChoices(list, field_names, CurrentSource->schema.numFields);
	LstSetHeight(list, CurrentSource->schema.numFields);

	/* Initialize the table control. */
	TopVisibleField = 0;
	table = GetObjectPtr(form, ctlID_ListViewEditor_Table);
	InitTable(table);

	/* Draw the form. */
	FrmDrawForm(form);

	return true;

    case frmCloseEvent:
        /* Ensure that the table no longer has focus. */
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListViewEditor_Table);
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	}

	/* Store and free any state associated with local copy. */
	if (cols) {
	    PutDefinition(ListViewEditor_ViewNum, false);
	}

	/* Free the space allocated for the field_names array. */
	MemPtrFree(field_names);
	field_names = 0;

	break;

    case tblSelectEvent:
	if (event->data.tblSelect.column == col_popup) {
            HandleContextMenu(FrmGetActiveForm(),
                              event->data.tblSelect.pTable,
                              event->data.tblSelect.row);
            return true;
	} else if (event->data.tblSelect.column == col_field) {
	    UInt16 colNum = TblGetRowID(event->data.tblSelect.pTable,
					event->data.tblSelect.row);
	    UInt16 fieldNum = TblGetItemInt(event->data.tblSelect.pTable,
					    event->data.tblSelect.row,
					    event->data.tblSelect.column);

	    if (! CurrentSource->ops.Capable(CurrentSource,
					     DS_CAPABLE_LISTVIEW_FIELD_EDIT)
		&& CurrentSource->ops.Capable(CurrentSource,
					      DS_CAPABLE_LISTVIEW_WIDTH_EDIT)){
		FieldType* fld;

		/* Width is editable and field name is not. Put the
		 * focus on the width.
		 */
		TblReleaseFocus(event->data.tblSelect.pTable);
		TblUnhighlightSelection(event->data.tblSelect.pTable);
		TblGrabFocus(event->data.tblSelect.pTable,
			     event->data.tblSelect.row, col_width);
		fld = TblGetCurrentField(event->data.tblSelect.pTable);
		if (fld) {
		    FldGrabFocus(fld);
		    FldMakeFullyVisible(fld);
		}
	    } else {
		/* Jot down the field number selected in the popup. */
		cols[colNum].fieldNum = fieldNum;
	    }
	    return true;
	}
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_ListViewEditor_DoneButton:
	    /* If the data source is in read-only mode, discard any edits. */
	    if (! CurrentSource->ops.Capable(CurrentSource,
					     DS_CAPABLE_LISTVIEW_EDIT)) {
		PutDefinition(ListViewEditor_ViewNum, false);
		FrmGotoForm(formID_NamesEditor);
		return true;
	    }
	    if (SaveFieldWidths()) {
		PutDefinition(ListViewEditor_ViewNum, true);
		FrmGotoForm(formID_NamesEditor);
	    }
	    return true;

	case ctlID_ListViewEditor_CancelButton:
	    PutDefinition(ListViewEditor_ViewNum, false);
	    FrmGotoForm(formID_NamesEditor);
	    return true;

	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_ListViewEditor_UpButton:
	    Scroll(winUp);
	    break;

	case ctlID_ListViewEditor_DownButton:
	    Scroll(winDown);
	    break;
	}
	break;

    case menuEvent:
	return HandleCommonMenuEvent(event->data.menu.itemID);

    default:
	break;

    }

    return false;
}
