/*
 * DB: Database I/O code
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <PalmOS.h>
#include <TxtGlue.h>

#include "db.h"
#include "enum.h"

DataSource * CurrentSource = 0;

typedef struct
{
    DataSourceSortCallback callback;
    void * callback_data;
    DataSource * source;
    DataSourceGetter getter;
} SortCallbackData;

/* While this pointer breaks the reentrant style of the data source
 * layer, it is necessary since PalmOS doesn't let you pass a pointer
 * to the sort callback but rather an integer instead.
 */
static SortCallbackData * sort_callback_data;


/*
 * Generic implementations of data source driver methods.
 *
 * These helper routines are called directly and indirectly by the
 * actual data source drivers. Generally, they implement routines
 * which are only specific to PalmOS database and not the specific
 * format.
 */

Boolean
DS_PDB_CheckTitle(const char* title)
{
    LocalID dbID;

    dbID = DmFindDatabase(0, title);
    return (dbID == 0);
}

void
DS_PDB_Delete(void * key_data, UInt32 key_size)
{
    DataSourceChooserKey * key = (DataSourceChooserKey *) key_data;
    LocalID dbID;

    dbID = DmFindDatabase(key->cardNo, key->name);
    if (dbID) {
	DmDeleteDatabase(key->cardNo, dbID);
    }
}

void
DS_PDB_Enumerate(DataSourceEnumerationFunc callback, void * arg,
		 UInt32 type, UInt32 creator, UInt16 sourceID)
{
    DmSearchStateType state;
    Boolean flag;
    LocalID dbID;
    DataSourceChooserKey key;

    flag = true;
    while (DmGetNextDatabaseByTypeCreator(flag, &state, type, creator,
					  false, &key.cardNo, &dbID) == 0) {
	MemSet(&key, sizeof(key), 0);
	DmDatabaseInfo(key.cardNo, dbID, key.name, 0,0,0,0,0,0,0,0,0,0);
	callback(arg, key.name, sourceID, &key, sizeof(key));
	flag = false;
    }
}

Boolean
DS_PDB_GlobalFind(DataSourceGlobalFindFunc callback, Boolean continuation,
		  FindParamsPtr params, UInt32 type, UInt32 creator,
		  Err (*OpenPDB)(DataSource *, UInt16, UInt16, LocalID))
{
    struct {
	DmSearchStateType state;
	UInt16 cardNo;
	LocalID dbID;
    } data;
    Err err;
    Boolean r;

    /* Fill in the search state from preferences if we are continuing
     * or initialize if this is the first time.  */
    if (! continuation) {
	err = DmGetNextDatabaseByTypeCreator(true, &data.state,
					     type, creator, false,
					     &data.cardNo, &data.dbID);
	if (err != 0) {
	    /* No more databases of this type to be found. */
	    return false;
	}
    } else {
	UInt16 prefsSize = sizeof(data);

	PrefGetAppPreferences(DBCreatorID, prefID_GFind_DataSource, &data,
			      &prefsSize, false);
    }

    do {
	DataSource * source = MemPtrNew(sizeof(DataSource));

	/* Open the data source. */
	err = OpenPDB(source, params->dbAccesMode, data.cardNo, data.dbID);
	if (err != 0) {
	    /* If we cannot open the database, jump to the next one. */
	    MemPtrFree(source);
	    goto next_db;
	}

	/* Reset the record number if this is not a continuation. */
	if (continuation)
	    continuation = false;
	else
	    params->recordNum = 0;

	/* Let the global find code have a go at the database. */
	r = callback(params, source, data.cardNo, data.dbID);

	/* Close the data source. */
	source->ops.Close(source);
	MemPtrFree(source);

	/* Check if global find code wants us to save our position. */
	if (r) {
	    PrefSetAppPreferences(DBCreatorID, prefID_GFind_DataSource, 0,
				  &data, sizeof(data), false);
	    return true;
	}

	/* Get the next data source. */
    next_db:
	err = DmGetNextDatabaseByTypeCreator(false, &data.state,
					     type, creator, false,
					     &data.cardNo, &data.dbID);
    } while (!err);

    /* No more databases to be found. */
    return false;
}

Boolean
DS_PDB_IsPresent(void * key_data, UInt32 key_size,
		 UInt32 ds_type, UInt32 ds_creator)
{
    DataSourceChooserKey* key = (DataSourceChooserKey *) key_data;
    LocalID dbID;
    UInt32 creator, type;

    if (key_size == sizeof(DataSourceChooserKey)
	&& (dbID = DmFindDatabase(key->cardNo, key->name)) != 0
	&& DmDatabaseInfo(key->cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			  &type, &creator) == 0
	&& creator == ds_creator
	&& type == ds_type)
	return true;

    return false;
}

Boolean
DS_PDB_CompareKey(void * key1_data, UInt16 key1_size,
		  void * key2_data, UInt16 key2_size)
{
    DataSourceChooserKey * key1 = (DataSourceChooserKey *) key1_data;
    DataSourceChooserKey * key2 = (DataSourceChooserKey *) key2_data;

    return (key1_size == key2_size)
	&& (key1->cardNo == key2->cardNo)
	&& (StrCompare(key1->name, key2->name) == 0);
}

static Err
WriteDBData(const void * dataP, UInt32 * sizeP, void * userDataP)
{
    Err err;

    /* Try to send as many bytes as were requested by the caller. */
    *sizeP = ExgSend((ExgSocketPtr)userDataP, (void *)dataP, *sizeP, &err);
    return err;
}

Err 
DS_PDB_Beam(void * key_data, UInt16 key_size)
{
    DataSourceChooserKey * key = (DataSourceChooserKey *) key_data;
    LocalID dbID;
    char name[dmDBNameLength + 4];
    ExgSocketType exgSocket;
    Err err;

    /* Make sure that the key has a valid size. */
    if (key_size != sizeof(DataSourceChooserKey))
	return 0;

    /* Retrieve the location in memory of this database. */
    dbID = DmFindDatabase(key->cardNo, key->name);
    if (!dbID)
        return DmGetLastErr();

    /* Form the file name that we will send. */
    DmDatabaseInfo(key->cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    StrCat(name, ".pdb");

    /* Create exgSocket structure. */
    MemSet(&exgSocket, sizeof(exgSocket), 0);
    exgSocket.description = name;
    exgSocket.name = name;

    /* Initiate the exchange. */
    err = ExgPut(&exgSocket);
    if (!err) {
	err = ExgDBWrite(WriteDBData, &exgSocket, NULL, dbID, key->cardNo);
	err = ExgDisconnect(&exgSocket, err);
    }

    return err;
}


/*
 * Generic implementations of data source methods.
 *
 * These helper routines are called directly and indirectly by the
 * actual data source functions. Generally, they implement routines
 * which are only specific to PalmOS database and not the specific
 * format.
 */

UInt16
DS_PDB_GetKey(DataSource * source, void * buffer)
{
    LocalID dbID;
    DataSourceChooserKey key;

    if (buffer) {
        DmOpenDatabaseInfo(source->db, &dbID, 0, 0, &key.cardNo, 0);
        DmDatabaseInfo(key.cardNo, dbID, key.name, 0,0,0,0,0,0,0,0,0,0);
        MemMove(buffer, &key, sizeof(key));
    }

    return sizeof(key);
}

void
DS_PDB_GetPDB(DataSource * source, UInt16 * cardNo_ptr, LocalID * dbID_ptr)
{
    DmOpenDatabaseInfo(source->db, dbID_ptr, 0, 0, cardNo_ptr, 0);
}

void
DS_PDB_GetTitle(DataSource * source, char * buf, UInt32 len)
{
    char name[dmDBNameLength];
    UInt16 cardNo;
    LocalID dbID;

    DmOpenDatabaseInfo(source->db, &dbID, 0, 0, &cardNo, 0);
    DmDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    StrNCopy(buf, name, len);
    buf[len-1] = '\0';
}

void
DS_PDB_SetTitle(DataSource * source, const char * title,
		void ** key_data_ptr, UInt16 * key_size_ptr)
{
    char name[dmDBNameLength];
    UInt16 cardNo;
    LocalID dbID;
    DataSourceChooserKey * key;

    /* Make the title fit within the PalmOS limits. */
    StrNCopy(name, title, sizeof(name));
    name[sizeof(name)-1] = '\0';

    /* Set the database title. */
    DmOpenDatabaseInfo(source->db, &dbID, 0, 0, &cardNo, 0);
    DmSetDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

    /* Return a new key for the chooser. */
    key = (DataSourceChooserKey *) MemPtrNew(sizeof(DataSourceChooserKey));
    MemSet(key, sizeof(*key), 0);
    key->cardNo = cardNo;
    StrNCopy(key->name, name, sizeof(key->name));
    key->name[sizeof(key->name) - 1] = '\0';
    *key_data_ptr = key;
    *key_size_ptr = sizeof(*key);
}


/*
 * Data source driver routines for DB's native format.
 */

/* Provide convenient reference to chunks in the app info block. */

typedef struct {
    UInt8 * ptr;
    UInt32 size;
} AppInfoBlock;

typedef struct {
    UInt16 chunk_type;
    UInt16 chunk_size;
    UInt8 * chunk_data;
} AppInfoChunk;

typedef struct {
    UInt16 fieldNum;
    UInt16 width;
} ListViewColumn;

typedef struct {
    UInt16 flags;
    UInt16 numCols;
    Char name[32];
    ListViewColumn * cols;
} ListViewDefinition;

/* Valid chunk types in the database header. Read docs/format.txt for info. */
#define CHUNK_FIELD_NAMES         0U
#define CHUNK_FIELD_TYPES         1U
#define CHUNK_LISTVIEW_DEFINITION 64U
#define CHUNK_LISTVIEW_OPTIONS    65U
#define CHUNK_LFIND_OPTIONS       128U

/* Valid values for the flags word at the start of the database header. */
#define FLAG_GFIND_FORCE_DISABLE   0x0001
#define FLAG_GFIND_FORCE_ENABLE    0x0002

static void GetAppInfoBlock(DataSource * source, AppInfoBlock * info)
{
    UInt16 cardNo;
    LocalID appInfoID;

    DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
    appInfoID = DmGetAppInfoID(source->db);
    info->ptr = MemLocalIDToLockedPtr(appInfoID, cardNo);
    info->size = MemPtrSize(info->ptr);
}

static void PutAppInfoBlock(DataSource * source, AppInfoBlock * info)
{
    MemPtrUnlock(info->ptr);
}

/* Validate that the chunks fill the app info block. */
static Boolean
ValidateAppInfoBlock(AppInfoBlock * info)
{
    AppInfoChunk chunk;
    UInt32 i;
    Boolean result = false;

    if (info->size >= 4UL) {
	/* Loop through each chunk in the block while data remains. */
	i = 4UL;
	while (i < info->size) {
	    /* Stop the loop if there is not enough room for even one
	     * chunk header.
	     */
	    if (i + 4UL >= info->size)
		break;

	    /* Copy the chunk type and size into the local buffer. */
	    MemMove(&(chunk.chunk_type), info->ptr + i, sizeof(UInt16));
	    MemMove(&(chunk.chunk_size), info->ptr + i + 2, sizeof(UInt16));
	    i += 4UL;

	    /* Advance the index by the size of the chunk. */
	    i += chunk.chunk_size;
	}

	/* If everything was correct, then we should be exactly at the
	 * end of the block. 
	 */
	if (i == info->size)
	    result = true;
    }

    return result;
}

/* Iterate over each chunk. The 'token' argument should be zero to start. */
static Boolean
GetNextChunk(AppInfoBlock * info, UInt32 * token, AppInfoChunk * chunk)
{
    /* If the token is 0, then point at the first chunk. */
    if (! (*token)) *token = 4UL;

    /* If the token is past the end of the app info block, signal done. */
    if (*token >= info->size)
	return false;

    /* Retrieve the header and size of the next chunk. */
    MemMove(&(chunk->chunk_type), info->ptr + (*token), sizeof(UInt16));
    MemMove(&(chunk->chunk_size), info->ptr + (*token) + 2, sizeof(UInt16));

    /* Point at the start of the chunk data. */
    chunk->chunk_data = info->ptr + (*token) + 4;

    /* Advance the token to the next chunk. */
    (*token) += 2 * sizeof(UInt16) + chunk->chunk_size;

    /* Align the pointer to a 16-bit (UInt16) boundary. */
    /* (*token) = ( ( (*token) + 1ul ) & 1ul ); */

    return true;
}

/* Find the first chunk with the matching type and return it. */
static Boolean
GetChunkByType(AppInfoBlock * info, AppInfoChunk * chunk,
	       UInt16 chunk_type, UInt16 index)
{
    UInt32 token = 0;
    
    while (GetNextChunk(info, &token, chunk)) {
	if (chunk->chunk_type == chunk_type) {
	    if (index == 0) {
		/* We found a worthy chunk. Return to the caller. */
		return true;
	    }
	    index--;
	}
    }

    return false;
}

/* Append the given data to a normal memory handle as a new chunk. */
static Err
AppendChunk(MemHandle h, UInt16 chunk_type,
	    UInt8 * chunk_data, UInt16 chunk_size)
{
    Err err;
    UInt8 * p;
    UInt32 size, orig_size;

    /* Resize the handle to make room for the new chunk. */
    orig_size = MemHandleSize(h);
    size = orig_size + 2 * sizeof(UInt16) + chunk_size;
    err = MemHandleResize(h, size);
    if (err)
	return err;

    /* Store the new chunk at the end of the handle. */
    p = MemHandleLock(h);
    MemMove(p + orig_size, &chunk_type, sizeof(UInt16));
    MemMove(p + orig_size + 2UL, &chunk_size, sizeof(UInt16));
    MemMove(p + orig_size + 4UL, chunk_data, chunk_size);
    MemHandleUnlock(h);

    return 0;
}

/* Finish off a chunk operation by installing the new app info block. */
static Err
DB_InstallAppInfoBlock(DataSource* source, MemHandle tmpAppInfoHand)
{
    MemHandle appInfoHand, oldAppInfoHand;
    UInt16 cardNo;
    LocalID dbID, appInfoID, oldAppInfoID;
    void *p, *q;
    UInt32 size;

    /* Allocate a handle in the storage memory for the new block. */
    size = MemHandleSize(tmpAppInfoHand);
    appInfoHand = DmNewHandle(source->db, size);
    if (!appInfoHand)
	return DmGetLastErr();

    /* Copye the new block into what will be its final location. */
    p = MemHandleLock(appInfoHand);
    q = MemHandleLock(tmpAppInfoHand);
    DmWrite(p, 0, q, size);
    MemPtrUnlock(q);
    MemPtrUnlock(p);

    /* Point the Data Manager at the new block. */
    DmOpenDatabaseInfo(source->db, &dbID, 0, 0, &cardNo, 0);
    oldAppInfoID = DmGetAppInfoID(source->db);
    appInfoID = MemHandleToLocalID(appInfoHand);
    DmSetDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0);

    /* Finally, free the old block. */
    p = MemLocalIDToLockedPtr(oldAppInfoID, cardNo);
    oldAppInfoHand = MemPtrRecoverHandle(p);
    MemHandleUnlock(oldAppInfoHand);
    MemHandleFree(oldAppInfoHand);

    /* Success! */
    return 0;
}

/* Append the given data as a new chunk. */
static Err
DB_Chunk_Append(DataSource * source, UInt16 new_chunk_type,
		UInt8 * new_chunk_data, UInt16 new_chunk_size)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    MemHandle h;
    UInt32 token = 0;
    Err err;
    void * p;

    /* Lock down the current block. */
    GetAppInfoBlock(source, &info);

    /* Allocate a temporary handle to build new block with. */
    h = MemHandleNew(4ul);
    p = MemHandleLock(h);
    MemMove(p, info.ptr, 4);
    MemPtrUnlock(p);

    /* Loop through each chunk and move them to the temporary handle. */
    while (GetNextChunk(&info, &token, &chunk)) {
	AppendChunk(h, chunk.chunk_type, chunk.chunk_data, chunk.chunk_size);
    }

    /* Append the new chunk after the end of the other chunks. */
    AppendChunk(h, new_chunk_type, new_chunk_data, new_chunk_size);

    /* Release the current block. */
    PutAppInfoBlock(source, &info);

    /* Install the new block. */
    err = DB_InstallAppInfoBlock(source, h);
    MemHandleFree(h);
    return err;
}

/* Replace an existing chunk with new given data. */
static Err
DB_Chunk_Replace(DataSource * source, UInt16 chunk_type, UInt16 index,
		 UInt8 * new_chunk_data, UInt16 new_chunk_size)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    MemHandle h;
    UInt32 token = 0;
    Boolean found_victim_chunk;
    Err err;
    void * p;

    /* Lock down the current block. */
    GetAppInfoBlock(source, &info);

    /* Allocate a temporary handle to build new block with. */
    h = MemHandleNew(4ul);
    p = MemHandleLock(h);
    MemMove(p, info.ptr, 4);
    MemPtrUnlock(p);

    /* Loop through each chunk and move them to the temporary
     * handle. We stop when the chunk to replace has been found.
     */
    found_victim_chunk = false;
    while (GetNextChunk(&info, &token, &chunk)) {
	if (chunk.chunk_type == chunk_type) {
	    if (index == 0) {
		/* Bail out since we have found the chunk to replace. */
		found_victim_chunk = true;
		break;
	    } else {
		/* Otherwise, decrement the index and fall through
		 * into the normal chunk copying code.
		 */
		index--;
	    }
	}

	/* Copy the current chunk into the temporary handle. */
	AppendChunk(h, chunk.chunk_type, chunk.chunk_data, chunk.chunk_size);
    }

    /* If we found the chunk to replace, then do the actual
     * replacement. If the victim was not found, then that means we
     * scanned all the chunks without finding it.
     */
    if (found_victim_chunk) {
	/* Place the new chunk into position where the old one was. */
	AppendChunk(h, chunk_type, new_chunk_data, new_chunk_size);

	/* Copy the remainder of the chunks into place. */
	while (GetNextChunk(&info, &token, &chunk)) {
	    AppendChunk(h, chunk.chunk_type,
			chunk.chunk_data, chunk.chunk_size);
	}
    }

    /* Release the current block. */
    PutAppInfoBlock(source, &info);

    /* Install the new app info block into place. */
    err = DB_InstallAppInfoBlock(source, h);
    MemHandleFree(h);
    return err;
}

/* Delete the given chunk. */
static Err
DB_Chunk_Delete(DataSource * source, UInt16 chunk_type, UInt16 index)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    MemHandle h;
    UInt32 token = 0;
    Boolean found_victim_chunk;
    Err err;
    void * p;

    /* Lock down the current block. */
    GetAppInfoBlock(source, &info);

    /* Allocate a temporary handle to build new block with. */
    h = MemHandleNew(4ul);
    p = MemHandleLock(h);
    MemMove(p, info.ptr, 4);
    MemPtrUnlock(p);

    /* Loop through each chunk and move them to the temporary
     * handle. We stop when the chunk to delete has been found.
     */
    found_victim_chunk = false;
    while (GetNextChunk(&info, &token, &chunk)) {
	if (chunk.chunk_type == chunk_type) {
	    if (index == 0) {
		/* Bail out since we have found the chunk to delete. */
		found_victim_chunk = true;
		break;
	    } else {
		/* Otherwise, decrement the index and fall through
		 * into the normal chunk copying code.
		 */
		index--;
	    }
	}

	/* Copy the current chunk into the temporary handle. */
	AppendChunk(h, chunk.chunk_type, chunk.chunk_data, chunk.chunk_size);
    }

    /* If we found the chunk to replace, then just copy the remaining
     * chunks to the temporary handle. If the victim was not found,
     * then that means we scanned all the chunks without finding it.
     */
    if (found_victim_chunk) {
	while (GetNextChunk(&info, &token, &chunk)) {
	    AppendChunk(h, chunk.chunk_type,
			chunk.chunk_data, chunk.chunk_size);
	}
    }

    /* Release the current block. */
    PutAppInfoBlock(source, &info);

    /* Install the new app info block into place. */
    err = DB_InstallAppInfoBlock(source, h);
    MemHandleFree(h);
    return err;
}

static Err
ExtractSchema(DataSource * source, AppInfoBlock * info)
{
    AppInfoChunk names_chunk, types_chunk;
    UInt16 i;
    UInt32 size;
    char *p, *q;
    UInt16 type;

    /* Retrieve and verify the chunks containing schema information. */
    if (! GetChunkByType(info, &names_chunk, CHUNK_FIELD_NAMES, 0)
	|| ! GetChunkByType(info, &types_chunk, CHUNK_FIELD_TYPES, 0)
	|| (source->schema.numFields*sizeof(UInt16)) != types_chunk.chunk_size)
	return dmErrCorruptDatabase;

    /* Calculate how much space to allocate. */
    size = source->schema.numFields * sizeof(DataSourceFieldInfo);
    p = names_chunk.chunk_data;
    for (i = 0; i < source->schema.numFields; ++i) {
	size += StrLen(p) + 1;
	p += StrLen(p) + 1;
    }

    /* Allocate the field info structure and extra space for strings. */
    source->schema.fields = MemPtrNew(size);

    /* Fill out the field info structure. */
    p = (char *) names_chunk.chunk_data;
    q = (char *) (source->schema.fields + source->schema.numFields);
    for (i = 0; i < source->schema.numFields; ++i) {
	/* Copy and set the field name. */
	StrCopy(q, p);
	source->schema.fields[i].name = q;
	p += StrLen(p) + 1;
	q += StrLen(q) + 1;

	/* Set the field type. */
	MemMove(&type, types_chunk.chunk_data + i * sizeof(UInt16),
		sizeof(UInt16));
	switch (type) {
	case 0:
	    source->schema.fields[i].type = FIELD_TYPE_STRING;
	    break;

	case 1:
	    source->schema.fields[i].type = FIELD_TYPE_BOOLEAN;
	    break;

	case 2:
	    source->schema.fields[i].type = FIELD_TYPE_INTEGER;
	    break;

	case 3:
	    source->schema.fields[i].type = FIELD_TYPE_DATE;
	    break;

	case 4:
	    source->schema.fields[i].type = FIELD_TYPE_TIME;
	    break;
	}
    }

    return 0;
}

static void
DB_Close(DataSource * source)
{
    DmCloseDatabase(source->db);
    source->db = 0;
    MemPtrFree(source->schema.fields);
    if (source->data) MemPtrFree(source->data);
    source->schema.numFields = 0;
    source->schema.fields = 0;
}

static Boolean
DB_Seek(DataSource * source, UInt16 * indexP, Int16 offset, Int16 direction)
{
    DmSeekRecordInCategory(source->db, indexP, offset, direction,
			   dmAllCategories);
    if (DmGetLastErr())
	return false;

    return true;
}

static UInt32
DB_PackedRecordSize(DataSource * source,
		    DataSourceGetter * getter, void * data)
{
    UInt32 size;
    UInt16 i;
    char * str;

    size = source->schema.numFields * sizeof(UInt16);
    for (i = 0; i < source->schema.numFields; i++) {
	switch (source->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	    str = getter->GetString(data, i);
	    size += StrLen(str) + 1;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    size += sizeof(UInt8);
	    break;

	case FIELD_TYPE_INTEGER:
	    size += sizeof(Int32);
	    break;

	case FIELD_TYPE_DATE:
	    size += sizeof(UInt16) + 2 * sizeof(UInt8);
	    break;

	case FIELD_TYPE_TIME:
	    size += 2 * sizeof(UInt8);
	    break;

	default:
	    ErrDisplay("DB_PackedRecordSize: unsupported field type");
	    break;
	}
    }

    return size;
}

static void
DB_PackRecord(DataSource * source, void * dest,
	      DataSourceGetter * getter, void * data)
{
    UInt32 offset;
    Int32 value;
    UInt16 i;
    UInt8 b;
    char * str;
    UInt16 offsets[source->schema.numFields];

    offset = source->schema.numFields * sizeof(UInt16);

    for (i = 0; i < source->schema.numFields; i++) {
	/* Mark the starting offset of this field. */
	offsets[i] = offset;

	/* Pack the field into the buffer. */
	switch (source->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	    str = getter->GetString(data, i);
	    DmWrite(dest, offset, str, StrLen(str) + 1);
	    offset += StrLen(str) + 1;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    b = (getter->GetBoolean(data, i)) ? 1 : 0;
	    DmWrite(dest, offset, &b, sizeof(UInt8));
	    offset += sizeof(UInt8);
	    break;

	case FIELD_TYPE_INTEGER:
	    value = getter->GetInteger(data, i);
	    DmWrite(dest, offset, &value, sizeof(Int32));
	    offset += sizeof(Int32);
	    break;

	case FIELD_TYPE_DATE:
	    {
		UInt16 year;
		UInt8 month, day;

		getter->GetDate(data, i, &year, &month, &day);

		DmWrite(dest, offset, &year, sizeof(year));
		offset += sizeof(year);

		DmWrite(dest, offset, &month, sizeof(month));
		offset += sizeof(month);

		DmWrite(dest, offset, &day, sizeof(day));
		offset += sizeof(day);
	    }
	    break;

	case FIELD_TYPE_TIME:
	    {
		UInt8 hour, minute;

		getter->GetTime(data, i, &hour, &minute);

		DmWrite(dest, offset, &hour, sizeof(hour));
		offset += sizeof(hour);

		DmWrite(dest, offset, &minute, sizeof(minute));
		offset += sizeof(minute);
	    }
	    break;

	default:
	    ErrDisplay("DB_PackRecord: unsupported field type");
	    break;
	}
    }

    /* Write the offset table into the start of the record. */
    DmWrite(dest, 0, &offsets[0], sizeof(offsets));
}

static char *
DB_Getter_GetString(void * data, UInt16 fieldNum)
{
    UInt16 * offsets = (UInt16 *) data;

    return ((UInt8 *) data) + offsets[fieldNum];
}

static Int32
DB_Getter_GetInteger(void * data, UInt16 fieldNum)
{
    UInt16 * offsets = (UInt16 *) data;
    Int32 value;

    MemMove(&value, ((UInt8 *) data) + offsets[fieldNum], sizeof(Int32));
    return value;
}

static Boolean
DB_Getter_GetBoolean(void * data, UInt16 fieldNum)
{
    UInt16 * offsets = (UInt16 *) data;

    if ( *(((UInt8 *) data) + offsets[fieldNum]) )
	return true;
    else
	return false;
}

static void
DB_Getter_GetDate(void * data, UInt16 fieldNum,
		  UInt16* year, UInt8* month, UInt8* day)
{
    UInt16 * offsets = (UInt16 *) data;
    UInt8 * ptr =  ((UInt8 *) data) + offsets[fieldNum];

    MemMove(year, ptr, sizeof(UInt16));
    MemMove(month, ptr + sizeof(UInt16), sizeof(UInt8));
    MemMove(day, ptr + sizeof(UInt16) + sizeof(UInt8), sizeof(UInt8));
}

static void
DB_Getter_GetTime(void * data, UInt16 fieldNum,
		  UInt8* hour, UInt8* minute)
{
    UInt16 * offsets = (UInt16 *) data;
    UInt8 * ptr =  ((UInt8 *) data) + offsets[fieldNum];

    MemMove(hour, ptr, sizeof(UInt8));
    MemMove(minute, ptr + sizeof(UInt8), sizeof(UInt8));
}

static void
DB_LockRecord(DataSource * source, UInt16 index, Boolean writable,
	      DataSourceGetter * getter, void ** data_ptr)
{
    MemHandle h;

    if (writable)
        h = DmGetRecord(source->db, index);
    else
	h = DmQueryRecord(source->db, index);

    *data_ptr = MemHandleLock(h);
    getter->GetString  = DB_Getter_GetString;
    getter->GetInteger = DB_Getter_GetInteger;
    getter->GetBoolean = DB_Getter_GetBoolean;
    getter->GetDate    = DB_Getter_GetDate;
    getter->GetTime    = DB_Getter_GetTime;
}

static void
DB_UnlockRecord(DataSource * source, UInt16 index, Boolean writable,
		void * record_data)
{
    MemPtrUnlock(record_data);

    if (writable)
        DmReleaseRecord(source->db, index, true);
}

static Err
DB_UpdateRecord(DataSource * source, UInt16 index, Boolean doInsert,
		DataSourceGetter * getter, void * data)
{
    UInt32 size;
    MemHandle recH;
    void * packed;

    /* Determine the packed size of the new record. */
    size = DB_PackedRecordSize(source, getter, data);

    if (index == dmMaxRecordIndex || doInsert) {
	/* Create a new record in the database. */
	recH = DmNewRecord(source->db, &index, size);
	if (recH == 0) {
	    /* If we couldn't make the record, bail out. */
	    return DmGetLastErr();
	}
    } else {
	/* Otherwise, retrieve and resize an existing record. */
	DmResizeRecord(source->db, index, size);
	recH = DmGetRecord(source->db, index);
    }

    /* Pack the new data into the record. */
    packed = MemHandleLock(recH);
    DB_PackRecord(source, packed, getter, data);
    MemPtrUnlock(packed);

    /* Release the write lock on the record. */
    DmReleaseRecord(source->db, index, true);

    return 0;
}

static void
DB_SetField(DataSource * source, void * data, UInt16 fieldNum,
	    DataSourceGetter * getter, void * getter_arg)
{
    UInt16 * offsets = (UInt16 *) data;
    UInt8 b;
    Int32 value;

    switch (source->schema.fields[fieldNum].type) {
    case FIELD_TYPE_BOOLEAN:
	b = getter->GetBoolean(getter_arg, fieldNum) ? 1 : 0;
	DmWrite(data, offsets[fieldNum], &b, sizeof(UInt8));
	break;

    case FIELD_TYPE_INTEGER:
	value = getter->GetInteger(getter_arg, fieldNum);
	DmWrite(data, offsets[fieldNum], &value, sizeof(Int32));
	break;

    default:
	ErrDisplay("field type not supported");
	break;
    }
}

static void
DB_DeleteRecord(DataSource * source, UInt16 index)
{
    DmRemoveRecord(source->db, index);
}

static UInt32
DB_GetRecordID(DataSource * source, UInt16 index)
{
    UInt32 uniqueID;

    DmRecordInfo(source->db, index, 0, &uniqueID, 0);
    return uniqueID;
}

static UInt16
DB_GetNumOfViews(DataSource * source)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    UInt32 token = 0;
    UInt16 numViews = 0;

    GetAppInfoBlock(source, &info);
    while (GetNextChunk(&info, &token, &chunk)) {
	if (chunk.chunk_type == CHUNK_LISTVIEW_DEFINITION) numViews++;
    }
    PutAppInfoBlock(source, &info);

    return numViews;
}

static DataSourceView *
DB_GetView(DataSource * source, UInt16 viewIndex)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    DataSourceView * view;

    /* Retrieve the list view definition. */
    GetAppInfoBlock(source, &info);
    if (GetChunkByType(&info, &chunk, CHUNK_LISTVIEW_DEFINITION, viewIndex)) {
	UInt16 flags, numCols, i;
	UInt8 * p;

	/* Allocate space for the view structure. */
	view = (DataSourceView *) MemPtrNew(sizeof(*view));

        /* Copy the header of this definition into local storage. */
        MemMove(&flags,   chunk.chunk_data,     sizeof(UInt16));
        MemMove(&numCols, chunk.chunk_data + 2, sizeof(UInt16));

        StrNCopy(view->name, chunk.chunk_data + 4, sizeof(view->name));
	view->numCols = numCols;
        view->cols = MemPtrNew(view->numCols * sizeof(DataSourceViewColumn));

        /* Copy the column info of this definition into local storage. */
        p = chunk.chunk_data + 2 + 2 + 32;
        for (i = 0; i < view->numCols; ++i) {
	    UInt16 fieldNum, width;

            MemMove(&fieldNum, p,     sizeof(UInt16));
            MemMove(&width,    p + 2, sizeof(UInt16));
            p += 4ul;

	    view->cols[i].fieldNum = fieldNum;
	    view->cols[i].width = width;
        }
    } else {
	view = 0;
    }
    PutAppInfoBlock(source, &info);

    return view;
}

static void
DB_PutView(DataSource * source, UInt16 viewIndex, DataSourceView * view)
{
    MemPtrFree(view->cols);
    MemPtrFree(view);
}

static void
DB_StoreView(DataSource * source, UInt16 viewIndex, DataSourceView * view)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    UInt8 *p, *q;
    Boolean exists;
    UInt16 flags = 0, i;

    /* Construct the chunk that we will store. */
    p = MemPtrNew(2ul + 2ul + 32ul + view->numCols * (2ul + 2ul));
    MemMove(p, &flags, sizeof(UInt16));
    MemMove(p + 2ul, &view->numCols, sizeof(UInt16));
    MemMove(p + 4ul, view->name, 32);
    for (i = 0, q = p + 2 + 2 + 32; i < view->numCols; ++i) {
	MemMove(q, &(view->cols[i].fieldNum), sizeof(UInt16));
	MemMove(q + 2ul, &(view->cols[i].width), sizeof(UInt16));
	q += 4ul;
    }

    /* See if this view is an existing one that we can replace. */
    GetAppInfoBlock(source, &info);
    exists = GetChunkByType(&info, &chunk,
			    CHUNK_LISTVIEW_DEFINITION, viewIndex);
    PutAppInfoBlock(source, &info);

    if (exists) {
	/* We can replace the existing chunk. */
	DB_Chunk_Replace(source, CHUNK_LISTVIEW_DEFINITION, viewIndex,
			 p, MemPtrSize(p));
    } else {
	/* Append this chunk to the app info block instead. */
	DB_Chunk_Append(source, CHUNK_LISTVIEW_DEFINITION, p, MemPtrSize(p));
    }

    /* Free the temporary copy of the chunk. */
    MemPtrFree(p);
}

static void
DB_DeleteView(DataSource * source, UInt16 index)
{
    UInt16 numViews, activeView;

    /* Sanity check: Ensure that at least 1 view will remain and that
     * the index is valid.
     */
    numViews = source->ops.GetNumOfViews(source);
    if (numViews <= 1 || index >= numViews)
	return;

    /* Delete the list view's chunk from the app info block. */
    DB_Chunk_Delete(source, CHUNK_LISTVIEW_DEFINITION, index);

    /* Update the active list view index if its list view shifted down. */
    activeView = source->ops.GetOption_UInt16(source,
					      optionID_ListView_ActiveView);
    if (activeView == index) {
	/* active view was the one just deleted, only safe one is the first */
	activeView = 0;
    } else if (activeView > index) {
	/* active view was above so it is now been shifted down one */
	activeView--;
    }
    source->ops.SetOption_UInt16(source, optionID_ListView_ActiveView, activeView);
    source->ops.SetOption_UInt16(source, optionID_ListView_TopVisibleRecord, 0);
}

static void
DB_CheckUpdateSchema(DataSource * source,
		     DataSourceSchema * schema, UInt32* reorder,
		     Boolean* changed, Boolean* rebuild)
{
    UInt16 i;

    /* By default, we assume no changes happened whatsoever. */
    *changed = false;
    *rebuild = false;

    /* Adding or deleting fields requires a rebuild. */
    if (source->schema.numFields != schema->numFields) {
	*changed = true;
	*rebuild = true;
	return;
    }

    for (i = 0; i < source->schema.numFields; ++i) {
	/* Changing the order of the fields or changing the type
	 * requires a full rebuild.
	 */
	if (source->schema.fields[i].type != schema->fields[i].type
	    || reorder[i] != i) {
	    *changed = true;
	    *rebuild = true;
	    return;
	}

	/* Field name changes don't need a rebuild. */
	if (StrCompare(source->schema.fields[i].name,
		       schema->fields[i].name) != 0) {
	    *changed = true;
	}
    }
}

static void
DB_UpdateSchema(DataSource * source,
		DataSourceSchema * schema, UInt32 * reorder)
{
    UInt16 i;
    UInt8 *p, *q;
    UInt32 size;

    /* We can only reach this point if DB_CheckUpdateSchema() verified
     * that only field name changes were done. Thus, we can safely
     * just go ahead and change the field names.
     */

    /* Determine the size of the new field name chunk. */
    size = 0;
    for (i = 0; i < schema->numFields; ++i) {
	size += StrLen(schema->fields[i].name) + 1;
    }

    /* Allocate, fill, and install the new field name chunk. */
    p = q = MemPtrNew(size);
    for (i = 0; i < schema->numFields; ++i) {
	StrCopy(q, schema->fields[i].name);
	q += StrLen(schema->fields[i].name) + 1;
    }
    DB_Chunk_Replace(source, CHUNK_FIELD_NAMES, 0, p, size);
    MemPtrFree(p);

    {
	AppInfoBlock info;
	GetAppInfoBlock(source, &info);
	if (! ValidateAppInfoBlock(&info))
	    DbgPrintF("invalid!!");
	PutAppInfoBlock(source, &info);
    }

    /* Now we need to update the schema. */
    MemPtrFree(source->schema.fields);
    CopySchema(&source->schema, schema);
}

static Boolean
DB_GetOption_Boolean(DataSource * source, UInt16 optionID)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    Boolean result = false;
    UInt16 flags;

    GetAppInfoBlock(source, &info);
    switch (optionID) {
    case optionID_DisableGlobalFind:
	flags = *((UInt16 *) info.ptr);
	if ((flags & FLAG_GFIND_FORCE_ENABLE) != 0)
	    result = false;
	else if ((flags & FLAG_GFIND_FORCE_DISABLE) != 0)
	    result = true;
	else
	    result = false;
	break;

    case optionID_LFind_CaseSensitive:
    case optionID_LFind_WholeWord:
	if (GetChunkByType(&info, &chunk, CHUNK_LFIND_OPTIONS, 0)) {
	    MemMove(&flags, chunk.chunk_data, sizeof(flags));
	    switch (optionID) {
	    case optionID_LFind_CaseSensitive:
		result = (flags & 0x0001) != 0;
		break;

	    case optionID_LFind_WholeWord:
		result = (flags & 0x0002) != 0;
		break;
	    }
	} else {
	    result = false;
	}
	break;

    default:
	result = false;
	break;
    }
    PutAppInfoBlock(source, &info);

    return result;
}

static void
DB_SetOption_Boolean(DataSource * source, UInt16 optionID, Boolean value)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    UInt16 flags;

    GetAppInfoBlock(source, &info);
    switch (optionID) {
    case optionID_DisableGlobalFind:
	flags = *((UInt16 *) info.ptr);
	flags &= ~(FLAG_GFIND_FORCE_DISABLE | FLAG_GFIND_FORCE_ENABLE);
	if (value) flags |= FLAG_GFIND_FORCE_DISABLE;
	DmWrite(info.ptr, 0, &flags, sizeof(flags));
	break;

    case optionID_LFind_CaseSensitive:
    case optionID_LFind_WholeWord:
	if (GetChunkByType(&info, &chunk, CHUNK_LFIND_OPTIONS, 0)) {
	    UInt16 bits;

	    MemMove(&flags, chunk.chunk_data, sizeof(UInt16));
	    switch (optionID) {
	    case optionID_LFind_CaseSensitive:
		bits = 0x0001;
		break;

	    case optionID_LFind_WholeWord:
		bits = 0x0002;
		break;

	    default:
		bits = 0;
		break;
	    }
	    if (value)
		flags |= bits;
	    else
		flags &= ~(bits);
	    DmWrite(info.ptr, chunk.chunk_data - info.ptr,
		    &flags, sizeof(flags));
	}
	break;
    }
    PutAppInfoBlock(source, &info);
}

static UInt16
DB_GetOption_UInt16(DataSource * source, UInt16 optionID)
{
    AppInfoBlock info;
    AppInfoChunk chunk;
    UInt16 result;

    switch (optionID) {
    case optionID_ListView_ActiveView:
    case optionID_ListView_TopVisibleRecord:
	GetAppInfoBlock(source, &info);
	if (GetChunkByType(&info, &chunk, CHUNK_LISTVIEW_OPTIONS, 0)) {
	    switch (optionID) {
	    case optionID_ListView_ActiveView:
		MemMove(&result, ((UInt16 *) chunk.chunk_data) + 0,
			sizeof(UInt16));
		break;

	    case optionID_ListView_TopVisibleRecord:
		MemMove(&result, ((UInt16 *) chunk.chunk_data) + 1,
			sizeof(UInt16));
		break;
	    }
	} else
	    result = 0;
	PutAppInfoBlock(source, &info);
	break;

    default:
	result = 0;
	break;
    }

    return result;
}

static void
DB_SetOption_UInt16(DataSource * source, UInt16 optionID, UInt16 value)
{
    AppInfoBlock info;
    AppInfoChunk chunk;

    switch (optionID) {
    case optionID_ListView_ActiveView:
    case optionID_ListView_TopVisibleRecord:
	GetAppInfoBlock(source, &info);
	if (GetChunkByType(&info, &chunk, CHUNK_LISTVIEW_OPTIONS, 0)) {
	    switch (optionID) {
	    case optionID_ListView_ActiveView:
		DmWrite(info.ptr, chunk.chunk_data - info.ptr,
			&value, sizeof(value));
		break;

	    case optionID_ListView_TopVisibleRecord:
		DmWrite(info.ptr, chunk.chunk_data-info.ptr + 1*sizeof(UInt16),
			&value, sizeof(value));
		break;
	    }
	}
	PutAppInfoBlock(source, &info);
	break;
    }
}

static Int16
DB_SortCallback(void * rec1, void * rec2, Int16 other,
		SortRecordInfoPtr rec1SortInfo, SortRecordInfoPtr rec2SortInfo,
		MemHandle appInfoH)
{
    SortCallbackData * data = (SortCallbackData *) sort_callback_data;
    int result;

    result = data->callback(data->source,
			    &data->getter, rec1,
			    &data->getter, rec2,
			    data->callback_data);

    return result;
}

static void
DB_Sort(DataSource * source,
	DataSourceSortCallback callback, void * callback_data)
{
    SortCallbackData data;

    data.callback = callback;
    data.callback_data = callback_data;
    data.source = source;
    data.getter.GetString = DB_Getter_GetString;
    data.getter.GetInteger = DB_Getter_GetInteger;
    data.getter.GetBoolean = DB_Getter_GetBoolean;
    data.getter.GetDate = DB_Getter_GetDate;
    data.getter.GetTime = DB_Getter_GetTime;
    sort_callback_data = &data;
    DmQuickSort(source->db, DB_SortCallback, 0);
}

static Boolean
DB_Capable(DataSource * source, DataSourceCapability capability)
{
    switch (capability) {
    case DS_CAPABLE_LISTVIEW_EDIT:
	return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_INSERT:
	return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_DELETE:
	return ! source->readonly && source->ops.GetNumOfViews(source) > 1;
    case DS_CAPABLE_LISTVIEW_NAME_EDIT:
	return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_FIELD_EDIT:
	return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_FIELD_INSERT:
	return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_FIELD_DELETE:
	return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_WIDTH_EDIT:
	return ! source->readonly;
    default:
	return false;
    }
}

static Err
DB_Open(DataSource * source, UInt16 mode, void * key_data, UInt16 key_size)
{
    DataSourceChooserKey * key = (DataSourceChooserKey *) key_data;
    Err err;
    UInt32 dbType;
    UInt16 numFields;
    AppInfoBlock info;
    LocalID dbID;

    /* Make sure that the key has a valid size. */
    if (key_size != sizeof(DataSourceChooserKey))
	return 0;
    
    /* Fill in the operations structure. */
    source->ops.Close = DB_Close;
    source->ops.Seek = DB_Seek;
    source->ops.LockRecord = DB_LockRecord;
    source->ops.UnlockRecord = DB_UnlockRecord;
    source->ops.UpdateRecord = DB_UpdateRecord;
    source->ops.SetField = DB_SetField;
    source->ops.DeleteRecord = DB_DeleteRecord;
    source->ops.GetRecordID = DB_GetRecordID;
    source->ops.GetNumOfViews = DB_GetNumOfViews;
    source->ops.GetView = DB_GetView;
    source->ops.PutView = DB_PutView;
    source->ops.StoreView = DB_StoreView;
    source->ops.DeleteView = DB_DeleteView;
    source->ops.CheckUpdateSchema = DB_CheckUpdateSchema;
    source->ops.UpdateSchema = DB_UpdateSchema;
    source->ops.GetOption_Boolean = DB_GetOption_Boolean;
    source->ops.SetOption_Boolean = DB_SetOption_Boolean;
    source->ops.GetOption_UInt16 = DB_GetOption_UInt16;
    source->ops.SetOption_UInt16 = DB_SetOption_UInt16;
    source->ops.GetKey = DS_PDB_GetKey;
    source->ops.GetPDB = DS_PDB_GetPDB;
    source->ops.GetTitle = DS_PDB_GetTitle;
    source->ops.SetTitle = DS_PDB_SetTitle;
    source->ops.Sort = DB_Sort;
    source->ops.Capable = DB_Capable;

    /* We do not have any private data for this source. */
    source->data = 0;

    /* Retrieve the location in memory of this database. */
    dbID = DmFindDatabase(key->cardNo, key->name);
    if (!dbID)
	return DmGetLastErr();

    /* Check to see if we support this type of database. */
    DmDatabaseInfo(key->cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, 0, 0, &dbType, 0);
    if (dbType != 'DB00')
	return dmErrCantOpen;

    /* Open this database. */
    source->db = DmOpenDatabase(key->cardNo, dbID, mode);
    if (! source->db)
	return DmGetLastErr();

    /* Retrieve the app info block which contains all the metadata. */
    GetAppInfoBlock(source, &info);

    /* Validate the app info block before using it. */
    if (! ValidateAppInfoBlock(&info)) {
	PutAppInfoBlock(source, &info);
	DmCloseDatabase(source->db);
	source->db = 0;
	return dmErrCorruptDatabase;
    }

    /* Extract the number of fields from the header. */
    MemMove(&numFields, info.ptr + sizeof(UInt16), sizeof(numFields));
    source->schema.numFields = numFields;

    /* Extract the entire schema into local storage. */
    err = ExtractSchema(source, &info);
    if (err) {
	PutAppInfoBlock(source, &info);
	DmCloseDatabase(source->db);
	source->db = 0;
	return err;
    }

    /* Put away our reference to the app info block. */
    PutAppInfoBlock(source, &info);

    return 0;
}

static Err
DB_OpenPDB(DataSource * source, UInt16 mode, UInt16 cardNo, LocalID dbID)
{
    DataSourceChooserKey key;

    key.cardNo = cardNo;
    DmDatabaseInfo(cardNo, dbID, key.name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return DB_Open(source, mode, &key, sizeof(key));
}

static Err
DB_Create(const char* name, DataSourceSchema* schema,
	  void ** key_data_ptr, UInt16 * key_size_ptr)
{
    Err err;
    LocalID dbID, appInfoID;
    MemHandle appInfoHand, h;
    void *p, *q;
    UInt16 flags, i;
    UInt32 size;
    ListViewDefinition def;
    DmOpenRef db;
    DataSourceChooserKey *key;

    err = DmCreateDatabase(0, name, DBCreatorID, 'DB00', false);
    if (err)
	return err;

    dbID = DmFindDatabase(0, name);

    db = DmOpenDatabase(0, dbID, dmModeReadWrite);
    if (! db)
	return DmGetLastErr();

    /* Create a memory handle with no chunks. */
    h = MemHandleNew(2 * sizeof(UInt16));
    p = MemHandleLock(h);
    flags = 0;
    MemMove(p, &flags, sizeof(UInt16)); /* flags */
    MemMove(p + sizeof(UInt16), &(schema->numFields), sizeof(UInt16));
    MemPtrUnlock(p);
    
    /* Determine how large the field name chunk should be. */
    size = 0;
    for (i = 0; i < schema->numFields; ++i) {
	size += StrLen(schema->fields[i].name) + 1;
    }
    p = q = MemPtrNew(size);
    for (i = 0; i < schema->numFields; ++i) {
	StrCopy(q, schema->fields[i].name);
	q += StrLen(schema->fields[i].name) + 1;
    }
    AppendChunk(h, CHUNK_FIELD_NAMES, p, size);
    MemPtrFree(p);
    
    /* Build the field types chunk. */
    p = MemPtrNew(schema->numFields * sizeof(UInt16));
    for (i = 0, q = p; i < schema->numFields; ++i) {
	UInt16 type;

	switch (schema->fields[i].type) {
	case FIELD_TYPE_STRING:
	    type = 0;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    type = 1;
	    break;

	case FIELD_TYPE_INTEGER:
	    type = 2;
	    break;

	case FIELD_TYPE_DATE:
	    type = 3;
	    break;

	case FIELD_TYPE_TIME:
	    type = 4;
	    break;

	default:
	    ErrDisplay("field type not supported");
	    break;
	}

	MemMove(q, &type, sizeof(UInt16));
	q += 2UL;
    }
    AppendChunk(h, CHUNK_FIELD_TYPES, p, schema->numFields * sizeof(UInt16));
    MemPtrFree(p);

    /* Build an in-core list view definition encompassing all the fields. */
    def.flags = 0;
    def.numCols = schema->numFields;
    p = GetStringResource(stringID_AllFields);
    StrNCopy(def.name, p, sizeof(def.name) - 1);
    def.name[sizeof(def.name) - 1] = '\0';
    PutStringResource(p);
    def.cols = MemPtrNew(def.numCols * sizeof(ListViewColumn));
    for (i = 0; i < def.numCols; ++i) {
	def.cols[i].fieldNum = i;
	def.cols[i].width = 80;
    }

    /* Construct the chunk that we will store. */
    p = MemPtrNew(2ul + 2ul + 32ul + def.numCols * (2ul + 2ul));
    MemMove(p, &def.flags, sizeof(UInt16));
    MemMove(p + 2ul, &def.numCols, sizeof(UInt16));
    MemMove(p + 4ul, def.name, 32);
    for (i = 0, q = p + 2 + 2 + 32; i < def.numCols; ++i) {
	MemMove(q, &(def.cols[i].fieldNum), sizeof(UInt16));
	MemMove(q + 2ul, &(def.cols[i].width), sizeof(UInt16));
	q += 4ul;
    }

    /* Append the chunk and free the temporary space. */
    AppendChunk(h, CHUNK_LISTVIEW_DEFINITION, p, MemPtrSize(p));
    MemPtrFree(p);
    MemPtrFree(def.cols);

    /* Create and append the chunk that will hold list view options. */
    p = MemPtrNew(2 * sizeof(UInt16));
    MemSet(p, MemPtrSize(p), 0);
    AppendChunk(h, CHUNK_LISTVIEW_OPTIONS, p, MemPtrSize(p));
    MemPtrFree(p);

    /* Create and append the chunk that will hold the local find options. */
    p = MemPtrNew(1 * sizeof(UInt16));
    MemSet(p, MemPtrSize(p), 0);
    AppendChunk(h, CHUNK_LFIND_OPTIONS, p, MemPtrSize(p));
    MemPtrFree(p);

    /* Create and fill the handle that will actually store the data. */
    size = MemHandleSize(h);
    appInfoHand = DmNewHandle(db, size);
    appInfoID = MemHandleToLocalID(appInfoHand);
    p = MemLocalIDToLockedPtr(appInfoID, 0);
    q = MemHandleLock(h);
    DmWrite(p, 0, q, MemHandleSize(h));
    MemPtrUnlock(p);
    MemPtrUnlock(q);

    /* Store a handle to the app info block. */
    DmSetDatabaseInfo(0, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0);

    /* Close the database. */
    DmCloseDatabase(db);

    /* Add this database into the chooser screen. */
    key = (DataSourceChooserKey *) MemPtrNew(sizeof(DataSourceChooserKey));
    MemSet(key, sizeof(*key), 0);
    key->cardNo = 0;
    StrNCopy(key->name, name, sizeof(key->name) - 1);
    key->name[sizeof(key->name)-1] = '\0';
    *key_data_ptr = key;
    *key_size_ptr = sizeof(*key);

    return 0;
}

static Boolean
DB_SupportsFieldType(DataSourceFieldType type)
{
    switch (type) {
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_INTEGER:
    case FIELD_TYPE_BOOLEAN:
    case FIELD_TYPE_DATE:
    case FIELD_TYPE_TIME:
	return true;

    default:
	return false;
    }
}

static void
DB_Enumerate(DataSourceEnumerationFunc callback, void * arg)
{
    return DS_PDB_Enumerate(callback, arg, 'DB00', DBCreatorID, sourceID_DB);
}

static Boolean
DB_GlobalFind(DataSourceGlobalFindFunc callback, Boolean continuation,
	      FindParamsPtr params)
{
    return DS_PDB_GlobalFind(callback, continuation, params, 'DB00',
			     DBCreatorID, DB_OpenPDB);
}

static Boolean
DB_IsPresent(void * key_data, UInt32 key_size)
{
    return DS_PDB_IsPresent(key_data, key_size, 'DB00', DBCreatorID);
}

static Boolean
DB_CanHandlePDB(UInt16 cardNo, LocalID dbID, UInt32 type, UInt32 creator)
{
    return (creator == DBCreatorID) && (type == 'DB00');
}

static DataSourceDriver *
DB_GetDriver(void)
{
    DataSourceDriver * driver = MemPtrNew(sizeof(DataSourceDriver));

    StrCopy(driver->name, "DB");
    driver->sourceID = sourceID_DB;
    driver->class_ops.Open = DB_Open;
    driver->class_ops.OpenPDB = DB_OpenPDB;
    driver->class_ops.CheckTitle = DS_PDB_CheckTitle;
    driver->class_ops.Create = DB_Create;
    driver->class_ops.SupportsFieldType = DB_SupportsFieldType;
    driver->class_ops.Delete = DS_PDB_Delete;
    driver->class_ops.Enumerate = DB_Enumerate;
    driver->class_ops.GlobalFind = DB_GlobalFind;
    driver->class_ops.IsPresent = DB_IsPresent;
    driver->class_ops.CompareKey = DS_PDB_CompareKey;
    driver->class_ops.CanHandlePDB = DB_CanHandlePDB;
    driver->class_ops.Beam = DS_PDB_Beam;

    return driver;
}


/*
 * Support for the legacy 0.2.x database format.
 */

#ifdef CONFIG_LEGACY_SUPPORT

static char *
OldDB_Getter_GetString(void * data, UInt16 fieldNum)
{
    void * * ptrs = data;

    return ptrs[fieldNum];
}

static Int32
OldDB_Getter_GetInteger(void * data, UInt16 fieldNum)
{
    void * * ptrs = data;
    Int32 value;

    MemMove(&value, ptrs[fieldNum], sizeof(Int32));
    return value;
}

static Boolean
OldDB_Getter_GetBoolean(void * data, UInt16 fieldNum)
{
    void * * ptrs = data;

    if ( *((UInt8 *) ptrs[fieldNum]))
	return true;
    else
	return false;
}

static void
OldDB_Getter_GetDate(void * data, UInt16 fieldNum,
		     UInt16* year, UInt8* mon, UInt8* day)
{
    ErrDisplay("OldDB_Getter_GetDate: no support");
}

static void
OldDB_Getter_GetTime(void * data, UInt16 fieldNum, UInt8* hour, UInt8* minute)
{
    ErrDisplay("OldDB_Getter_GetTime: no support");
}

static void
OldDB_UnpackRecord(DataSource * source, void * packed, void * unpacked[])
{
    UInt8 * rec = packed;
    UInt16 i;

    for (i = 0; i < source->schema.numFields; i++) {
	switch (source->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	    unpacked[i] = rec;
	    rec += StrLen(rec) + 1;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    unpacked[i] = rec;
	    rec += 1;
	    break;

	case FIELD_TYPE_INTEGER:
	    unpacked[i] = rec;
	    rec += 4;
	    break;

	default:
	    ErrDisplay("unknown field type");
	    break;
	}
    }
}

static UInt32
OldDB_PackedRecordSize(DataSource * source,
		       DataSourceGetter * getter, void * data)
{
    UInt32 size = 0;
    UInt16 i;
    char * str;

    for (i = 0; i < source->schema.numFields; i++) {
	switch (source->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	    str = getter->GetString(data, i);
	    size += StrLen(str) + 1;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    size += sizeof(UInt8);
	    break;

	case FIELD_TYPE_INTEGER:
	    size += sizeof(Int32);
	    break;

	default:
	    ErrDisplay("unsupported field type");
	    break;
	}
    }

    return size;
}

static void
OldDB_PackRecord(DataSource * source, void * dest,
		 DataSourceGetter * getter, void * data)
{
    UInt32 offset = 0;
    Int32 value;
    UInt16 i;
    UInt8 b;
    char * str;

    for (i = 0; i < source->schema.numFields; i++) {
	switch (source->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	    str = getter->GetString(data, i);
	    DmWrite(dest, offset, str, StrLen(str) + 1);
	    offset += StrLen(str) + 1;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    b = (getter->GetBoolean(data, i)) ? 1 : 0;
	    DmWrite(dest, offset, &b, sizeof(UInt8));
	    offset += sizeof(UInt8);
	    break;

	case FIELD_TYPE_INTEGER:
	    value = getter->GetInteger(data, i);
	    DmWrite(dest, offset, &value, sizeof(Int32));
	    offset += sizeof(Int32);
	    break;

	default:
	    ErrDisplay("unsupported field type");
	    break;
	}
    }
}

static void
OldDB_LockRecord(DataSource * source, UInt16 index, Boolean writable,
		 DataSourceGetter * getter, void ** data_ptr)
{
    void ** ptrs, *packed;
    MemHandle h;

    ptrs = MemPtrNew(source->schema.numFields * sizeof(void *));

    if (writable)
        h = DmGetRecord(source->db, index);
    else
	h = DmQueryRecord(source->db, index);

    packed = MemHandleLock(h);
    OldDB_UnpackRecord(source, packed, ptrs);

    getter->GetString  = OldDB_Getter_GetString;
    getter->GetInteger = OldDB_Getter_GetInteger;
    getter->GetBoolean = OldDB_Getter_GetBoolean;
    getter->GetDate    = OldDB_Getter_GetDate;
    getter->GetTime    = OldDB_Getter_GetTime;
    *data_ptr = ptrs;
}

static void
OldDB_UnlockRecord(DataSource * source, UInt16 index, Boolean writable,
		void * record_data)
{
    void * * ptrs = record_data;

    MemPtrUnlock(ptrs[0]);

    if (writable)
        DmReleaseRecord(source->db, index, true);

    MemPtrFree(ptrs);
}

static Err
OldDB_UpdateRecord(DataSource * source, UInt16 index, Boolean doInsert,
		   DataSourceGetter * getter, void * data)
{
    UInt32 size;
    MemHandle recH;
    void * packed;

    /* Determine the packed size of the new record. */
    size = OldDB_PackedRecordSize(source, getter, data);

    if (index == dmMaxRecordIndex || doInsert) {
	/* Create a new record in the database. */
	recH = DmNewRecord(source->db, &index, size);
	if (recH == 0) {
	    /* If we couldn't make the record, bail out. */
	    return DmGetLastErr();
	}
    } else {
	/* Otherwise, retrieve and resize an existing record. */
	DmResizeRecord(source->db, index, size);
	recH = DmGetRecord(source->db, index);
    }

    /* Pack the new data into the record. */
    packed = MemHandleLock(recH);
    OldDB_PackRecord(source, packed, getter, data);
    MemPtrUnlock(packed);

    /* Release the write lock on the record. */
    DmReleaseRecord(source->db, index, true);

    return 0;
}

static void
OldDB_SetField(DataSource * source, void * arg, UInt16 fieldNum,
	       DataSourceGetter * getter, void * data)
{
    void **ptrs = (void **) arg;
    UInt8 b;
    Int32 value;

    /* Specific behavior depends on the type of field. */
    switch (source->schema.fields[fieldNum].type) {
    case FIELD_TYPE_BOOLEAN:
	b = getter->GetBoolean(data, fieldNum) ? 1 : 0;
	DmWrite(ptrs[0], ptrs[fieldNum] - ptrs[0], &b, sizeof(UInt8));
	break;

    case FIELD_TYPE_INTEGER:
	value = getter->GetInteger(data, fieldNum);
	DmWrite(ptrs[0], ptrs[fieldNum] - ptrs[0], &value, sizeof(Int32));
	break;

    default:
	ErrDisplay("field type not supported");
	break;
    }
}

static UInt16
OldDB_GetNumOfViews(DataSource * source)
{
    return 1;
}

static DataSourceView *
OldDB_GetView(DataSource * source, UInt16 viewIndex)
{
    DataSourceView * view;
    UInt16 i, *widths = (UInt16 *) source->data;
    char * p;

    if (viewIndex != 0)
	return 0;

    view = MemPtrNew(sizeof(*view));
    p = GetStringResource(stringID_AllFields);
    StrNCopy(view->name, p, sizeof(view->name) - 1);
    view->name[sizeof(view->name) - 1] = '\0';
    PutStringResource(p);
    view->numCols = source->schema.numFields;
    view->cols = MemPtrNew(view->numCols * sizeof(DataSourceViewColumn));
    for (i = 0; i < source->schema.numFields; ++i) {
	view->cols[i].fieldNum = i;
	view->cols[i].width = widths[i];
    }

    return view;
}

static void
OldDB_StoreView(DataSource * source, UInt16 viewIndex, DataSourceView * view)
{
    UInt16 * widths = (UInt16 *) source->data;
    UInt16 i;
    Boolean changed;

    /* We only support a single view of the data. */
    if (viewIndex != 0 || view->numCols != source->schema.numFields)
	return;

    /* Bail out if the field numbers don't match up. */
    for (i = 0; i < source->schema.numFields; ++i) {
	if (view->cols[i].fieldNum != i) return;
    }

    /* Scan the field widths and look for any changes. */
    changed = false;
    for (i = 0; i < source->schema.numFields; ++i) {
	if (view->cols[i].width != widths[i]) {
	    changed = true;
	    widths[i] = view->cols[i].width;
	}
    }

    /* Store any changes back into the database. */
    if (changed) {
	UInt16 cardNo;
	void * appInfo;
	UInt32 offset;
	LocalID appInfoID;

	/* Retrieve a pointer into the app info block. */
	DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
	appInfoID = DmGetAppInfoID(source->db);
	appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);

	offset = 6;
	for (i = 0; i < source->schema.numFields; ++i) {
	    /* Store the field width. */
	    DmWrite(appInfo, offset + 2 + 32 + 2, &widths[i], sizeof(UInt16));

	    /* Advance to the next entry. */
	    offset += 2 + 32 + 2 + 2;
	}
	
	/* Put away the pointer to the app info block. */
	MemPtrUnlock(appInfo);
    }
}

static void
OldDB_UpdateSchema(DataSource * source,
		   DataSourceSchema * schema, UInt32 * reorder)
{
    UInt16 cardNo, i;
    void * appInfo;
    UInt32 offset;
    LocalID appInfoID;
    Char name[32];

    /* Retrieve a pointer into the app info block. */
    DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
    appInfoID = DmGetAppInfoID(source->db);
    appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);

    offset = 6;
    for (i = 0; i < source->schema.numFields; ++i) {
	/* Build a local copy of the field name. */
	StrNCopy(name, schema->fields[i].name, sizeof(name) - 1);
	name[sizeof(name)-1] = '\0';

	/* Store the field name into the app info block. */
	DmWrite(appInfo, offset + 2, &name[0], 32);

	/* Advance to the next entry. */
	offset += 2 + 32 + 2 + 2;
    }
	
    /* Put away the pointer to the app info block. */
    MemPtrUnlock(appInfo);

    /* Now we need to update the schema. */
    MemPtrFree(source->schema.fields);
    CopySchema(&source->schema, schema);
}

static Boolean
OldDB_GetOption_Boolean(DataSource * source, UInt16 optionID)
{
    UInt16 cardNo;
    void * appInfo;
    LocalID appInfoID;
    Boolean result;

    /* Retrieve a pointer into the app info block. */
    DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
    appInfoID = DmGetAppInfoID(source->db);
    appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);

    switch (optionID) {
    case optionID_DisableGlobalFind:
	result = ( *((UInt16 *) appInfo) & 0x0001 ) != 0;
	break;

    case optionID_LFind_CaseSensitive:
	result = ( *((UInt16 *) appInfo) & 0x0002 ) != 0;
	break;

    case optionID_LFind_WholeWord:
	result = ( *((UInt16 *) appInfo) & 0x0004 ) != 0;
	break;

    default:
	result = false;
	break;
    }

    MemPtrUnlock(appInfo);

    return result;
}

static void
OldDB_SetOption_Boolean(DataSource * source, UInt16 optionID, Boolean value)
{
    UInt16 cardNo;
    void * appInfo;
    LocalID appInfoID;
    UInt16 flags, bits;

    /* Retrieve a pointer into the app info block. */
    DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
    appInfoID = DmGetAppInfoID(source->db);
    appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);

    flags = *((UInt16 *) appInfo);

    switch (optionID) {
    case optionID_DisableGlobalFind:
	bits = 0x0001;
	break;

    case optionID_LFind_CaseSensitive:
	bits = 0x0002;
	break;

    case optionID_LFind_WholeWord:
	bits = 0x0004;
	break;

    default:
	bits = 0x0000;
	break;
    }

    if (value)
	flags |= bits;
    else
	flags &= ~(bits);

    DmWrite(appInfo, 0, &flags, sizeof(UInt16));
    MemPtrUnlock(appInfo);
}

static UInt16
OldDB_GetOption_UInt16(DataSource * source, UInt16 optionID)
{
    void * appInfo;
    LocalID appInfoID;
    UInt16 result, cardNo;

    switch (optionID) {
    case optionID_ListView_ActiveView:
	result = 0;
	break;

    case optionID_ListView_TopVisibleRecord:
	DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
	appInfoID = DmGetAppInfoID(source->db);
	appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);
	result = *( ((UInt16 *) appInfo) + 1 );
	MemPtrUnlock(appInfo);
	break;

    default:
	result = 0;
	break;
    }

    return result;
}

static void
OldDB_SetOption_UInt16(DataSource * source, UInt16 optionID, UInt16 value)
{
    if (optionID == optionID_ListView_TopVisibleRecord) {
	void * appInfo;
	LocalID appInfoID;
	UInt16 cardNo;

	DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
	appInfoID = DmGetAppInfoID(source->db);
	appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);
	DmWrite(appInfo, 1 * sizeof(UInt16), &value, sizeof(value));
	MemPtrUnlock(appInfo);
    }
}

static Int16
OldDB_SortCallback(void * rec1, void * rec2, Int16 other,
		   SortRecordInfoPtr rec1SortInfo,
		   SortRecordInfoPtr rec2SortInfo, MemHandle appInfoH)
{
    SortCallbackData * data = (SortCallbackData *) sort_callback_data;
    void * ptrs1[data->source->schema.numFields];
    void * ptrs2[data->source->schema.numFields];
    int result;

    OldDB_UnpackRecord(data->source, rec1, ptrs1);
    OldDB_UnpackRecord(data->source, rec2, ptrs2);

    result = data->callback(data->source,
			    &data->getter, &ptrs1,
			    &data->getter, &ptrs2,
			    data->callback_data);

    return result;
}

static void
OldDB_Sort(DataSource * source,
	   DataSourceSortCallback callback, void * callback_data)
{
    SortCallbackData data;

    data.callback = callback;
    data.callback_data = callback_data;
    data.source = source;
    data.getter.GetString = OldDB_Getter_GetString;
    data.getter.GetInteger = OldDB_Getter_GetInteger;
    data.getter.GetBoolean = OldDB_Getter_GetBoolean;
    sort_callback_data = &data;
    DmQuickSort(source->db, OldDB_SortCallback, 0);
}

static Boolean
OldDB_Capable(DataSource * source, DataSourceCapability capability)
{
    switch (capability) {
    case DS_CAPABLE_LISTVIEW_EDIT:
	return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_INSERT:
	return false;
    case DS_CAPABLE_LISTVIEW_DELETE:
	return false;
    case DS_CAPABLE_LISTVIEW_NAME_EDIT:
	return false;
    case DS_CAPABLE_LISTVIEW_FIELD_EDIT:
	return false;
    case DS_CAPABLE_LISTVIEW_FIELD_INSERT:
	return false;
    case DS_CAPABLE_LISTVIEW_FIELD_DELETE:
	return false;
    case DS_CAPABLE_LISTVIEW_WIDTH_EDIT:
	return ! source->readonly;
    default:
	return false;
    }
}

static Err
OldDB_Open(DataSource * source, UInt16 mode, void * key_data, UInt16 key_size)
{
    DataSourceChooserKey * key = (DataSourceChooserKey *) key_data;
    UInt32 dbType, size;
    UInt16 numFields, i;
    LocalID dbID, appInfoID;
    UInt8* appInfo, *p, *q;
    UInt16* widths;

    /* Make sure that the key has a valid size. */
    if (key_size != sizeof(DataSourceChooserKey))
	return 0;
    
    /* Fill in the operations structure. */
    source->ops.Close = DB_Close;
    source->ops.Seek = DB_Seek;
    source->ops.LockRecord = OldDB_LockRecord;
    source->ops.UnlockRecord = OldDB_UnlockRecord;
    source->ops.UpdateRecord = OldDB_UpdateRecord;
    source->ops.SetField = OldDB_SetField;
    source->ops.DeleteRecord = DB_DeleteRecord;
    source->ops.GetRecordID = DB_GetRecordID;
    source->ops.GetNumOfViews = OldDB_GetNumOfViews;
    source->ops.GetView = OldDB_GetView;
    source->ops.PutView = DB_PutView;
    source->ops.StoreView = OldDB_StoreView;
    source->ops.CheckUpdateSchema = DB_CheckUpdateSchema;
    source->ops.UpdateSchema = OldDB_UpdateSchema;
    source->ops.GetOption_Boolean = OldDB_GetOption_Boolean;
    source->ops.SetOption_Boolean = OldDB_SetOption_Boolean;
    source->ops.GetOption_UInt16 = OldDB_GetOption_UInt16;
    source->ops.SetOption_UInt16 = OldDB_SetOption_UInt16;
    source->ops.GetKey = DS_PDB_GetKey;
    source->ops.GetPDB = DS_PDB_GetPDB;
    source->ops.GetTitle = DS_PDB_GetTitle;
    source->ops.SetTitle = DS_PDB_SetTitle;
    source->ops.Sort = OldDB_Sort;
    source->ops.Capable = OldDB_Capable;

    /* Retrieve the location in memory of this database. */
    dbID = DmFindDatabase(key->cardNo, key->name);
    if (!dbID)
	return DmGetLastErr();

    /* Check to see if we support this type of database. */
    DmDatabaseInfo(key->cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, 0, 0, &dbType, 0);
    if (dbType != 'DB99')
	return dmErrCantOpen;

    /* Open this database. */
    source->db = DmOpenDatabase(key->cardNo, dbID, mode);
    if (! source->db)
	return DmGetLastErr();

    /* Retrieve the app info block which contains all the metadata. */
    appInfoID = DmGetAppInfoID(source->db);
    appInfo = MemLocalIDToLockedPtr(appInfoID, key->cardNo);
    size = MemPtrSize(appInfo);

    /* Extract the number of fields from the header. */
    MemMove(&numFields, appInfo + 2*sizeof(UInt16), sizeof(numFields));
    source->schema.numFields = numFields;
    source->data = MemPtrNew(numFields * sizeof(UInt16));
    widths = (UInt16 *) source->data;

    /* Ensure that we have enough space for all the fields. */
    if (size < (numFields * (2 + 32 + 2 + 2))) {
	MemPtrUnlock(appInfo);
	DmCloseDatabase(source->db);
	return dmErrCantOpen;
    }

    /* Determine how much space needs to be allocated. */
    p = appInfo + 6;
    size = numFields * sizeof(DataSourceFieldInfo);
    for (i = 0; i < numFields; ++i) {
	size += StrLen(p + 2) + 1;
	p += 2 + 32 + 2 + 2;
    }
    source->schema.fields = MemPtrNew(size);

    /* Extract the remainder of the schema from the block. */
    p = appInfo + 6;
    q = (UInt8 *) (source->schema.fields + numFields);
    for (i = 0; i < numFields; ++i) {
	UInt16 type;

	MemMove(&type, p + 0, sizeof(UInt16));
	switch (type) {
	case 0:
	    source->schema.fields[i].type = FIELD_TYPE_STRING;
	    break;
	case 1:
	    source->schema.fields[i].type = FIELD_TYPE_BOOLEAN;
	    break;
	case 2:
	    source->schema.fields[i].type = FIELD_TYPE_INTEGER;
	    break;
	}

	/* Copy the field name over. */
	source->schema.fields[i].name = q;
	StrCopy((char *) q, (char *) (p + 2));
	q += StrLen((char *) (p + 2)) + 1;

	/* Copy the width into the private data section of source. */
	MemMove(&widths[i], p + 2 + 32 + 2, sizeof(UInt16));

	/* Advance to the next entry. */
	p += 2 + 32 + 2 + 2;
    }

    /* Put away our reference to the app info block. */
    MemPtrUnlock(appInfo);

    return 0;
}

static Err
OldDB_Create(const char* title, DataSourceSchema* schema,
	     void ** key_data_ptr, UInt16 * key_size_ptr)
{
    Err err;
    LocalID dbID, appInfoID;
    MemHandle appInfoHand;
    DmOpenRef db;
    DataSourceChooserKey *key;
    UInt32 size, offset;
    void *p;
    UInt16 v, i;

    err = DmCreateDatabase(0, title, DBCreatorID, 'DB99', false);
    if (err)
        return err;

    dbID = DmFindDatabase(0, title);
    if (!dbID)
	return DmGetLastErr();

    db = DmOpenDatabase(0, dbID, dmModeReadWrite);
    if (!db)
        return DmGetLastErr();

    /* Allocate a handle for the app info block. */
    size = 6 + (schema->numFields * (2 + 32 + 2 + 2));
    appInfoHand = DmNewHandle(db, size);
    p = MemHandleLock(appInfoHand);
    DmSet(p, 0, size, 0);

    /* Store the number of fields. */
    v = schema->numFields;
    DmWrite(p, 2 * sizeof(UInt16), &v, sizeof(UInt16));

    /* Store the field information. */
    offset = 6;
    for (i = 0; i < schema->numFields; ++i) {
	/* Store the field type. */
	switch (schema->fields[i].type) {
	case FIELD_TYPE_STRING:
	    v = 0;
	    break;
	case FIELD_TYPE_BOOLEAN:
	    v = 1;
	    break;
	case FIELD_TYPE_INTEGER:
	    v = 2;
	    break;
	default:
	    ErrDisplay("field type not supported");
	    break;
	}
	DmWrite(p, offset, &v, sizeof(UInt16));

	/* Store the field name. */
	DmStrCopy(p, offset + 2, schema->fields[i].name);

	/* Set the initial column width. */
	v = 80;
	DmWrite(p, offset + 2 + 32 + 2, &v, sizeof(UInt16));

	/* Advance the offset to the next field. */
	offset += 2 + 32 + 2 + 2;
    }

    /* Store the app info handle into the database. */
    appInfoID = MemHandleToLocalID(appInfoHand);
    DmSetDatabaseInfo(0, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0); 

    /* Close the database. */
    DmCloseDatabase(db);

    /* Add this database into the chooser screen. */
    key = (DataSourceChooserKey *) MemPtrNew(sizeof(DataSourceChooserKey));
    MemSet(key, sizeof(*key), 0);
    key->cardNo = 0;
    StrNCopy(key->name, title, sizeof(key->name) - 1);
    key->name[sizeof(key->name)-1] = '\0';
    *key_data_ptr = key;
    *key_size_ptr = sizeof(*key);

    return 0;
}

static Boolean
OldDB_SupportsFieldType(DataSourceFieldType type)
{
    switch (type) {
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_INTEGER:
    case FIELD_TYPE_BOOLEAN:
        return true;
 
    default:
        return false;
    }
}

static void
OldDB_Enumerate(DataSourceEnumerationFunc callback, void * arg)
{
    return DS_PDB_Enumerate(callback, arg, 'DB99', DBCreatorID,
			    sourceID_DB02x);
}

static Err
OldDB_OpenPDB(DataSource * source, UInt16 mode, UInt16 cardNo, LocalID dbID)
{
    DataSourceChooserKey key;

    key.cardNo = cardNo;
    DmDatabaseInfo(cardNo, dbID, key.name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return OldDB_Open(source, mode, &key, sizeof(key));
}

static Boolean
OldDB_GlobalFind(DataSourceGlobalFindFunc callback, Boolean continuation,
		 FindParamsPtr params)
{
    return DS_PDB_GlobalFind(callback, continuation, params, 'DB99',
			     DBCreatorID, OldDB_OpenPDB);
}

static Boolean
OldDB_IsPresent(void * key_data, UInt32 key_size)
{
    return DS_PDB_IsPresent(key_data, key_size, 'DB99', DBCreatorID);
}

static Boolean
OldDB_CanHandlePDB(UInt16 cardNo, LocalID dbID, UInt32 type, UInt32 creator)
{
    return (creator == DBCreatorID) && (type == 'DB99');
}

static DataSourceDriver *
OldDB_GetDriver(void)
{
    DataSourceDriver * driver = MemPtrNew(sizeof(DataSourceDriver));

    StrCopy(driver->name, "DB 0.2.x");
    driver->sourceID = 1;
    driver->class_ops.Open = OldDB_Open;
    driver->class_ops.OpenPDB = OldDB_OpenPDB;
    driver->class_ops.CheckTitle = DS_PDB_CheckTitle;
    driver->class_ops.Create = OldDB_Create;
    driver->class_ops.SupportsFieldType = OldDB_SupportsFieldType;
    driver->class_ops.Delete = DS_PDB_Delete;
    driver->class_ops.Enumerate = OldDB_Enumerate;
    driver->class_ops.GlobalFind = OldDB_GlobalFind;
    driver->class_ops.IsPresent = OldDB_IsPresent;
    driver->class_ops.CompareKey = DS_PDB_CompareKey;
    driver->class_ops.CanHandlePDB = OldDB_CanHandlePDB;
    driver->class_ops.Beam = DS_PDB_Beam;

    return driver;
}

#endif /* CONFIG_LEGACY_SUPPORT */

DataSourceDriver *
GetInternalDataSources(void)
{
    DataSourceDriver* driver_DB = DB_GetDriver();
#ifdef CONFIG_LEGACY_SUPPORT
    DataSourceDriver* driver_OldDB = OldDB_GetDriver();
#endif

#ifdef CONFIG_LEGACY_SUPPORT
    driver_DB->next = driver_OldDB;
    driver_OldDB->next = 0;
#else
    driver_DB->next = 0;
#endif
    
    return driver_DB;
}

DataSourceDriver *
GetDataSources(void)
{
    DataSourceDriver * drivers = GetInternalDataSources();
    DataSourceDriver* driver_MobileDB = MobileDB_GetDriver();
    DataSourceDriver* driver_JFile3 = JFile3_GetDriver();
    DataSourceDriver* last;

    /* XXX: Make it so we can disable other data source drivers. */
    last = drivers;
    while (last->next != 0) last = last->next;
    last->next = driver_MobileDB;
    driver_MobileDB->next = driver_JFile3;
    driver_JFile3->next = 0;

    return drivers;
}

void
PutDataSources(DataSourceDriver * drivers)
{
    DataSourceDriver * tmp;

    while (drivers != 0) {
	tmp = drivers;
	drivers = drivers->next;
	MemPtrFree(tmp);
    }
}

void
DeleteDataSource(DataSourceDriver * drivers, UInt16 sourceID,
		 void * key_data, UInt32 key_size) {
    while (drivers) {
	if (sourceID == drivers->sourceID) {
	    drivers->class_ops.Delete(key_data, key_size);
	    return;
	}
	drivers = drivers->next;
    }
}

DataSource *
OpenDataSource(DataSourceDriver * drivers, UInt16 mode, UInt16 sourceID,
	       void * key_data, UInt16 key_size)
{
    DataSource * source;
    Err err;

    source = MemPtrNew(sizeof(DataSource));
    MemSet(source, sizeof(*source), 0);
    if ((mode & dmModeReadWrite) == dmModeReadOnly)
	source->readonly = true;

    while (drivers) {
	if (sourceID == drivers->sourceID) {
	    err = drivers->class_ops.Open(source, mode, key_data, key_size);
	    if (err == 0) {
		source->driver = drivers;
		return source;
	    }
	}
	drivers = drivers->next;
    }

    MemPtrFree(source);
    return 0;
}

DataSource *
OpenDataSourcePDB(DataSourceDriver * drivers, UInt16 mode,
		  UInt16 cardNo, LocalID dbID)
{
    DataSource * source;
    Err err;
    UInt32 type, creator;

    DmDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, 0, 0, &type, &creator);

    source = MemPtrNew(sizeof(DataSource));
    MemSet(source, sizeof(*source), 0);
    if ((mode & dmModeReadWrite) == dmModeReadOnly)
	source->readonly = true;

    while (drivers) {
	if (drivers->class_ops.CanHandlePDB
	    && drivers->class_ops.CanHandlePDB(cardNo, dbID, type, creator)) {
	    err = drivers->class_ops.OpenPDB(source, mode, cardNo, dbID);
	    if (err == 0) {
		source->driver = drivers;
		return source;
	    }
	}
	drivers = drivers->next;
    }

    MemPtrFree(source);
    return 0;
}

void
CloseDataSource(DataSource * source)
{
    source->ops.Close(source);
    MemPtrFree(source);
}

void CopySchema(DataSourceSchema * dst, DataSourceSchema * src)
{
    UInt32 size;
    UInt16 i;
    char * p;

    /* Destination will have the same number of fields. */
    dst->numFields = src->numFields;

    /* Calclate how much space is needed for strings. */
    size = dst->numFields * sizeof(DataSourceFieldInfo);
    for (i = 0; i < src->numFields; ++i)
	size += StrLen(src->fields[i].name) + 1;

    /* Allocate space for the field information and name strings. */
    dst->fields = MemPtrNew(size);

    /* Copy all of the data over. */
    p = (char *) (dst->fields + dst->numFields);
    for (i = 0; i < dst->numFields; ++i) {
	dst->fields[i].name = p;
	dst->fields[i].type = src->fields[i].type;
	StrCopy(p, src->fields[i].name);
	p += StrLen(src->fields[i].name) + 1;
    }
}

/* Search the drivers list for a specific ID. */
DataSourceDriver *
GetDriverByID(UInt16 sourceID)
{
    DataSourceDriver * driver = DriverList;

    while (driver) {
	if (driver->sourceID == sourceID) return driver;
	driver = driver->next;
    }

    return 0;
}

Err
BeamDataSource(DataSourceDriver * drivers, UInt16 sourceID,
	       void * key_data, UInt16 key_size)
{
    while (drivers) {
	if (drivers->sourceID == sourceID) {
	    return drivers->class_ops.Beam(key_data, key_size);
	}
	drivers = drivers->next;
    }

    return 0;
}

int
DS_StandardSortCallback(DataSource * source,
			DataSourceGetter * getter1, void * rec1_data,
			DataSourceGetter * getter2, void * rec2_data,
			void * callback_data)
{
    DataSourceSortInfo * info = (DataSourceSortInfo *) callback_data;
    UInt16 i;
    char *p1, *p2;
    Int16 result = -1;
    Int32 v1, v2;
    Boolean b1, b2;
    UInt16 match1, match2;

    for (i = 0; i < info->numFields; ++i) {
	UInt16 fieldNum = info->fields[i].fieldNum;
	DataSourceSortDirection direction = info->fields[i].direction;
	Boolean case_sensitive = info->fields[i].case_sensitive;

	switch (CurrentSource->schema.fields[fieldNum].type) {
	case FIELD_TYPE_STRING:
	    p1 = getter1->GetString(rec1_data, fieldNum);
	    p2 = getter2->GetString(rec2_data, fieldNum);
	    if (case_sensitive)
		result = TxtGlueCompare(p1, StrLen(p1), &match1,
					p2, StrLen(p2), &match2);
	    else
		result = TxtGlueCaselessCompare(p1, StrLen(p1), &match1,
						p2, StrLen(p2), &match2);
	    break;

	case FIELD_TYPE_INTEGER:
	    v1 = getter1->GetInteger(rec1_data, fieldNum);
	    v2 = getter2->GetInteger(rec2_data, fieldNum);
	    if (v1 < v2)
		result = -1;
	    else if (v1 > v2)
		result = 1;
	    else
		result = 0;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    b1 = getter1->GetBoolean(rec1_data, fieldNum);
	    b2 = getter2->GetBoolean(rec2_data, fieldNum);
	    if (b1 == b2)
		result = 0;
	    else if (!b1)
		result = -1;
	    else
		result = 1;
	    break;

	case FIELD_TYPE_DATE:
	    {
		UInt16 y1, y2;
		UInt8 m1, m2, d1, d2;

		getter1->GetDate(rec1_data, fieldNum, &y1, &m1, &d1);
		getter2->GetDate(rec2_data, fieldNum, &y2, &m2, &d2);

		if (y1 < y2)
		    result = -1;
		else if (y1 > y2)
		    result = 1;
		else {
		    if (m1 < m2)
			result = -1;
		    else if (m1 > m2)
			result = 1;
		    else {
			if (d1 < d2)
			    result = -1;
			else if (d1 > d2)
			    result = 1;
			else
			    result = 0;
		    }
		}
	    }
	    break;

	case FIELD_TYPE_TIME:
	    {
		UInt8 h1, h2, m1, m2;

		getter1->GetTime(rec1_data, fieldNum, &h1, &m1);
		getter2->GetTime(rec2_data, fieldNum, &h2, &m2);

		if (h1 < h2)
		    result = -1;
		else if (h1 > h2)
		    result = 1;
		else {
		    if (m1 < m2)
			result = -1;
		    else if (m1 > m2)
			result = 1;
		    else
			result = 0;
		}
	    }
	    break;

	default:
	    /* Assume that any types not known to us are equal. */
	    result = 0;
	    break;
	}

	/* If we are going in descending order, invert the result. */
	if (direction == SORT_DIRECTION_DESCENDING) {
	    if (result < 0)
		result = 1;
	    else if (result > 0)
		result = -1;
	}

	/* If we found a difference (nonzero basically), stop the loop. */
	if (result != 0) break;
    }

    return result;
}
