/*
 * Grid Widget Definitions
 * Copyright (C) 2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _GRID_H
#define _GRID_H

#if defined(__GNUC__)
#define GRIDSECT __attribute__ ((__section__("gridsect")))
#else
#define GRIDSECT
#endif


typedef enum {
    gridCellTypeString,
    gridCellTypeBoolean,
} GridCellType;

typedef struct {
    /* Total number of columns in the grid. Some may not be visible. */
    UInt16 (*GetColumnCount)(void * GridModelData);

    /* Store a string containing the column name. */
    void (*GetColumnName)(void * GridModelData, UInt16 col,
			  Char * col_name, UInt32 len);

    /* Width in pixels of a particular column. */
    UInt16 (*GetColumnWidth)(void * GridModelData, UInt16 col);

    /* Set the width in pixels of a particular column. */
    void (*SetColumnWidth)(void * GridModelData, UInt16 col, UInt16 width);

    /* Lock and unlock a particular index in the model. */
    void * (*LockIndex)(void * GridModelData, UInt16 index, Boolean writable);
    void (*UnlockIndex)(void * GridModelData, UInt16 index, Boolean writable,
			void * data);

    /* Retrieve the type of the cell at the given row and column. */
    GridCellType (*GetCellType)(void * GridModelData, UInt16 index, UInt16 col);

    /* Retrieve the contents of the cell as a string. */
    void (*GetCellString)(void * GridModelData, UInt16 index, UInt16 col,
			  Char * data, UInt32 len, void * row_data);

    /* Retrieve the contents of the cell as a boolean. */
    Boolean (*GetCellBoolean)(void * GridModelData, UInt16 index, UInt16 col,
			      void * row_data);

    /* Store new contents in a boolean cell. */
    void (*SetCellBoolean)(void * GridModelData, UInt16 index, UInt16 col,
			   Boolean value, void * row_data);

    /* Retrieve a unqiue ID for a row. */
    void (*GetCellID)(void * GridModelData, UInt16 index, UInt32 * idP);

    /* Seek in the grid. Similar to DmSeekRecordInCategory(). */
    Boolean (*Seek)(void * GridModelData, UInt16 * indexP,
		    Int16 offset, Int16 direction);

    /* Return true if a column is editable. Only queried for editable
     * columns like checkbox columns.
     */
    Boolean (*IsColumnEditable)(void * GridModelData, UInt16 col);

} GridModelType, *GridModelPtr;

typedef struct {
    UInt16  height;         /* Height in pixels of this row on screen. */
    UInt32 modelRowID;     /* Unique ID number for this row in the model. */
    UInt16  modelIndex;     /* Index number for this row in the model. */

    Boolean valid;        /* We have queried the model for data already. */
    Boolean redraw;       /* This row needs to be drawn (again). */
    Boolean highlighted;  /* This rows should be highlighted. */
} GridScreenRow;

typedef struct {
    UInt16 x;             /* X offset of the column. */
    UInt16 width;         /* Visible width of the column. */
    Boolean visible;    /* True if this column should be visible. */
} GridScreenColumn;

typedef struct {
    /* The widget uses a model to gain access to the data. */
    GridModelPtr model;
    void * model_data;

    /* Area of the screen that we take up. */
    RectangleType bounds;
    UInt16 controlID;
    UInt16 ctl_checkbox_id;
    UInt16 ctl_checkbox_index;
    ControlPtr ctl_checkbox;
    FormPtr form;

    /* Offscreen buffer for flicker avoidance. */
    WinHandle win;

    /* First visible column. */
    UInt16 leftColumn, rightColumn;
    UInt16 topmostRow;
    UInt16 lineHeight;

    /* Keep track of which UI components should be used. */
    Boolean showHeader, showHeaderSep, invertHeader, showColumnSep;
    FontID headerFont;

    /* Keep track of UI components that have been drawn. */
    Boolean headerVisible, columnSepVisible;

    /* Pixel widths/heights of various components. */
    UInt16 headerExtent, columnGutterExtent;

    /* Information on the rows visible on the screen. */
    UInt16 numScreenRows;
    GridScreenRow * screenRows;

    /* Information on the rows visible on the screen. */
    UInt16 numCols;
    GridScreenColumn * cols;

    /* Information on indexes that are highlighted. */
    UInt16 numHighlights;
    UInt16 * highlights;
} GridType, *GridPtr;

enum {
    gridSelectEvent = firstUserEvent + 100U,
    gridUpdateVScrollersEvent,
    gridUpdateHScrollersEvent,
};

extern Err GridInit(GridPtr grid, FormPtr form, UInt16 grid_gadget_id,
		    UInt16 grid_checkbox_id,
		    GridModelPtr model, void * model_data) GRIDSECT;

extern void GridDraw(GridPtr grid) GRIDSECT;

extern Boolean GridHandleEvent(GridPtr grid, EventPtr event) GRIDSECT;

extern void GridFree(GridPtr grid) GRIDSECT;

extern void GridScrollVertical(GridPtr grid, WinDirectionType direction,
			       UInt16 increment) GRIDSECT;

extern void GridScrollHorizontal(GridPtr grid, WinDirectionType direction,
				 UInt16 increment) GRIDSECT;

extern void GridSetHeaderLook(GridPtr grid, Boolean showHeader,
			      Boolean showHeaderSep, Boolean invertHeader,
			      FontID headerFont) GRIDSECT;

extern void GridSetColumnLook(GridPtr grid, Boolean showColumnSep) GRIDSECT;

extern UInt16 GridGetNumberOfRows(GridPtr grid) GRIDSECT;

extern void GridHighlightIndex(GridPtr grid, UInt16 index) GRIDSECT;

extern void GridUnhighlightAll(GridPtr grid) GRIDSECT;

extern void GridMakeIndexVisible(GridPtr grid, UInt16 index) GRIDSECT;

#endif
