/*
 * Grid Widget Implementation
 * Copyright (C) 2000-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include <stdarg.h>

#include "db.h"
#include "grid.h"
#include "enum.h"

typedef enum {
    gridPositionNone,
    gridPositionHeaderCell,
    gridPositionHeaderColumnSep,
    gridPositionCell,
    gridPositionColumnSep,
} GridPosition;

static void FillColumnInfo(GridPtr grid) GRIDSECT;
static void InitRowInfo(GridPtr grid) GRIDSECT;
static void InvalidateGrid(GridPtr grid) GRIDSECT;
static void LoadGrid(GridPtr grid) GRIDSECT;
static void GetCellBounds(GridPtr grid, UInt16 screenRow,
			  UInt16 col, RectanglePtr r) GRIDSECT;
static void GetRowBounds(GridPtr grid, UInt16 screenRow,
			 RectanglePtr r) GRIDSECT;
static GridPosition PtToPosition(GridPtr grid, UInt16 * row,
				 UInt16 * col, Int16 x, Int16 y) GRIDSECT;
static void DrawCellBoolean(GridPtr grid, RectanglePtr r, Boolean value)
     GRIDSECT;
static void DrawCell(GridPtr grid, UInt16 row, UInt16 col, void * data)
     GRIDSECT;
static void SelectBoolean(GridPtr grid, UInt16 row, UInt16 col,
			  EventPtr event) GRIDSECT;

static void FillColumnInfo(GridPtr grid)
{
    EventType e;
    UInt16 width;
    Int16 x;
    Boolean rightColumnClipped = false;

    x = 0;
    while (x < grid->bounds.extent.x && grid->rightColumn < grid->numCols) {
	/* Ask the grid model for the desired width of this column. */
	width = grid->model->GetColumnWidth(grid->model_data,
					    grid->rightColumn);

	/* Clip the requested width to the actual space remaining. */
	if (width > grid->bounds.extent.x - x) {
	    width = grid->bounds.extent.x - x;
	    rightColumnClipped = true;
	}

	/* If this is the rightmost column, allocate it all remaining space. */
	if (grid->rightColumn == grid->numCols - 1
	    && width < grid->bounds.extent.x - x)
	    width = grid->bounds.extent.x - x;

	/* Assign the column to this x-offset and allocated width. */
	grid->cols[grid->rightColumn].x = x;
	grid->cols[grid->rightColumn].width = width;

	/* Bump the column number up. */
	grid->rightColumn += 1;

	/* Move the x-offset ahead by the width of the field, both
	 * column sep gutters, and the column sep if necessary.
	 */
	x += width + 2 * grid->columnGutterExtent;
	if (grid->showColumnSep) x += 1;
    }

    /* We incremented one too many times. */
    grid->rightColumn -= 1;

    /* Send an event to the form to enable/disable scroll buttons. */
    e.eType = gridUpdateHScrollersEvent;
    e.data.generic.datum[0] = grid->controlID;
    e.data.generic.datum[1] = (grid->leftColumn > 0);
    e.data.generic.datum[2] = (grid->rightColumn < grid->numCols - 1)
	|| ( rightColumnClipped && (grid->leftColumn < (grid->numCols - 1)) );
    EvtAddEventToQueue(&e);
}

static void InitRowInfo(GridPtr grid)
{
    Int16 height, i;

    /* Free any existing screenRows array. */
    if (grid->screenRows) {
	MemPtrFree(grid->screenRows);
	grid->screenRows = 0;
    }

    height = grid->bounds.extent.y - grid->headerExtent;
    grid->numScreenRows = height / grid->lineHeight;
    grid->screenRows = MemPtrNew(grid->numScreenRows * sizeof(GridScreenRow));
    for (i = 0; i < grid->numScreenRows; ++i) {
	grid->screenRows[i].valid = false;
	grid->screenRows[i].redraw = true;
	grid->screenRows[i].highlighted = false;
	grid->screenRows[i].height = grid->lineHeight;
    }
}

Err GridInit(GridPtr grid, FormPtr form, UInt16 grid_gadget_id,
	     UInt16 grid_checkbox_id,
	     GridModelPtr model, void * model_data)
{
    WinHandle winH;
    UInt16 i;
    FontID oldFont;
    Err err;

    /* Clear out the entire structure. */
    MemSet(grid, sizeof(*grid), 0);

    /* Fill in the model provided by the user. */
    grid->model = model;
    grid->model_data = model_data;
    grid->controlID = grid_gadget_id;
    grid->ctl_checkbox_id = grid_checkbox_id;
    grid->ctl_checkbox_index = FrmGetObjectIndex(form, grid_checkbox_id);
    grid->ctl_checkbox = FrmGetObjectPtr(form, grid->ctl_checkbox_index);
    grid->form = form;

    /* Use the form to pull the bounds out of the gadget. */
    FrmGetObjectBounds(form, FrmGetObjectIndex(form, grid_gadget_id),
		       &(grid->bounds));

    /* Allocate an offscreen window for flicker-free drawing. */
    grid->win = WinCreateOffscreenWindow(grid->bounds.extent.x,
					 grid->bounds.extent.y,
					 genericFormat, &err);

    /* Clear the offscreen window. */
    winH = WinGetDrawWindow();
    WinSetDrawWindow(grid->win);
    WinEraseWindow();
    WinSetDrawWindow(winH);

    /* For now, we use the standard font to draw all rows. */
    oldFont = FntSetFont(stdFont);
    grid->lineHeight = FntLineHeight();
    FntSetFont(oldFont);

    /* Set which UI components will be drawn by default. */
    grid->showHeader = true;
    grid->showHeaderSep = true;
    grid->invertHeader = false;
    grid->showColumnSep = true;
    grid->headerFont = stdFont;

    /* Nothing has been drawn yet. */
    grid->headerVisible = false;
    grid->columnSepVisible = false;

    /* Initialize various pixel widths. */
    oldFont = FntSetFont(boldFont);
    grid->headerExtent = FntLineHeight() + 1;
    FntSetFont(oldFont);
    grid->columnGutterExtent = 1;

    /* Initialize the row information. */
    InitRowInfo(grid);

    /* Fill in the column information. */
    grid->numCols = grid->model->GetColumnCount(grid->model_data);
    grid->cols = MemPtrNew(grid->numCols * sizeof(GridScreenColumn));
    for (i = 0; i < grid->numCols; ++i) {
	grid->cols[i].width = grid->model->GetColumnWidth(grid->model_data, i);
    }

    /* Now determine what columns are visible and their offsets. */
    grid->leftColumn = grid->rightColumn = 0;
    FillColumnInfo(grid);

    /* Fill in the remainder of the fields. */
    grid->topmostRow = 0;

    return 0;
}

void GridFree(GridPtr grid)
{
    MemPtrFree(grid->screenRows);
    MemPtrFree(grid->cols);
    if (grid->highlights)
	MemPtrFree(grid->highlights);
    WinDeleteWindow(grid->win, false);
}

/* Invalidate the entire grid and redraw it. */
static void InvalidateGrid(GridPtr grid)
{
    UInt16 i;
    WinHandle winH;

    /* Erase the entire offscreen window. */
    winH = WinGetDrawWindow();
    WinSetDrawWindow(grid->win);
    WinEraseWindow();
    WinSetDrawWindow(winH);

    /* Set all state variables to the undrawn position. */
    grid->headerVisible = false;
    grid->columnSepVisible = false;
    for (i = 0; i < grid->numScreenRows; ++i) {
	grid->screenRows[i].valid = false;
	grid->screenRows[i].redraw = true;
    }
}

/* Figure out what rows to display in the grid. */
static void LoadGrid(GridPtr grid)
{
    UInt16 row, index, j;
    UInt32 uniqueID;
    Boolean scrollUp, scrollDown;
    EventType e;

    /* Adjust the top visible row so it is valid. */
    grid->model->Seek(grid->model_data, &(grid->topmostRow), 0, dmSeekForward);

    /* Make sure that the entire display area is used. */
    index = grid->topmostRow;
    if (! grid->model->Seek(grid->model_data, &index,
			    grid->numScreenRows - 1, dmSeekForward)) {
        grid->topmostRow = dmMaxRecordIndex;
        if (! grid->model->Seek(grid->model_data, &(grid->topmostRow),
				grid->numScreenRows - 1, dmSeekBackward)) {
            /* There are not enough records to fill one page so make
             * the first row visible.
	     */
            grid->topmostRow = 0;
            grid->model->Seek(grid->model_data, &(grid->topmostRow),
			      0, dmSeekForward);
        }
    }

    index = grid->topmostRow;
    for (row = 0; row < grid->numScreenRows; ++row) {
	/* Stop the loop if no more rows are available. */
	if (! grid->model->Seek(grid->model_data, &index, 0, dmSeekForward))
	    break;

	/* Retrieve the unique ID for this row from the model. O(1) */
	grid->model->GetCellID(grid->model_data, index, &uniqueID);

	/* Determine if we need to redraw this row. */
	if (! grid->screenRows[row].valid
	    || grid->screenRows[row].modelRowID != uniqueID) {
	    grid->screenRows[row].valid = true;
	    grid->screenRows[row].redraw = true;
	    grid->screenRows[row].modelRowID = uniqueID;
	    grid->screenRows[row].modelIndex = index;
	    grid->screenRows[row].height = grid->lineHeight;

	    /* Determine if this row should be highlighted. */
	    grid->screenRows[row].highlighted = false;
	    for (j = 0; j < grid->numHighlights; ++j)
		if (grid->highlights[j] == index)
		    grid->screenRows[row].highlighted = true;
	}

	/* Increment the row index. */
	++index;
    }

    /* Mark the remaining rows as invalid. */
    while (row < grid->numScreenRows) {
	grid->screenRows[row].valid = false;
	grid->screenRows[row].redraw = true;
	row++;
    }

    /* Inform the scrolling callback if we can scroll or not. */
    scrollUp = false;
    if (grid->screenRows[0].valid) {
	index = grid->screenRows[0].modelIndex;
	if (grid->model->Seek(grid->model_data, &index, 1, dmSeekBackward))
	    scrollUp = true;
    }
    scrollDown = false;
    if (grid->screenRows[grid->numScreenRows - 1].valid) {
	index = grid->screenRows[grid->numScreenRows - 1].modelIndex;
	if (grid->model->Seek(grid->model_data, &index, 1, dmSeekForward))
	    scrollDown = true;
    }

    /* Send a notification event about the vertical scrollers. */
    e.eType = gridUpdateVScrollersEvent;
    e.data.generic.datum[0] = grid->controlID;
    e.data.generic.datum[1] = scrollUp;
    e.data.generic.datum[2] = scrollDown;
    EvtAddEventToQueue(&e);
}

static void
GetCellBounds(GridPtr grid, UInt16 screenRow, UInt16 col, RectanglePtr r)
{
    UInt16 i;

    /* Determine the top of the cell. */
    r->extent.y = grid->screenRows[screenRow].height;
    r->topLeft.y = grid->headerExtent;
    for (i = 0; screenRow > 0; ++i, --screenRow) {
	r->topLeft.y += grid->screenRows[i].height;
    }

    /* Determine the left of the cell. */
    r->topLeft.x = grid->cols[col].x;
    r->extent.x = grid->cols[col].width;
}

static void
GetRowBounds(GridPtr grid, UInt16 screenRow, RectanglePtr r)
{
    UInt16 i;

    /* Determine the top of the cell. */
    r->extent.y = grid->screenRows[screenRow].height;
    r->topLeft.y = grid->headerExtent;
    for (i = 0; screenRow > 0; ++i, --screenRow) {
	r->topLeft.y += grid->screenRows[i].height;
    }

    r->topLeft.x = 0;
    r->extent.x = grid->bounds.extent.x;
}

static GridPosition
PtToPosition(GridPtr grid, UInt16 * row, UInt16 * col, Int16 x, Int16 y)
{
    /* Normalize the point into local coordinates. */
    x -= grid->bounds.topLeft.x;
    y -= grid->bounds.topLeft.y;

    /* Determine if the header row was hit. */
    if (grid->showHeader) {
	if (y < grid->headerExtent) {
	    *col = grid->leftColumn;
	    while (x >= grid->cols[*col].width + 2*grid->columnGutterExtent
		   + ((grid->showColumnSep) ? 1 : 0)) {
		x -= grid->cols[*col].width - 2*grid->columnGutterExtent
		    - ((grid->showColumnSep) ? 1 : 0);
		*col += 1;
	    }

	    if (x < grid->cols[*col].width)
		return gridPositionHeaderCell;
	    else if (x >= grid->cols[*col].width
		     && x < (grid->cols[*col].width
			     + 2*grid->columnGutterExtent
			     + ((grid->showColumnSep) ? 1 : 0)))
		return gridPositionHeaderColumnSep;
	}

	/* Remove the header row from the y coordinate. */
	y -= grid->headerExtent;
    }

    /* Determine the row where the point lies. */
    *row = 0;
    while (*row < grid->numScreenRows && y >= grid->screenRows[*row].height) {
	y -= grid->screenRows[*row].height;
	*row += 1;
    }

    /* Determine the column where the point lies. */
    *col = grid->leftColumn;
    while (x >= grid->cols[*col].width + 2*grid->columnGutterExtent
	   + ((grid->showColumnSep) ? 1 : 0)) {
	x -= grid->cols[*col].width - 2*grid->columnGutterExtent
	    - ((grid->showColumnSep) ? 1 : 0);
	*col += 1;
    }

    if (x < grid->cols[*col].width)
	return gridPositionCell;
    else if (x >= grid->cols[*col].width
	     && x < (grid->cols[*col].width + 2*grid->columnGutterExtent
		     + ((grid->showColumnSep) ? 1 : 0)))
	return gridPositionColumnSep;
    else
	return gridPositionNone;
}

static void DrawCellBoolean(GridPtr grid, RectanglePtr r, Boolean value)
{
    /* Clear out any extra stuff from the cell. */
    WinEraseRectangle(r, 0);

    /* Setup the checkbox structure for this cell. */
    CtlSetEnabled(grid->ctl_checkbox, 1);
    CtlSetUsable(grid->ctl_checkbox, 1);
    FrmSetObjectBounds(grid->form, grid->ctl_checkbox_index, r);
    CtlSetValue(grid->ctl_checkbox, value);

    /* Now instruct the checkbox to draw itself. */
    CtlDrawControl(grid->ctl_checkbox);

    /* Make the sure the checkbox is not usable again. */
    CtlSetEnabled(grid->ctl_checkbox, 0);
    CtlSetUsable(grid->ctl_checkbox, 0);
}

static void DrawCell(GridPtr grid, UInt16 row, UInt16 col, void * data)
{
    RectangleType r;
    Char buf[128];
    Boolean b;

    /* Retrieve the location of this cell. */
    GetCellBounds(grid, row, col, &r);

    switch (grid->model->GetCellType(grid->model_data,
				     grid->screenRows[row].modelIndex, col)) {
    case gridCellTypeString:
	grid->model->GetCellString(grid->model_data,
				   grid->screenRows[row].modelIndex,
				   col, buf, sizeof(buf), data);
	buf[sizeof(buf)-1] = '\0';

	WinEraseRectangle(&r, 0);
	MyWinDrawTruncChars(buf, StrLen(buf), r.topLeft.x, r.topLeft.y,
			    r.extent.x);
	break;

    case gridCellTypeBoolean:
	b = grid->model->GetCellBoolean(grid->model_data,
					grid->screenRows[row].modelIndex,
					col, data);
	DrawCellBoolean(grid, &r, b);
	break;
    }
}

void GridDraw(GridPtr grid)
{
    UInt16 col, row;
    RectangleType r;
    Char buf[128];
    WinHandle winH;
    FontID oldFont;
    void * row_data;

    LoadGrid(grid);

    /* Switch the draw window to the offscreen window. */
    winH = WinGetDrawWindow();
    WinSetDrawWindow(grid->win);

    /* Draw the column headers and column markers. */
    if (grid->showHeader && ! grid->headerVisible) {
	/* Mark the header as visible. */
	grid->headerVisible = true;

	/* Draw the column names. */
	for (col = grid->leftColumn; col <= grid->rightColumn; ++col) {
	    grid->model->GetColumnName(grid->model_data, col, buf, sizeof buf);
	    buf[sizeof(buf)-1] = '\0';
	    oldFont = FntSetFont(grid->headerFont);
	    WinDrawChars(buf, StrLen(buf), grid->cols[col].x, 0);
	    FntSetFont(oldFont);
	}

	/* Invert the header row if necessry. */
	if (grid->invertHeader) {
	    r.topLeft.x = 0;
	    r.topLeft.y = 0;
	    r.extent.x = grid->bounds.extent.x;
	    r.extent.y = grid->headerExtent - 1;
	    WinInvertRectangle(&r, 0);
	}

	/* Draw the seperator between the header and data rows. */
	if (grid->showHeaderSep)
	    WinDrawLine(0, grid->headerExtent - 1,
			grid->bounds.extent.x, grid->headerExtent - 1);
    }

    /* Draw the vertical cell seperator lines. */
    if (grid->showColumnSep && ! grid->columnSepVisible) {
	for (col = grid->leftColumn + 1; col <= grid->rightColumn; ++col) {
	    WinDrawLine(grid->cols[col].x - grid->columnGutterExtent - 1,
			0,
			grid->cols[col].x - grid->columnGutterExtent - 1,
			grid->bounds.extent.y);
	}
	grid->columnSepVisible = true;
    }

    /* Scan through each row and see what cells need to be redrawn. */
    for (row = 0; row < grid->numScreenRows; ++row) {
	if (grid->screenRows[row].valid) {
	    if (grid->screenRows[row].redraw) {
		row_data = grid->model->LockIndex(grid->model_data,
						  grid->screenRows[row].modelIndex,
						  false);
		for (col = grid->leftColumn; col <= grid->rightColumn; ++col)
		    DrawCell(grid, row, col, row_data);
		grid->model->UnlockIndex(grid->model_data,
					 grid->screenRows[row].modelIndex,
					 false, row_data);
		grid->screenRows[row].redraw = false;

		/* Highlight the row if it is called for. */
		if (grid->screenRows[row].highlighted) {
		    GetRowBounds(grid, row, &r);
		    WinInvertRectangle(&r, 0);
		}
	    }
	} else {
	    if (grid->screenRows[row].redraw) {
		for (col = grid->leftColumn; col <= grid->rightColumn; ++col) {
		    GetCellBounds(grid, row, col, &r);
		    WinEraseRectangle(&r, 0);
		}
		grid->screenRows[row].redraw = false;
	    }
	}
    }

    /* Reset back to the original draw window. */
    WinSetDrawWindow(winH);

    /* Copy the offscreen window to the actual screen. */
    r.topLeft.x = 0;
    r.topLeft.y = 0;
    r.extent.x = grid->bounds.extent.x;
    r.extent.y = grid->bounds.extent.y;
    WinCopyRectangle(grid->win, (WindowType *) FrmGetActiveForm(), &r,
		     grid->bounds.topLeft.x, grid->bounds.topLeft.y,
		     winPaint);
}

static void SelectBoolean(GridPtr grid, UInt16 row, UInt16 col, EventPtr event)
{
    RectangleType screen_rect, offscreen_rect;
    Boolean value, orig_value;
    WinHandle winH;
    void * row_data;
    EventType new_event;

    /* Determine the location of this cell on-screen and off-screen. */
    GetCellBounds(grid, row, col, &offscreen_rect);
    RctCopyRectangle(&offscreen_rect, &screen_rect);
    screen_rect.topLeft.x += grid->bounds.topLeft.x;
    screen_rect.topLeft.y += grid->bounds.topLeft.y;

    /* Lock down this index in the model. */
    row_data = grid->model->LockIndex(grid->model_data,
				      grid->screenRows[row].modelIndex,
				      true);

    /* Query the model for the current value of the cell. */
    value = grid->model->GetCellBoolean(grid->model_data,
					grid->screenRows[row].modelIndex,
					col, row_data);
    orig_value = value;

    /* Setup the checkbox structure for this cell. */
    CtlSetEnabled(grid->ctl_checkbox, 1);
    CtlSetUsable(grid->ctl_checkbox, 1);
    FrmSetObjectBounds(grid->form, grid->ctl_checkbox_index, &screen_rect);
    CtlSetValue(grid->ctl_checkbox, value);
    CtlDrawControl(grid->ctl_checkbox);

    /* Prepare a ctlEnterEvent for the checkbox. */
    EvtCopyEvent(event, &new_event);
    new_event.eType = ctlEnterEvent;
    new_event.data.ctlEnter.controlID = grid->ctl_checkbox_id;
    new_event.data.ctlEnter.pControl = grid->ctl_checkbox;

    while (true) {
	if (new_event.eType == ctlEnterEvent) {
	    CtlHandleEvent(grid->ctl_checkbox, &new_event);
	} else if (new_event.eType == ctlSelectEvent) {
	    value = CtlGetValue(grid->ctl_checkbox);
	    /* SendTableSelectEvent (table, row, column); */
	} else if (new_event.eType == ctlExitEvent) {
	    /* Post a tableExitEvent. */
	    /* SendTableExitEvent (table, row, column); */
	} else {
	    /* If we got an event that we didn't expect, put it back
	     * in the queue.
	     */
	    EvtAddEventToQueue(&new_event);
	    break;
	}
                                                
	EvtGetEvent(&new_event, evtWaitForever);
    }

    /* Disable the checkbox so it not picked up by the form. */
    CtlSetEnabled(grid->ctl_checkbox, 0);
    CtlSetUsable(grid->ctl_checkbox, 0);

    /* If the user completed the selection, then inform the model. */
    if (value != orig_value) {
	grid->model->SetCellBoolean(grid->model_data,
				    grid->screenRows[row].modelIndex,
				    col, value, row_data);

	winH = WinGetDrawWindow();
	WinSetDrawWindow(grid->win);
	DrawCellBoolean(grid, &offscreen_rect, value);
	WinSetDrawWindow(winH);
    }

    /* Unlock this index in the model. */
    grid->model->UnlockIndex(grid->model_data,
			     grid->screenRows[row].modelIndex,
			     true, row_data);
}

Boolean GridHandleEvent(GridPtr grid, EventPtr event)
{
    UInt16 row, col;
    Int16 x, y;
    Boolean penDown;
    Boolean selected;
    RectangleType r;
    EventType e;

    switch (event->eType) {
    case penDownEvent:
	if (! RctPtInRectangle(event->screenX, event->screenY, &grid->bounds))
	    return false;

	switch (PtToPosition(grid, &row, &col,
			     event->screenX, event->screenY)) {
	case gridPositionNone:
	case gridPositionHeaderCell:
	case gridPositionHeaderColumnSep:
	    break;

	case gridPositionColumnSep:
	    break;

	case gridPositionCell:
	    if (! grid->screenRows[row].valid)
		break;

	    switch (grid->model->GetCellType(grid->model_data,
					     grid->screenRows[row].modelIndex,
					     col)) {
	    case gridCellTypeBoolean:
		if (grid->model->IsColumnEditable(grid->model_data, col)) {
		    SelectBoolean(grid, row, col, event);
		    break;
		}
		/* fall-through if column is not editable */

	    default:
		GetRowBounds(grid, row, &r);
		r.topLeft.x += grid->bounds.topLeft.x;
		r.topLeft.y += grid->bounds.topLeft.y;

		WinInvertRectangle(&r, 0);
		selected = true;

		while (true) {
		    PenGetPoint(&x, &y, &penDown);
		    if (!penDown)
			break;

		    if (RctPtInRectangle(x, y, &r)) {
			if (! selected) {
			    WinInvertRectangle(&r, 0);
			    selected = true;
			}			
		    } else {
			if (selected) {
			    WinInvertRectangle(&r, 0);
			    selected = false;
			}
		    }
		}
		
		/* Uninvert the selection if the pen is still over it. */
		if (selected) {
		    WinInvertRectangle(&r, 0);
		    e.eType = gridSelectEvent;
		    e.data.tblSelect.tableID = grid->controlID;
		    e.data.tblSelect.pTable = (TableType *) grid;
		    e.data.tblSelect.row = grid->screenRows[row].modelIndex;
		    e.data.tblSelect.column = col;
		    EvtAddEventToQueue(&e);
		}
	    }
	    break;

	}
	return true;

    default:
	break;

    }

    return false;
}

void GridScrollVertical(GridPtr grid, WinDirectionType direction,
			UInt16 increment)
{
    int i;
    UInt16 newTopmostRow, distance;
    RectangleType r, dummy;
    WinHandle winH;

    /* If the increment is greater than or equal to the number of
     * visible rows, then it is simpler to invalidate every row and
     * redraw everthing. GridDraw() outputs everything to the screen
     * at once so there will be no flicker.
     */

    if (increment >= grid->numScreenRows) {
	if (direction == winDown) {
	    grid->model->Seek(grid->model_data, &grid->topmostRow,
			      increment, dmSeekForward);
	} else {
	    if (! grid->model->Seek(grid->model_data, &grid->topmostRow,
				    increment, dmSeekBackward))
		grid->topmostRow = 0;
	}

	/* Invalidate all of the rows onscreen. */
	for (i = 0; i < grid->numScreenRows; ++i)
	    grid->screenRows[i].valid = false;

	/* Redraw the screen. */
	GridDraw(grid);

	return;
    }

    /* Otherwise, we will move the rows that will stay on screen up
     * the offscreen buffer and then force a redraw on the new rows
     * that have been opened up.
     */

    /* Move the rows within the screenRows array to new positions. */
    if (direction == winDown) {
	for (i = increment; i < grid->numScreenRows; ++i) {
	    grid->screenRows[i - increment] = grid->screenRows[i];
	}
	for (i = grid->numScreenRows-increment; i < grid->numScreenRows; ++i) {
	    grid->screenRows[i].valid = false;
	    grid->screenRows[i].redraw = true;
	}
    } else {
	for (i = grid->numScreenRows - increment - 1; i >= 0; --i) {
	    grid->screenRows[i + increment] = grid->screenRows[i];
	}
	for (i = 0; i < increment; ++i) {
	    grid->screenRows[i].valid = false;
	    grid->screenRows[i].redraw = true;
	}
    }

    /* Switch to the offscreen window. */
    winH = WinGetDrawWindow();
    WinSetDrawWindow(grid->win);

    /* Move the rows within the offscreen window. */
    r.topLeft.x = 0;
    r.topLeft.y = grid->headerExtent;
    r.extent.x = grid->bounds.extent.x;
    r.extent.y = grid->bounds.extent.y - grid->headerExtent;
    distance = 0;
    if (direction == winDown) {
	for (i = 0; i < increment; ++i) distance += grid->screenRows[i].height;
	WinScrollRectangle(&r, winUp, distance, &dummy);
    } else {
	for (i = grid->numScreenRows - increment; i < grid->numScreenRows; ++i)
	    distance += grid->screenRows[i].height;
	WinScrollRectangle(&r, winDown, distance, &dummy);
    }
    WinEraseRectangle(&dummy, 0);

    /* Draw the column seperators in the vacated area. */
    if (grid->showColumnSep) {
        for (i = grid->leftColumn + 1; i <= grid->rightColumn; ++i) {
            WinDrawLine(grid->cols[i].x - grid->columnGutterExtent - 1,
			dummy.topLeft.y,
                        grid->cols[i].x - grid->columnGutterExtent - 1,
                        dummy.topLeft.y + dummy.extent.y);
        }
    }

    /* Switch back to the original draw window. */
    WinSetDrawWindow(winH);

    /* Determine what the new topmost index will be. */
    newTopmostRow = grid->topmostRow;
    if (direction == winDown) {
	grid->model->Seek(grid->model_data, &newTopmostRow,
			  increment, dmSeekForward);
    } else {
	if (! grid->model->Seek(grid->model_data, &newTopmostRow,
				increment, dmSeekBackward))
	    newTopmostRow = 0;
    }
    grid->topmostRow = newTopmostRow;

    /* Draw the rows that have been displaced. */
    GridDraw(grid);
}

void GridScrollHorizontal(GridPtr grid, WinDirectionType direction,
			  UInt16 increment)
{
    if (direction == winRight) {
	if (! (grid->leftColumn + increment >= grid->numCols))
	    grid->leftColumn += increment;
    } else {
	if (increment > grid->leftColumn)
	    grid->leftColumn = 0;
	else
	    grid->leftColumn -= increment;
    }

    grid->rightColumn = grid->leftColumn;
    FillColumnInfo(grid);
    InvalidateGrid(grid);
}

void GridSetHeaderLook(GridPtr grid, Boolean showHeader,
		       Boolean showHeaderSep, Boolean invertHeader,
		       FontID headerFont)
{
    FontID oldFont;

    grid->showHeader = showHeader;
    grid->showHeaderSep = showHeaderSep;
    grid->invertHeader = invertHeader;
    grid->headerFont = headerFont;

    grid->headerExtent = 0;
    if (grid->showHeader) {
	oldFont = FntSetFont(grid->headerFont);
	grid->headerExtent += FntLineHeight();
	FntSetFont(oldFont);

	if (grid->showHeaderSep)
	    grid->headerExtent += 1;
    }

    InitRowInfo(grid);
    InvalidateGrid(grid);
}

void GridSetColumnLook(GridPtr grid, Boolean showColumnSep)
{
    grid->showColumnSep = showColumnSep;

    if (grid->showColumnSep) {
	grid->columnGutterExtent = 1;
    } else {
	grid->columnGutterExtent = 0;
    }

    grid->rightColumn = grid->leftColumn;
    FillColumnInfo(grid);
    InvalidateGrid(grid);
}

UInt16 GridGetNumberOfRows(GridPtr grid)
{
    return grid->numScreenRows;
}

void GridHighlightIndex(GridPtr grid, UInt16 index)
{
    UInt16 i, * new_highlights;
    
    /* Ensure this index is not already in the list of highlighted rows. */
    for (i = 0; i < grid->numHighlights; ++i) {
	if (grid->highlights[i] == index) return;
    }

    /* Add this index to the list of highlighted indexes. */
    new_highlights = MemPtrNew((grid->numHighlights + 1) * sizeof(UInt16));
    if (grid->highlights) {
	MemMove(new_highlights, grid->highlights,
		grid->numHighlights * sizeof(UInt16));
	MemPtrFree(grid->highlights);
	grid->highlights = 0;
    }
    grid->highlights = new_highlights;
    grid->highlights[grid->numHighlights] = index;
    ++(grid->numHighlights);

    /* Now scan the list of visible indexes and force a redraw for the
     * index if it is present.
     */
    for (i = 0; i < grid->numScreenRows; ++i) {
	if (grid->screenRows[i].valid)
	    if (grid->screenRows[i].modelIndex == index) {
		grid->screenRows[i].redraw = true;
		grid->screenRows[i].highlighted = true;
	    }
    }
}

void GridUnhighlightAll(GridPtr grid)
{
    UInt16 i;

    /* Force a redraw for any rows that are highlighted. */
    for (i = 0; i < grid->numScreenRows; ++i) {
	if (grid->screenRows[i].valid && grid->screenRows[i].highlighted) {
	    grid->screenRows[i].redraw = true;
	    grid->screenRows[i].highlighted = false;
	}
    }

    /* Destroy the list of highlighted rows. */
    if (grid->highlights) {
	MemPtrFree(grid->highlights);
	grid->highlights = 0;
	grid->numHighlights = 0;
    }
}

void GridMakeIndexVisible(GridPtr grid, UInt16 index)
{
    UInt16 row;

    /* Check to see if the row is already visible. If so, return. */
    for (row = 0; row < grid->numScreenRows; ++row) {
	if (grid->screenRows[row].valid
	    && grid->screenRows[row].modelIndex == index)
	    return;
    }

    /* Shift to the requested index and invalidate all rows. */
    grid->topmostRow = index;
    for (row = 0; row < grid->numScreenRows; ++row) {
	grid->screenRows[row].valid = false;
	grid->screenRows[row].redraw = false;
    }
}
