/*
 * DB: Database Design Form
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * The design form is activated from two different contexts in the
 * program. The first context is when creating a new data source. In
 * this case. the following hold:
 *
 *   CurrentSource == NULL
 *   DesignViewDriver points at the data source driver to use
 *   DesignViewInitialSchema:
 *     NULL for blank schema 
 *     points at schema to fill the form with (freed when form closes)
 *   DesignViewName points at the name to use (freed when form closes)
 *
 * The second context is when an existing data source's schema is
 * being viewed or modified. In this case, the following hold:
 *
 *   CurrentSource points at the open data source
 *   DesignViewDriver == CurrentSource->driver
 *   DesignViewInitialSchema == NULL (ignored by code)
 *   DesignViewName == NULL (ignored by code)
 *
 * The difference between the modes can be determined based on the
 * state of the CurrentSource variable.
 */

#include <PalmOS.h>

#include "enum.h"
#include "db.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define col_popup   0
#define col_oldname 1
#define col_name    2
#define col_type    3

typedef struct
{
    MemHandle name;            /* Handle used for this field's name. */
    UInt32 uniqueID;           /* Unique ID number for changed/new fields. */
    DataSourceFieldType type;  /* Type of data for this field. */
    Char old_name[32];         /* Old name of this field (for the label). */
} SchemaDesignField;
    
static UInt16 FirstField;
static MemHandle SchemaFieldsHandle = 0;
static SchemaDesignField * SchemaFields = 0;
static UInt16 numFields = 0;
static UInt32 NextUniqueID = 0;
static char ** popup_strings = 0;
static UInt16 popup_strings_count = 0;

static struct types_t {
    DataSourceFieldType type;
    UInt16 nameID;
    Int16 index;
} types[] = {
    { FIELD_TYPE_STRING,  stringID_String,  -1 },
    { FIELD_TYPE_BOOLEAN, stringID_Boolean, -1 },
    { FIELD_TYPE_INTEGER, stringID_Integer, -1 },
    { FIELD_TYPE_DATE,    stringID_Date,    -1 },
    { FIELD_TYPE_TIME,    stringID_Time,    -1 }
};

/* Global variables to allow other portions of the DB code to tell
 * this form if a blank schema should be used or if there is some
 * initial schema that should be used to fill the display.
 */

DataSourceSchema *DesignViewInitialSchema = 0;
DataSourceDriver *DesignViewDriver = 0;
char *DesignViewName = 0;

static void DrawPopupIndicator(void *, Int16, Int16, RectangleType *) SECT_UI;
static void UpdateScrollers(void) SECT_UI;
static Err LoadData(void *, Int16, Int16, Boolean, MemHandle *, Int16 *,
		    Int16 *, FieldPtr) SECT_UI;
static Boolean SaveData(void *, Int16, Int16) SECT_UI;
static void LoadTable(TablePtr) SECT_UI;
static void SetColumnWidths(TablePtr) SECT_UI;
static void InitTable(TablePtr) SECT_UI;
static void Scroll(WinDirectionType) SECT_UI;
static Boolean UpdateNewDesign(void) SECT_UI;
static Boolean UpdateExistingDesign(void) SECT_UI;
static void InsertField(UInt16) SECT_UI;
static void DeleteField(UInt16) SECT_UI;
static void HandleContextMenu(FormType *, TableType *, Int16) SECT_UI;


static void
DrawPopupIndicator(void * table, Int16 row, Int16 col, RectangleType *b)
{
    Coord x, y, width;
 
    width = 8;
    x = b->topLeft.x + 1;
    y = b->topLeft.y + (TblGetRowHeight(table, row) - (width / 2)) / 2;

    while (width > 0) {
        WinDrawLine(x, y, x + width - 1, y);
        x += 1;
        y += 1;
        width -= 2;
    }
} 

static void
UpdateScrollers(void)
{
    UInt16 upIndex, downIndex, fieldNum;
    Boolean scrollUp, scrollDown;
    Int16 row;
    FormPtr form;
    TablePtr table;

    scrollUp = FirstField != 0 && numFields > 0;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_DesignView_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1 && row != 0) {
	fieldNum = TblGetRowID(table, row);
	scrollDown = fieldNum < numFields - 1;
    } else {
	scrollDown = false;
    }

    upIndex = FrmGetObjectIndex(form, ctlID_DesignView_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_DesignView_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static Err
LoadData(void * table, Int16 row, Int16 column,
	 Boolean editing, MemHandle * textH, Int16 * textOffset,
	 Int16 * textAllocSize, FieldPtr fld)
{
    UInt16 fieldNum = TblGetRowID(table, row);

    *textH = SchemaFields[fieldNum].name;
    *textOffset = 0;
    *textAllocSize = MemHandleSize(SchemaFields[fieldNum].name);

    if (fld) {
	FieldAttrType attr;

	FldGetAttributes(fld, &attr);
	attr.singleLine = 1;
	attr.dynamicSize = 0;
	attr.underlined = 1;
	FldSetAttributes(fld, &attr);
	
	FldSetMaxChars(fld, 31);
    }

    return 0;
}           

static Boolean
SaveData(void * table, Int16 row, Int16 col)
{
    return false;
}

static void
LoadTable(TablePtr table)
{
    UInt16 numRows, row, fieldNum, i;
    ListPtr list;
    FormPtr form;

    form = FrmGetActiveForm();
    list = GetObjectPtr(form, ctlID_DesignView_PopupList);

    numRows = TblGetNumberOfRows(table);
    fieldNum = FirstField;
    for (row = 0; row < numRows; row++) {
	/* Stop the loop if no more fields are available. */
	if (fieldNum >= numFields)
	    break;

	/* Update this row if anything has changed. */
	if (! TblRowUsable(table, row)
	    || TblGetRowData(table, row) != SchemaFields[fieldNum].uniqueID) {
	    /* Mark the row as usable. */
	    TblSetRowUsable(table, row, true);
	    TblSetRowHeight(table, row, FntLineHeight());
	    TblSetRowID(table, row, fieldNum);
	    TblSetRowData(table, row, SchemaFields[fieldNum].uniqueID);
	    TblSetRowSelectable(table, row, true);
	    TblMarkRowInvalid(table, row);

	    /* popup indicator */
	    TblSetItemStyle(table, row, col_popup, customTableItem);
	    
	    /* old name */
	    TblSetItemStyle(table, row, col_oldname, labelTableItem);
	    TblSetItemPtr(table, row, col_oldname,
			  SchemaFields[fieldNum].old_name);

	    /* current name */
	    TblSetItemStyle(table, row, col_name, textTableItem);

	    /* field type */
	    TblSetItemStyle(table, row, col_type, popupTriggerTableItem);
	    TblSetItemPtr(table, row, col_type, list);
	    for (i = 0; i < (sizeof(types) / sizeof(struct types_t)); ++i) {
		if (types[i].type == SchemaFields[fieldNum].type) {
		    TblSetItemInt(table, row, col_type, types[i].index);
		    break;
		}
	    }
	}

	/* Advance the field number. */
	fieldNum++;
    }

    /* Mark the remainder of the rows as unusable. */
    while (row < numRows) {
        TblSetRowUsable(table, row, false);
        row++;
    }

    /* Update the scroll arrows. */
    UpdateScrollers();
}

static void
SetColumnWidths(TableType* table)
{
    Coord oldname_width, fieldtype_width;
    Int16 i, w;
    FontID oldFont;
    RectangleType bounds;
    FormPtr form;
    ListPtr lst;
    UInt16 lstIndex;

    /* Determine and set the width needed for the "old name" column. */
    oldFont = FntSetFont(stdFont);
    oldname_width = 0;
    for (i = 0; i < numFields; i++) {
	w = FntCharsWidth(SchemaFields[i].old_name,
			  StrLen(SchemaFields[i].old_name))
	    + FntCharsWidth(":", 1);
	if (w > oldname_width)
	    oldname_width = w;
    }
    FntSetFont(oldFont);

    /* Determine and set the width of the field type column. */
    fieldtype_width = 0;
    for (i = 0; i < popup_strings_count; ++i) {
	w = FntCharsWidth(popup_strings[i], StrLen(popup_strings[i]));
	if (w > fieldtype_width)
	    fieldtype_width = w;
    }

    /* The width of the field entry column is the remaining width. */
    TblGetBounds(table, &bounds);
    TblSetColumnWidth(table, col_popup, 10);
    TblSetColumnWidth(table, col_oldname, oldname_width);
    TblSetColumnWidth(table, col_type, fieldtype_width + 15);
    TblSetColumnWidth(table, col_name, bounds.extent.x - 10 - oldname_width
		      - (fieldtype_width + 15));

    form = FrmGetActiveForm();
    lstIndex = FrmGetObjectIndex(form, ctlID_DesignView_PopupList);
    lst = FrmGetObjectPtr(form, lstIndex);

    LstSetListChoices(lst, popup_strings, popup_strings_count);
    LstSetHeight(lst, popup_strings_count);

    FrmGetObjectBounds(form, lstIndex, &bounds);
    bounds.extent.x = fieldtype_width + 10;
    FrmSetObjectBounds(form, lstIndex, &bounds);
}

static void
InitTable(TablePtr table)
{
    UInt16 row, numRows;

    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++) {
	TblSetRowUsable(table, row, false);
    }

    TblSetColumnUsable(table, col_popup, true);
    TblSetColumnUsable(table, col_oldname, true);
    TblSetColumnUsable(table, col_name, true);
    TblSetColumnUsable(table, col_type, true);

    TblSetCustomDrawProcedure(table, col_popup, DrawPopupIndicator);
    TblSetLoadDataProcedure(table, col_name, LoadData);
    TblSetSaveDataProcedure(table, col_name, SaveData);

    SetColumnWidths(table);

    LoadTable(table);
}

static void
Scroll(WinDirectionType direction)
{
    UInt16 newFirstField;

    newFirstField = FirstField;

    if (direction == winDown) {
	newFirstField++;
    } else {
	newFirstField--;
    }

    if (FirstField != newFirstField) {
	FormPtr form;
	TablePtr table;

	FirstField = newFirstField;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	LoadTable(table);
	TblRedrawTable(table);
    }
}

static Boolean
UpdateNewDesign(void)
{
    DataSourceSchema schema;
    int i;
    void * key_data;
    UInt16 key_size;
    Err err;

    /* Determine the number of fields by finding the last nonempty field. */
    if (! DesignViewInitialSchema) {
	/* Blank schema, so search for number of fields to use. */
	for (i = numFields - 1; i >= 0; i--) {
	    char * p = MemHandleLock(SchemaFields[i].name);
	    Int16 len = StrLen(p);
	    MemPtrUnlock(p);

	    if (len > 0) break;
	}
	schema.numFields = i + 1;
    } else {
	/* Had an initial schema so just use the current number of fields. */
	schema.numFields = numFields;
    }

    /* Flag an error if no fields were defined. */
    if (schema.numFields == 0) {
	FrmAlert(alertID_NoFields);
	return false;
    }

    /* Build a schema from the data. */
    schema.fields = MemPtrNew(schema.numFields * sizeof(DataSourceFieldInfo));
    for (i = 0; i < schema.numFields; ++i) {
	schema.fields[i].name = MemHandleLock(SchemaFields[i].name);
	schema.fields[i].type = SchemaFields[i].type;
    }

    /* Create the database. Creation routine will display any errors. */
    err = DesignViewDriver->class_ops.Create(DesignViewName, &schema,
					     &key_data, &key_size);
    if (err == 0) {
	Chooser_AddDatabase(DesignViewName, DesignViewDriver->sourceID,
			    key_data, key_size);
	MemPtrFree(key_data);
    }

    /* Unlock and free any temporary data. */
    for (i = 0; i < schema.numFields; ++i) {
	MemPtrUnlock(schema.fields[i].name);
    }
    MemPtrFree(schema.fields);

    return true;
}

typedef struct {
    /* Info needed to get old data out of source database. */
    DataSourceGetter getter;
    void * rec_data;

    /* Info on the source database so conversion can be determined. */
    DataSource * src;
    UInt32 * reorder;
} RebuildData;

static char * Rebuild_GetString(void * ptr, UInt16 fieldNum) SECT_UI;
static Int32 Rebuild_GetInteger(void * ptr, UInt16 fieldNum) SECT_UI;
static Boolean Rebuild_GetBoolean(void * ptr, UInt16 fieldNum) SECT_UI;
static void Rebuild_GetDate(void * ptr, UInt16 fieldNum,
			    UInt16* year, UInt8* month, UInt8* day) SECT_UI;
static void Rebuild_GetTime(void * ptr, UInt16 fieldNum,
			    UInt8* hour, UInt8* minute) SECT_UI;
static void Rebuild(DataSource *, DataSourceSchema *, UInt32 *) SECT_UI;

static char * Rebuild_GetString(void * ptr, UInt16 fieldNum)
{
    static char buf[32];

    RebuildData * data = (RebuildData *) ptr;
    UInt16 srcFieldNum = data->reorder[fieldNum];
    Int32 v;
    char *p;

    if (srcFieldNum >= data->src->schema.numFields) {
	buf[0] = '\0';
	return buf;
    }

    switch (data->src->schema.fields[srcFieldNum].type) {
    case FIELD_TYPE_STRING:
	p = data->getter.GetString(data->rec_data, srcFieldNum);
	break;

    case FIELD_TYPE_INTEGER:
	v = data->getter.GetInteger(data->rec_data, srcFieldNum);
	StrPrintF(buf, "%ld", v);
	p = buf;
	break;

    case FIELD_TYPE_BOOLEAN:
	if (data->getter.GetBoolean(data->rec_data, srcFieldNum))
	    buf[0] = '1';
	else
	    buf[0] = '0';
	buf[1] = '\0';
	p = buf;
	break;

    default:
	buf[0] = '\0';
	p = buf;
    }

    return p;
}

static Int32 Rebuild_GetInteger(void * ptr, UInt16 fieldNum)
{
    RebuildData * data = (RebuildData *) ptr;
    UInt16 srcFieldNum = data->reorder[fieldNum];
    Int32 v;
    char * p;

    if (srcFieldNum >= data->src->schema.numFields)
	return 0;

    switch (data->src->schema.fields[srcFieldNum].type) {
    case FIELD_TYPE_STRING:
	p = data->getter.GetString(data->rec_data, srcFieldNum);
	v = String2Long(p);
	break;

    case FIELD_TYPE_INTEGER:
	v = data->getter.GetInteger(data->rec_data, srcFieldNum);
	break;

    case FIELD_TYPE_BOOLEAN:
	if (data->getter.GetBoolean(data->rec_data, srcFieldNum))
	    v = 1;
	else
	    v = 0;
	break;

    default:
	v = 0;
	break;
    }

    return v;
}

static Boolean Rebuild_GetBoolean(void * ptr, UInt16 fieldNum)
{
    RebuildData * data = (RebuildData *) ptr;
    UInt16 srcFieldNum = data->reorder[fieldNum];
    Boolean b;

    if (srcFieldNum >= data->src->schema.numFields)
	return false;

    switch (data->src->schema.fields[srcFieldNum].type) {
    case FIELD_TYPE_STRING:
	b = false;
	break;

    case FIELD_TYPE_INTEGER:
	if (data->getter.GetInteger(data->rec_data, srcFieldNum))
	    b = true;
	else
	    b = false;
	break;

    case FIELD_TYPE_BOOLEAN:
	b = data->getter.GetBoolean(data->rec_data, srcFieldNum);
	break;

    default:
	b = false;
	break;
    }

    return b;
}

static void
Rebuild_GetDate(void * ptr, UInt16 fieldNum,
		UInt16* year, UInt8* month, UInt8* day)
{
    RebuildData * data = (RebuildData *) ptr;
    UInt16 srcFieldNum = data->reorder[fieldNum];
    DateTimeType datetime;

    /* For new fields, just return the current date. */
    if (srcFieldNum >= data->src->schema.numFields) {
	TimSecondsToDateTime(TimGetSeconds(), &datetime);
	*year = datetime.year;
	*month = datetime.month;
	*day = datetime.day;
	return;
    }

    switch (data->src->schema.fields[srcFieldNum].type) {
    case FIELD_TYPE_DATE:
	data->getter.GetDate(data->rec_data, srcFieldNum, year, month, day);
	break;

    default:
	TimSecondsToDateTime(TimGetSeconds(), &datetime);
	*year = datetime.year;
	*month = datetime.month;
	*day = datetime.day;
	break;
    }
}

static void
Rebuild_GetTime(void * ptr, UInt16 fieldNum, UInt8* hour, UInt8* minute)
{
    RebuildData * data = (RebuildData *) ptr;
    UInt16 srcFieldNum = data->reorder[fieldNum];

    DateTimeType datetime;

    /* For new fields, just return the current date. */
    if (srcFieldNum >= data->src->schema.numFields) {
	TimSecondsToDateTime(TimGetSeconds(), &datetime);
	*hour = datetime.hour;
	*minute = datetime.minute;
	return;
    }

    switch (data->src->schema.fields[srcFieldNum].type) {
    case FIELD_TYPE_TIME:
	data->getter.GetTime(data->rec_data, srcFieldNum, hour, minute);
	break;

    default:
	TimSecondsToDateTime(TimGetSeconds(), &datetime);
	*hour = datetime.hour;
	*minute = datetime.minute;
	break;
    }
}

static DataSourceGetter rebuild_getter = {
    GetString: Rebuild_GetString,
    GetInteger: Rebuild_GetInteger,
    GetBoolean: Rebuild_GetBoolean,
    GetDate: Rebuild_GetDate,
    GetTime: Rebuild_GetTime
};

static void
Rebuild(DataSource * src, DataSourceSchema * schema, UInt32 * reorder)
{
    RebuildData data;
    DataSource * dst;
    void * key_data, *new_key_data;
    UInt16 key_size, new_key_size, recNum, i;
    Err err;
    void * context;

    /* Display an on-screen "Working..." message. */
    context = DrawWorkingIndicator();

    /* Create a temporary database to move the fields into. */
    err = src->driver->class_ops.Create("DBOStmp", schema,
					&key_data, &key_size);
    if (err != 0) {
	EraseWorkingIndicator(context);
	FrmAlert(alertID_RebuildFailed);
	return;
    }

    /* Open the temporary data source for writing. */
    dst = OpenDataSource(DriverList, dmModeReadWrite, src->driver->sourceID,
			 key_data, key_size);
    if (!dst) {
	EraseWorkingIndicator(context);
	FrmAlert(alertID_RebuildFailed);
	return;
    }

    /* Fill in the fields of the rebuild data that we know of. */
    data.src = src;
    data.reorder = reorder;

    /* Convert the records into the temporary database. */
    recNum = 0;
    while (1) {
	/* Make sure that this is a valid record. */
	if (! src->ops.Seek(src, &recNum, 0, dmSeekForward))
	    break;

	/* Copy and convert the record to the destination. */
	src->ops.LockRecord(src, recNum, false, &data.getter, &data.rec_data);
	err = dst->ops.UpdateRecord(dst, dmMaxRecordIndex, false,
				    &rebuild_getter, &data);
	src->ops.UnlockRecord(src, recNum, false, data.rec_data);

	/* If the copy failed, then notify the user. */
	if (err != 0) {
	    /* Flag the error. */
	    EraseWorkingIndicator(context);
	    FrmAlert(DeviceFullAlert);

	    /* Close the temporary database and delete it. */
	    CloseDataSource(dst);
	    DeleteDataSource(DriverList, src->driver->sourceID,
			     key_data, key_size);
	    MemPtrFree(key_data);
	    return;
	}

	/* Advance the record number. */
	recNum++;
    }

    /* Now copy each view in the source over. */
    for (i = 0; i < src->ops.GetNumOfViews(src); ++i) {
	DataSourceView * view;
	UInt16 col, j, newFieldNum;

	/* Retrieve a copy of this view. */
	view = src->ops.GetView(src, i);

	/* Validate the view in light of any fields added or removed. */
	for (col = 0; col < view->numCols; ++col) {
	    /* Assume that we don't find anything. Use first field. */
	    newFieldNum = 0;

	    /* Look up this field number in the reordering list. */
	    for (j = 0; j < schema->numFields; ++j) {
		if (view->cols[col].fieldNum == reorder[j])
		    newFieldNum = j;
	    }

	    /* Remap this column. */
	    view->cols[col].fieldNum = newFieldNum;
	}

	/* Store the view in the destination and free view data. */
	dst->ops.StoreView(dst, i, view);
	src->ops.PutView(src, i, view);
    }

    /* Remove the "Working..." indicator since we need to close all forms. */
    EraseWorkingIndicator(context);

    /* Close and delete the active source database. */
    {
	UInt16 sourceID = src->driver->sourceID, src_key_size;
	void * src_key_data;
	char title[64];

	/* Retrieve the name of the source database. */
	src->ops.GetTitle(src, title, sizeof(title));

	/* Get a key we can use to delete the source database. */
	src_key_size = src->ops.GetKey(src, 0);
	src_key_data = MemPtrNew(src_key_size);
	src->ops.GetKey(src, src_key_data);

	/* Close the source database. */
	CloseDataSource(CurrentSource);
	CurrentSource = dst;

	/* Delete the source database. */
	DeleteDataSource(DriverList, sourceID, src_key_data, src_key_size);
	MemPtrFree(src_key_data);

	/* Rename the destination database to the name the source had. */
	dst->ops.SetTitle(dst, title, &new_key_data, &new_key_size);
	Chooser_RenameDatabase(dst->driver->sourceID,
			       key_data, key_size,
			       new_key_data, new_key_size, title);
	MemPtrFree(new_key_data);
    }

    /* Free any temporary allocations. */
    MemPtrFree(key_data);
}

static Boolean
UpdateExistingDesign(void)
{
    Boolean changed, rebuild, result;
    DataSourceSchema schema;
    UInt32 *reorder;
    UInt16 i;

    /* Build a schema from the current design to pass to the data source. */
    schema.numFields = numFields;
    schema.fields = MemPtrNew(numFields * sizeof(DataSourceFieldInfo));
    reorder = MemPtrNew(numFields * sizeof(UInt32));
    for (i = 0; i < numFields; ++i) {
	schema.fields[i].name = MemHandleLock(SchemaFields[i].name);
	schema.fields[i].type = SchemaFields[i].type;
	reorder[i] = SchemaFields[i].uniqueID;
    }

    /* Ask the data source about the extent of the changes. */
    CurrentSource->ops.CheckUpdateSchema(CurrentSource, &schema, reorder,
					 &changed, &rebuild);

    result = true;
    if (changed) {
	if (rebuild) {
	    /* The changes require a rebuild, inform the user. */
	    if (FrmAlert(alertID_RebuildRequired) == 0) {
		Rebuild(CurrentSource, &schema, reorder);
	    } else {
		/* User does not want to do a rebuild. Stay in design mode. */
		result = false;
	    }
	} else {
	    /* User made changes that the data source can handle itself. */
	    CurrentSource->ops.UpdateSchema(CurrentSource, &schema, reorder);
	}
    }

    /* Free the schema that was built to pass to the data source. */
    for (i = 0; i < numFields; ++i) MemPtrUnlock(schema.fields[i].name);
    MemPtrFree(schema.fields);
    MemPtrFree(reorder);

    return result;
}

static void
InsertField(UInt16 WhichField)
{
    Char * str1, *str2, *p;
    UInt16 i;

    /* Allocate more space for the additional field. */
    MemHandleUnlock(SchemaFieldsHandle);
    MemHandleResize(SchemaFieldsHandle,
		    (numFields + 1) * sizeof(SchemaDesignField));
    SchemaFields = MemHandleLock(SchemaFieldsHandle);

    /* Move the fields after the insert position to the right. */
    if (WhichField < numFields) {
	for (i = numFields; i != WhichField; --i) {
	    MemMove(&SchemaFields[i], &SchemaFields[i - 1],
		    sizeof(SchemaFields[i]));
	}
    }

    str1 = GetStringResource(stringID_NewField);
    str2 = GetStringResource(stringID_New);

    SchemaFields[WhichField].name = MemHandleNew(StrLen(str1) + 1);
    p = MemHandleLock(SchemaFields[WhichField].name);
    StrCopy(p, str1);
    MemPtrUnlock(p);
    SchemaFields[WhichField].uniqueID = NextUniqueID++;
    SchemaFields[WhichField].type = FIELD_TYPE_STRING;
    StrNCopy(SchemaFields[WhichField].old_name, str2,
	     sizeof(SchemaFields[WhichField].old_name));

    PutStringResource(str1);
    PutStringResource(str2);

    /* Increment the number of fields in the design. */
    numFields++;
}

static void
DeleteField(UInt16 WhichField)
{
    Int16 i;

    /* Move the other fields down. */
    for (i = WhichField; i < numFields - 1; i++) {
	MemMove(&(SchemaFields[i]), &(SchemaFields[i + 1]),
		sizeof(SchemaFields[i]));
    }
    numFields--;

    /* Reallocate the space in the handle to remove last position. */
    MemHandleUnlock(SchemaFieldsHandle);
    MemHandleResize(SchemaFieldsHandle,
		    numFields * sizeof(SchemaDesignField));
    SchemaFields = MemHandleLock(SchemaFieldsHandle);
}

static void
HandleContextMenu(FormType* form, TableType* table, Int16 row)
{
    ListType* list;
    UInt16 list_index, i;
    Coord width;
    RectangleType cell_bounds, list_bounds;
    char * choices[3];
 
    /* Remove any selection UI that the table control does. */
    TblUnhighlightSelection(table);
 
    /* Make copies of strings to be used for the list choices. */
    choices[0] = CopyStringResource(stringID_InsertBefore);
    choices[1] = CopyStringResource(stringID_InsertAfter);
    choices[2] = CopyStringResource(stringID_Remove);

    /* Determine the width of the popup list. */
    width = 0;
    for (i = 0; i < 3; ++i) {
	UInt16 new_width;

	new_width = FntCharsWidth(choices[i], StrLen(choices[i]));
	if (new_width > width)
	    width = new_width;
    }
    width += 5;

    /* Retrieve the popup choices list. */
    list_index = FrmGetObjectIndex(form, ctlID_DesignView_InserterList);
    list = FrmGetObjectPtr(form, list_index);
 
    /* Set the location for the popup list. */
    TblGetItemBounds(table, row, col_popup, &cell_bounds);
    FrmGetObjectBounds(form, list_index, &list_bounds);
    list_bounds.topLeft.x = cell_bounds.topLeft.x;
    list_bounds.topLeft.y = cell_bounds.topLeft.y;
    list_bounds.extent.x = width;
    FrmSetObjectBounds(form, list_index, &list_bounds);
 
    /* Set the choices which will be presented to the user. */
    LstSetSelection(list, noListSelection);
    if (numFields == 1) {
        LstSetListChoices(list, choices, 2);
        LstSetHeight(list, 2);
    } else {
        LstSetListChoices(list, choices, 3);
        LstSetHeight(list, 3);
    }
 
    /* Let the user choose an action. */
    LstPopupList(list);

    /* Insert or delete a column as necessary. */
    if (LstGetSelection(list) != noListSelection) {
        UInt16 entry = TblGetRowID(table, row);
 
        switch (LstGetSelection(list)) {
        case 0:
            InsertField(entry);
            break;
 
        case 1:
            InsertField(entry + 1);
            break;
 
        case 2:
            DeleteField(entry);
            break;
        }
 
        /* Redraw the table now that we have changed the ordering. */
        LoadTable(table);
        TblRedrawTable(table);
    }

    /* Free the list choices that we allocated earlier. */
    MemPtrFree(choices[0]);
    MemPtrFree(choices[1]);
    MemPtrFree(choices[2]);
} 

static void
FillLocalSchema(void)
{
    DataSourceSchema * schema;
    Int16 i;
    char * p;

    /* Determine the number of fields that we will display. */
    if (CurrentSource) {
	schema = &CurrentSource->schema;
	numFields = schema->numFields;
    } else if (DesignViewInitialSchema) {
	schema = DesignViewInitialSchema;
	numFields = schema->numFields;
    } else {
	schema = 0;
	numFields = MAX_NUM_FIELDS_FOR_NEW_DB;
    }

    /* Allocate the temporary memory space. */
    SchemaFieldsHandle = MemHandleNew(numFields * sizeof(SchemaDesignField));
    SchemaFields = MemHandleLock(SchemaFieldsHandle);
    NextUniqueID = 0;

    if (schema) {
	for (i = 0; i < numFields; i++) {
	    UInt32 len = StrLen(schema->fields[i].name);
	    SchemaFields[i].name = MemHandleNew(1 + len);
	    p = MemHandleLock(SchemaFields[i].name);
	    StrCopy(p, schema->fields[i].name);
	    MemPtrUnlock(p);
	    SchemaFields[i].uniqueID = NextUniqueID++;
	    SchemaFields[i].type = schema->fields[i].type;
	    StrNCopy(SchemaFields[i].old_name, schema->fields[i].name,
		     sizeof(SchemaFields[i].old_name));
	}
    } else {
	char * FldN;

	FldN = GetStringResource(stringID_FldN);
	for (i = 0; i < numFields; i++) {
	    SchemaFields[i].name = MemHandleNew(32);
	    p = MemHandleLock(SchemaFields[i].name);
	    MemSet(p, 32, 0);
	    MemPtrUnlock(p);
	    SchemaFields[i].uniqueID = NextUniqueID++;
	    SchemaFields[i].type = FIELD_TYPE_STRING;
	    StrPrintF(SchemaFields[i].old_name, FldN, (Int16) (i + 1));
	}
	PutStringResource(FldN);
    }
}

static void
SetupPopupList(void)
{
    UInt16 i, index;

    /* Count the number of valid field types. */
    popup_strings_count = 0;
    for (i = 0; i < (sizeof(types) / sizeof(struct types_t)); ++i) {
	if (DesignViewDriver->class_ops.SupportsFieldType(types[i].type))
	    popup_strings_count++;
    }

    /* Allocate space for the popup list strings. */
    popup_strings = MemPtrNew(popup_strings_count * sizeof(char *));

    /* Fill in the popup list strings. */
    for (i = index = 0; i < (sizeof(types) / sizeof(struct types_t)); ++i) {
	if (DesignViewDriver->class_ops.SupportsFieldType(types[i].type)) {
	    /* Copy the field type name string into place. */
	    popup_strings[index] = CopyStringResource(types[i].nameID);

	    /* Advance the index inside the popup_strings array. */
	    types[i].index = index;
	    index++;
	} else {
	    types[i].index = -1;
	}
    }
}

Boolean
DesignViewHandleEvent(EventPtr event)
{
    FormPtr form;
    FieldPtr fld;
    TablePtr table;
    Char * name;
    UInt16 i;

    form = FrmGetActiveForm();

    switch (event->eType) {
    case frmOpenEvent:
	/* Initialize the working copy of the schema. */
	FillLocalSchema();

	/* Setup the popup list used for data types. */
	SetupPopupList();

	/* Set the title of depending on what mode we are in. */
	if (CurrentSource)
	    name = GetStringResource(stringID_DatabaseDesign);
	else
	    name = GetStringResource(stringID_NewDatabase);
	FrmCopyTitle(form, name);
	PutStringResource(name);

	/* Initialize the table that holds the design information. */
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	FirstField = 0;
	InitTable(table);

	/* Draw the form with field #1 appearing at the top. */
	FrmDrawForm(form);
	return true;

    case frmCloseEvent:
	/* Make sure the table no longer has focus. */
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	}

	/* Free the local working copy of the schema. */
	if (SchemaFields) {
	    MemHandleUnlock(SchemaFieldsHandle);
	    MemHandleFree(SchemaFieldsHandle);
	    SchemaFields = 0;
	    SchemaFieldsHandle = 0;
	}

	/* Free the popup list strings. */
	if (popup_strings) {
	    MemPtrFree(popup_strings);
	    popup_strings = 0;
	    popup_strings_count = 0;
	}

	/* Free the name selected by the creation dialog. */
	if (DesignViewName) {
	    MemPtrFree(DesignViewName);
	    DesignViewName = 0;
	}

	/* Free any initial schema. */
        if (DesignViewInitialSchema) {
            MemPtrFree(DesignViewInitialSchema->fields);
            MemPtrFree(DesignViewInitialSchema);
            DesignViewInitialSchema = 0;
        }

	break;

    case tblSelectEvent:
	if (event->data.tblSelect.column == col_popup) {
	    HandleContextMenu(FrmGetActiveForm(),
                              event->data.tblSelect.pTable,
                              event->data.tblSelect.row);
            return true;
	} else if (event->data.tblSelect.column == col_type) {
	    UInt16 fieldNum, index;

	    fieldNum = TblGetRowID(event->data.tblSelect.pTable,
				   event->data.tblSelect.row);
	    index = TblGetItemInt(event->data.tblSelect.pTable,
				  event->data.tblSelect.row,
				  event->data.tblSelect.column);
	    for (i = 0; i < (sizeof(types) / sizeof(struct types_t)); ++i) {
		if (types[i].index >= 0 && types[i].index == index) {
		    SchemaFields[fieldNum].type = types[i].type;
		}
	    }
	    return true;
	} else if (event->data.tblSelect.column == col_oldname) {
	    /* A tap on the label edits the field name. */
	    TblReleaseFocus(event->data.tblSelect.pTable);
	    TblUnhighlightSelection(event->data.tblSelect.pTable);
	    TblGrabFocus(event->data.tblSelect.pTable,
			 event->data.tblSelect.row, col_name);
	    fld = TblGetCurrentField(event->data.tblSelect.pTable);
	    if (fld) {
		FldGrabFocus(fld);
		FldMakeFullyVisible(fld);
	    }
	    return true;
	}
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_DesignView_DoneButton:
	    if (CurrentSource) {
		/* Discard all changes in read-only mode. */
		if (CurrentSource->readonly) {
		    FrmGotoForm(formID_ListView);
		    return true;
		}

		/* Try to implement any edits the user wants. */
		if (! UpdateExistingDesign())
		    return true;
	    } else {
		/* Try to create the database as specified. */
		if (! UpdateNewDesign())
		    return true;
	    }

	case ctlID_DesignView_CancelButton:
	    if (CurrentSource)
		FrmGotoForm(formID_ListView);
	    else
		FrmGotoForm(formID_Chooser);

	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_DesignView_UpButton:
	    Scroll(winUp);
	    break;

	case ctlID_DesignView_DownButton:
	    Scroll(winDown);
	    break;
	}
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_Help:
	    FrmHelp(stringID_Help_DesignView);
	    return true;

	default:
	    return HandleCommonMenuEvent(event->data.menu.itemID);
	}
	break;

    default:
	break;
    }

    return false;
}
