/*
 * palm-db-tools: Support Library: String Parsing Utility Functions
 * Copyright (C) 1999-2000 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __LIBSUPPORT_STROP_H__
#define __LIBSUPPORT_STROP_H__

#include <stdexcept>
#include <vector>
#include <string>
#include <strstream>

namespace StrOps {

    // This exception is thrown whenever an error is encountered in
    // csv_to_array and str_to_array.
    class csv_parse_error : public std::runtime_error {
    public:
	csv_parse_error(const std::string& msg) : std::runtime_error(msg) { }
    };

    // The results of any parse are returned as this type.
    typedef std::vector<std::string> string_list_t;


    /**
     * Convert all uppercase characters in a string to lowercase.
     */
    void lower(std::string& str);

    /**
     * Convert a string to a boolean value.
     *
     * @param str The string containing a boolean value to convert.
     */
    bool string2boolean(const std::string& str);

    /**
     * Strip trailing characters from a string.
     *
     * @param str  The string to strip characters from.
     * @param what The string containing characters to strip.
     */
    std::string strip_back(const std::string& str, const std::string& what);

    /**
     * Strip leading characters from a string.
     *
     * @param str  The string to strip characters from.
     * @param what The string containing characters to strip.
     */
    std::string strip_front(const std::string& str, const std::string& what);

    /**
     * Convert a string to a target type using a strstream.
     */
    template<class T>
	inline void convert_string(const std::string& str, T& result) {
	std::istrstream(str.c_str()) >> result;
    }

    /**
     * Parse a string in CSV (comman-seperated values) format and
     * return it as a list.
     *
     * @param str The string containing the CSV fields.
     * @param delim The field delimiter. Defaults to a comma.
     */
    string_list_t csv_to_array(const std::string& str, char delim = ',');


    /**
     * Parse an argv-style array and return it as a list.
     *
     * @param str             The string to parse.
     * @param delim           String containing the delimiter characters.
     * @param multiple_delim  Should multiple delimiters count as one?
     * @param handle_comments Handle # as a comment character.
     */
    string_list_t str_to_array(const std::string& str,
			       const std::string& delim,
			       bool multiple_delim,
			       bool handle_comments);

} // namespace StrOps

#endif
