/*
 * palm-db-tools: Support Library: String Parsing Utility Functions
 * Copyright (C) 1999-2000 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string>
#include <vector>
#include <strstream>
#include <algorithm>
#include <cctype>

#include "strop.h"

void StrOps::lower(std::string& str)
{
    for (std::string::iterator p = str.begin(); p != str.end(); ++p) {
	if (isupper(*p))
	    *p = tolower(*p);
    }
}

bool StrOps::string2boolean(const std::string& str)
{
    std::string value(str);

    StrOps::lower(value);

    if (value == "on")       return true;
    else if (str == "off")   return false;
    else if (str == "true")  return true;
    else if (str == "t")     return true;
    else if (str == "false") return false;
    else if (str == "f")     return false;
    else {
	int num = 0;

	std::istrstream(str.c_str()) >> num;
	return num != 0 ? true : false;
    }
}

std::string StrOps::strip_back(const std::string& str, const std::string& what)
{
    std::string result(str);
    std::string::reverse_iterator p = result.rbegin();

    while (p != result.rend()
	   && (std::find(what.begin(), what.end(), *p) != what.end())) ++p;

    result.erase(p.base(), result.end());

    return result;
}

std::string StrOps::strip_front(const std::string& str,const std::string& what)
{
    std::string result(str);
    std::string::iterator p = result.begin();

    while (p != result.end()
	   && (std::find(what.begin(), what.end(), *p) != what.end())) ++p;

    result.erase(result.begin(), p);

    return result;
}

StrOps::string_list_t StrOps::csv_to_array(const std::string& str, char delim)
{
    enum { STATE_NORMAL, STATE_QUOTES } state;
    StrOps::string_list_t result;
    std::string elem;

    state = STATE_NORMAL;
    for (std::string::const_iterator p = str.begin(); p != str.end(); ++p) {
	switch (state) {
	case STATE_NORMAL:
	    if (*p == '"') {
		state = STATE_QUOTES;
	    } else if (*p == delim) {
		result.push_back(elem);
		elem = "";
	    } else {
		elem += *p;
	    }
	    break;

	case STATE_QUOTES:
	    if (*p == '"') {
		if ((p + 1) != str.end() && *(p+1) == '"') {
		    ++p;
		    elem += '"';
		} else {
		    state = STATE_NORMAL;
		}
	    } else {
		elem += *p;
	    }
	    break;
	}
    }

    switch (state) {
    case STATE_NORMAL:
	result.push_back(elem);
	break;
    case STATE_QUOTES:
	throw StrOps::csv_parse_error("unterminated quotes");
	break;
    }

    return result;
}

StrOps::string_list_t
StrOps::str_to_array(const std::string& str, const std::string& delim,
		     bool multiple_delim, bool handle_comments)
{
    enum { STATE_NORMAL, STATE_COMMENT, STATE_QUOTE_DOUBLE, STATE_QUOTE_SINGLE,
	   STATE_BACKSLASH, STATE_BACKSLASH_DOUBLEQUOTE } state;
    StrOps::string_list_t result;
    std::string elem;

    state = STATE_NORMAL;
    for (std::string::const_iterator p = str.begin(); p != str.end(); ++p) {
	switch (state) {
	case STATE_NORMAL:
	    if (*p == '"') {
		state = STATE_QUOTE_DOUBLE;
	    } else if (*p == '\'') {
		state = STATE_QUOTE_SINGLE;
	    } else if (std::find(delim.begin(), delim.end(), *p) != delim.end()) {
		if (multiple_delim) {
		    ++p;
		    while (p != str.end()
			   && std::find(delim.begin(), delim.end(), *p)
			   != delim.end()) {
			++p;
		    }
		    --p;
		}
		result.push_back(elem);
		elem = "";
	    } else if (*p == '\\') {
		state = STATE_BACKSLASH;
	    } else if (handle_comments && *p == '#') {
		state = STATE_COMMENT;
	    } else {
		elem += *p;
	    }
	    break;

	case STATE_COMMENT:
	    break;

	case STATE_QUOTE_DOUBLE:
	    if (*p == '"')
		state = STATE_NORMAL;
	    else if (*p == '\\')
		state = STATE_BACKSLASH_DOUBLEQUOTE;
	    else
		elem += *p;
	    break;

	case STATE_QUOTE_SINGLE:
	    if (*p == '\'')
		state = STATE_NORMAL;
	    else
		elem += *p;
	    break;
	    
	case STATE_BACKSLASH:
	    elem += *p;
	    state = STATE_NORMAL;
	    break;
	    
	case STATE_BACKSLASH_DOUBLEQUOTE:
	    switch (*p) {
	    case '\\':
		elem += '\\';
		break;

	    case 'n':
		elem += '\n';
		break;

	    case 'r':
		elem += '\r';
		break;

	    case 't':
		elem += '\t';
		break;

	    case 'v':
		elem += '\v';
		break;

	    case '"':
		elem += '"';
		break;

	    case 'x':
		{
		    char buf[3];

		    // Extract and check the first hexadecimal character.
		    if ((p + 1) == str.end())
			throw StrOps::csv_parse_error("truncated escape");
		    if (! isxdigit(*(p + 1)))
			throw StrOps::csv_parse_error("invalid hex character");
		    buf[0] = *++p;

		    // Extract and check the second (if any) hex character.
		    if ((p + 1) != str.end() && isxdigit(*(p + 1))) {
			buf[1] = *++p;
			buf[2] = '\0';
		    } else {
			buf[1] = buf[2] = '\0';
		    }

		    std::istrstream stream(buf);
		    stream.setf(std::ios::hex, std::ios::basefield);
		    unsigned value;
		    stream >> value;

		    elem += static_cast<char> (value & 0xFFu);
		}
		break;
	    }

	    // Escape is done. Go back to the normal double quote state.
	    state = STATE_QUOTE_DOUBLE;
	    break;
	}
    }

    switch (state) {
    case STATE_NORMAL:
	result.push_back(elem);
	break;

    case STATE_QUOTE_DOUBLE:
	throw StrOps::csv_parse_error("unterminated double quotes");
	break;

    case STATE_QUOTE_SINGLE:
	throw StrOps::csv_parse_error("unterminated single quotes");
	break;

    case STATE_BACKSLASH:
    case STATE_BACKSLASH_DOUBLEQUOTE:
	throw StrOps::csv_parse_error("an escape character must follow a backslash");
	break;

    default:
	break;
    }

    return result;
}
