/*
 * palm-db-tools: Read/write PDB files from disk.
 * Copyright (C) 2000 by Tom Dyas (tdyas@users.sourceforge.net)
 */

#ifndef __PALMLIB_FILE_H__
#define __PALMLIB_FILE_H__

#include <map>
#include <vector>

#include "Database.h"

namespace PalmLib {

    class File : public Database {
    public:
	File(bool resourceDB = false);
	File(const char * fname);
	virtual ~File();

	// Load a PDB file into the object.
	void load(const char * fname);

	// Save the object to a PDB file.
	void save(const char * fname);

	// Return the total number of records/resources in this
	// database.
	virtual unsigned getNumRecords() const { return m_records.size(); }

	// Return the database's application info block as a Block
	// object. The caller owns the returned object.
	virtual Block getAppInfoBlock() const { return m_app_info; }

	// Set the database's app info block to the contents of the
	// passed Block object.
	virtual void setAppInfoBlock(const Block& new_app_info)
	    { m_app_info = new_app_info; }

	// Return the database's sort info block as a Block
	// object. The caller owns the returned object.
	virtual Block getSortInfoBlock() const { return m_sort_info; }

	// Set the database's sort info block to the contents of the
	// passed Block object.
	virtual void setSortInfoBlock(const Block& new_sort_info)
	    { m_sort_info = new_sort_info; }

	// Return the record identified by the given index. The caller
	// owns the returned RawRecord object.
	virtual Record getRecord(unsigned index) const;

	// Set the record identified by the given index to the given
	// record.
	virtual void setRecord(unsigned index, const Record& rec);

	// Append the given record to the database.
	virtual void appendRecord(const Record& rec);

	// returned if the specified (type, ID) combination is not
	// present in the database. The caller owns the returned
	// RawRecord object.
	virtual Resource getResourceByType(pi_uint32_t type,
					   pi_uint32_t id) const;
	
	// Return the resource present at the given index. NULL is
	// returned if the index is invalid. The caller owns the
	// returned RawRecord object.
	virtual Resource getResourceByIndex(unsigned index) const;

	// Set the resouce at given index to passed Resource object.
	virtual void setResource(unsigned index, const Resource& rsrc);

    private:
	Block m_app_info;
	Block m_sort_info;
	pi_int32_t m_next_record_list_id;
	pi_int32_t m_ent_hdr_size;

    protected:
	typedef std::vector<Block *> record_list_t;
	typedef std::map<pi_uint32_t, Record *> uid_map_t;

	record_list_t m_records;
	uid_map_t m_uid_map;
    };

};

#endif
