/*
 * palm-db-tools: Read/write Palm Pilot database files.
 * Copyright (C) 1999-2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <fstream>
#include <utility>
#include <stdexcept>
#include <algorithm>

#include <string.h>
#include <memory.h>

#include "File.h"

using namespace std;
using namespace PalmLib;

static const int PI_HDR_SIZE          = 78;
static const int PI_RESOURCE_ENT_SIZE = 10;
static const int PI_RECORD_ENT_SIZE   = 8;

PalmLib::File::File(bool resourceDB)
    : Database(resourceDB),
      m_app_info(), m_sort_info(), m_next_record_list_id(0)
{
    m_ent_hdr_size = isResourceDB()? PI_RESOURCE_ENT_SIZE : PI_RECORD_ENT_SIZE;
}

PalmLib::File::File(const char * fname)
    : Database(false), m_app_info(), m_sort_info(), m_next_record_list_id(0)
{
    load(fname);
}

PalmLib::File::~File()
{
    for (record_list_t::iterator i = m_records.begin();
	 i != m_records.end(); ++i) {
	delete (*i);
    }
}

void PalmLib::File::load(const char * fname)
{
    ifstream f;
    pi_char_t buf[PI_HDR_SIZE];
    pi_uint32_t app_info_offset, sort_info_offset;
    size_t app_info_size = 0, sort_info_size = 0;
    streampos file_size;
    pi_int32_t offset;

    f.open(fname, ios::in | ios::binary);
    if (!f)
	throw PalmLib::error("unable to open the file");

    // Determine the size of the file.
    f.seekg(0, ios::end);
    file_size = f.tellg();
    f.seekg(0, ios::beg);

    // Read and parse the PDB header.
    f.read(reinterpret_cast<char *> (buf), PI_HDR_SIZE);
    if (!f)
	throw PalmLib::error("unable to read header");
    {
	char tmp_name[32];
	strncpy(tmp_name, (char *) buf, sizeof(tmp_name));
	tmp_name[sizeof(tmp_name)-1] = '\0';
	name(tmp_name);
    }
    flags(get_short(buf + 32));
    version(get_short(buf + 34));
    creation_time(get_long(buf + 36));
    modification_time(get_long(buf + 40));
    backup_time(get_long(buf + 44));
    modnum(get_long(buf + 48));
    app_info_offset = get_long(buf + 52);
    sort_info_offset = get_long(buf + 56);
    type(get_long(buf + 60));
    creator(get_long(buf + 64));
    unique_id_seed(get_long(buf + 68));

    m_next_record_list_id = get_long(buf + 72);
    pi_int16_t num_entries = get_short(buf + 76);

    typedef vector< pair<Block *, pair< streampos, size_t > > > recinfo_t;
    recinfo_t recinfo;

    if (isResourceDB())
	m_ent_hdr_size = PI_RESOURCE_ENT_SIZE;
    else
	m_ent_hdr_size = PI_RECORD_ENT_SIZE;

    // Read all of the record entries into the object.
    for (int i = 0; i < num_entries; ++i) {
	pi_char_t *hdr = new pi_char_t[m_ent_hdr_size];
	Block * entry = 0;

	f.read(reinterpret_cast<char *> (hdr), m_ent_hdr_size);
	if (!f)
	    throw PalmLib::error("unable to read record header");

	if (isResourceDB()) {
	    entry = new Resource(get_long(hdr), get_short(hdr + 4), 0);
	    recinfo.push_back(make_pair(entry, make_pair(get_long(hdr + 6), 0)));
	} else {
	    entry = new Record(hdr[4], get_treble(hdr + 5), 0);
	    recinfo.push_back(make_pair(entry, make_pair(get_long(hdr), 0)));
	}

	delete [] hdr;
    }

    // Now we iterate in reverse over the vector so figure out the size
    // of each entry.
    offset = file_size;
    for (recinfo_t::reverse_iterator p = recinfo.rbegin();
	 p != recinfo.rend(); ++p) {
	streampos & entry_offset = (*p).second.first;
	size_t & entry_size = (*p).second.second;

	entry_size = offset - entry_offset;
	offset = entry_offset;
    }

    // Determine the size of the sort info block.
    if (sort_info_offset > 0) {
	sort_info_size = offset - sort_info_offset;
	offset = sort_info_offset;
    }

    // Determine the size of the app info block.
    if (app_info_offset > 0) {
	app_info_size = offset - app_info_offset;
	offset = app_info_offset;
    }

    // Read the app info block.
    if (app_info_size > 0) {
	pi_char_t * app_info = new pi_char_t[app_info_size];
	f.seekg(app_info_offset);
	f.read(reinterpret_cast<char *> (app_info), app_info_size);
	if (!f)
	    throw PalmLib::error("unable to read application info block");
	m_app_info.set_raw(app_info, app_info_size);
	delete [] app_info;
    }

    // Read the sort info block.
    if (sort_info_size > 0) {
	pi_char_t * sort_info = new pi_char_t[sort_info_size];
	f.seekg(sort_info_offset);
	f.read(reinterpret_cast<char *> (sort_info), sort_info_size);
	if (!f)
	    throw PalmLib::error("unable to read sort info block");
	m_sort_info.set_raw(sort_info, sort_info_size);
	delete [] sort_info;
    }
    // Read all of the records into memory.
    for (recinfo_t::iterator q = recinfo.begin(); q != recinfo.end(); ++q) {
	Block * entry = (*q).first;
	streampos entry_offset = (*q).second.first;
	size_t entry_size = (*q).second.second;

	pi_char_t * data = new pi_char_t[entry_size];
	f.seekg(entry_offset);
	f.read(reinterpret_cast<char *> (data), entry_size);
	if (!f)
	    throw PalmLib::error("unable to read record data");
	entry->set_raw(data, entry_size);
	delete [] data;

	m_records.push_back(entry);
    }

    f.close();
}

void PalmLib::File::save(const char * fname)
{
    ofstream f;
    pi_char_t buf[PI_HDR_SIZE];
    streampos offset;

    f.open(fname, ios::out | ios::binary);
    if (!f)
	throw PalmLib::error("unable to open file for output");

    offset = PI_HDR_SIZE + m_records.size() * m_ent_hdr_size + 2;

    memcpy(buf, name().c_str(), 32);
    set_short(buf + 32, flags());
    set_short(buf + 34, version());
    set_long(buf + 36, creation_time());
    set_long(buf + 40, modification_time());
    set_long(buf + 44, backup_time());
    set_long(buf + 48, modnum());
    if (m_app_info.raw_size() > 0) {
	set_long(buf + 52, offset);
	offset += m_app_info.raw_size();
    } else {
	set_long(buf + 52, 0);
    }
    if (m_sort_info.raw_size() > 0) {
	set_long(buf + 56, offset);
	offset += m_sort_info.raw_size();
    } else {
	set_long(buf + 56, 0);
    }
    set_long(buf + 60, type());
    set_long(buf + 64, creator());
    set_long(buf + 68, unique_id_seed());
    set_long(buf + 72, m_next_record_list_id);
    set_short(buf + 76, m_records.size());

    // Write the PDB/PRC header to the file.
    f.write(reinterpret_cast<char *> (buf), sizeof(buf));
    if (!f)
	throw PalmLib::error("unable to write header");

    for (record_list_t::iterator i = m_records.begin();
	 i != m_records.end(); ++i) {
	Block * entry = *i;

	if (isResourceDB()) {
	    Resource * resource = reinterpret_cast<Resource *> (entry);
	    set_long(buf, resource->type());
	    set_short(buf + 4, resource->id());
	    set_long(buf + 6, offset);
	} else {
	    Record * record = reinterpret_cast<Record *> (entry);
	    set_long(buf, offset);
	    buf[4] = record->attrs();
	    set_treble(buf + 5, record->unique_id());
	}

	f.write(reinterpret_cast<char *> (buf), m_ent_hdr_size);
	if (!f)
	    throw PalmLib::error("unable to write record header");

	offset += entry->raw_size();
    }

    f.write("\0", 1);
    f.write("\0", 1);

    if (m_app_info.raw_size() > 0) {
	f.write((char *) m_app_info.raw_data(), m_app_info.raw_size());
	if (!f)
	    throw PalmLib::error("unable to write application info block");
    }

    if (m_sort_info.raw_size() > 0) {
	f.write((char *) m_sort_info.raw_size(), m_sort_info.raw_size());
	if (!f)
	    throw PalmLib::error("unable to write sort info block");
    }

    for (record_list_t::iterator q = m_records.begin();
	 q != m_records.end(); ++q) {
	Block * entry = *q;
	f.write((char *) entry->raw_data(), entry->raw_size());
	if (!f)
	    throw PalmLib::error("unable to write records");
    }

    f.close();
}

// Return the record identified by the given index. The caller owns
// the returned RawRecord object.
Record PalmLib::File::getRecord(unsigned index) const
{
    if (index >= m_records.size()) throw std::out_of_range("invalid index");
    return *(reinterpret_cast<Record *> (m_records[index]));
}

// Set the record identified by the given index to the given record.
void PalmLib::File::setRecord(unsigned index, const Record& rec)
{
    if (index >= m_records.size()) throw std::out_of_range("invalid index");
    *(reinterpret_cast<Record *> (m_records[index])) = rec;
}

// Append the given record to the database.
void PalmLib::File::appendRecord(const Record& rec)
{
    Record* record = new Record(rec);

    // If this new record has a unique ID that duplicates any other
    // record, then reset the unique ID to an unused value.
    if (m_uid_map.find(record->unique_id()) != m_uid_map.end()) {
	uid_map_t::iterator iter = max_element(m_uid_map.begin(),
					       m_uid_map.end());
	pi_uint32_t maxuid = (*iter).first;

	// The new unique ID becomes the max plus one.
	record->unique_id(maxuid + 1);
    }

    m_uid_map[record->unique_id()] = record;
    m_records.push_back(record);
}

// Return the resource with the given type and ID. NULL is returned if
// the specified (type, ID) combination is not present in the
// database. The caller owns the returned RawRecord object.
Resource PalmLib::File::getResourceByType(pi_uint32_t type, pi_uint32_t id) const
{
    for (record_list_t::const_iterator i = m_records.begin();
	 i != m_records.end(); ++i) {
	Resource* resource = reinterpret_cast<Resource *> (*i);
	if (resource->type() == type && resource->id() == id)
	    return *resource;
    }
    throw std::out_of_range("not found");
}

// Return the resource present at the given index. NULL is returned if
// the index is invalid. The caller owns the returned RawRecord
// object.
Resource PalmLib::File::getResourceByIndex(unsigned index) const
{
    if (index >= m_records.size()) throw std::out_of_range("invalid index");
    return *(reinterpret_cast<Resource *> (m_records[index]));
}

// Set the resouce at given index to passed RawResource object.
void PalmLib::File::setResource(unsigned index, const Resource& resource)
{
    if (index >= m_records.size()) throw std::out_of_range("invalid index");
    *(reinterpret_cast<Resource *> (m_records[index])) = resource;
}
