/*
 * This class provides access to "List" format databases.
 */

#ifndef __PALMLIB_FLATFILE_LISTDB_H__
#define __PALMLIB_FLATFILE_LISTDB_H__


#include <vector>
#include <string>

#include "libpalm/Database.h"
#include "Database.h"

namespace PalmLib {
    namespace FlatFile {

	class ListDB : public Database {
	public:
	    /**
	     * Return true if this class can handle the given PalmOS
	     * database.
	     *
	     * @param pdb PalmOS database to check for support.
	     */
	    static bool classify(PalmLib::Database& pdb);

	    /**
	     * Return true if this class is the database identified by
	     * name.
	     *
	     * @param name A database type name to check.
	     */
	    static bool match_name(const std::string& name);

	    /**
	     * Default constructor for an initially empty database.
	     */
	    ListDB()
		: m_display_style(FIELD1_FIELD2), m_write_protect(false) { }

	    /**
	     * Constructor which fills the flat-file structure from a
	     * PalmOS database.
	     */
	    ListDB(PalmLib::Database&);

	    // destructor
	    virtual ~ListDB() { }

            /**
             * After all processing to add fields and records is done,
             * outputPDB is called to create the actual file format
             * used by the flat-file database product.
             *
             * @param pdb An instance of PalmLib::Database.
	     */
            virtual void outputPDB(PalmLib::Database& pdb) const;

            /**
             * Return the maximum number of fields allowed in the
             * database. This class returns 0 since there is no limit.
	     */
            virtual unsigned getMaxNumOfFields() const;

	    /**
	     * Return true for the field types that this class
	     * currently supports. Returns false otherwise.
	     *
	     * @param type The field type to check for support.
	     */
	    virtual bool supportsFieldType(const Field::FieldType& type) const;

            /**
             * Return the maximum number of views supported by this
             * type of flat-file database.
	     */
	    virtual unsigned getMaxNumOfListViews() const;

	    /**
	     * Set a extra option.
	     *
	     * @param opt_name  The name of the option to set.
	     * @param opt_value The value to assign to this option.
	     */
	    virtual void setOption(const std::string& opt_name,
				   const std::string& opt_value);

	    /**
	     * Get a list of extra options.
	     */
	    virtual options_list_t getOptions(void) const;

	    /**
	     * Hook function which should be invoked by a caller after
	     * all calls the meta-deta functions have completed. This
	     * allows the database type-specific code to do final
	     * checks on the meta-data. An exception will be throw if
	     * there is an error. Otherwise, nothing will happen.
	     */
	    virtual void doneWithSchema();

	private:
	    enum display_style_t { FIELD1_FIELD2, FIELD2_FIELD1 };

	    struct ListAppInfoType {
		pi_uint16_t renamedCategories;
		std::string categoryLabels[16];
		pi_char_t categoryUniqIDs[16];
		pi_char_t lastUniqID;

		display_style_t displayStyle;
		bool writeProtect;
		pi_char_t lastCategory;
		std::string customField1;
		std::string customField2;

		/**
		 * Pack the above data into binary form.
		 */
		PalmLib::Block pack();

		/**
		 * Unpack the binary form into the above data. Throws
		 * an exception if the block is invalid.
		 *
		 * @param block The block to extract data from.
		 */
		void unpack(const PalmLib::Block& block);
	    };
	    friend struct ListAppInfoType;

	    display_style_t m_display_style;
	    bool m_write_protect;
	};

    }
}

#endif
