/*
 * This class provides access to JFile v3 databases.
 */

#ifndef __PALMLIB_FLATFILE_JFILE3_H__
#define __PALMLIB_FLATFILE_JFILE3_H__

#include <vector>
#include <string>

#include "libpalm/Database.h"
#include "Database.h"

namespace PalmLib {
    namespace FlatFile {

	class JFile3 : public Database {
	public:
            /**
             * Return true if this class can handle the given PalmOS
             * database.
             *
             * @param pdb PalmOS database to check for support.
             */
            static bool classify(const PalmLib::Database& pdb);

            /**
             * Return true if this class is the database identified by
             * name.
             *
             * @param name A database type name to check.
             */
            static bool match_name(const std::string& name);

            /**
             * Default constructor for an initially empty database.
             */
            JFile3() : m_flags(0) { }

            /**
             * Constructor which fills the flat-file structure from a
             * PalmOS database.
             */
            JFile3(const PalmLib::Database&);

            // destructor
            virtual ~JFile3() { }

            /**
             * After all processing to add fields and records is done,
             * outputPDB is called to create the actual file format
             * used by the flat-file database product.
             *
             * @param pdb An instance of PalmLib::Database.
             */
            virtual void outputPDB(PalmLib::Database& pdb) const;

            /**
             * Return the maximum number of fields allowed in the
             * database. This class returns 0 since there is no limit.
             */
            virtual unsigned getMaxNumOfFields() const;

            /**
             * Return true for the field types that this class
             * currently supports. Returns false otherwise.
             *
             * @param type The field type to check for support.
             */
            virtual bool supportsFieldType(const Field::FieldType& type) const;

            /**
             * Return the maximum number of views supported by this
             * type of flat-file database.
             */
            virtual unsigned getMaxNumOfListViews() const;

            /**
             * Hook the end of schema processing.
             */
            virtual void doneWithSchema();

            /**
             * Set a extra option.
             *
             * @param opt_name  The name of the option to set.
             * @param opt_value The value to assign to this option.
             */
            virtual void setOption(const std::string& opt_name,
                                   const std::string& opt_value);

            /**
             * Get a list of extra options.
             */
            virtual options_list_t getOptions(void) const;

	private:
	    struct JFileAppInfoType {
		std::string fieldNames[20];
		int fieldTypes[20];
		unsigned numFields;
		unsigned version;
		unsigned showDBColumnWidths[20];
		unsigned showDataWidth;
		int sortFields[3];
		int findField;
		int filterField;
		std::string findString;
		std::string filterString;
		unsigned flags;
		int firstColumnToShow;
		std::string password;

		// Pack the above data into binary form.
                PalmLib::Block pack() const;

                // Unpack the binary form into the above data.
                void unpack(const PalmLib::Block& block);
	    };

	    /**
	     * Take a record and extract the null-terminated strings
	     * that are contained within it.
	     */
	    std::vector<std::string> parse_record(const PalmLib::Record&) const;

	    /**
	     * Take a series of strings and produce a PalmLib::Record
	     * with the data packed into it.
	     */
	    PalmLib::Record build_record(const std::vector<std::string>&) const;

	    std::string m_password;
	    pi_uint16_t m_flags;
	};

    } // namespace FlatFile
} // namespace PalmLib

#endif
