/*
 * palm-db-tools: Read/write DB-format databases
 * Copyright (C) 1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <vector> 
#include <string>
#include <stdexcept>

#include <cstring>

#include "libsupport/strop.h"

#include "DB.h"

using namespace PalmLib::FlatFile;
using namespace PalmLib;

static const pi_uint16_t CHUNK_FIELD_NAMES         = 0;
static const pi_uint16_t CHUNK_FIELD_TYPES         = 1;
static const pi_uint16_t CHUNK_LISTVIEW_DEFINITION = 64;
static const pi_uint16_t CHUNK_LISTVIEW_OPTIONS    = 65;
static const pi_uint16_t CHUNK_LFIND_OPTIONS       = 128;

template <class Map, class Key>
static inline bool has_key(const Map& map, const Key& key)
{
    return map.find(key) != map.end();
}

bool PalmLib::FlatFile::DB::classify(PalmLib::Database& pdb)
{
    return (pdb.creator() == PalmLib::mktag('D','B','O','S'))
	&& (pdb.type()    == PalmLib::mktag('D','B','0','0'));
}

bool PalmLib::FlatFile::DB::match_name(const std::string& name)
{
    return (name == "DB") || (name == "db");
}

void PalmLib::FlatFile::DB::extract_chunks(const PalmLib::Block& appinfo)
{
    size_t i;
    pi_uint16_t chunk_type;
    pi_uint16_t chunk_size;

    if (appinfo.raw_size() > 4) {
	// Loop through each chunk in the block while data remains.
        i = 4;
        while (i < appinfo.raw_size()) {
            /* Stop the loop if there is not enough room for even one
             * chunk header.
             */
            if (i + 4 >= appinfo.raw_size())
                throw PalmLib::error("header is corrupt");

            // Copy the chunk type and size into the local buffer.
	    chunk_type = get_short(appinfo.raw_data() + i);
	    chunk_size = get_short(appinfo.raw_data() + i + 2);
            i += 4;

	    // Copy the chunk into seperate storage.
	    Chunk chunk(appinfo.raw_data() + i, chunk_size);
	    chunk.chunk_type = chunk_type;
	    m_chunks[chunk.chunk_type].push_back(chunk);

            /* Advance the index by the size of the chunk. */
            i += chunk.raw_size();
        }

        // If everything was correct, then we should be exactly at the
	// end of the block. 
        if (i != appinfo.raw_size())
            throw PalmLib::error("header is corrupt");
    } else {
	throw PalmLib::error("header is corrupt");
    }
}

void PalmLib::FlatFile::DB::extract_schema(unsigned numFields)
{
    unsigned i;

    if (!has_key(m_chunks, CHUNK_FIELD_NAMES)
	|| !has_key(m_chunks, CHUNK_FIELD_TYPES))
	throw PalmLib::error("database is missing its schema");

    Chunk names_chunk = m_chunks[CHUNK_FIELD_NAMES][0];
    Chunk types_chunk = m_chunks[CHUNK_FIELD_TYPES][0];
    pi_char_t* p = names_chunk.raw_data();
    pi_char_t* q = types_chunk.raw_data();

    // Ensure that the types chunk has the expected size.
    if (types_chunk.raw_size() != numFields * sizeof(pi_uint16_t))
	throw PalmLib::error("types chunk is corrupt");

    // Loop for each field and extract the name and type.
    for (i = 0; i < numFields; ++i) {
	Field::FieldType type;
	int len;

	// Determine the length of the name string. Ensure that the
	// string does not go beyond the end of the chunk.
	pi_char_t* null_p = reinterpret_cast<pi_char_t*>
	    (memchr(p, 0, names_chunk.size() - (p - names_chunk.data())));
	if (!null_p)
	    throw PalmLib::error("names chunk is corrupt");
	len = null_p - p;

	switch (PalmLib::get_short(q)) {
	case 0:
	    type = Field::STRING;
	    break;
	case 1:
	    type = Field::BOOLEAN;
	    break;
	case 2:
	    type = Field::INTEGER;
	    break;
	default:
	    throw PalmLib::error("unsupported field type");
	}

	appendField(std::string((char *) p, len), type);

	p += len + 1;
	q += 2;
    }
}

void PalmLib::FlatFile::DB::extract_listviews()
{
    if (!has_key(m_chunks, CHUNK_LISTVIEW_DEFINITION))
	throw PalmLib::error("no list views in database");

    const std::vector<Chunk>& chunks = m_chunks[CHUNK_LISTVIEW_DEFINITION];

    for (std::vector<Chunk>::const_iterator iter = chunks.begin();
	 iter != chunks.end(); ++iter) {
	const Chunk& chunk = (*iter);
	PalmLib::FlatFile::ListView lv;

	if (chunk.size() < (2 + 2 + 32))
	    throw PalmLib::error("list view is corrupt");

	// pi_uint16_t flags = PalmLib::get_short(chunk.data());
	pi_uint16_t num_cols = PalmLib::get_short(chunk.data() + 2);

	if (chunk.size() != static_cast<unsigned> (2 + 2 + 32 + num_cols * 4))
	    throw PalmLib::error("list view is corrupt");

	// Determine the length of the name string.
	pi_char_t* null_ptr = reinterpret_cast<pi_char_t*>
	    (memchr(chunk.data() + 4, 0, 32));
	if (null_ptr)
	    lv.name = std::string((char *) (chunk.data() + 4),
				  null_ptr - (chunk.data() + 4));
	else
	    lv.name = "Unknown";

	const pi_char_t* p = chunk.data() + 2 + 2 + 32;
	for (int i = 0; i < num_cols; ++i) {
	    pi_uint16_t field = PalmLib::get_short(p);
	    pi_uint16_t width = PalmLib::get_short(p + 2);
	    p += 2 * sizeof(pi_uint16_t);

	    if (field >= getNumOfFields())
		throw PalmLib::error("list view is corrupt");

	    PalmLib::FlatFile::ListViewColumn col(field, width);
	    lv.push_back(col);
	}

	appendListView(lv);
    }
}

void PalmLib::FlatFile::DB::parse_record(PalmLib::Record& record,
					 std::vector<pi_char_t *>& ptrs,
					 std::vector<size_t>& sizes)
{
    unsigned i;

    // Ensure that enough space for the offset table exists.
    if (record.raw_size() < getNumOfFields() * sizeof(pi_uint16_t))
	throw PalmLib::error("record is corrupt");

    // Extract the offsets from the record. Determine field pointers.
    std::vector<pi_uint16_t> offsets(getNumOfFields());
    for (i = 0; i < getNumOfFields(); ++i) {
	offsets[i] = get_short(record.raw_data() + i * sizeof(pi_uint16_t));
	if (offsets[i] >= record.raw_size())
	    throw PalmLib::error("record is corrupt");
	ptrs.push_back(record.raw_data() + offsets[i]);
    }

    // Determine the field sizes.
    for (i = 0; i < getNumOfFields() - 1; ++i) {
	sizes.push_back(offsets[i + 1] - offsets[i]);
    }
    sizes.push_back(record.raw_size() - offsets[getNumOfFields() - 1]);
}

PalmLib::FlatFile::DB::DB(PalmLib::Database& pdb)
    : Database(pdb), m_flags(0)
{
    // Split the application information block into its component chunks.
    extract_chunks(pdb.getAppInfoBlock());

    // Pull the header fields and schema out of the databasse.
    m_flags = get_short(pdb.getAppInfoBlock().raw_data());
    unsigned numFields = get_short(pdb.getAppInfoBlock().raw_data() + 2);
    extract_schema(numFields);

    // Extract all of the list views.
    extract_listviews();

    for (unsigned i = 0; i < pdb.getNumRecords(); ++i) {
	PalmLib::Record record = pdb.getRecord(i);
	Record rec;

	std::vector<pi_char_t *> ptrs;
	std::vector<size_t> sizes;
	parse_record(record, ptrs, sizes);

	for (unsigned j = 0; j < getNumOfFields(); ++j) {
	    PalmLib::FlatFile::Field f;

	    switch (field_type(j)) {
	    case Field::STRING:
		f.type = Field::STRING;
		f.v_string = std::string((char *) ptrs[j], sizes[j] - 1);
		break;

	    case Field::BOOLEAN:
		if (sizes[j] != 1)
		    throw PalmLib::error("record is corrupt");
		f.type = Field::BOOLEAN;
		if (*(ptrs[j]))
		    f.v_boolean = true;
		else
		    f.v_boolean = false;
		break;

	    case Field::INTEGER:
		if (sizes[j] != sizeof(pi_int32_t))
		    throw PalmLib::error("record is corrupt");
		f.type = Field::INTEGER;
		f.v_integer = get_long(ptrs[j]);
		break;

	    default:
		throw PalmLib::error("unknown field type");
	    }

	    rec.push_back(f);
	}

	appendRecord(rec);
    }
}  

void PalmLib::FlatFile::DB::make_record(PalmLib::Record& pdb_record,
					const Record& record) const
{
    PalmLib::FlatFile::Record::const_iterator iter;

    // Determine the packed size of this record.
    size_t size = record.size() * sizeof(pi_uint16_t);
    for (iter = record.begin(); iter != record.end(); ++iter) {
	const Field& field = (*iter);
	switch (field.type) {
	case Field::STRING:
	    size += field.v_string.length() + 1;
	    break;

	case Field::BOOLEAN:
	    size += 1;
	    break;

	case Field::INTEGER:
	    size += 4;
	    break;

	default:
	    throw PalmLib::error("unsupported field type");
	}
    }

    // Allocate a block for the packed record and setup the pointers.
    pi_char_t* buf = new pi_char_t[size];
    pi_char_t* p = buf + record.size() * sizeof(pi_uint16_t);
    pi_char_t* offsets = buf;

    // Pack the fields into the buffer.
    for (iter = record.begin(); iter != record.end(); ++iter) {
	const Field& field = (*iter);

	// Mark the offset to the start of this field in the offests table.
	PalmLib::set_short(offsets, static_cast<pi_uint16_t> (p - buf));
	offsets += sizeof(pi_uint16_t);

	// Pack the field.
	switch (field.type) {
	case Field::STRING:
	    memcpy(p, field.v_string.c_str(), field.v_string.length() + 1);
	    p += field.v_string.length() + 1;
	    break;

	case Field::BOOLEAN:
	    *p++ = ((field.v_boolean) ? 1 : 0);
	    break;

	case Field::INTEGER:
	    PalmLib::set_long(p, field.v_integer);
	    p += sizeof(pi_int32_t);
	    break;

	default:
	    throw PalmLib::error("unsupported field type");
	}
    }

    // Place the packed data into the PalmOS record.
    pdb_record.set_raw(buf, size);
    delete [] buf;
}

void PalmLib::FlatFile::DB::build_standard_chunks(std::vector<DB::Chunk>& chunks) const
{
    pi_char_t* buf;
    pi_char_t* p;
    unsigned i;

    // Determine the size needed for the names chunk.
    size_t names_chunk_size = 0;
    for (i = 0; i < getNumOfFields(); ++i) {
	names_chunk_size += field_name(i).length() + 1;
    }

    // Build the names chunk.
    buf = new pi_char_t[names_chunk_size];
    p = buf;
    for (i = 0; i < getNumOfFields(); ++i) {
	const std::string name = field_name(i);
	memcpy(p, name.c_str(), name.length() + 1);
	p += name.length() + 1;
    }
    Chunk names_chunk(buf, names_chunk_size);
    names_chunk.chunk_type = CHUNK_FIELD_NAMES;
    delete [] buf;

    // Build the types chunk.
    buf = new pi_char_t[getNumOfFields() * sizeof(pi_uint16_t)];
    p = buf;
    for (i = 0; i < getNumOfFields(); ++i) {
	// Pack the type of the current field.
	switch (field_type(i)) {
	case Field::STRING:
	    PalmLib::set_short(p, 0);
	    break;

	case Field::BOOLEAN:
	    PalmLib::set_short(p, 1);
	    break;

	case Field::INTEGER:
	    PalmLib::set_short(p, 2);
	    break;

	default:
	    throw PalmLib::error("unsupported field type");
	}

	// Advance to the next position.
	p += sizeof(pi_uint16_t);
    }
    Chunk types_chunk(buf, getNumOfFields() * sizeof(pi_uint16_t));
    types_chunk.chunk_type = CHUNK_FIELD_TYPES;
    delete [] buf;

    // Build the list view options chunk.
    buf = new pi_char_t[2 * sizeof(pi_uint16_t)];
    PalmLib::set_short(buf, 0);
    PalmLib::set_short(buf + sizeof(pi_uint16_t), 0);
    Chunk listview_options_chunk(buf, 2 * sizeof(pi_uint16_t));
    listview_options_chunk.chunk_type = CHUNK_LISTVIEW_OPTIONS;
    delete [] buf;

    // Build the local find options chunk.
    buf = new pi_char_t[sizeof(pi_uint16_t)];
    PalmLib::set_short(buf, 0);
    Chunk lfind_options_chunk(buf, 1 * sizeof(pi_uint16_t));
    lfind_options_chunk.chunk_type = CHUNK_LFIND_OPTIONS;
    delete [] buf;

    // Add all the chunks to the chunk list.
    chunks.push_back(names_chunk);
    chunks.push_back(types_chunk);
    chunks.push_back(listview_options_chunk);
    chunks.push_back(lfind_options_chunk);
}

void PalmLib::FlatFile::DB::build_listview_chunk(std::vector<DB::Chunk>& chunks,
						 const ListView& lv) const
{
    // Calculate size and allocate space for the temporary buffer.
    size_t size = 2 * sizeof(pi_uint16_t) + 32
	+ lv.size() * (2 * sizeof(pi_uint16_t));
    pi_char_t* buf = new pi_char_t[size];

    // Fill in the header details.
    PalmLib::set_short(buf, 0);
    PalmLib::set_short(buf + sizeof(pi_uint16_t), lv.size());
    memset((char *) (buf + 4), 0, 32);
    strncpy((char *) (buf + 4), lv.name.c_str(), 32);

    // Fill in the column details.
    pi_char_t* p = buf + 4 + 32;
    for (ListView::const_iterator i = lv.begin(); i != lv.end(); ++i) {
	const ListViewColumn& col = (*i);
	PalmLib::set_short(p, col.field);
	PalmLib::set_short(p + sizeof(pi_uint16_t), col.width);
	p += 2 * sizeof(pi_uint16_t);
    }

    // Create the chunk and place it in the chunks list.
    Chunk chunk(buf, size);
    chunk.chunk_type = CHUNK_LISTVIEW_DEFINITION;
    delete [] buf;
    chunks.push_back(chunk);
}

void PalmLib::FlatFile::DB::build_appinfo_block(const std::vector<DB::Chunk>& chunks, PalmLib::Block& appinfo) const
{
    std::vector<Chunk>::const_iterator iter;

    // Determine the size of the final app info block.
    size_t size = 2 * sizeof(pi_uint16_t);
    for (iter = chunks.begin(); iter != chunks.end(); ++iter) {
	const Chunk& chunk = (*iter);
	size += 2 * sizeof(pi_uint16_t) + chunk.raw_size();
    }

    // Allocate the temporary buffer. Fill in the header.
    pi_char_t* buf = new pi_char_t[size];
    PalmLib::set_short(buf, m_flags);
    PalmLib::set_short(buf + sizeof(pi_uint16_t), getNumOfFields());

    // Pack the chunks into the buffer.
    size_t i = 4;
    for (iter = chunks.begin(); iter != chunks.end(); ++iter) {
	const Chunk& chunk = (*iter);

	// Set the chunk type and size.
	PalmLib::set_short(buf + i, chunk.chunk_type);
	PalmLib::set_short(buf + i + 2, chunk.raw_size());
	i += 4;

	// Copy the chunk data into the buffer.
	memcpy(buf + i, chunk.raw_data(), chunk.raw_size());
	i += chunk.raw_size();
    }

    // Finally, move the buffer into the provided appinfo block.
    appinfo.set_raw(buf, size);
    delete [] buf;
}

void PalmLib::FlatFile::DB::outputPDB(PalmLib::Database& pdb) const
{
    unsigned i;

    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile, Database, outputPDB, (pdb));

    // Set the database's type and creator.
    pdb.type(PalmLib::mktag('D','B','0','0'));
    pdb.creator(PalmLib::mktag('D','B','O','S'));

    // Create the app info block.
    std::vector<Chunk> chunks;
    build_standard_chunks(chunks);
    for (i = 0; i < getNumOfListViews(); ++i) {
	build_listview_chunk(chunks, getListView(i));
    }
    PalmLib::Block appinfo;
    build_appinfo_block(chunks, appinfo);
    pdb.setAppInfoBlock(appinfo);

    // Output each record to the PalmOS database.
    for (i = 0; i < getNumRecords(); ++i) {
	Record record = getRecord(i);
	PalmLib::Record pdb_record;

	make_record(pdb_record, record);
	pdb.appendRecord(pdb_record);
    }
}

unsigned PalmLib::FlatFile::DB::getMaxNumOfFields() const
{
    return 0;
}

bool
PalmLib::FlatFile::DB::supportsFieldType(const Field::FieldType& type) const
{
    switch (type) {
    case Field::STRING:
    case Field::BOOLEAN:
    case Field::INTEGER:
	return true;
    default:
	return false;
    }
}

unsigned PalmLib::FlatFile::DB::getMaxNumOfListViews() const
{
    return 0;
}

void PalmLib::FlatFile::DB::doneWithSchema()
{
    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile, Database, doneWithSchema, ());

    if (getNumOfListViews() < 1)
	throw PalmLib::error("at least one list view must be specified");
}

void PalmLib::FlatFile::DB::setOption(const std::string& name,
				      const std::string& value)
{
    if (name == "find") {
	if (StrOps::string2boolean(value))
	    m_flags &= ~(1);
	else
	    m_flags |= 1;
    } else {
	SUPERCLASS(PalmLib::FlatFile, Database, setOption, (name, value));
    }
}

PalmLib::FlatFile::Database::options_list_t
PalmLib::FlatFile::DB::getOptions(void)
{
    typedef PalmLib::FlatFile::Database::options_list_t::value_type value;
    PalmLib::FlatFile::Database::options_list_t result;

    result = SUPERCLASS(PalmLib::FlatFile, Database, getOptions, ());

    if (m_flags & 1)
	result.push_back(value("find", "true"));
    else
	result.push_back(value("find", "false"));

    return result;
}
