/*
 * palm-db-tools: PDB->CSV conversion tool
 * Copyright (C) 1998-2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <fstream>
#include <string>
#include <strstream>
#include <fstream>

#include <ctype.h>

#ifndef WIN32
# if TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
# else
#  if HAVE_SYS_TIME_H
#   include <sys/time.h>
#  else
#   include <time.h>
#  endif
# endif
#else
# include <time.h>
#endif


#include "libsupport/clp.h"
#include "libflatfile/Factory.h"
#include "libflatfile/Database.h"
#include "libpalm/File.h"

using namespace std;
using namespace PalmLib::FlatFile;

static bool extended_csv_mode = false;
static std::string program;
static std::ostream* err = &std::cerr;

static std::string
quote_string(std::string str, bool extended_mode)
{
    std::string result;

    if (extended_mode) {
	result += '"';
	for (std::string::iterator c = str.begin(); c != str.end(); ++c) {
	    switch (*c) {
	    case '\\':
		result += '\\';
		result += '\\';
		break;

	    case '\r':
		result += '\\';
		result += 'r';
		break;

	    case '\n':
		result += '\\';
		result += 'n';
		break;

	    case '\t':
		result += '\\';
		result += 't';
		break;

	    case '\v':
		result += '\\';
		result += 'v';
		break;

	    case '"':
		result += '\\';
		result += '"';
		break;

	    default:
		if (isprint(*c)) {
		    result += *c;
		} else {
		    std::ostrstream buf;
		    buf.width(2);
		    buf.setf(std::ios::hex, std::ios::basefield);
		    buf << ((static_cast<unsigned> (*c)) & 0xFF) << std::ends;

		    result += "\\x";
		    result += buf.str();
		}
		break;
	    }
	}
	result += '"';
    } else {
	result += '"';
	for (std::string::iterator c = str.begin(); c != str.end(); ++c) {
	    if (*c == '"') {
		result += "\"\"";
	    } else if (*c == '\n' || *c == '\r') {
		*err << program << "use extended csv mode for newlines\n";
		exit(1);
	    } else {
		result += *c;
	    }
	}
	result += '"';
    }

    return result;
}

static void output_csv_file(std::string fname,
			    const PalmLib::FlatFile::Database& db,
			    const std::string& sep,
			    const std::string& format_date,
			    const std::string& format_time,
			    const std::string& format_datetime)
{
    int numRecords = db.getNumRecords();

    std::ofstream csv(fname.c_str());
    if (!csv) {
	*err << "unable to create the CSV file\n";
	exit(1);
    }

    for (int i = 0; i < numRecords; ++i) {
	bool first = true;
	PalmLib::FlatFile::Record record = db.getRecord(i);

	for (PalmLib::FlatFile::Record::const_iterator f = record.begin();
	     f != record.end(); ++f) {
	    // If this is not the first field, output the field seperator.
	    if (!first) csv << sep;
	    first = false;

	    // Do not output anything if the field is NULL.
	    if ((*f).no_value)
		continue;

	    // Output the field contents.
	    switch ((*f).type) {
	    case PalmLib::FlatFile::Field::STRING:
		csv << quote_string((*f).v_string, extended_csv_mode);
		break;
		
	    case PalmLib::FlatFile::Field::BOOLEAN:
		if ((*f).v_boolean)
		    csv << "true";
		else
		    csv << "false";
		break;

	    case PalmLib::FlatFile::Field::INTEGER:
		csv << (*f).v_integer;
		break;

	    case PalmLib::FlatFile::Field::FLOAT:
		csv << (*f).v_float;
		break;

	    case PalmLib::FlatFile::Field::DATE:
		{
		    char buf[1024];
		    struct tm tm;

		    // Clear out the output buffer.
		    memset(buf, 0, sizeof(buf));

		    // Fill in the date and normalize the remaining fields.
		    tm.tm_mon = (*f).v_date.month - 1;
		    tm.tm_mday = (*f).v_date.day;
		    tm.tm_year = (*f).v_date.year - 1900;
		    tm.tm_hour = 0;
		    tm.tm_min = 0;
		    tm.tm_sec = 0;
		    tm.tm_wday = 0;
		    tm.tm_yday = 0;
		    tm.tm_isdst = -1;
		    (void) ::mktime(&tm);

		    // Convert and output the date using the format.
		    ::strftime(buf, sizeof(buf), format_date.c_str(), &tm);
		    csv << buf;
		    break;
		}
	    
	    case PalmLib::FlatFile::Field::TIME:
		{
		    char buf[1024];
		    struct tm tm;
		    const struct tm * tm_ptr;
		    time_t now;

		    // Clear out the output buffer.
		    memset(buf, 0, sizeof(buf));

		    // Get a normalized structure and change the time.
		    ::time(&now);
		    tm_ptr = ::localtime(&now);
		    memcpy(&tm, tm_ptr, sizeof(tm));
		    tm.tm_hour = (*f).v_time.hour;
		    tm.tm_min = (*f).v_time.minute;
		    tm.tm_sec = 0;

		    // Convert and output the time using the format.
		    ::strftime(buf, sizeof(buf), format_time.c_str(), &tm);
		    csv << buf;
		    break;
		}

	    case PalmLib::FlatFile::Field::DATETIME:
		{
		    char buf[1024];
		    struct tm tm;

		    // Clear out the output buffer.
		    memset(buf, 0, sizeof(buf));

		    // Fill in the datetime and normalize the remaining fields.
		    tm.tm_mon = (*f).v_date.month - 1;
		    tm.tm_mday = (*f).v_date.day;
		    tm.tm_year = (*f).v_date.year - 1900;
		    tm.tm_hour = (*f).v_time.hour;
		    tm.tm_min = (*f).v_time.minute;
		    tm.tm_sec = 0;
		    tm.tm_wday = 0;
		    tm.tm_yday = 0;
		    tm.tm_isdst = -1;
		    (void) mktime(&tm);

		    // Convert and output the time using the format.
		    ::strftime(buf, sizeof(buf), format_datetime.c_str(), &tm);
		    csv << buf;
		    break;
		}

	    default:
		// Leave the field blank if we don't support the field type.
		break;
	    }
	}

	// Finally, output the newline to finish the line.
	csv << endl;
    }

    // Close the CSV file.
    csv.close();
}

static std::string getTypeName(Field::FieldType t)
{
    switch (t) {
    case PalmLib::FlatFile::Field::STRING:
	return "string";

    case PalmLib::FlatFile::Field::BOOLEAN:
	return "boolean";

    case PalmLib::FlatFile::Field::INTEGER:
	return "integer";

    case PalmLib::FlatFile::Field::FLOAT:
	return "float";

    case PalmLib::FlatFile::Field::DATE:
	return "date";

    case PalmLib::FlatFile::Field::TIME:
	return "time";

    case PalmLib::FlatFile::Field::DATETIME:
	return "datetime";

    default:
	// If we don't support the field type, then fake it as a string.
	return "string";
    }
}

static bool
has_trivial_listview(const PalmLib::FlatFile::Database& flatfile)
{
    if (flatfile.getMaxNumOfListViews() != 1)
	return false;

    if (flatfile.getNumOfListViews() != 1)
	return false;

    PalmLib::FlatFile::ListView lv = flatfile.getListView(0);

    unsigned field = 0;
    PalmLib::FlatFile::ListView::const_iterator iter = lv.begin();
    for (; iter != lv.end(); ++iter, ++field) {
	const PalmLib::FlatFile::ListViewColumn& col = (*iter);
	if (col.field != field) return false;
    }

    if (field != flatfile.getNumOfFields())
	return false;

    return true;
}

static void
output_info_file(const std::string& fname,
		 const PalmLib::FlatFile::Database& flatfile,
		 const PalmLib::Database& pdb)
{
    unsigned i;

    // Open the output file.
    std::ofstream info(fname.c_str());
    if (!info) {
	*err << program << ": unable to open metadata file\n";
	exit(1);
    }

    /* Output the database title. */
    info << "title " << quote_string(flatfile.title(), true) << "\n";

    /* Output the database structure. */
    if (flatfile.getNumOfListViews() == 0 || !has_trivial_listview(flatfile)) {
	for (i = 0; i < flatfile.getNumOfFields(); ++i) {
	    info << "field " << quote_string(flatfile.field_name(i), true)
		 << " " << getTypeName(flatfile.field_type(i)) << std::endl;
	}
    } else {
	PalmLib::FlatFile::ListView lv = flatfile.getListView(0);
	PalmLib::FlatFile::ListView::const_iterator p = lv.begin();

	for (i = 0; i < flatfile.getNumOfFields(); ++i, ++p) {
	    info << "field " << quote_string(flatfile.field_name(i), true)
		 << " " << getTypeName(flatfile.field_type(i)) << " "
		 << (*p).width << std::endl;
	}
    }

    // If this database supports multiple views, output them all.
    if (! has_trivial_listview(flatfile)) {
	for (i = 0; i < flatfile.getNumOfListViews(); ++i) {
	    ListView lv = flatfile.getListView(i);

	    info << "view " << quote_string(lv.name, true) << " ";
	    for (ListView::const_iterator p = lv.begin(); p != lv.end(); ++p) {
		const ListViewColumn& col = (*p);
		info << " " << quote_string(flatfile.field_name(col.field), true)
		     << " " << col.width;
	    }
	    info << std::endl;
	}
    }

    // Output any extra options that this database format supports.
    const Database::options_list_t opts = flatfile.getOptions();
    for (Database::options_list_t::const_iterator p = opts.begin();
	 p != opts.end(); ++p) {
	info << "option " << (*p).first << ' ' << (*p).second << endl;
    }

    /* Output extended CSV mode. */
    if (extended_csv_mode)
	info << "extended on\n";
    else
	info << "extended off\n";
    
    // Close the output file.
    info.close();
}

static void usage(void)
{
    cout << "usage: " << program << " [options] PDB_FILE CSV_FILE INFO_FILE\n";
    cout << "  -e, --extended               Use extended CSV mode.\n";
    cout << "  -s SEP, --separator=SEP      Change the default field separator. (Default: \",\")\n";
    cout << "  -h, --help                   Display this help screen.\n";
    cout << "  -v, --version                Display the program version.\n";
    cout << "  -n FILE, --errors=FILE       Send all error messages to FILE.\n";
    cout << "  -d FORMAT, --date=FORMAT     Change the output format for date fields.\n";
    cout << "  -t FORMAT, --time=FORMAT     Change the output format for time fields.\n";
    cout << "  -D FORMAT, --datetime=FORMAT Change the output format for datetime fields.\n";
}

int main(int argc, char *argv[])
{
    static const CLP::option_t options[] = {
	{ "extended", 'e', "extended", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
	{ "sep", 's', "separator", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ "help", 'h', "help", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
	{ "version", 'v', "version", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
	{ "errors", 'n', "errors", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ "date", 'd', "date", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ "time", 't', "time", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ "datetime", 'D', "datetime", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ 0 },
    };
    CLP::args_t args;
    CLP::option_map_t opts;
    std::string field_sep(",");
    std::string format_date("%m/%d/%Y");
    std::string format_time("%H:%M");
    std::string format_datetime("%m/%d/%Y %H:%M");

    program = argv[0];

    try {
	CLP::parse(argc - 1, argv + 1, options, opts, args);
    } catch (CLP::missing_value_error e) {
	*err << argv[0] << ": option `" << e.option_name()
	     << "' takes an argument\n";
	usage();
	return 1;
    } catch (CLP::invalid_option_error e) {
	*err << argv[0] << ": unknown option `" << e.option_name()
	     << "' was passed\n";
	usage();
	return 1;
    } catch (CLP::value_present_error e) {
	*err << argv[0] << ": option `" << e.option_name()
	     << "' does not take an argument\n";
	usage();
	return 1;
    }

    // Display the usage screen if requested or no arguments present.
    if (argc == 1 || opts.find("help") != opts.end()) {
	usage();
	return 0;
    }

    // Display the program version if requested.
    if (opts.find("version") != opts.end()) {
	cout << "pdb2csv (" << PACKAGE << ' ' << VERSION << ')' << endl;
	return 0;
    }

    // Redirect errors to a file. (Added to support a GUI wraparound on Win32.)
    if (opts.find("errors") != opts.end()) {
        std::ofstream* f = new std::ofstream(opts["errors"].c_str());
        if (!f) {
            *err << program << ": unable to open error file '"
		 << opts["errors"] << "'\n";
        } else {
	    err = f;
	}
    }

    // Check to see if extended CSV mode should be enabled.
    if (opts.find("extended") != opts.end())
	extended_csv_mode = true;

    // Change the field seperator if requested.
    if (opts.find("sep") != opts.end())
	field_sep = opts["sep"];

    // Check to see if any of the date/time formats were changed.
    if (opts.find("date") != opts.end())
	format_date = opts["date"];
    if (opts.find("time") != opts.end())
	format_time = opts["time"];
    if (opts.find("datetime") != opts.end())
	format_datetime = opts["datetime"];

    // Ensure that all 3 arguements were specified.
    if (args.size() != 3) {
	*err << program << ": 3 arguments were expected\n";
	usage();
	return 1;
    }

    // Assign better names to the arguements.
    std::string pdb_fname(args[0]);
    std::string csv_fname(args[1]);
    std::string info_fname(args[2]);

    // Read the PDB file into memory.
    PalmLib::File pdb;
    try {
	pdb.load(pdb_fname.c_str());
    } catch (const PalmLib::error& e) {
	*err << pdb_fname << ": " << e.what() << endl;
	return 1;
    }

    // Instantiate an object based on parameters.
    PalmLib::FlatFile::Factory factory;
    PalmLib::FlatFile::Database* db = 0;
    try {
	db = factory.makeDatabase(pdb);
	if (!db)
	    throw PalmLib::error("unable to determine database type");
    } catch (const PalmLib::error& e) {
	*err << pdb_fname << ": " << e.what() << endl;
	return 1;
    }

    // Output the metadata file.
    output_info_file(info_fname, *db, pdb);

    // Output all of the records.
    output_csv_file(csv_fname, *db, field_sep, format_date,
		    format_time, format_datetime);

    return 0;
}
