/*
 * palm-db-tools: CSV->PDB conversion tool
 * Copyright (C) 1998-2000 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <fstream>
#include <string>
#include <stdexcept>
#include <utility>

#ifndef WIN32
# if TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
# else
#  if HAVE_SYS_TIME_H
#   include <sys/time.h>
#  else
#   include <time.h>
#  endif
# endif
#else
# include <time.h>
#endif

#include "libpalm/File.h"
#include "libflatfile/Database.h"
#include "libflatfile/Factory.h"
#include "libsupport/clp.h"
#include "libsupport/strop.h"
extern "C" {
#include "getdate.h"
}

using namespace PalmLib;

struct parsing_state
{
    PalmLib::FlatFile::ListView implicit_listview;
    bool extended_csv_mode;
};

static std::string program;
static std::ostream* err = &std::cerr;

// Read a line from an istream w/o concern for buffer limitations.
static std::string readline(std::istream& stream)
{
    std::string line;
    char buf[1024];

    while (1) {
	// Read the next line (or part thereof) from the stream.
	stream.getline(buf, sizeof(buf));

	// Bail out of the loop if the stream has reached end-of-file.
	if (stream.eof() || stream.bad())
	    break;

	// Append the data read to the result string.
	line.append(buf);

	// If the stream is good, then stop. Otherwise, clear the
	// error indicator so that getline will work again.
	if (stream.good())
	    break;
	stream.clear();
    }

    return line;
}

PalmLib::FlatFile::Field::FieldType string2type(std::string typestr)
{
    StrOps::lower(typestr);
    if (typestr == "string")
	return PalmLib::FlatFile::Field::STRING;
    else if (typestr == "str")
	return PalmLib::FlatFile::Field::STRING;
    else if (typestr == "bool")
	return PalmLib::FlatFile::Field::BOOLEAN;
    else if (typestr == "boolean")
	return PalmLib::FlatFile::Field::BOOLEAN;
    else if (typestr == "integer")
	return PalmLib::FlatFile::Field::INTEGER;
    else if (typestr == "int")
	return PalmLib::FlatFile::Field::INTEGER;
    else if (typestr == "float")
	return PalmLib::FlatFile::Field::FLOAT;
    else if (typestr == "date")
	return PalmLib::FlatFile::Field::DATE;
    else if (typestr == "time")
	return PalmLib::FlatFile::Field::TIME;
    else if (typestr == "datetime")
	return PalmLib::FlatFile::Field::DATETIME;
    else
	throw std::invalid_argument("unknown field type");
}

void read_csv_file(PalmLib::FlatFile::Database& db, const char* fname,
		   const std::string& sep, bool extended_csv_mode)
{
    // Open the CSV file.
    std::ifstream f(fname);
    if (!f)
	return;

    int linenum = 0;
    while (1) {
	// Read the next line from the file.
	std::string line(readline(f));
	++linenum;
	if (!f)
	    break;

	// Strip the trailing newline character.
	line = StrOps::strip_back(line, "\r\n");

	// Convert the text into an argv-style array.
	StrOps::string_list_t array;
	try {
	    if (extended_csv_mode)
		array = StrOps::str_to_array(line, sep, false, false);
	    else
		array = StrOps::csv_to_array(line, sep[0]);
	} catch (const StrOps::csv_parse_error& e) {
	    *err << fname << ':' << linenum << ": " << e.what() << std::endl;
	    exit(1);
	}

	/* Make sure that the correct number of fields were received. */
	if (array.size() != db.getNumOfFields()) {
	    *err << fname << ':' << linenum
		 << ": number of fields doesn't match\n";
	    exit(1);
	}

	PalmLib::FlatFile::Record record;
	for (unsigned i = 0; i < db.getNumOfFields(); ++i) {
	    PalmLib::FlatFile::Field field;

	    switch (db.field_type(i)) {
	    case PalmLib::FlatFile::Field::STRING:
		field.type = PalmLib::FlatFile::Field::STRING;
		field.v_string = array[i];
		break;

	    case PalmLib::FlatFile::Field::BOOLEAN:
		field.type = PalmLib::FlatFile::Field::BOOLEAN;
		field.v_boolean = StrOps::string2boolean(array[i]);
		break;

	    case PalmLib::FlatFile::Field::INTEGER:
		field.type = PalmLib::FlatFile::Field::INTEGER;
		StrOps::convert_string(array[i], field.v_integer);
		break;

	    case PalmLib::FlatFile::Field::FLOAT:
		field.type = PalmLib::FlatFile::Field::FLOAT;
		StrOps::convert_string(array[i], field.v_float);
		break;

	    case PalmLib::FlatFile::Field::DATE:
	    case PalmLib::FlatFile::Field::TIME:
	    case PalmLib::FlatFile::Field::DATETIME:
		{
		    time_t when;
		    struct tm tm;

		    when = ::parse_datetime_string(array[i].c_str(), 0, &tm);
		    if (when == -1) {
			*err << fname << ':' << linenum
			     << ": invalid date in field " << (i + 1) << "\n";
			exit(1);
		    }

		    field.type = db.field_type(i);
		    field.v_date.month = tm.tm_mon + 1;
		    field.v_date.day = tm.tm_mday;
		    field.v_date.year = tm.tm_year + 1900;
		    field.v_time.hour = tm.tm_hour;
		    field.v_time.minute = tm.tm_min;
		}
		break;

	    default:
		*err << fname << ':' << linenum << ": unsupported field type\n";
		exit(1);

	    }

	    record.push_back(field);
	}
	
	// Add the record to the database.
	db.appendRecord(record);
    }

    f.close();
}

void handle_field(PalmLib::FlatFile::Database& db, parsing_state& state,
		  const std::string& name, const std::string& typestr,
		  const std::string& widthstr)
{
    PalmLib::FlatFile::Field::FieldType type = string2type(typestr);

    // Make sure that the database supports the target type.
    if (! db.supportsFieldType(type))
	throw std::string("field type not supported by this database type");

    // Make sure that we don't exceed any maximum number of fields.
    if (db.getMaxNumOfFields() != 0
	&& db.getNumOfFields() + 1 > db.getMaxNumOfFields())
	throw std::string("too many fields for this database type");

    // Add this field to the database.
    db.appendField(name, type);

    // If a field width was given, then add it to the implicit listview.
    if (! widthstr.empty()) {
	int width;

	StrOps::convert_string(widthstr, width);

	if (width < 10 || width > 160)
	    throw std::string("field width must be greater than 10 and less than 160");

	PalmLib::FlatFile::ListViewColumn col(db.getNumOfFields() - 1, width);
	state.implicit_listview.push_back(col);
    }
}

void read_info_file(PalmLib::FlatFile::Database& db, const char* fname,
		    parsing_state& state)
{
    int linenum;

#define errorout(s) *err << fname << ':' << linenum << ": " << (s) << std::endl
#define warning(s) *err << fname << ':' << linenum << ": warning: " << (s) << std::endl

    // Open the information file.
    std::ifstream f(fname);
    if (!f) {
	*err << program << "unable to open '" << fname << "'\n";
	exit(1);
    }

    linenum = 0;
    while (1) {
	// Read the next line into the buffer.
	std::string line(readline(f));
	++linenum;
	if (!f)
	    break;

	// Strip any trailing newline characters.
	line = StrOps::strip_back(line, "\r\n");

	// Strip trailing whitespace.
	line = StrOps::strip_back(line, " \t");

	// Strip leading whitespace.
	line = StrOps::strip_front(line, " \t");

	// Skip this line if it is blank.
	if (line.length() == 0)
	    continue;
	
	// Parse the line into an argv-style array.
	std::vector<std::string> array;
	try {
	    array = StrOps::str_to_array(line, " \t", true, true);
	} catch (const StrOps::csv_parse_error& e) {
	    errorout(e.what());
	    exit(1);
	}

	// Skip this line if no arguments were found.
	if (array.size() == 0)
	    continue;
	    
	StrOps::lower(array[0]);
	if (array[0] == "title") {
	    if (array.size() != 2) {
		errorout("title directive only takes 1 argument");
		exit(1);
	    }
	    db.title(array[1]);
	} else if (array[0] == "field") {
	    if (array.size() < 3 || array.size() > 4) {
		errorout("field directive takes 3 or 4 arguments");
		exit(1);
	    }
	    
	    try {
		if (array.size() == 3)
		    handle_field(db, state, array[1], array[2], "");
		else
		    handle_field(db, state, array[1], array[2], array[3]);
	    } catch (const std::string& errstr) {
		errorout(errstr);
		exit(1);
	    } catch (const std::exception& e) {
		errorout(e.what());
		exit(1);
	    }
	} else if (array[0] == "view") {
	    PalmLib::FlatFile::ListView lv;

	    // Ignore attempts to go beyond the maximum number of list views.
	    if (db.getMaxNumOfListViews() != 0
		&& db.getNumOfListViews() + 1 > db.getMaxNumOfListViews()) {
		warning("too many view directives for this database type");
		continue;
	    }

	    // Ensure that we have enough arguments.
	    if (array.size() < 3) {
		errorout("view directive takes at least 3 arguments");
		exit(1);
	    }
	    if ((array.size() % 2) != 0) {
		errorout("missing field name or width in view directive");
		exit(1);
	    }

	    // The first argument is the view name.
	    lv.name = array[1];

	    // Loop through the remaining arguments to find the fields
	    // and widths for the rest of the list view.
	    for (unsigned i = 2; i < array.size() - 1; i += 2) {
		const std::string& field_name = array[i];
		const std::string& width_string = array[i + 1];
		int field_index = -1;
		int width = 80;

		// Search for the field index corresponding to the field name.
		for (unsigned j = 0; j < db.getNumOfFields(); ++j) {
		    if (field_name == db.field_name(j)) field_index = j;
		}
		if (field_index < 0) {
		    errorout("unknown field name in field directive");
		    exit(1);
		}

		// Extract the field width from the string.
		StrOps::convert_string(width_string, width);

		// Check the width for validity.
		if (width < 10 || width > 160) {
		    errorout("field width must be greater than 10 and less than 160");
		    exit(1);
		}

		// Add this column to the list view.
		PalmLib::FlatFile::ListViewColumn col(field_index, width);
		lv.push_back(col);
	    }

	    // Add this list view to the database.
	    try {
		db.appendListView(lv);
	    } catch (const std::exception& e) {
		errorout(e.what());
		exit(1);
	    }
	} else if (array[0] == "extended") {
	    if (array.size() != 2) {
		errorout("the extended directive takes 1 argument");
		exit(1);
	    }
	    state.extended_csv_mode = StrOps::string2boolean(array[1]);
	} else if (array[0] == "option") {
	    if (array.size() != 3) {
		errorout("option directives take 2 arguments");
		exit(1);
	    }
	    db.setOption(array[1], array[2]);
	}
    }

#undef warning
#undef errorout

    f.close();
}

void usage(void)
{
    std::cout << "usage: " << program <<  " [options] CSV PDB\n";
    std::cout << "  -b, --backup            Set pdb backup flag\n";
    std::cout << "  -e, --extended          Use extended mode when reading the CSV file\n";
    std::cout << "  -t TITLE, --title=TITLE Set the database's title\n";
    std::cout << "  -i FILE, --info=FILE    Read database metadeta commands from FILE\n";
    std::cout << "  -o NAME=VALUE\n";
    std::cout << "     --option=NAME=VALUE  Set the database option NAME to VALUE\n";
    std::cout << "  -f SPEC, --field=SPEC   Add the field described by SPEC to the database.\n";
    std::cout << "                          SPEC format is \"NAME,TYPE,[WIDTH]\"\n";
    std::cout << "  -h, --help              Display this help screen\n";
    std::cout << "  -v, --version           Display the program version\n";
    std::cout << "  -s SEP, --separator=SEP Change the CSV field separator from a comma to SEP\n";
    std::cout << "  -n FILE, --errors=FILE  Send all error messages to FILE\n";
    std::cout << "  -t TYPE, --type=TYPE    Set the target format. Choices are:\n";
    std::cout << "                            DB, db (DB 0.3.x format) (Default)\n";
    std::cout << "                            OldDB, olddb (DB 0.2.x format)\n";
    std::cout << "                            MobileDB, mdb (MobileDB format)\n";
    std::cout << "                            List, listdb, list (Freeware List format)\n";
    std::cout << "                            JFile3, jfile3, JF3, jf3 (JFile v3.x)\n";
}

static pi_uint32_t get_current_time(void)
{
    time_t now;

    ::time(&now);
    return static_cast<pi_uint32_t> (now) + pi_uint32_t(2082844800);
}

typedef std::vector< std::pair<std::string, std::string> > schema_args_t;

static void opt_callback(const std::string& key, const std::string& value,
			 void * data)
{
    schema_args_t* schema_args = reinterpret_cast<schema_args_t*> (data);
    (*schema_args).push_back(std::make_pair(key, value));
}

int main(int argc, char *argv[])
{
    schema_args_t schema_args;
    static const CLP::option_t options[] = {
	{ "type", 't', "type", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ "help", 'h', "help", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
	{ "version", 'v', "version", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
	{ "errors", 'n', "errors", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ "sep", 's', "separator", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
	{ "extended", 'e', "extended", CLP::ARGUMENT_NONE,
	  CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
	{ "backup", 'b', "backup", CLP::ARGUMENT_NONE,
	  CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
	{ "title", 't', "title", CLP::ARGUMENT_STRING,
	  CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
	{ "info", 'i', "info", CLP::ARGUMENT_STRING,
	  CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
	{ "field", 'f', "field", CLP::ARGUMENT_STRING,
	  CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
	{ "option", 'o', "option", CLP::ARGUMENT_STRING,
	  CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
	{ 0 },
    };
    CLP::args_t args;
    CLP::option_map_t opts;
    parsing_state state;
    char separator;

    program = argv[0];

    try {
	CLP::parse(argc - 1, argv + 1, &options[0], opts, args);
    } catch (const CLP::missing_value_error& e) {
	*err << program << ": option `" << e.option_name()
	     << "' takes an argument\n";
	usage();
	return 1;
    } catch (const CLP::invalid_option_error& e) {
	*err << program << ": unknown option `" << e.option_name()
	     << "' was passed\n";
	usage();
	return 1;
    } catch (const CLP::value_present_error& e) {
        *err << program << ": option `" << e.option_name()
             << "' does not take an argument\n";
        usage();
        return 1;
    }

    // Redirect errors to a file. (Added to support a GUI wraparound on Win32.)
    if (opts.find("errors") != opts.end()) {
        std::ofstream* f = new std::ofstream(opts["errors"].c_str());
        if (!f) {
            *err << program << ": warning: unable to open error file `"
                 << opts["errors"] << "'\n";
        } else {
            err = f;
        }
    }

    // Display the usage screen if requested or no arguments present.
    if (argc == 1 || opts.find("help") != opts.end()) {
        usage();
        return 0;
    }

    // Display the program version if requested.
    if (opts.find("version") != opts.end()) {
        std::cout << "pdb2csv (" << PACKAGE << " " << VERSION << ")\n";
        return 0;
    }

    // Determine the CSV separator to use.
    if (opts.find("sep") != opts.end()) {
	if (opts["sep"].length() != 1) {
	    *err << program << ": -s option must be a single character\n";
	    return 1;
	}
	separator = opts["sep"][0];
    } else {
	separator = ',';
    }

    // Ensure that all 3 arguements were specified.
    if (args.size() != 2) {
        *err << program << ": 2 arguments were expected\n";
        usage();
        return 1;
    }

    // Assign better names to the arguments.
    std::string csv_fname(args[0]);
    std::string pdb_fname(args[1]);

    // Create the flat-file database object that will receive the data.
    PalmLib::FlatFile::Factory factory;
    PalmLib::FlatFile::Database* flatfile;
    if (opts.find("type") != opts.end())
	flatfile = factory.newDatabase(opts["type"]);
    else
	flatfile = factory.newDatabase("db");

    // If the factory couldn't create the object, tell the user.
    if (!flatfile) {
	*err << program << ": an unknown database type `" << opts["type"]
	     << "' was specified\n";
	return 1;
    }

    // Process all of the arguments that determine the schema and
    // database options.
    state.extended_csv_mode = false;
    for (schema_args_t::const_iterator iter = schema_args.begin();
	 iter != schema_args.end(); ++iter) {
	const std::string& key = (*iter).first;
	const std::string& value = (*iter).second;

	if (key == "info") {
	    read_info_file(*flatfile, value.c_str(), state);
	} else if (key == "extended") {
	    state.extended_csv_mode = true;
	} else if (key == "title") {
	    flatfile->title(value);
	} else if (key == "backup") {
	    flatfile->setOption("backup", "true");
	} else if (key == "option") {
	    std::string opt_name;
	    std::string opt_value;

	    // Determine if the option has an associated value.
	    std::string::size_type equals_pos = value.find("=", 0);
	    if (equals_pos != std::string::npos) {
		opt_name = value.substr(0, equals_pos);
		opt_value = value.substr(equals_pos + 1);
	    } else {
		opt_name = value;
		opt_value = "";
	    }

	    // Tell the database code about the option.
	    flatfile->setOption(opt_name, opt_value);
	} else if (key == "field") {
	    StrOps::string_list_t array;

	    // Retrieve the name, field type, and optional width.
	    try {
		array = StrOps::str_to_array(value, ",", false, false);
	    } catch (const std::exception& e) {
		*err << program << ": bad -f option: " << e.what() << "\n";
		return 1;
	    }

	    // Make sure the user gave the correct number of subarguments.
	    if (array.size() < 2 || array.size() > 3) {
		*err << program
		     << ": -f option takes comma-delimited field descriptor\n";
		return 1;
	    }

	    // Add this described field to the database.
	    try {
		if (array.size() == 2)
		    handle_field(*flatfile, state, array[0], array[1], "");
		else
		    handle_field(*flatfile, state, array[0], 
				 array[1], array[2]);
	    } catch (const std::exception& e) {
		*err << program << ": bad -f option: " << e.what() << "\n";
		return 1;
	    } catch (const std::string& s) {
		*err << program << ": bad -f option: " << s << "\n";
		return 1;
	    }
	}
    }

    // If an implict list view was given and there are also explicit
    // list views, then complain to the user.
    if (flatfile->getNumOfListViews() >= 1
	&& state.implicit_listview.size() > 0) {
	*err << program << ": warning: ignoring implicit list view due to view directives\n";
    }

    // If no explicit list views were specified, then we need to check
    // for an implicit list view from widths given in field arguments.
    if (flatfile->getNumOfListViews() == 0) {
	if (state.implicit_listview.size() == 0) {
	    *err << program << ": at least one list view must be specified\n";
	    return 1;
	} else {
	    try {
		flatfile->appendListView(state.implicit_listview);
	    } catch (const std::exception& e) {
		*err << program << ": " << e.what() << std::endl;
		return 1;
	    }
	}
    }

    // Allow the database type code to perform final checks on the
    // schema. After this call, we promise the database type to not
    // change the schema or any other metadata.
    try {
	flatfile->doneWithSchema();
    } catch (const std::exception& e) {
	*err << program << ": " << e.what() << std::endl;
	return 1;
    }

    // Process the CSV file.
    read_csv_file(*flatfile, csv_fname.c_str(),
		  std::string(1, separator), state.extended_csv_mode);

    // Create the PDB file object and set the creation time.
    PalmLib::File pdb;
    pi_int32_t now = get_current_time();
    pdb.creation_time(now);
    pdb.modification_time(now);
    pdb.backup_time(now);

    // Tell the flat-file database to output to the PDB.
    try {
	flatfile->outputPDB(pdb);
    } catch (...) {
	*err << program << ": error while copying into PDB object\n";
	return 1;
    }

    // Save the database as a PDB file.
    try {
	pdb.save(pdb_fname.c_str());
    } catch (const PalmLib::error& e) {
	*err << pdb_fname << ": " << e.what() << std::endl;
	return 1;
    } catch (const std::exception& e) {
	*err << pdb_fname << ": " << e.what() << std::endl;
    }

    return 0;
}
