/*
 * palm-db-tools: String Parsing Utility Functions
 * Copyright (C) 1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string>
#include <vector>
#include <strstream>
#include <algorithm>

#include <cstring>

#include "util.h"

using namespace std;

string strip_back(const string & str, const string & what)
{
    char *buf = strdup(str.c_str()), *p;

    p = buf + strlen(buf) - 1;

    while (p >= buf  && (find(what.begin(), what.end(), *p) != what.end()))
	*p-- = '\0';

    return string(buf);
}

string strip_front(const string & str, const string & what)
{
    char *buf = strdup(str.c_str()), *p;

    p = buf;
    while (*p && (find(what.begin(), what.end(), *p) != what.end()))
	++p;

    return string(p);
}

vector<string> csv_to_array(const string & str)
{
    enum { STATE_NORMAL, STATE_QUOTES } state;
    vector<string> result;
    string elem;

    state = STATE_NORMAL;
    for (string::const_iterator p = str.begin(); p != str.end(); ++p) {
	switch (state) {
	case STATE_NORMAL:
	    if (*p == '"') {
		state = STATE_QUOTES;
	    } else if (*p == ',') {
		result.push_back(elem);
		elem = "";
	    } else {
		elem += *p;
	    }
	    break;

	case STATE_QUOTES:
	    if (*p == '"') {
		if ((p + 1) != str.end() && *(p+1) == '"') {
		    ++p;
		    elem += '"';
		} else {
		    state = STATE_NORMAL;
		}
	    } else {
		elem += *p;
	    }
	    break;
	}
    }

    switch (state) {
    case STATE_NORMAL:
	result.push_back(elem);
	break;
    case STATE_QUOTES:
	throw csv_parse_error("unterminated quotes");
	break;
    }

    return result;
}

vector<string>
str_to_array(const string & str, const string & delim, bool multiple_delim)
{
    enum { STATE_NORMAL, STATE_QUOTE_DOUBLE, STATE_QUOTE_SINGLE,
	   STATE_BACKSLASH, STATE_BACKSLASH_DOUBLEQUOTE } state;
    vector<string> result;
    string elem;

    state = STATE_NORMAL;
    for (string::const_iterator p = str.begin(); p != str.end(); ++p) {
	switch (state) {
	case STATE_NORMAL:
	    if (*p == '"') {
		state = STATE_QUOTE_DOUBLE;
	    } else if (*p == '\'') {
		state = STATE_QUOTE_SINGLE;
	    } else if (find(delim.begin(), delim.end(), *p) != delim.end()) {
		if (multiple_delim) {
		    ++p;
		    while (p != str.end()
			   && find(delim.begin(), delim.end(), *p)
			   != delim.end()) {
			++p;
		    }
		    --p;
		}
		result.push_back(elem);
		elem = "";
	    } else if (*p == '\\') {
		state = STATE_BACKSLASH;
	    } else {
		elem += *p;
	    }
	    break;

	case STATE_QUOTE_DOUBLE:
	    if (*p == '"')
		state = STATE_NORMAL;
	    else if (*p == '\\')
		state = STATE_BACKSLASH_DOUBLEQUOTE;
	    else
		elem += *p;
	    break;

	case STATE_QUOTE_SINGLE:
	    if (*p == '\'')
		state = STATE_NORMAL;
	    else
		elem += *p;
	    break;
	    
	case STATE_BACKSLASH:
	    elem += *p;
	    state = STATE_NORMAL;
	    break;
	    
	case STATE_BACKSLASH_DOUBLEQUOTE:
	    switch (*p) {
	    case '\\':
		elem += '\\';
		break;
	    case 'n':
		elem += '\n';
		break;
	    case 'r':
		elem += '\r';
		break;
	    case 't':
		elem += '\t';
		break;
	    case '"':
		elem += '"';
		break;
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		string d(1, *p);
		while (p+1 != str.end() && (*(p+1) >= '0' && *(p+1) <= '9')) {
		    d += *(p+1);
		    ++p;
		}
		int num = -1;
		istrstream(d.c_str()) >> num;
		if (num < 0 || num > 255)
		    throw csv_parse_error("bad number");
		elem += static_cast<char> (num & 0xFF);
		break;
	    }
	    state = STATE_QUOTE_DOUBLE;
	    break;
	}
    }

    switch (state) {
    case STATE_NORMAL:
	result.push_back(elem);
	break;
    case STATE_QUOTE_DOUBLE:
    case STATE_QUOTE_SINGLE:
	throw csv_parse_error("unterminated quotes");
	break;
    case STATE_BACKSLASH:
    case STATE_BACKSLASH_DOUBLEQUOTE:
	throw csv_parse_error("character must follow a backslash");
	break;
    }

    return result;
}
