/*
 * palm-db-tools: PDB->CSV conversion tool
 * Copyright (C) 1998,1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <fstream>
#include <string>
#include <strstream>
#include <fstream>

#include <ctype.h>

#include "DBDatabase.h"
#include "MobileDB.h"
#include "SimpleCmdlineParser.h"

using namespace std;

static bool extended_csv_mode = false;
static string program;
static ostream * err = &cerr;

static string
quote_string(string str, bool extended_mode)
{
    string result;

    if (extended_mode) {
	result += '"';
	for (string::iterator c = str.begin(); c != str.end(); ++c) {
	    switch (*c) {
	    case '\\':
		result += '\\';
		result += '\\';
		break;

	    case '\r':
		result += '\\';
		result += 'r';
		break;

	    case '\n':
		result += '\\';
		result += 'n';
		break;

	    case '\t':
		result += '\\';
		result += 't';
		break;

	    case '"':
		result += '\\';
		result += '"';
		break;

	    default:
		if (isprint(*c)) {
		    result += *c;
		} else {
		    ostrstream buf;

		    result += '\\';
		    buf << ( (int) (*c) );
		    result += buf.str();
		}
		break;
	    }
	}
	result += '"';
    } else {
	result += '"';
	for (string::iterator c = str.begin(); c != str.end(); ++c) {
	    if (*c == '"') {
		result += "\"\"";
	    } else if (*c == '\n' || *c == '\r') {
		*err << program << "use extended csv mode for newlines\n";
		exit(1);
	    } else {
		result += *c;
	    }
	}
	result += '"';
    }

    return result;
}

static void output_csv_file(string fname, const DatabaseInterface & db)
{
    int numRecords = db.getNumRecords();
    int numFields = db.getNumFields();

    ofstream csv(fname.c_str());
    if (!csv) {
	*err << "unable to create the CSV file\n";
	exit(1);
    }

    for (int i = 0; i < numRecords; ++i) {
	vector<DatabaseInterface::FieldData> record = db.getRecord(i);
	for (int j = 0; j < numFields; ++j) {
	    if (j != 0) {
		csv << ",";
	    }
	    switch (db.getField(j).type) {
	    case DatabaseInterface::STRING:
		csv << quote_string(record[j].s, extended_csv_mode);
		break;

	    case DatabaseInterface::BOOLEAN:
		if (record[j].b)
		    csv << "true";
		else
		    csv << "false";
		break;

	    case DatabaseInterface::INTEGER:
		csv << record[j].i;
		break;

	    case DatabaseInterface::DATE:
	    case DatabaseInterface::TIME:
	    case DatabaseInterface::FLOAT:
		csv << "N/A";
		break;
	    }
	}
	csv << endl;
    }

    // Close the CSV file.
    csv.close();
}

static string getTypeName(DatabaseInterface::FieldType t)
{
    switch (t) {
    case DatabaseInterface::STRING:
	return "string";
    case DatabaseInterface::INTEGER:
	return "integer";
    case DatabaseInterface::BOOLEAN:
	return "boolean";
    case DatabaseInterface::DATE:
	return "date";
    case DatabaseInterface::TIME:
	return "time";
    case DatabaseInterface::FLOAT:
	return "float";
    }
    return "unknown";
}

static void
output_info_file(const string & fname, const DatabaseInterface & db)
{
    // Open the output file.
    ofstream info(fname.c_str());
    if (!info) {
	*err << program << ": unable to open metadata file\n";
	exit(1);
    }

    /* Output the database structure. */
    int numFields = db.getNumFields();
    for (int i = 0; i < numFields; ++i) {
	const DatabaseInterface::FieldInfo & field = db.getField(i);

	info << "field \"" << field.name << "\" " << getTypeName(field.type);
	info << " " << field.width << endl;
    }

    /* Output the database title. */
    info << "title " << quote_string(db.getName(), extended_csv_mode) << "\n";

    /* Output the backup bit status. */
    if (db.getBackupFlag())
	info << "backup on\n";
    else
	info << "backup off\n";

    /* Output the global find disable bit */
    if (db.getFindDisabledFlag())
	info << "find off\n";
    else
	info << "find on\n";

    /* Output extended CSV mode. */
    if (extended_csv_mode)
	info << "extended on\n";
    else
	info << "extended off\n";

    // Close the output file.
    info.close();
}

static void usage(void)
{
    cout << "usage: " << program << " [options] PDB_FILE CSV_FILE INFO_FILE\n";
    cout << "  -e, --extended     Output records using extended CSV mode\n";
    cout << "  -h, --help         Display this help screen\n";
    cout << "  -v, --version      Display program version\n";
    cout << "  -m, --mobiledb     Force PDB to be read in MobileDB-format.\n";
    cout << "  -d, --db           Force PDB to be read in DB-format.\n";
    cout << "  -n FILE\n";
    cout << "      --errors=FILE  Send all error messages to FILE\n";
    exit(0);
}

class MyParser : public SimpleCmdlineParser
{
public:
    enum { FMT_DETERMINE, FMT_DB, FMT_MOBILEDB } fmt;
    string pdb_fname;
    string csv_fname;
    string info_fname;

    MyParser() : fmt(FMT_DETERMINE), pdb_fname(), csv_fname(),
	info_fname() { }

    virtual const SimpleCmdlineParser::OptionMapping * getOptionMapping();

    virtual void foundOption(const CmdlineParser::OptionDescription * descr,
			     const string & opt,
			     const string & value);
    
    virtual void normalArguments(const vector<string> & args);
};

const SimpleCmdlineParser::OptionMapping * MyParser::getOptionMapping()
{
  static const SimpleCmdlineParser::OptionMapping mappings[] = {
      { 'e', "extended", { 'e', false } },
      { 'h', "help", { 'h', false } },
      { 'd', "db", { 'd', false } },
      { 'm', "mobiledb", { 'm', false } },
      { 'v', "version", { 'v', false } },
      { 'n', "errors", { 'n', true } },
      { '\0', "", { '\0', false } }
  };

  return &mappings[0];
}

void MyParser::foundOption(const CmdlineParser::OptionDescription * descr,
                           const string & opt, const string & value)
{
    switch (descr->cookie) {
    case 'e':
	extended_csv_mode = true;
	break;
    case 'd':
	fmt = FMT_DB;
	break;
    case 'm':
	fmt = FMT_MOBILEDB;
	break;
    case 'h':
	usage();
	break;
    case 'v':
	cout << "pdb2csv (" << PACKAGE << ' ' << VERSION << ')' << endl;
	exit(0);
	break;
    case 'n':
        ofstream * f = new ofstream(value.c_str());
        if (!f) {
            *err << program << ": unable to open error file '"
                 << value << "'\n";
            break;
        }
        err = f;
        break;
    }
}

void MyParser::normalArguments(const vector<string> & args)
{
    if (args.size() != 3) {
	usage();
    }

    pdb_fname = args[0];
    csv_fname = args[1];
    info_fname = args[2];
}

DatabaseInterface *
getDatabaseObject(const string & fname)
{
    ifstream f;
    pi_char_t buf[78];

    f.open(fname.c_str(), ios::in | ios::binary);
    if (!f) return 0;

    // Read enough of the file so we can figure out what object to use.
    f.read((char *) buf, 78);
    if (!f) return 0;

    f.close();

    // Extract the creator and type information.
    pi_int32_t creator = get_long(buf + 64);
    pi_int32_t type = get_long(buf + 60);

    if (creator == PalmDatabase::mktag('D','B','O','S')) {
	return new DBDatabase();
    } else if (creator == PalmDatabase::mktag('T','K','D','3')
	       && type == PalmDatabase::mktag('D','B','9','9')) {
	return new DBDatabase();
    } else if (creator == PalmDatabase::mktag('M','d','b','1')) {
	return new MobileDB();
    } else {
	return 0;
    }

}

int main(int argc, char *argv[])
{
    MyParser parser;
    DatabaseInterface * db = 0;

    program = parser.program = argv[0];

    if (! parser.parse(argc - 1, argv + 1)) {
	usage();
	return 1;
    }

    // Instantiate an object based on parameters.
    switch (parser.fmt) {
    case MyParser::FMT_DB:
	db = new DBDatabase();
	break;
    case MyParser::FMT_MOBILEDB:
	db = new MobileDB();
	break;
    case MyParser::FMT_DETERMINE:
	db = getDatabaseObject(parser.pdb_fname);
	if (!db) {
	    *err << argv[0] << ": unable to determine database type\n";
	    return 1;
	}
	break;
    }

    // Open the PDB file.
    try {
	db->load(parser.pdb_fname.c_str());
    } catch (PalmDatabase::error e) {
	*err << argv[0] << ": " << parser.pdb_fname << ": ";
	*err << e.what() << endl;
	return 1;
    }

    // Output the metadata file.
    output_info_file(parser.info_fname, *db);

    // Output all of the records.
    output_csv_file(parser.csv_fname, *db);

    return 0;
}
