/*
 * palm-db-tools: CSV->PDB conversion tool
 * Copyright (C) 1998,1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <fstream>
#include <string>
#include <cctype>
#include <strstream>
#include <stdexcept>

#ifndef WIN32
# if TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
# else
#  if HAVE_SYS_TIME_H
#   include <sys/time.h>
#  else
#   include <time.h>
#  endif
# endif
#else
# include <time.h>
#endif

#include "DBDatabase.h"
#include "MobileDB.h"
#include "SimpleCmdlineParser.h"

#include "util.h"

using namespace std;

static string program;
static bool extended_csv_mode = false;
static ostream * err = &cerr;

void strlower(string & str)
{
    for (string::iterator p = str.begin(); p != str.end(); ++p) {
	if (isupper(*p))
	    *p = tolower(*p);
    }
}

bool string2boolean(string & str)
{
    strlower(str);
    if (str == "on")
	return true;
    else if (str == "off")
	return false;
    else if (str == "true")
	return true;
    else if (str == "t")
	return true;
    else if (str == "false")
	return false;
    else if (str == "f")
	return false;
    else {
	int num = 0;

	istrstream(str.c_str()) >> num;
	return num != 0 ? true : false;
    }
}

DatabaseInterface::FieldType string2type(string typestr)
{
    strlower(typestr);
    if (typestr == "string")
	return DatabaseInterface::STRING;
    else if (typestr == "str")
	return DatabaseInterface::STRING;
    else if (typestr == "bool")
	return DatabaseInterface::BOOLEAN;
    else if (typestr == "boolean")
	return DatabaseInterface::BOOLEAN;
    else if (typestr == "integer")
	return DatabaseInterface::INTEGER;
    else if (typestr == "int")
	return DatabaseInterface::INTEGER;
    else
	throw invalid_argument("unknown field type");
}

void read_csv_file(DatabaseInterface & db, const char * fname)
{
    char buf[1024];
    
    // Open the CSV file.
    ifstream f(fname);
    if (!f)
	return;

    int linenum = 0;
    while (1) {
	// Read the next line from the file.
	f.getline(buf, sizeof(buf));
	if (!f)
	    break;
	++linenum;
	string line = buf;

	// Strip the trailing newline character.
	line = strip_back(line, "\r\n");

	// Convert the text into an argv-style array.
	vector<string> array;
	try {
	    if (extended_csv_mode)
		array = str_to_array(line, ",", false);
	    else
		array = csv_to_array(line);
	} catch (csv_parse_error e) {
	    *err << fname << ':' << linenum << ": " << e.what() << endl;
	    exit(1);
	}

	/* Make sure that the correct number of fields were received. */
	if (array.size() != db.getNumFields()) {
	    *err << fname << ':' << linenum << ": number of fields doesn't match\n";
	    exit(1);
	}

	vector<DatabaseInterface::FieldData> record;
	for (int i = 0; i < db.getNumFields(); i++) {
	    DatabaseInterface::FieldData data;

	    switch (db.getField(i).type) {
	    default:
		break;

	    case DatabaseInterface::STRING:
		data.type = DatabaseInterface::STRING;
		data.s = array[i];
		break;

	    case DatabaseInterface::BOOLEAN:
		data.type = DatabaseInterface::BOOLEAN;
		data.b = string2boolean(array[i]);
		break;

	    case DatabaseInterface::INTEGER:
		data.type = DatabaseInterface::INTEGER;
		const char *s = array[i].c_str();
		istrstream(s) >> data.i;
		break;

	    }

	    record.push_back(data);
	}
	
	// Add the record to the database.
	db.addRecord(record);
    }

    f.close();
}

string handle_field(DatabaseInterface & db,
		    const string & name,
		    const string & typestr,
		    const string & widthstr)
{
    DatabaseInterface::FieldInfo info;

    info.name = name;
    try {
	info.type = string2type(typestr);
    } catch (invalid_argument e) {
	return e.what();
    }
    istrstream(widthstr.c_str()) >> info.width;
    if (info.width < 10 || info.width > 160)
	return "column widths must between 10 and 160";

    db.addField(info);

    return "";
}

void read_info_file(DatabaseInterface & db, const char * fname)
{
    char buf[2048];
    int linenum;

#define errorout(s) *err << fname << ':' << linenum << ": " << (s) << endl

    // Open the information file.
    ifstream f(fname);
    if (!f) {
	*err << program << "unable to open '" << fname << "'\n";
	exit(1);
    }

    linenum = 0;
    while (1) {
	// Read the next line into the buffer.
	f.getline(buf, sizeof(buf));
	if (!f)
	    break;
	++linenum;
	string line = buf;

	// Strip any trailing newline characters. */
	line = strip_back(line, "\r\n");

	// Strip trailing whitespace.
	line = strip_back(line, " \t");

	// Strip leading whitespace.
	line = strip_front(line, " \t");

	/* Skip this line if it is blank. */
	if (line.length() == 0)
	    continue;
	
	/* Parse the line into an argv-style array. */
	vector<string> array = str_to_array(line, " \t", true);
	    
	strlower(array[0]);
	if (array[0] == "title") {
	    if (array.size() != 2) {
		errorout("title only takes 1 parameter");
		exit(1);
	    }
	    db.setName(array[1].c_str());
	} else if (array[0] == "field") {
	    string errstr;

	    // Make sure we received the correct number of arguments.
	    if (array.size() < 3 || array.size() > 4) {
		errorout("bad field directive");
		exit(1);
	    }
	    
	    if (array.size() == 3)
		errstr = handle_field(db, array[1], array[2], "80");
	    else
		errstr = handle_field(db, array[1], array[2], array[3]);
	    if (errstr.length() > 0) {
		errorout(errstr);
		exit(1);
	    }
	} else if (array[0] == "extended") {
	    if (array.size() != 2) {
		errorout("the extended directive takes 1 argument");
		exit(1);
	    }
	    extended_csv_mode = string2boolean(array[1]);
	} else if (array[0] == "backup") {
	    if (array.size() != 2) {
		errorout("the backup directive takes 1 argument");
		exit(1);
	    }

	    db.setBackupFlag(string2boolean(array[1]));
	} else if (array[0] == "find") {
	    if (array.size() != 2) {
		errorout("the find directive takes 1 argument");
		exit(1);
	    }
	    db.setFindDisabledFlag(! string2boolean(array[1]));
	}
    }

#undef errorout

    f.close();
}

void field_directive(DatabaseInterface & db, const string & str)
{
    string errstr;

    vector<string> array = str_to_array(str, ",", false);
    if (array.size() < 2 || array.size() > 3) {
	*err << program << "-f option takes comma-delimited field description\n";
	exit(1);
    }

    if (array.size() == 2)
	errstr = handle_field(db, array[0], array[1], "80");
    else
	errstr = handle_field(db, array[0], array[1], array[2]);
    if (errstr.length() > 0) {
	*err << program << ": -f option error: " << errstr << endl;
	exit(1);
    }
}

void usage(void)
{
    cout << "usage: " << program <<  ": [-d|-m] [options] CSV PDB\n";
    cout << "       -b, --backup       Set pdb backup bit\n";
    cout << "       -e, --extended     Use extended csv mode\n";
    cout << "       -t TITLE\n";
    cout << "           --title=TITLE  Set database title\n";
    cout << "       -i FILE\n";
    cout << "           --info=FILE    Read database metadeta from FILE\n";
    cout << "       -f SPEC\n";
    cout << "           --field=SPEC   Add field described by SPEC to the\n";
    cout << "                          database. SPEC format: NAME,TYPE,[WIDTH]\n";
    cout << "       -n FILE\n";
    cout << "           --errors=FILE  Send all error messages to FILE\n";
    cout << "       -d, --db           Output a DB format file.\n";
    cout << "       -m, --mobiledb     Output a MobileDB format file.\n";
    cout << "       -h, --help         Display this help screen\n";
    cout << "       -v, --version      Display program version\n";
    exit(1);
}


class MyParser : public SimpleCmdlineParser
{
public:
    DatabaseInterface *dbP;
    string csv_fname;
    string pdb_fname;
    bool sawDBOption, sawRegularOption;

    MyParser() : sawDBOption(false), sawRegularOption(false)
	{ dbP = new DBDatabase(); }
    virtual const OptionMapping * getOptionMapping();
    virtual void foundOption(const CmdlineParser::OptionDescription * descr,
                             const string & opt,
                             const string & value);
    
    virtual void normalArguments(const vector<string> & args);
};

const SimpleCmdlineParser::OptionMapping * MyParser::getOptionMapping()
{
    static const SimpleCmdlineParser::OptionMapping mappings[] = {
	{ 'b', "backup", { 'b', false } },
	{ 'e', "extended", { 'e', false } },
	{ 't', "title", { 't', true } },
	{ 'i', "info", { 'i', true } },
	{ 'f', "field", { 'f', true } },
	{ 'm', "mobiledb", { 'm', false } },
	{ 'd', "db", { 'd', false } },
	{ 'h', "help", { 'h', false } },
	{ 'v', "version", { 'v', false } },
	{ 'n', "errors", { 'n', true } },
	{ '\0', "", { '\0', false } }
    };

    return &mappings[0];
}

void MyParser::foundOption(const CmdlineParser::OptionDescription * descr,
                           const string & opt, const string & value)
{
    switch (descr->cookie) {
    case 'b':
	dbP->setBackupFlag(true);
	sawRegularOption = true;
	break;

    case 'e':
	extended_csv_mode = true;
	sawRegularOption = true;
	break;

    case 't':
	dbP->setName(value.c_str());
	sawRegularOption = true;
	break;

    case 'i':
	read_info_file(*dbP, value.c_str());
	sawRegularOption = true;
	break;

    case 'f':
	field_directive(*dbP, value);
	sawRegularOption = true;
	break;

    case 'm':
    case 'd':
	if (sawRegularOption) {
	    *err << program << ": " << opt << " must appear first\n";
	    exit(1);
	}
	if (sawDBOption) {
	    *err << program << ": the -m and -d options are mutually exclusive\n";
	    exit(1);
	}

	// Create the database object that the user wants.
	switch (descr->cookie) {
	case 'm':
	    dbP = new MobileDB();
	    break;
	case 'd':
	    dbP = new DBDatabase();
	    break;
	}
	sawDBOption = true;

	break;

    case 'h':
	usage();
	sawRegularOption = true;
	break;

    case 'v':
	cout << "csv2pdb (" << PACKAGE << ' ' << VERSION << ")" << endl;
	sawRegularOption = true;
	exit(0);
	break;

    case 'n':
	ofstream * f = new ofstream(value.c_str());
	if (!f) {
	    *err << program << ": unable to open error file '"
		 << value << "'\n";
	    break;
	}
	err = f;
	break;
    }
}

void MyParser::normalArguments(const vector<string> & args)
{
    if (args.size() != 2) {
        usage();
    }

    csv_fname = args[0];
    pdb_fname = args[1];
}

pi_int32_t get_current_time(void)
{
    time_t now;

    ::time(&now);
    return pi_int32_t(now) + pi_int32_t(2082844800);
}

int main(int argc, char *argv[])
{
    MyParser parser;

    parser.program = program = argv[0];

    if (! parser.parse(argc - 1, argv + 1)) {
	usage();
	return 1;
    }

    /* Make sure that we have something to do. */
    if (parser.dbP->getNumFields() == 0) {
	*err << program << ": you must specify field definitions\n";
	return 1;
    } else if (strlen(parser.dbP->getName()) == 0) {
	*err << program << ": you must specify a title for the database\n";
	return 1;
    }

    // Process the CSV file.
    read_csv_file(*(parser.dbP), parser.csv_fname.c_str());

    // Set the creation time for the PDB file.
    pi_int32_t now = get_current_time();
    parser.dbP->setCreationTime(now);
    parser.dbP->setModificationTime(now);
    parser.dbP->setBackupTime(now);

    // Save the database as a PDB file.
    try {
	parser.dbP->save(parser.pdb_fname.c_str());
    } catch (PalmDatabase::error e) {
	*err << program << ": " << parser.pdb_fname << ": " << e.what() << endl;
	return 1;
    }

    return 0;
}
