/*
 * This file contains a class to read and write PDB/PRC files used by
 * a Palm Pilot.
 */

#ifndef _PALMDATABASE_H
#define _PALMDATABASE_H

#include <vector>
#include <string>
#include <stdexcept>

#include "palmtypes.h"
#include "PalmRecord.h"

class PalmDatabase
{
 public:
    class error : public std::runtime_error {
    public:
	error(const std::string & what_arg) : std::runtime_error(what_arg) { }
    };

    static pi_int32_t mktag(pi_char_t c1, pi_char_t c2,
			    pi_char_t c3, pi_char_t c4)
	{ return (((c1)<<24)|((c2)<<16)|((c3)<<8)|(c4)); }

    PalmDatabase(bool resourceDB = false);
    virtual ~PalmDatabase();

    // Load a PDB file into the object.
    virtual void load(const char * fname);

    // Save the object to a PDB file.
    virtual void save(const char * fname);

    // Accessor functions for header data.
    const char * getName() const { return name; }
    void setName(const char *);

    pi_int32_t getType() const { return type; }
    void setType(pi_int32_t t) { type = t; }

    pi_int32_t getCreator() const { return creator; }
    void setCreator(pi_int32_t c) { creator = c; }

    pi_int16_t getVersion() const { return version; }
    void setVersion(pi_int16_t v) { version = v; }

    pi_int32_t getCreationTime() const { return creation_time; }
    void setCreationTime(pi_int32_t ct) { creation_time = ct; }

    pi_int32_t getModificationTime() const { return modification_time; }
    void setModificationTime(pi_int32_t mt) { modification_time = mt; }

    pi_int32_t getBackupTime() const { return backup_time; }
    void setBackupTime(pi_int32_t bt) { backup_time = bt; }

    bool getBackupFlag() const { return (flags & dlpDBFlagBackup) != 0; }
    void setBackupFlag(bool state) {
	if (state)
	    flags |= dlpDBFlagBackup;
	else
	    flags &= ~(dlpDBFlagBackup);
    }

    void getAppInfo(pi_char_t * & data, int & size)
	{ data = app_info; size = app_info_size; }
    void setAppInfo(pi_char_t * data, int size);

    void getSortInfo(pi_char_t * & data, int & size)
	{ data = sort_info; size = sort_info_size; }
    void setSortInfo(pi_char_t * data, int size);

 private:
    char name[32];
    pi_int16_t flags;
    pi_int16_t version;
    pi_int32_t creation_time;
    pi_int32_t modification_time;
    pi_int32_t backup_time;
    pi_int32_t modification;
    pi_int32_t type;
    pi_int32_t creator;
    pi_int32_t unique_id_seed;

    pi_int32_t next_record_list_id;
    pi_int32_t ent_hdr_size;
    bool is_resource_db;

    pi_char_t * app_info;
    int app_info_size;

    pi_char_t * sort_info;
    int sort_info_size;

 protected:
    std::vector<PalmRecord *> records;
};

#endif
