/*
 * palm-db-tools: Generic Palm Pilot database routines
 * Copyright (C) 1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <memory.h>

#include <iostream>
#include <fstream>

#include "PalmDatabase.h"

using namespace std;


/*
 * Routines to read/write a Palm Pilot database file.
 */

static const int PI_HDR_SIZE          = 78;
static const int PI_RESOURCE_ENT_SIZE = 10;
static const int PI_RECORD_ENT_SIZE   = 8;

PalmDatabase::PalmDatabase(bool resourceDB)
{
    memset(name, 0, sizeof(name));

    version = 0;
    creation_time = 0;
    modification_time = 0;
    backup_time = 0;
    modification = 0;
    type = 0;
    creator = 0;
    unique_id_seed = 0;
    next_record_list_id = 0;

    app_info = 0;
    app_info_size = 0;

    sort_info = 0;
    sort_info_size = 0;

    is_resource_db = resourceDB;
    flags = is_resource_db ? dlpDBFlagResource : 0;
    ent_hdr_size = is_resource_db ? PI_RESOURCE_ENT_SIZE : PI_RECORD_ENT_SIZE;
}

PalmDatabase::~PalmDatabase()
{
    if (app_info)
	delete app_info;

    if (sort_info)
	delete sort_info;
}

void PalmDatabase::load(const char * fname)
{
    ifstream f;
    pi_char_t buf[PI_HDR_SIZE];
    pi_int32_t app_info_offset, sort_info_offset;
    streampos file_size;
    pi_int32_t offset;

    f.open(fname, ios::in | ios::binary);
    if (!f)
	throw error("unable to open the file");

    // Determine the size of the file.
    f.seekg(0, ios::end);
    file_size = f.tellg();
    f.seekg(0, ios::beg);

    // Read and parse the PDB header.
    f.read((char *) buf, PI_HDR_SIZE);
    if (!f)
	throw error("unable to read header");
    memcpy(name, buf, 32);
    flags = get_short(buf + 32);
    version = get_short(buf + 34);
    creation_time = get_long(buf + 36);
    modification_time = get_long(buf + 40);
    backup_time = get_long(buf + 44);
    modification = get_long(buf + 48);
    app_info_offset = get_long(buf + 52);
    sort_info_offset = get_long(buf + 56);
    type = get_long(buf + 60);
    creator = get_long(buf + 64);
    unique_id_seed = get_long(buf + 68);

    next_record_list_id = get_long(buf + 72);
    pi_int16_t num_entries = get_short(buf + 76);

    if (flags & dlpDBFlagResource) {
	is_resource_db = true;
	ent_hdr_size = PI_RESOURCE_ENT_SIZE;
    } else {
	is_resource_db = false;
	ent_hdr_size = PI_RECORD_ENT_SIZE;
    }

    // Read all of the record entries into the object.
    for (int i = 0; i < num_entries; ++i) {
	pi_char_t *hdr = new pi_char_t[ent_hdr_size];
	PalmRecord * entry = new PalmRecord;

	f.read((char *) hdr, ent_hdr_size);
	if (!f)
	    throw error("unable to read record header");

	if (is_resource_db) {
	    entry->type = get_long(hdr);
	    entry->id = get_short(hdr + 4);
	    entry->offset = get_long(hdr + 6);
	} else {
	    entry->offset = get_long(hdr);
	    entry->attrs = hdr[4];
	    entry->uid = get_treble(hdr + 5);
	}

	records.push_back(entry);
	delete [] hdr;
  }

    // Now we iterate in reverse over the vector so figure out the size
    // of each entry.
    offset = file_size;
    for (vector<PalmRecord *>::reverse_iterator p = records.rbegin(); 
	 p != records.rend(); ++p) {
	PalmRecord * entry = *p;
	entry->data_size = offset - entry->offset;
	offset = entry->offset;
    }

    // Determine the size of the sort info block.
    if (sort_info_offset > 0) {
	sort_info_size = offset - sort_info_offset;
	offset = sort_info_offset;
    }

    // Determine the size of the app info block.
    if (app_info_offset > 0) {
	app_info_size = offset - app_info_offset;
	offset = app_info_offset;
    }

    // Read the app info block.
    if (app_info_size > 0) {
	app_info = new pi_char_t[app_info_size];
	f.seekg(app_info_offset);
	f.read((char *) app_info, app_info_size);
	if (!f)
	    throw error("unable to read application info block");
    } else {
	app_info = 0;
    }

    // Read the sort info block.
    if (sort_info_size > 0) {
	sort_info = new pi_char_t[sort_info_size];
	f.seekg(sort_info_offset);
	f.read((char *) sort_info, sort_info_size);
	if (!f)
	    throw error("unable to read sort info block");
    } else {
	sort_info = 0;
    }

    // Read all of the records into memory.
    for (vector<PalmRecord *>::iterator q = records.begin();
	 q != records.end(); ++q) {
	PalmRecord * entry = *q;
	entry->data = new pi_char_t[entry->data_size];
	f.seekg(entry->offset);
	f.read((char *) entry->data, entry->data_size);
	if (!f)
	    throw error("unable to read record data");
    }

    f.close();
}

void PalmDatabase::save(const char * fname)
{
    ofstream f;
    pi_char_t buf[PI_HDR_SIZE], *p;
    streampos offset;

    f.open(fname, ios::out | ios::binary);
    if (!f)
	throw error("unable to open file");

    offset = PI_HDR_SIZE + records.size() * ent_hdr_size + 2;

    p = buf;
    memcpy(p, name, 32);
    set_short(p + 32, flags);
    set_short(p + 34, version);
    set_long(p + 36, creation_time);
    set_long(p + 40, modification_time);
    set_long(p + 44, backup_time);
    set_long(p + 48, modification);
    set_long(p + 52, app_info_size ? offset : 0); offset += app_info_size;
    set_long(p + 56, sort_info_size ? offset : 0); offset += sort_info_size; 
    set_long(p + 60, type);
    set_long(p + 64, creator);
    set_long(p + 68, unique_id_seed);
    set_long(p + 72, next_record_list_id);
    set_short(p + 76, records.size());

    f.write((char *) buf, PI_HDR_SIZE);
    if (!f)
	throw error("unable to write header");

    int unique_id = 0;
    for (vector<PalmRecord *>::iterator i = records.begin();
	 i != records.end(); ++i) {
	PalmRecord * entry = *i;

	entry->offset = offset;
	entry->uid = unique_id; unique_id++;

	p = buf;
	if (is_resource_db) {
	    set_long(p, entry->type);
	    set_short(p + 4, entry->id);
	    set_long(p + 6, entry->offset);
	} else {
	    set_long(p, entry->offset);
	    p[4] = entry->attrs;
	    set_treble(p + 5, entry->uid);
	}

	f.write((char *) buf, ent_hdr_size);
	if (!f)
	    throw error("unable to write record header");

	offset += entry->data_size;
    }

    f.write("\0", 1);
    f.write("\0", 1);

    if (app_info) {
	f.write((char *) app_info, app_info_size);
	if (!f)
	    throw error("unable to write application info block");
    }

    if (sort_info) {
	f.write((char *) sort_info, sort_info_size);
	if (!f)
	    throw error("unable to write sort info block");
    }

    for (vector<PalmRecord *>::iterator q = records.begin();
	 q != records.end(); ++q) {
	PalmRecord * entry = *q;
	f.write((char *) entry->data, entry->data_size);
	if (!f)
	    throw error("unable to write records");
    }

    f.close();
}

void PalmDatabase::setAppInfo(pi_char_t * data, int size)
{
    if (app_info) {
	delete app_info;
	app_info = 0;
	app_info_size = 0;
    }
    if (data && size > 0) {
	app_info = new pi_char_t[size];
	memcpy(app_info, data, size);
	app_info_size = size;
    }
}

void PalmDatabase::setSortInfo(pi_char_t * data, int size)
{
    if (sort_info) {
	delete sort_info;
	sort_info = 0;
	sort_info_size = 0;
    }
    if (data && size > 0) {
	sort_info = new pi_char_t[size];
	memcpy(sort_info, data, size);
	sort_info_size = size;
    }
}

void PalmDatabase::setName(const char * n)
{
    strncpy(name, n, 32);
    name[sizeof(name)-1] = '\0';
}
