/*
 * palm-db-tools: Read/write MobileDB databases
 * Copyright (C) 1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <cstdlib>
#include <cstring>
#include <stdexcept>
#include <strstream>
#include <algorithm>

#include <memory.h>

#include "MobileDB.h"

using namespace std;

const int MobileDB::mdbMaxFields                    = 20;

const pi_char_t MobileDB::mdbCategoryFieldLabels     = 1;
const pi_char_t MobileDB::mdbCategoryDataRecords     = 2;
const pi_char_t MobileDB::mdbCategoryDataRecordsFout = 3;
const pi_char_t MobileDB::mdbCategoryPreferences     = 4;
const pi_char_t MobileDB::mdbCategoryDataType        = 5;
const pi_char_t MobileDB::mdbCategoryFieldLengths    = 6;

MobileDB::MobileDB()
{
    setType(mktag('M','d','b','1'));
    setCreator(mktag('M','d','b','1'));
}

int MobileDB::find_metadata_by_category(pi_char_t theCategory)
{
    int foundRecNum;

    foundRecNum = -1;
    for (int i = 0; i < records.size(); ++i) {
        if (records[i]->category() == theCategory) {
            if (foundRecNum != -1) {
            }
            foundRecNum = i;
        }
    }

    return foundRecNum;
}

void MobileDB::parse_record(PalmRecord * record, vector<pi_char_t *> & ptrs,
			    vector<int> & sizes) const
{
    int fieldNum;
    bool reachedEnd;
    pi_char_t *p, *q;

    /* Verify that record header exists. */
    if (record->data_size < 6) {
        throw domain_error("record is too small");
    }

    /* Verify that the header fields are correct. */
    if (record->data[3] != 0x01 || record->data[5] != 0x00) {
        throw domain_error("record header type not supported");
    }

    /* Point at the start of the record. */
    p = record->data + 6;
    fieldNum = 0;
    reachedEnd = false;
    while (p < record->data + record->data_size) {
        /* Make sure that there is a terminator. */
        if (*p != 0) {
            throw domain_error("could not find record terminator");
        }
        p++;

        /* Check the field number. */
        if ( (static_cast<int> (*p)) == 0xFF) {
            reachedEnd = true;
            break;
        } else if ( (static_cast<int> (*p)) != fieldNum) {
            throw domain_error("field number mismatch");
        }
        p++;

        /* Mark down the start of this field. */
        ptrs.push_back(p);

        /* Search for the end of this field. */
        q = (pi_char_t *) memchr((void *) p, 0x00, record->data_size);
        if (!q || q > record->data + record->data_size) {
            throw domain_error("could not find end of field\n");
	}

        /* Mark down the size of this field. */
	sizes.push_back( (static_cast<int> (q - p)) );

        /* Point at the field terminator. */
        p = q;

        /* Advance the field number. */
        fieldNum++;
        if (fieldNum > mdbMaxFields + 1) {
	    throw domain_error("maximum number of fields exceeded");
        }
    }

    /* Make sure that we are at the very end of the string. */
    if (! reachedEnd) {
	throw domain_error("no record end marker");
    }
}

void MobileDB::make_appinfo(const MobileAppInfoType & mai)
{
    int len = 2 + 16 * 16 + 16 + 1 + 1 + 2, i;
    pi_char_t *buf = new pi_char_t[len], *p = buf;

    set_short(p, mai.renamedCategories);
    p += 2;

    for (i = 0; i < 16; ++i) {
	memcpy(p, mai.categoryLabels[i], 16);
	p += 16;
    }
    for (i = 0; i < 16; ++i) {
	*p++ = mai.categoryUniqIDs[i];
    }
    *p++ = mai.lastUniqID;
    *p++ = mai.reserved1;
    set_short(p, mai.reserved2);
    p += 2;

    setAppInfo(buf, len);
    delete [] buf;
}

void MobileDB::load(const char *fname)
{
    vector<pi_char_t *> name_ptrs, type_ptrs, width_ptrs;
    vector<int> name_sizes, type_sizes, width_sizes;
    int recNum;

    // Let the superclass load the raw database.
    PalmDatabase::load(fname);

    // Verify that the type and creator match.
    if (getType() != mktag('M','d','b','1')
	|| getCreator() != mktag('M','d','b','1'))
	throw error("not a MobileDB database");

    recNum = find_metadata_by_category(mdbCategoryFieldLabels);
    parse_record(records[recNum], name_ptrs, name_sizes);

    recNum = find_metadata_by_category(mdbCategoryDataType);
    parse_record(records[recNum], type_ptrs, type_sizes);

    recNum = find_metadata_by_category(mdbCategoryFieldLengths);
    parse_record(records[recNum], width_ptrs, width_sizes);

    for (int i = 0; i < name_ptrs.size(); ++i) {
	FieldInfo data;

	data.name = string((char *) name_ptrs[i], name_sizes[i]);
	data.type = STRING;

	istrstream buf((char *) width_ptrs[i], width_sizes[i]);
	buf >> data.width;

	fields.push_back(data);
    }
}

void MobileDB::save(const char *fname)
{
    PalmRecord * record;
    vector<string> v;
    vector<PalmRecord *> tmp;
    int i;

    // Create the record which contains the field names.
    for (i = 0; i < fields.size(); ++i) {
	v.push_back(fields[i].name);
    }
    record = new PalmRecord();
    make_record(*record, v);
    record->category(mdbCategoryFieldLabels);
    tmp.push_back(record);

    // Create the record which contains the field types.
    v.clear();
    for (i = 0; i < fields.size(); ++i) {
	v.push_back(string("str"));
    }
    record = new PalmRecord();
    make_record(*record, v);
    record->category(mdbCategoryDataType);
    tmp.push_back(record);

    // Create the record which contains the widths.
    v.clear();
    for (i = 0; i < fields.size(); ++i) {
	ostrstream s;
	s << fields[i].width;
	v.push_back(string(s.str()));
    }
    record = new PalmRecord();
    make_record(*record, v);
    record->category(mdbCategoryFieldLengths);
    tmp.push_back(record);

    // Create the preferences record.
    v.clear();
    for (i = 0; i < fields.size(); ++i) {
	v.push_back(string(1, char(0x01)));
    }
    record = new PalmRecord();
    make_record(*record, v);
    record->category(mdbCategoryPreferences);
    tmp.push_back(record);

    // Insert these new records into the front of the database.
    records.insert(records.begin(), tmp.begin(), tmp.end());

    // Fill in the application info block.
    MobileAppInfoType hdr;
    hdr.renamedCategories = 0;
    strncpy(hdr.categoryLabels[0], "Unfiled         ", 16);
    strncpy(hdr.categoryLabels[1], "FieldLabels     ", 16);
    strncpy(hdr.categoryLabels[2], "DataRecords     ", 16);
    strncpy(hdr.categoryLabels[3], "DataRecordsFout ", 16);
    strncpy(hdr.categoryLabels[4], "Preferences     ", 16);
    strncpy(hdr.categoryLabels[5], "DataType        ", 16);
    strncpy(hdr.categoryLabels[6], "FieldLengths    ", 16);
    for (i = 7; i < 16; ++i) {
        memset(hdr.categoryLabels[i], 0, sizeof(hdr.categoryLabels[i]));
    }
    for (i = 0; i < 16; i++) {
        hdr.categoryUniqIDs[i] = i;
    }
    hdr.lastUniqID = 16 - 1;
    hdr.reserved1 = 0x0;
    hdr.reserved2 = 0x0;

    make_appinfo(hdr);

    PalmDatabase::save(fname);
}

int MobileDB::getNumFields() const
{
    return fields.size();
}

const DatabaseInterface::FieldInfo & MobileDB::getField(int i) const
{
    return fields[i];
}

void MobileDB::addField(const DatabaseInterface::FieldInfo & info)
{
    switch (info.type) {
    case STRING:
	break;
    default:
	throw invalid_argument("unsupported field type");
    }

    fields.push_back(info);
}

bool MobileDB::supportsFieldType(const FieldType & type)
{
    if (type == STRING)
	return true;
    else
	return false;
}

int MobileDB::getNumRecords() const
{
    int count = 0;

    for (vector<PalmRecord *>::const_iterator i = records.begin();
	 i != records.end(); ++i) {
	switch ((*i)->category()) {
	case mdbCategoryDataRecords:
	case mdbCategoryDataRecordsFout:
	    ++count;
	    break;
	}
    }

    return count;
}

vector<DatabaseInterface::FieldData> MobileDB::getRecord(int n) const
{
    vector<pi_char_t *> ptrs;
    vector<int> sizes;
    vector<FieldData> record;
    vector<PalmRecord *>::const_iterator i;

    for (i = records.begin(); i != records.end(); ++i) {
	pi_char_t cat = (*i)->category();

	if (cat == mdbCategoryDataRecords
	    || cat == mdbCategoryDataRecordsFout) {
	    if (n == 0)
		break;
	    --n;
	}
    }

    parse_record(*i, ptrs, sizes);

    for (int j = 0; j < ptrs.size(); ++j) {
	FieldData data;

	data.type = STRING;
	data.s = string((char *) ptrs[j], sizes[j]);

	record.push_back(data);
    }

    return record;
}

void MobileDB::make_record(PalmRecord & rec,
			   const vector<string> & array)
{
    int i;

    // Calculate the size of the allocated record.
    int len = 6 + 2;
    for (i = 0; i < array.size(); ++i) {
	len += (2 + array[i].length());
    }

    // Allocate the temporary buffer.
    pi_char_t *buf = new pi_char_t[len];
    pi_char_t *p = buf;

    // Place the header into the buffer.
    *p++ = 0xFF;
    *p++ = 0xFF;
    *p++ = 0xFF;
    *p++ = 0x01;
    *p++ = 0xFF;
    *p++ = 0x00;

    // Now copy the fields into the buffer.
    for (i = 0; i < array.size(); ++i) {
	// Field header.
	*p++ = 0x00;
	*p++ = static_cast<pi_char_t> (i & 0xFF);

	// Copy the field into the buffer.
	const char * str = array[i].c_str();
	while (*str) {
	    *p = static_cast<pi_char_t> (*str);
	    ++p;
	    ++str;
	}
    }

    // Tack on the trailing bytes.
    *p++ = 0x00;
    *p++ = 0xFF;

    // Now make the record reflect the buffer.
    rec.setData(buf, len);
}

void MobileDB::addRecord(vector<DatabaseInterface::FieldData> recData)
{
    // Make sure the vector is of the right size.
    if (recData.size() != fields.size())
	throw invalid_argument("number of fields mismatch");

    // Build a vector of strings.
    vector<string> v;
    for (int i = 0; i < recData.size(); ++i) {
	// Make sure that the field types match.
	if (recData[i].type != fields[i].type)
	    throw invalid_argument("field type mismatch");

	// Add the field to the vector.
	v.push_back(recData[i].s);
    }

    // Allocate and fill a new PalmRecord.
    PalmRecord * record = new PalmRecord();
    make_record(*record, v);
    record->category(mdbCategoryDataRecords);
    records.push_back(record);
}

bool MobileDB::getFindDisabledFlag() const
{
    return false;
}

void MobileDB::setFindDisabledFlag(bool state)
{
}
