/*
 * palm-db-tools: Read/write DB-format databases
 * Copyright (C) 1999 by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include <string.h>
#include <stdexcept>

#include "DBDatabase.h"

using namespace std;

DBDatabase::DBDatabase() : dbFlags(0)
{
    setCreator(mktag('D','B','O','S'));
    setType(mktag('D','B','9','9'));
}

void DBDatabase::load(const char *fname)
{
    pi_char_t *p;
    int size;

    // Let the superclass load the raw database.
    PalmDatabase::load(fname);

    // Verify that the creator and type match what we expect.
    if ((getCreator() != mktag('D','B','O','S')
	 && getCreator() != mktag('T','K','D','3'))
	|| getType() != mktag('D','B','9','9'))
	throw error("not a DB-format database");

    // Retrieve a pointer into the app info block.
    getAppInfo(p, size);
  
    // Parse the initial portion of the header.
    if (size < 6)
      throw error("database header is corrupt");
    dbFlags = get_short(p); /* flags */
    p += 2;

    p += 2;  /* skip topVisibleRecord */

    int numFields = get_short(p);  /* numFields */
    p += 2;

    // Make sure that we have enough header left for the field info.
    if (size < (numFields * (2 + 32 + 2 + 2)))
	throw error("database header is corrupt");

    for (int i = 0; i < numFields; ++i) {
	FieldInfo info;

	// type
	switch (get_short(p)) {
	case 0:
	    info.type = STRING;
	    break;
	case 1:
	    info.type = BOOLEAN;
	    break;
	case 2:
	    info.type = INTEGER;
	    break;
	default:
	    throw error("unknown field type in header");
	}
	p += 2;

	// name
	info.name = string((char *) p);
	p += 32;

	// skip length
	p += 2;

	// width
	info.width = get_short(p);
	p += 2;

	fields.push_back(info);
    }
}  

void DBDatabase::save(const char *fname)
{
    // Determine the size of the application info block.
    size_t s = 3 * (2) + fields.size() * (2 + 32 + 2 + 2);

    // Allocate a temporary buffer for the block.
    pi_char_t *buf = new pi_char_t[s];
    pi_char_t *p = buf;
    memset(buf, 0xEA, s);

    set_short(p, dbFlags);        // flags
    p += 2;

    set_short(p, 0);              // topVisibleRecord
    p += 2;

    set_short(p, fields.size());  // numFields
    p += 2;

    for (int i = 0; i < fields.size(); ++i) {
	// type
	switch (fields[i].type) {
	case STRING:
	    set_short(p, 0);
	    break;
	case BOOLEAN:
	    set_short(p, 1);
	    break;
	case INTEGER:
	    set_short(p, 2);
	    break;
	default:
	    break;
	}
        p += 2;

	// name
        memset(p, 0, 32);
        strncpy((char *) p, fields[i].name.c_str(), 31);
        p += 32;

	// length (unused)
        set_short(p, 32);
        p += 2;

	// column width
        set_short(p, fields[i].width);
        p += 2;
    }

    // Put the app info block into place.
    setAppInfo(buf, s);

    // Deallocate the temporary buffer.
    delete [] buf;

    PalmDatabase::save(fname);
}

int DBDatabase::getNumFields() const
{
    return fields.size();
}

const DatabaseInterface::FieldInfo & DBDatabase::getField(int i) const
{
    return fields[i];
}

void DBDatabase::addField(const DatabaseInterface::FieldInfo & info)
{
    fields.push_back(info);
}

bool DBDatabase::supportsFieldType(const FieldType & type)
{
    switch (type) {
    case STRING:
    case BOOLEAN:
    case INTEGER:
	return true;
    default:
	return false;
    }
}

int DBDatabase::getNumRecords() const
{
    return records.size();
}

vector<DatabaseInterface::FieldData> DBDatabase::getRecord(int recNum) const
{
    vector<FieldData> record;

    pi_char_t *p = records[recNum]->data;

    for (int i = 0; i < fields.size(); ++i) {
	FieldData data;

	data.type = fields[i].type;
	switch (fields[i].type) {
	case STRING:
	    data.s = string((char *) p);
	    p += strlen((char *) p) + 1;
	    break;
	case BOOLEAN:
	    data.b = (*p) ? true : false;
	    p += 1;
	    break;
	case INTEGER:
	    data.i = get_long(p);
	    p += 4;
	    break;
	default:
	    break;
	}

	record.push_back(data);
    }

    return record;
}

void DBDatabase::addRecord(vector<DatabaseInterface::FieldData> recData)
{
    int size = 0;
    int i;

    if (recData.size() != fields.size())
	throw invalid_argument("number of fields mismatch");

    for (i = 0; i < fields.size(); ++i) {
	if (recData[i].type != fields[i].type)
	    throw invalid_argument("field type mismatch");

	switch (recData[i].type) {
	case STRING:
	    size += recData[i].s.length() + 1;
	    break;
	case BOOLEAN:
	    size += 1;
	    break;
	case INTEGER:
	    size += 4;
	    break;
	default:
	    throw invalid_argument("unsupported field type");
	}
    }

    pi_char_t *buf = new pi_char_t[size];
    pi_char_t *p = buf;

    for (i = 0; i < fields.size(); ++i) {
	switch (recData[i].type) {
	case STRING:
	    memcpy((void *) p, recData[i].s.c_str(), recData[i].s.length());
	    p += recData[i].s.length();
	    *p = '\0';
	    p += 1;
	    break;
	case BOOLEAN:
	    if (recData[i].b)
		*p = 1;
	    else
		*p = 0;
	    p += 1;
	    break;
	case INTEGER:
	    set_long(p, recData[i].i);
	    p += 4;
	    break;
	default:
	    throw invalid_argument("unsupported field type");
	}
    }

    PalmRecord * rec = new PalmRecord;
    rec->data = buf;
    rec->data_size = size;

    records.push_back(rec);
}

bool DBDatabase::getFindDisabledFlag() const
{
    return ((dbFlags & 1) != 0);
}

void DBDatabase::setFindDisabledFlag(bool state)
{
    if (state)
	dbFlags |= 1;
    else
	dbFlags &= ~(1);
}
