/* DB: Utility Functions
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include <TxtGlue.h>
#include <IntlGlue.h>
#include <WinGlue.h>
#include <stdarg.h>

#include "enum.h"
#include "db.h"

Boolean
HandleCommonMenuEvent(UInt16 menuitemID)
{
    switch (menuitemID) {
    case menuitemID_AppPrefs:
	PopupAppPrefs();
	return true;

    case menuitemID_About:
	FrmAlert(alertID_About);
	return true;

    default:
	return false;
    } 
}

void
DbgPrintF(const Char * fmt, ...)
{
    Char buf[256];
    va_list args;

    va_start(args, fmt);
    StrVPrintF(buf, fmt, args);
    va_end(args);

    FrmCustomAlert(alertID_Debug, buf, " ", " ");
}

char *
GetStringResource(UInt16 stringID)
{
    MemHandle str_handle;
    char * str;

    str_handle = DmGetResource(strRsc, stringID);
    if (! str_handle)
	return 0;

    str = MemHandleLock(str_handle);
    return str;
}

void
PutStringResource(const char * str)
{
    MemHandle str_handle = MemPtrRecoverHandle((MemPtr) str);

    MemHandleUnlock(str_handle);
    DmReleaseResource(str_handle);
}

char *
CopyStringResource(UInt16 stringID)
{
    MemHandle str_handle;
    char * str;
    char * str_copy;

    str_handle = DmGetResource(strRsc, stringID);
    if (! str_handle)
	return 0;

    str = MemHandleLock(str_handle);

    str_copy = MemPtrNew(1 + StrLen(str));
    if (str_copy) {
	StrCopy(str_copy, str);
    }

    MemHandleUnlock(str_handle);
    DmReleaseResource(str_handle);

    return str_copy;
}

Boolean
IsNumber(const char * str)
{
    UInt16 len = StrLen(str), index = 0;
    int base = 10, value;
    WChar ch;

    /* Blank strings cannot be numbers. Or can they? */
    if (len == 0)
	return false;

    /* Skip over any - at the start of a string. */
    index += TxtGlueGetNextChar(str, index, &ch);
    if (ch == chrHyphenMinus) {
	if (index >= len) return false;
	index += TxtGlueGetNextChar(str, index, &ch);
    }

    /* Check for leading 0 or 0x which signifies octal and hexadecimal. */
    if (ch == chrDigitZero) {
	base = 8;

	if (index >= len) return true;
	index += TxtGlueGetNextChar(str, index, &ch);

	if (ch == chrSmall_X || ch == chrCapital_X) {
	    base = 16;
	    if (index >= len) return false;
	    index += TxtGlueGetNextChar(str, index, &ch);
	}
    }

    while (1) {
	/* Figure out what the value of the current digit is. */
	if (TxtGlueCharIsDigit(ch))
	    value = ch - chrDigitZero;
	else if (TxtGlueCharIsLower(ch))
	    value = ch - chrSmall_A + 10;
	else if (TxtGlueCharIsUpper(ch))
	    value = ch - chrCapital_A + 10;
	else
	    return false;

	/* If the digit exceeds the base, then it is not a valid number. */
	if (value >= base) return false;

	/* Drop out of the loop if we are at the end of string. */
	if (index >= len) break;

	/* Pull the next character from the string. */
	index += TxtGlueGetNextChar(str, index, &ch);
    }

    return true;
}

Int32
String2Long(const char * str)
{
    Int16 len = StrLen(str), index = 0;
    Int32 result = 0, value;
    int base = 10;
    Boolean positive = true;
    WChar ch;

    /* Skip over any - at the start of a string. */
    index += TxtGlueGetNextChar(str, index, &ch);
    if (ch == chrHyphenMinus) {
	positive = false;
	if (index >= len) return 0;
	index += TxtGlueGetNextChar(str, index, &ch);
    }

    /* Check for leading 0 or 0x which signifies octal and hexadecimal. */
    if (ch == chrDigitZero) {
	base = 8;
	if (index >= len) return 0;
	index += TxtGlueGetNextChar(str, index, &ch);
	if (ch == chrSmall_X || ch == chrCapital_X) {
	    base = 16;
	    if (index >= len) return 0;
	    index += TxtGlueGetNextChar(str, index, &ch);
	}
    }

    while (1) {
	if (TxtGlueCharIsDigit(ch))
	    value = ch - chrDigitZero;
	else if (TxtGlueCharIsLower(ch))
	    value = ch - chrSmall_A + 10;
	else if (TxtGlueCharIsUpper(ch))
	    value = ch - chrCapital_A + 10;
	else
	    break;

	/* If we exceed the base, then it is not a valid number. */
	if (value >= base) break;

	/* Move this digit into the result. */
	result = result * base + value;

	/* Drop out of the loop if we are at the end of string. */
	if (index >= len) break;

	/* Pull the next character from the string. */
	index += TxtGlueGetNextChar(str, index, &ch);
    }

    return positive ? result : (- result);
}

Boolean
MatchString(const char * str, const char * substr,
	    Boolean case_sensitive, Boolean wholeWord)
{
    UInt16 len_str, len_substr, offset;
    Int16 (*Compare)(const Char*, UInt16, UInt16 *,
		     const Char*, UInt16, UInt16*);
    UInt32 word_start, word_end;

    len_substr = StrLen(substr);
    if (len_substr == 0)
	return true;
    len_str = StrLen(str);

    /* Determine the comparison routine to use. */
    if (case_sensitive)
	Compare = IntlGlueGetRoutineAddress(TxtGlueCompare);
    else
	Compare = IntlGlueGetRoutineAddress(TxtGlueCaselessCompare);

    offset = 0;
    while (offset + len_substr <= len_str) {
	if (Compare(str + offset, len_substr, 0, substr, len_substr, 0) == 0) {
	    /* If we don't need to match a word, then a match was found. */
	    if (! wholeWord) return true;

	    /* Figure out where a word is in the current match. */
	    if (TxtGlueWordBounds(str, len_str, offset,
				  &word_start, &word_end)) {
		if (word_start == offset
		    && word_end - word_start == len_substr) return true;
	    }
	}

	/* Slide the offset to the right by one character. */
	offset += TxtGlueGetNextChar(str, offset, 0);
    }

    return false;
}

void MyWinDrawTruncChars(const char * str, UInt16 length,
			 Int16 x, Int16 y, Int16 maxWidth)
{
    Boolean fitInWidth;
    Int16 len, width;

    width = maxWidth;
    len = length;
    fitInWidth = false;
    FntCharsInWidth(str, &width, &len, &fitInWidth);

    if (fitInWidth) {
	WinDrawChars(str, length, x, y);
    } else {
	WChar ch;

	/* Retrieve the ellipsis character. */
	ch = TxtGlueGetHorizEllipsisChar();

	/* Figure out how many characters can be shown with the ellipsis. */
	width = maxWidth - TxtGlueCharWidth(ch);
	len = length;
	fitInWidth = false;
	FntCharsInWidth(str, &width, &len, &fitInWidth);

	/* Draw the string and the ellipsis. */
	WinDrawChars(str, len, x, y);
	WinGlueDrawChar(ch, x + width, y);
    }
}

struct indicator_context
{
    RectangleType r;
    WinHandle winH;
};

void *
DrawWorkingIndicator(void)
{
    struct indicator_context * context;
    RectangleType screen;
    FontID oldFont;
    char * msg;
    Err err;

    context = (struct indicator_context *) MemPtrNew(sizeof(*context));

    msg = GetStringResource(stringID_Working);
    oldFont = FntSetFont(stdFont);

    FrmGetFormBounds(FrmGetActiveForm(), &screen);
    context->r.extent.x = 5 + FntCharsWidth(msg, StrLen(msg)) + 5;
    context->r.extent.y = FntLineHeight() * 3;
    context->r.topLeft.x = screen.topLeft.x
	+ (screen.extent.x - context->r.extent.x) / 2;
    context->r.topLeft.y = screen.topLeft.y
	+ (screen.extent.y - context->r.extent.y) / 2;

    context->winH = WinSaveBits(&context->r, &err);
    WinDrawRectangle(&context->r, 0);
    WinEraseChars(msg, StrLen(msg),
                  context->r.topLeft.x + 5,
		  context->r.topLeft.y + FntLineHeight());

    PutStringResource(msg);
    FntSetFont(oldFont);

    return context;
}

void
EraseWorkingIndicator(void * ptr)
{
    struct indicator_context * context = (struct indicator_context *) ptr;

    WinRestoreBits(context->winH, context->r.topLeft.x, context->r.topLeft.y);
    MemPtrFree(context);
}
