/*
 * DB: Database Program for Pilot
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>

#include "enum.h"
#include "db.h"
#include "linkaware.h"

DBPrefsType prefs;
privateRecordViewEnum PrivateRecordStatus;
Boolean FeatureSet30, FeatureSet31, FeatureSet32, FeatureSet35;
DataSourceDriver * DriverList;

static Boolean
AppHandleEvent(EventPtr event)
{
    UInt16 formID;
    FormPtr form;

    if (event->eType == frmLoadEvent) {
	formID = event->data.frmLoad.formID;
	form = FrmInitForm(formID);
	FrmSetActiveForm(form);
	switch (formID) {
	case formID_Chooser:
	    FrmSetEventHandler(form, ChooserHandleEvent);
	    break;
	case formID_DesignView:
	    FrmSetEventHandler(form, DesignViewHandleEvent);
	    break;
	case formID_ListView:
	    FrmSetEventHandler(form, ListViewHandleEvent);
	    break;
	case formID_FindDialog:
	    FrmSetEventHandler(form, FindDialogHandleEvent);
	    break;
	case formID_EditView:
	    FrmSetEventHandler(form, EditViewHandleEvent);
	    break;
	case formID_ChooserInfoDialog:
	    FrmSetEventHandler(form, ChooserInfoDialogHandleEvent);
	    break;
	case formID_ListViewEditor:
	    FrmSetEventHandler(form, ListViewEditor_HandleEvent);
	    break;
	case formID_CreateDlg:
	    FrmSetEventHandler(form, CreateDlgHandleEvent);
	    break;
	case formID_DuplicateDlg:
	    FrmSetEventHandler(form, DuplicateDlgHandleEvent);
	    break;
	case formID_RenameDlg:
	    FrmSetEventHandler(form, RenameDlgHandleEvent);
	    break;
	case formID_SortEditor:
	    FrmSetEventHandler(form, SortEditor_HandleEvent);
	    break;
	case formID_NamesEditor:
	    FrmSetEventHandler(form, NamesEditorHandleEvent);
	    break;
	default:
	    ErrDisplay("unknown form ID");
	    return false;
	}
	return true;
    }
    return false;
}

static void
EventLoop(void)
{
    UInt16 err;
    EventType event;

    do {
	EvtGetEvent(&event, evtWaitForever);

	if (!SysHandleEvent(&event))
	    if (!MenuHandleEvent((void *)0, &event, &err))
		if (!AppHandleEvent(&event))
		    FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static void
InitPrefs(DBPrefsPtr prefs)
{
    MemSet(prefs, sizeof(*prefs), 0);

    prefs->flags = prefFlagEnableFind | prefFlagPageUpPage
	| prefFlagPageDownPage;
    prefs->ChooserCurrentCategory = dmAllCategories;
    prefs->ChooserShowAllCategories = true;
    prefs->ChooserSortOrder = chooserSortOrderManual;
}

static Err
CheckMinVersion(UInt16 launchFlags)
{
    UInt32 version;

    if (FtrGet(sysFtrCreator, sysFtrNumROMVersion, &version) == 0) {
	if (version < sysMakeROMVersion(2, 0, 0, sysROMStageRelease, 0)) {
	    if ((launchFlags &
		 (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
		(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) {
		FrmAlert(alertID_RomIncompatible);

		/* Pilot 1.0 will continuously relaunch this app unless we
		 * switch to another safe one.
		 */
		if (version < 0x02000000)
		    AppLaunchWithCommand(sysFileCDefaultApp,
					 sysAppLaunchCmdNormalLaunch, NULL);
	    }
            
	    return sysErrRomIncompatible;
	}
    }

    return 0;
}

static Err
StartApplication(void)
{
    UInt32 version;
    UInt16 prefsSize = sizeof(prefs);
    Int16 actualSize;

    /* Detect the PalmOS version so we know what features exist. */
    if (FtrGet(sysFtrCreator, sysFtrNumROMVersion, &version) == 0) {
#define ver(a,b) sysMakeROMVersion(a, b, 0, sysROMStageRelease, 0)
	FeatureSet30 = (version >= ver(3,0));
	FeatureSet31 = (version >= ver(3,1));
	FeatureSet32 = (version >= ver(3,2));
	FeatureSet35 = (version >= ver(3,5));
#undef ver
    }

    /* Read the current application preferences. */
    actualSize = PrefGetAppPreferences(DBCreatorID, prefID_GlobalPrefs,
				       &prefs, &prefsSize, false);

    /* Reset the preferences if they don't exist or have the wrong size. */
    if (actualSize == noPreferenceFound || prefsSize != sizeof(prefs)) {
	InitPrefs(&prefs);
	PrefSetAppPreferences(DBCreatorID, prefID_GlobalPrefs, 0,
			      &prefs, sizeof(prefs), false);
    }

    /* Detect what should be done with private records. */
    if (FeatureSet35) {
	PrivateRecordStatus = PrefGetPreference(prefShowPrivateRecords);
    } else {
	if (PrefGetPreference(prefHidePrivateRecordsV33))
	    PrivateRecordStatus = hidePrivateRecords;
	else
	    PrivateRecordStatus = showPrivateRecords;
    }

    /* Initialize any built-in data source drivers. */
    DriverList = GetDataSources();

    return 0;
}

static void
ChooseInitialForm(void)
{
    UInt16 formID = formID_Chooser;
    UInt8 * key_data;
    UInt16 key_size = 0;
    UInt16 sourceID;

    /* Open any previously active data source. */
    if (PrefGetAppPreferences(DBCreatorID, prefID_LastDataSource,
			      0, &key_size, false) != noPreferenceFound) {
	key_data = MemPtrNew(key_size);
	PrefGetAppPreferences(DBCreatorID, prefID_LastDataSource,
			      key_data, &key_size, false);

	sourceID = *((UInt16 *) key_data);
	CurrentSource = OpenDataSource(DriverList, dmModeReadWrite, sourceID,
				       key_data + 2, key_size - 2);
	if (CurrentSource)
	    formID = formID_ListView;

	MemPtrFree(key_data);
    }

    FrmGotoForm(formID);
}

static void
SaveCurrentDataSource(void)
{
    UInt8 * key_data;
    UInt32 key_size;

    if (CurrentSource) {
	/* Store a key to this source so we can open it later. */
	key_size = CurrentSource->ops.GetKey(CurrentSource, 0);
	key_data = MemPtrNew(sizeof(UInt16) + key_size);
	*((UInt16 *) key_data) = CurrentSource->driver->sourceID;
	CurrentSource->ops.GetKey(CurrentSource, key_data + 2);
	PrefSetAppPreferences(DBCreatorID, prefID_LastDataSource, 0,
			      key_data, (UInt16) (key_size + 2), false);
	MemPtrFree(key_data);

	/* Now close the data source for real. */
	CloseDataSource(CurrentSource);
	CurrentSource = 0;
    } else {
	PrefSetAppPreferences(DBCreatorID, prefID_LastDataSource, 0,
			      0, 0, false);
    }
}

static void
StopApplication(void)
{
    FrmSaveAllForms();
    FrmCloseAllForms();

    /* Close the active data source (if any) and remember it for later. */
    SaveCurrentDataSource();

    /* Save the application preferences. */
    PrefSetAppPreferences(DBCreatorID, prefID_GlobalPrefs, 0,
			  &prefs, sizeof(prefs), false);

    /* Free the list of available data sources. */
    PutDataSources(DriverList);
}

static void
GoToItem(GoToParamsPtr goToParams, Boolean launchingApp)
{
    EventType event;
    UInt16 recordNum = goToParams->recordNum;
 
    /* If the current record is blank, then it will be deleted, so we'll
     * the record's unique id to find the record index again, after all
     * the forms are closed.
     */
    if (! launchingApp) {
	UInt32 uniqueID;
	UInt16 cardNo;
	LocalID dbID;

	if (CurrentSource) {
	    DmOpenDatabaseInfo(CurrentSource->db, &dbID, 0, 0, &cardNo, 0);
	    if (cardNo == goToParams->dbCardNo && dbID == goToParams->dbID) {
		/* For same database, recordNum could change. */
		DmRecordInfo(CurrentSource->db, recordNum, NULL,
			     &uniqueID, NULL);
		FrmCloseAllForms();
		DmFindRecordByID(CurrentSource->db, uniqueID, &recordNum);
	    } else {
		/* Switch to the other database. */
		FrmCloseAllForms();
		CloseDataSource(CurrentSource);
		CurrentSource = OpenDataSourcePDB(DriverList, dmModeReadWrite,
						  goToParams->dbCardNo,
						  goToParams->dbID);
	    }
	} else {
	    CurrentSource = OpenDataSourcePDB(DriverList, dmModeReadWrite,
					      goToParams->dbCardNo,
					      goToParams->dbID);
	}
    }
 
    /* We go to the list view for now. */
    FrmGotoForm(formID_ListView);
 
    /* Send an event to select the matching text. */
    MemSet(&event, 0, sizeof(EventType));
 
    event.eType = frmGotoEvent;
    event.data.frmGoto.formID = formID_ListView;
    event.data.frmGoto.recordNum = goToParams->recordNum;
    event.data.frmGoto.matchPos = goToParams->matchPos;
    event.data.frmGoto.matchLen = goToParams->searchStrLen;
    event.data.frmGoto.matchFieldNum = goToParams->matchFieldNum;
    event.data.frmGoto.matchCustom = goToParams->matchCustom;
    EvtAddEventToQueue(&event);
}

UInt32
PilotMain(UInt16 cmd, void * cmdPBP, UInt16 launchFlags)
{
    Err err;

    /* Before we continue, make sure we are on PalmOS 2.0 or higher. */
    err = CheckMinVersion(launchFlags);
    if (err)
	return err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;

	ChooseInitialForm();
	EventLoop();
	StopApplication();
    } else if (cmd == sysAppLaunchCmdOpenDB) {
	SysAppLaunchCmdOpenDBType * db = (SysAppLaunchCmdOpenDBType *) cmdPBP;

	err = StartApplication();
	if (err)
	    return err;

	CurrentSource = OpenDataSourcePDB(DriverList, dmModeReadWrite,
					  db->cardNo, db->dbID);
	if (CurrentSource)
	    FrmGotoForm(formID_ListView);
	else
	    FrmGotoForm(formID_Chooser);

	EventLoop();
	StopApplication();
    } else if (cmd == sysAppLaunchCmdSyncNotify) {
	DBPrefsType prefs;
	UInt16 prefsSize = sizeof(prefs);

	if (PrefGetAppPreferences(DBCreatorID, prefID_GlobalPrefs, &prefs,
				  &prefsSize, false) == noPreferenceFound
	    || prefsSize != sizeof(prefs))
	    InitPrefs(&prefs);
	prefs.ChooserForceRescan = true;
	PrefSetAppPreferences(DBCreatorID, prefID_GlobalPrefs, 0,
			      &prefs, sizeof(prefs), false);
    } else if (cmd == linkAppLaunchRequestLink) {
	err = StartApplication();
	if (err)
	    return err;
	FrmGotoForm(formID_Chooser); /* Go to index when a link is requested */
	EventLoop();
	StopApplication();
    } else if (cmd == sysAppLaunchCmdFind) {
	UInt16 prefsSize = sizeof(prefs);
	DBPrefsType prefs;

	if (PrefGetAppPreferences(DBCreatorID, prefID_GlobalPrefs, &prefs,
				  &prefsSize, false) != noPreferenceFound) {
	    if (prefs.flags & prefFlagEnableFind) {
		GlobalSearch((FindParamsPtr)cmdPBP);
	    }
	}
    } else if (cmd == sysAppLaunchCmdGoTo) {
	Boolean launched;
	GoToParamsPtr go;

	launched = launchFlags & sysAppLaunchFlagNewGlobals;
	go = (GoToParamsPtr) cmdPBP;

	if (launched) {
	    err = StartApplication();
	    if (err)
		return err;

	    CurrentSource = OpenDataSourcePDB(DriverList, dmModeReadWrite,
					      go->dbCardNo, go->dbID);
	    if (CurrentSource) {
		GoToItem(go, launched);
		EventLoop();
		StopApplication();
	    }
	} else {
	    GoToItem(go, launched);
	}
    } else if (cmd == linkAppLaunchFollowLink
	       || cmd == linkAppLaunchFollowLinkSubCall) {
	GoToParamsPtr go;

	if (cmd == linkAppLaunchFollowLinkSubCall
	    && ! (launchFlags&sysAppLaunchFlagNewGlobals) ) {
	    DmSearchStateType state;
	    UInt16 cardNo;
	    LocalID dbID;

	    if (DmGetNextDatabaseByTypeCreator(true, &state,
					       sysFileTApplication,
					       DBCreatorID, false,
					       &cardNo, &dbID) == 0) {
		SysUIAppSwitch(cardNo, dbID, linkAppLaunchFollowLink, cmdPBP);
	    }
	    return 0;
	}
       
	go = LinkSimpleToGoToParams((LinkSimplePtr) cmdPBP);

	if (go) {
	    if (cmd == linkAppLaunchFollowLink) {
		err = StartApplication();
		if (err)
		    return err;

		CurrentSource = OpenDataSourcePDB(DriverList, dmModeReadWrite,
						  go->dbCardNo, go->dbID);
		if (CurrentSource) {
		    IsNewRecord = false;
		    CurrentRecord = go->recordNum;
		    FrmGotoForm(formID_EditView);
		    EventLoop();
		    StopApplication();
		}
	    } else {
		GoToItem(go, false);
	    }
	    MemPtrFree(go);
	} else {
	    FrmAlert(alertID_NoLinkRecord);
	}
    }

    return 0;
}
