/*
 * DB: Record List View
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>

#include "enum.h"
#include "db.h"
#include "grid.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))
#define noRecord dmMaxRecordIndex
#define updateRedrawGrid 1
#define updateModelChanged 2

static UInt16 ActiveViewIndex;
static DataSourceView * ActiveView;
static DataSourceGetter getter;

static UInt16 LFindRecord;    /* record that was found */
static UInt16 LFindField;     /* field to search, 0 for all fields */
static Boolean LFindCaseSens, LFindWholeWord;
static Char FindString[32]; /* previous search string */
static GridType grid;

static UInt16 ViewModel_GetColumnCount(void * __unused) SECT_UI;
static void ViewModel_GetColumnName(void *, UInt16, Char *, UInt32) SECT_UI;
static UInt16 ViewModel_GetColumnWidth(void *, UInt16) SECT_UI;
static void ViewModel_SetColumnWidth(void *, UInt16, UInt16) SECT_UI;
static void * ViewModel_LockIndex(void *, UInt16, Boolean) SECT_UI;
static void ViewModel_UnlockIndex(void *, UInt16, Boolean, void *) SECT_UI;
static GridCellType ViewModel_GetCellType(void *,UInt16,UInt16) SECT_UI;
static void ViewModel_GetCellString(void *, UInt16, UInt16, Char *,
				    UInt32, void *) SECT_UI;
static Boolean ViewModel_GetCellBoolean(void *, UInt16, UInt16, void *_data) SECT_UI;
static Boolean bool_getter_GetBoolean(void * data, UInt16 fieldNum) SECT_UI;
static void ViewModel_SetCellBoolean(void *, UInt16, UInt16, Boolean,
				     void *) SECT_UI;
static void ViewModel_GetCellID(void *, UInt16, UInt32 *) SECT_UI;
static Boolean ViewModel_Seek(void *, UInt16 *, Int16, Int16) SECT_UI;
static void UpdateFieldButtons(Boolean, Boolean) SECT_UI;
static void UpdateScrollers(Boolean, Boolean) SECT_UI;
static Boolean CmpField(DataSourceGetter *, void *, UInt16, const char *)
     SECT_UI;
static void LocalSearch(Char * str) SECT_UI;
static Boolean DoCommand(UInt16 itemID) SECT_UI;
static void PopupListOptions(void) SECT_UI;


static UInt16 ViewModel_GetColumnCount(void * __unused)
{
    return ActiveView->numCols;
}

static void ViewModel_GetColumnName(void * __unused, UInt16 col,
				    Char * name, UInt32 len)
{
    StrNCopy(name,
	     CurrentSource->schema.fields[ActiveView->cols[col].fieldNum].name,
	     len);
}

static UInt16 ViewModel_GetColumnWidth(void * __unused, UInt16 col)
{
    return ActiveView->cols[col].width;
}

static void ViewModel_SetColumnWidth(void * __unused, UInt16 col, UInt16 width)
{
}

static void * ViewModel_LockIndex(void * __unused,
				  UInt16 index, Boolean writable)
{
    void * row_data;

    CurrentSource->ops.LockRecord(CurrentSource, index, writable,
				  &getter, &row_data);
    return row_data;
}

static void ViewModel_UnlockIndex(void * __unused, UInt16 index,
				  Boolean writable, void * row_data)
{
    CurrentSource->ops.UnlockRecord(CurrentSource, index, writable, row_data);
}

static GridCellType ViewModel_GetCellType(void * __unused,
					  UInt16 index, UInt16 col)
{
    switch (CurrentSource->schema.fields[ActiveView->cols[col].fieldNum].type) {
    case FIELD_TYPE_BOOLEAN:
        return gridCellTypeBoolean;
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_INTEGER:
    default:
	return gridCellTypeString;
    }
}

static void ViewModel_GetCellString(void * __unused, UInt16 index, UInt16 col,
				    Char * data, UInt32 len, void * row_data)
{
    const UInt16 fieldNum = ActiveView->cols[col].fieldNum;
    char * p;
    Int32 value;
    UInt8 day, month, hour, minute;
    UInt16 year;

    switch (CurrentSource->schema.fields[fieldNum].type) {
    case FIELD_TYPE_STRING:
	p = getter.GetString(row_data, fieldNum);
	StrNCopy(data, p, len);
	break;

    case FIELD_TYPE_INTEGER:
	value = getter.GetInteger(row_data, fieldNum);
	StrIToA(data, value);
	break;

    case FIELD_TYPE_DATE:
	getter.GetDate(row_data, fieldNum, &year, &month, &day);
	DateToAscii(month, day, year, PrefGetPreference(prefDateFormat), data);
	break;

    case FIELD_TYPE_TIME:
	getter.GetTime(row_data, fieldNum, &hour, &minute);
	TimeToAscii(hour, minute, PrefGetPreference(prefTimeFormat), data);
	break;

    default:
	StrNCopy(data, "N/A", len);
	break;
    }
}

static Boolean ViewModel_GetCellBoolean(void * __unused, UInt16 index,
					UInt16 col, void * row_data)
{
    return getter.GetBoolean(row_data, ActiveView->cols[col].fieldNum);
}

static Boolean bool_getter_GetBoolean(void * data, UInt16 fieldNum)
{
    return (data != 0) ? true : false;
}

static DataSourceGetter bool_getter = {
    GetBoolean: bool_getter_GetBoolean,
};

static void ViewModel_SetCellBoolean(void * __unused, UInt16 index,
				     UInt16 col, Boolean b, void * row_data)
{
    CurrentSource->ops.SetField(CurrentSource, row_data,
				ActiveView->cols[col].fieldNum, &bool_getter,
				b ? ((void *) 1) : ((void *) 0));
}

static void ViewModel_GetCellID(void * __unused, UInt16 index, UInt32 * idP)
{
    UInt32 uniqueID;

    uniqueID = CurrentSource->ops.GetRecordID(CurrentSource, index);
    *idP = uniqueID;
}

static Boolean ViewModel_Seek(void * __unused, UInt16 * indexP,
			      Int16 offset, Int16 direction)
{
    return CurrentSource->ops.Seek(CurrentSource, indexP, offset, direction);
}

static Boolean ViewModel_IsColumnEditable(void * __unused, UInt16 col)
{
    Boolean result = false;

    if ((CurrentSource->schema.fields[ActiveView->cols[col].fieldNum].type)
	== FIELD_TYPE_BOOLEAN && (! CurrentSource->readonly))
	result = true;

    return result;
}

static GridModelType ViewModel = {
    GetColumnCount: ViewModel_GetColumnCount,
    GetColumnName: ViewModel_GetColumnName,
    GetColumnWidth: ViewModel_GetColumnWidth,
    SetColumnWidth: ViewModel_SetColumnWidth,
    LockIndex: ViewModel_LockIndex,
    UnlockIndex: ViewModel_UnlockIndex,
    GetCellType: ViewModel_GetCellType,
    GetCellString: ViewModel_GetCellString,
    GetCellBoolean: ViewModel_GetCellBoolean,
    SetCellBoolean: ViewModel_SetCellBoolean,
    GetCellID: ViewModel_GetCellID,
    Seek: ViewModel_Seek,
    IsColumnEditable: ViewModel_IsColumnEditable,
};

static void
UpdateFieldButtons(Boolean scrollLeft, Boolean scrollRight)
{
    FormPtr form;
    UInt16 leftIndex, rightIndex;

    form = FrmGetActiveForm();
    leftIndex = FrmGetObjectIndex(form, ctlID_ListView_LeftButton);
    rightIndex = FrmGetObjectIndex(form, ctlID_ListView_RightButton);

    if (scrollLeft)
	FrmShowObject(form, leftIndex);
    else
	FrmHideObject(form, leftIndex);

    if (scrollRight)
	FrmShowObject(form, rightIndex);
    else
	FrmHideObject(form, rightIndex);
}

static void
UpdateScrollers(Boolean scrollUp, Boolean scrollDown)
{
    FormPtr form;
    UInt16 upIndex, downIndex;

    form = FrmGetActiveForm();
    upIndex = FrmGetObjectIndex(form, ctlID_ListView_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_ListView_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static Boolean
CmpField(DataSourceGetter* getter, void* rec_data, UInt16 i, const char* str)
{
    char buf[32], *p;
    Int32 v;

    switch (CurrentSource->schema.fields[i].type) {
    case FIELD_TYPE_STRING:
	p = getter->GetString(rec_data, i);
	return MatchString(p, str, LFindCaseSens, LFindWholeWord);

    case FIELD_TYPE_INTEGER:
	v = getter->GetInteger(rec_data, i);
	StrPrintF(buf, "%ld", v);
	return MatchString(buf, str, LFindCaseSens, LFindWholeWord);

    case FIELD_TYPE_BOOLEAN:
	/* XXX Should really convert str to bool and ==. */
	buf[0] = getter->GetBoolean(rec_data, i) ? '1' : '0';
	buf[1] = '\0';
	return MatchString(buf, str, LFindCaseSens, LFindWholeWord);

    default:
	return false;
    }
}

static void
LocalSearch(Char * str)
{
    UInt16 recNum, foundRecNum;
    DataSourceGetter getter;
    void * rec_data;
    Boolean match;

    recNum = (LFindRecord == noRecord) ? 0 : (LFindRecord + 1);
    while (1) {
	if (! CurrentSource->ops.Seek(CurrentSource, &recNum, 0,
				      dmSeekForward)) {
	    foundRecNum = noRecord;
	    break;
	}

	/* Lock down the record. */
	CurrentSource->ops.LockRecord(CurrentSource, recNum, false,
				      &getter, &rec_data);

	match = false;
	if (LFindField == 0) {
	    UInt16 i;

	    /* Search all string fields for the string. */
	    for (i = 0; i < CurrentSource->schema.numFields; ++i) {
		/* Compare the string to the field. */
		match = CmpField(&getter, rec_data, i, str);

		/* Bail out if we found a match. */
		if (match) break;
	    }
	} else {
	    /* Search the single field. */
	    match = CmpField(&getter, rec_data, LFindField - 1, str);
	}

	/* Unlock the record. */
	CurrentSource->ops.UnlockRecord(CurrentSource, recNum,
					false, rec_data);

	/* If a match was found, then stop the search. */
	if (match) {
	    foundRecNum = recNum;
	    break;
	}

	/* Increment the record number. */
	recNum++;
    }

    /* Highlight the found record or scroll grid so record is visible. */
    GridUnhighlightAll(&grid);
    if (foundRecNum != noRecord) {
	GridHighlightIndex(&grid, foundRecNum);
	GridMakeIndexVisible(&grid, foundRecNum);
    } else {
	FrmAlert(alertID_MatchNotFound);
    }

    LFindRecord = foundRecNum;

    FrmUpdateForm(formID_ListView, updateRedrawGrid);
}

Boolean FindDialogHandleEvent(EventPtr event)
{
    static MemHandle strH;
    static char ** names;

    FormPtr form;
    FieldPtr fld;
    ControlPtr ctl;
    ListPtr lst;
    Char * str;
    UInt16 i, objIndex;
    Int16 width, w;
    RectangleType r;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();

	ctl = GetObjectPtr(form, ctlID_FindDialog_CaseSensCheckbox);
	CtlSetValue(ctl, LFindCaseSens ? 1 : 0);

	ctl = GetObjectPtr(form, ctlID_FindDialog_WholeWordCheckbox);
	CtlSetValue(ctl, LFindWholeWord ? 1 : 0);

	/* Allocate space for an array of string pointers for field list. */
	names = (char **) MemPtrNew((CurrentSource->schema.numFields + 1) * sizeof(char *));

	/* Duplicate the "All Fields" string. */
	names[0] = CopyStringResource(stringID_AllFields);

	/* Figure out the width needed for the field popup list. */
	width = FntCharsWidth(names[0], StrLen(names[0]));
	for (i = 0; i < CurrentSource->schema.numFields; i++) {
	    names[1 + i] = CurrentSource->schema.fields[i].name;
	    w = FntCharsWidth(names[1 + i], StrLen(names[1 + i]));
	    if (w > width)
		width = w;
	}
	width += 10;

	/* Set the width of the field popup list. */
	objIndex = FrmGetObjectIndex(form, ctlID_FindDialog_FieldList);
	lst = FrmGetObjectPtr(form, objIndex);
	FrmGetObjectBounds(form, objIndex, &r);
	r.extent.x = width;
	FrmSetObjectBounds(form, objIndex, &r);

	/* Set the remainder of the list parameters. */
	LstSetListChoices(lst, names, 1 + CurrentSource->schema.numFields);
	LstSetSelection(lst, LFindField);
	LstSetHeight(lst, 1 + CurrentSource->schema.numFields);

	ctl = GetObjectPtr(form, ctlID_FindDialog_FieldTrigger);
	CtlSetLabel(ctl, names[LFindField]);

	strH = MemHandleNew(32);
	str = MemHandleLock(strH);
	StrCopy(str, FindString);
	MemPtrUnlock(str);
    
	fld = GetObjectPtr(form, ctlID_FindDialog_Field);
	FldSetTextHandle(fld, strH);

	FrmDrawForm(form);
	FrmSetFocus(form, FrmGetObjectIndex(form, ctlID_FindDialog_Field));

	return true;

    case frmCloseEvent:
	if (names) {
	    MemPtrFree(names[0]);
	    MemPtrFree(names);
	    names = 0;
	}
	return false;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_FindDialog_FindButton:
	    form = FrmGetActiveForm();

	    str = MemHandleLock(strH);
	    if (LFindCaseSens)
		StrCopy(FindString, str);
	    else
		StrToLower(FindString, str);
	    MemPtrUnlock(str);

	    ctl = GetObjectPtr(form, ctlID_FindDialog_CaseSensCheckbox);
	    LFindCaseSens = (CtlGetValue(ctl)) ? true : false;

	    ctl = GetObjectPtr(form, ctlID_FindDialog_WholeWordCheckbox);
	    LFindWholeWord = (CtlGetValue(ctl)) ? true : false;

	    lst = GetObjectPtr(form, ctlID_FindDialog_FieldList);
	    LFindField = LstGetSelection(lst);

	    /* Update the database flags with the current checkbox settings. */
	    CurrentSource->ops.SetOption_Boolean(CurrentSource,
						 optionID_LFind_CaseSensitive,
						 LFindCaseSens);
	    CurrentSource->ops.SetOption_Boolean(CurrentSource,
						 optionID_LFind_WholeWord,
						 LFindWholeWord);

	    /* Remove the find dialog from the screen. */
	    FrmReturnToForm(0);
	    if (names) {
		MemPtrFree(names[0]);
		MemPtrFree(names);
		names = 0;
	    }

	    /* Do the actual find operation. */
	    if (FindString[0] != '\0') LocalSearch(FindString);

	    return true;

	case ctlID_FindDialog_CancelButton:
	    FrmReturnToForm(0);
	    if (names) {
		MemPtrFree(names[0]);
		MemPtrFree(names);
		names = 0;
	    }
	    return true;
	}
	break;

    default:
	break;
    }

    return false;
}

static Boolean
DoCommand(UInt16 itemID)
{
    switch (itemID) {
    case menuitemID_DatabaseDesign:
	DesignViewDriver = CurrentSource->driver;
	FrmGotoForm(formID_DesignView);
	return true;

    case menuitemID_EditListViews:
	FrmGotoForm(formID_NamesEditor);
	return true;

    case menuitemID_Sort:
	FrmGotoForm(formID_SortEditor);
	return true;

    case menuitemID_NewRecord:
	if (! CurrentSource->readonly) {
	    IsNewRecord = true;
	    FrmGotoForm(formID_EditView);
	}
	return true;

    case menuitemID_GotoTop:
	grid.topmostRow = 0;
	FrmUpdateForm(formID_ListView, updateRedrawGrid);
	return true;

    case menuitemID_GotoBottom:
	grid.topmostRow = dmMaxRecordIndex;
	FrmUpdateForm(formID_ListView, updateRedrawGrid);
	return true;

    case menuitemID_DatabasePrefs:
	MenuEraseStatus(0);
	PopupDBPrefs();
	return true;

    case menuitemID_ListDisplayOpts:
	MenuEraseStatus(0);
	PopupListOptions();
	return true;

    case menuitemID_Help:
	FrmHelp(stringID_Help_ListView);
	return true;

    default:
	return HandleCommonMenuEvent(itemID);

    }
}

static void
PopupListOptions(void)
{
    FormType * form;
    FormType * oldForm;
    ControlType * ctl;
    ListType * lst;
    UInt16 numViews, i;
    UInt16 new_view_index;
    char ** view_names;

    oldForm = FrmGetActiveForm();
    form = FrmInitForm(formID_ListViewOptionsDlg);
    FrmSetActiveForm(form);

    /* Construct an array of the names of the views in this database. */
    numViews = CurrentSource->ops.GetNumOfViews(CurrentSource);
    view_names = MemPtrNew(numViews * sizeof(char *));
    for (i = 0; i < numViews; ++i) {
	DataSourceView * v = CurrentSource->ops.GetView(CurrentSource, i);
	view_names[i] = MemPtrNew(1 + StrLen(v->name));
	StrCopy(view_names[i], v->name);
	CurrentSource->ops.PutView(CurrentSource, i, v);
    }

    /* Setup the list and popup trigger. */
    ctl = GetObjectPtr(form, ctlID_ListViewOptionsDlg_ViewTrigger);
    lst = GetObjectPtr(form, ctlID_ListViewOptionsDlg_ViewList);
    LstSetListChoices(lst, view_names, numViews);
    LstSetHeight(lst, numViews);
    LstSetSelection(lst, ActiveViewIndex);
    CtlSetLabel(ctl, view_names[ActiveViewIndex]);

    new_view_index = ActiveViewIndex;
    if (FrmDoDialog(form) == ctlID_ListViewOptionsDlg_OK) {
	lst = GetObjectPtr(form, ctlID_ListViewOptionsDlg_ViewList);
	if (LstGetSelection(lst) != noListSelection) {
	    new_view_index = LstGetSelection(lst);	
	}
    }

    /* Take down the dialog and free the view names. */
    FrmDeleteForm(form);
    FrmSetActiveForm(oldForm);
    for (i = 0; i < numViews; ++i) {
	MemPtrFree(view_names[i]);
    }
    MemPtrFree(view_names);

    /* If the user switched views, move to the new view. */
    if (new_view_index != ActiveViewIndex) {
	/* Free the existing data source view. */
	CurrentSource->ops.PutView(CurrentSource, ActiveViewIndex, ActiveView);

	/* Update the in-core variables to the new view. */
	ActiveViewIndex = new_view_index;
	ActiveView = CurrentSource->ops.GetView(CurrentSource, ActiveViewIndex);

	/* Tell the list form to reinitialize the grid. */
	FrmUpdateForm(formID_ListView, updateModelChanged);
    }
}

Boolean
ListViewHandleEvent(EventPtr event)
{
    static char title[64];

    FormPtr form;

    switch (event->eType) {
    case frmOpenEvent:
	/* Get the active list view definition. */
	ActiveViewIndex =
	    CurrentSource->ops.GetOption_UInt16(CurrentSource,
						optionID_ListView_ActiveView);
	ActiveView = CurrentSource->ops.GetView(CurrentSource, ActiveViewIndex);

	form = FrmGetActiveForm();
	LFindRecord = noRecord;
	LFindField = 0;
	LFindCaseSens = CurrentSource->ops.GetOption_Boolean(CurrentSource, optionID_LFind_CaseSensitive);
	LFindWholeWord = CurrentSource->ops.GetOption_Boolean(CurrentSource, optionID_LFind_WholeWord);
	FindString[0] = '\0';

	/* Make the form title reflect the database name. */
	StrCopy(title, "DB: ");
	CurrentSource->ops.GetTitle(CurrentSource, title+4, sizeof(title)-4);
	title[sizeof(title)-1] = '\0';
	FrmSetTitle(form, title);

	/* Hide the "new" button if we are in read-only mode. */
	if (CurrentSource->readonly) {
	    FrmHideObject(form, FrmGetObjectIndex(form,
						  ctlID_ListView_NewButton));
	}

	GridInit(&grid, form, ctlID_ListView_Grid,
		 ctlID_ListView_GridCheckbox, &ViewModel, 0);
	grid.topmostRow = CurrentSource->ops.GetOption_UInt16(CurrentSource, optionID_ListView_TopVisibleRecord);
	GridSetHeaderLook(&grid, true, false, false, boldFont);
	GridSetColumnLook(&grid, false);

	FrmDrawForm(form);
	GridDraw(&grid);

	return true;

    case frmCloseEvent:
	CurrentSource->ops.SetOption_UInt16(CurrentSource,
					    optionID_ListView_TopVisibleRecord,
					    grid.topmostRow);
	GridFree(&grid);
	CurrentSource->ops.PutView(CurrentSource, ActiveViewIndex, ActiveView);
	CurrentSource->ops.SetOption_UInt16(CurrentSource,
					    optionID_ListView_ActiveView,
					    ActiveViewIndex);
	break;

    case frmUpdateEvent:
	/* If the model changed, then reinitialize the grid widget. */
	if (event->data.frmUpdate.updateCode & updateModelChanged) {
	    GridFree(&grid);
	    GridInit(&grid, FrmGetActiveForm(), ctlID_ListView_Grid,
		     ctlID_ListView_GridCheckbox, &ViewModel, 0);
	    grid.topmostRow = CurrentSource->ops.GetOption_UInt16(CurrentSource, optionID_ListView_TopVisibleRecord);
	    GridSetHeaderLook(&grid, true, false, false, boldFont);
	    GridSetColumnLook(&grid, false);
	}

	/* Redraw the entire display. */
	FrmDrawForm(FrmGetActiveForm());
	GridDraw(&grid);
	return true;

    case frmGotoEvent:
	form = FrmGetActiveForm();
	grid.topmostRow = event->data.frmGoto.recordNum;
	GridDraw(&grid);
	return true;

    case penDownEvent:
	if (GridHandleEvent(&grid, event))
            return true;
        break;

    case gridSelectEvent:
	CurrentRecord = event->data.tblSelect.row;
	IsNewRecord = false;
	FrmGotoForm(formID_EditView);
	return true;

    case gridUpdateVScrollersEvent:
        UpdateScrollers(event->data.generic.datum[1],
			event->data.generic.datum[2]);
        return true;

    case gridUpdateHScrollersEvent:
        UpdateFieldButtons(event->data.generic.datum[1],
			   event->data.generic.datum[2]);
        return true;

    case menuEvent:
	return DoCommand(event->data.menu.itemID);

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_ListView_DoneButton:
	    FrmGotoForm(formID_Chooser);
	    return true;

	case ctlID_ListView_NewButton:
	    IsNewRecord = true;
	    FrmGotoForm(formID_EditView);
	    return true;

	case ctlID_ListView_FindButton:
	    FrmPopupForm(formID_FindDialog);
	    return true;

	case ctlID_ListView_RepeatButton:
	    if (FindString[0])
		LocalSearch(FindString);
	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_ListView_UpButton:
	    if (prefs.flags & prefFlagUpArrowPage)
		GridScrollVertical(&grid, winUp, GridGetNumberOfRows(&grid));
	    else
		GridScrollVertical(&grid, winUp, 1);
	    break;

	case ctlID_ListView_DownButton:
	    if (prefs.flags & prefFlagDownArrowPage)
		GridScrollVertical(&grid, winDown, GridGetNumberOfRows(&grid));
	    else
		GridScrollVertical(&grid, winDown, 1);
	    break;

	case ctlID_ListView_LeftButton:
            GridScrollHorizontal(&grid, winLeft, 1);
            GridDraw(&grid);
	    break;

	case ctlID_ListView_RightButton:
            GridScrollHorizontal(&grid, winRight, 1);
            GridDraw(&grid);
	    break;
	}
	break;

    case keyDownEvent:
	switch (event->data.keyDown.chr) {
	case pageUpChr:
	    if (prefs.flags & prefFlagPageUpPage)
		GridScrollVertical(&grid, winUp, GridGetNumberOfRows(&grid));
	    else
		GridScrollVertical(&grid, winUp, 1);
	    return true;

	case pageDownChr:
	    if (prefs.flags & prefFlagPageDownPage)
		GridScrollVertical(&grid, winDown, GridGetNumberOfRows(&grid));
	    else
		GridScrollVertical(&grid, winDown, 1);
	    return true;

	}
	break;

    default:
	break;
    }

    return false;
}
