/*
 * DB: Global Find Support
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include <TxtGlue.h>

#include "enum.h"
#include "db.h"

static Boolean
GFindCallback(FindParamsPtr params, DataSource * source,
	      UInt16 cardNo, LocalID dbID)
{
    UInt16 recNum, i;
    RectangleType bounds;
    char title[64];

    /* Determine if this database should be skipped. */
    if (source->ops.GetOption_Boolean(source, optionID_DisableGlobalFind)) {
	/* Pretend that we scanned the database. */
	return false;
    }

    /* Build the title displayed for this database. */
    StrCopy(title, "DB: ");
    source->ops.GetTitle(source, title + 4, sizeof(title) - 4);
    title[sizeof(title)-1] = '\0';

    /* Draw the header for this section. */
    if (FindDrawHeader(params, title)) {
	/* No space for header. Tell caller to postpone. */
	return true;
    }

    /* Iterate over all the records looking for a match */
    recNum = params->recordNum;  
    while (1) {
	DataSourceGetter getter;
	void * rec_data;
	Boolean match;
	char * str = 0;

	if (! source->ops.Seek(source, &recNum, 0, dmSeekForward)) {
	    /* No more records in this database. */
	    break;
	}

	/* Lock down this record. */
	source->ops.LockRecord(source, recNum, false, &getter, &rec_data);

	/* Try to find a match in this record. */
	match = false;
	for (i = 0; i < source->schema.numFields; i++) {
	    if (source->schema.fields[i].type == FIELD_TYPE_STRING) {
		UInt32 outPos;
		UInt16 outLen;

		str = getter.GetString(rec_data, i);
		match = TxtGlueFindString(str, params->strToFind,
					  &outPos, &outLen);
		if (match) break;
	    }
	}

	/* If a match was found, save it. */
	if (match) {
	    if (FindSaveMatch(params, recNum, 0, i, 0, cardNo, dbID)) {
		/* A true result means we have to stop the search here. */
		source->ops.UnlockRecord(source, recNum, false, rec_data);
		params->recordNum = recNum;
		return true;
	    }

	    /* Draw the record. */
	    FindGetLineBounds(params, &bounds);
	    WinDrawChars(str, StrLen(str), bounds.topLeft.x, bounds.topLeft.y);
	    params->lineNumber++;
	}

	/* Unlock the record so we can search again. */
	source->ops.UnlockRecord(source, recNum, false, rec_data);

	/* Increment the record number. */
	recNum++;
    }

    /* We have reached the end of the database. */
    return false;
}

void
GlobalSearch(FindParamsPtr params)
{
    DataSourceDriver *drivers, *driver;
    UInt16 sourceID;
    Boolean continuation;

    /* Initialize the data source layer. */
    drivers = GetInternalDataSources();

    /* By default, we assume there are more things to search. */
    params->more = true;

    if (! params->continuation) {
	/* Start with the first data source driver. */
	driver = drivers;
    } else {
	UInt16 size = sizeof(UInt16);

	/* Retrieve the last driver active during global find. */
	PrefGetAppPreferences(DBCreatorID, prefID_GFind_sourceID, &sourceID,
			      &size, false);

	/* Recover the driver based on the stored source ID. */
	driver = drivers;
	while (driver) {
	    if (sourceID == driver->sourceID) break;
	    driver = driver->next;
	}

	/* If the driver could not be recovered, then stop global find. */
	if (!driver) {
	    params->more = false;
	    PutDataSources(drivers);
	    return;
	}
    }

    /* Loop through each driver and ask it to apply global find to
     * each data source.
     */
    continuation = params->continuation;
    while (driver) {
	if (driver->class_ops.GlobalFind) {
	    /* Ask the driver to do a global find on its sources. */
	    if (driver->class_ops.GlobalFind(GFindCallback,
					     continuation, params)) {
		/* We have been requested to stop. */
		sourceID = driver->sourceID;
		PrefSetAppPreferences(DBCreatorID, prefID_GFind_sourceID, 0,
				      &sourceID, sizeof(sourceID), false);
		PutDataSources(drivers);
		return;
	    }
	}

	/* Make sure that other drivers start from the beginning. */
	continuation = false;

	/* Point at the next driver. */
	driver = driver->next;
    }

    /* The search is finished. */
    params->more = false;
    PutDataSources(drivers);
}
