/*
 * DB: Record Edit Form
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>

#include "enum.h"
#include "db.h"
#include "linkaware.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))
#define noFieldIndex 32

typedef struct {
    MemHandle h;
    Boolean b;
    DateTimeType datetime;
} EditViewFieldData;

UInt16 CurrentRecord;
Boolean IsNewRecord;

static UInt16 TopVisibleField, CurrentField;
static EditViewFieldData * fields;
static Boolean IsDirty;

static Err EditViewLoadRecord(void * table, Int16 row, Int16 column, 
			      Boolean editing, MemHandle * textH,
			      Int16 * textOffset, Int16 * textAllocSize,
			      FieldPtr fld) EDITSECT;
static Boolean EditViewSaveRecord(void * table,
				  Int16 row, Int16 column) EDITSECT;
static void EditViewUpdateScrollers(FormPtr frm, UInt16 bottomField,
				    Boolean lastItemClipped) EDITSECT;
static UInt16 EditViewGetFieldHeight(TablePtr table, UInt16 fieldNum,
				     UInt16 columnWidth, UInt16 maxHeight)
     EDITSECT;
static void EditViewInitTableRow(TablePtr table, UInt16 row, UInt16 fieldNum,
				 short rowHeight) EDITSECT;
static void EditViewLoadTable(TablePtr table) EDITSECT;
static void EditViewInitTable(TablePtr table) EDITSECT;
static void EditViewResizeDescription(EventPtr event) EDITSECT;
#if 0
static void EditViewHandleSelectField(UInt16 row, const UInt8 column) EDITSECT;
#endif
static void EditViewScroll(WinDirectionType direction) EDITSECT;
static Boolean ValidateData(void) EDITSECT;
static void SaveRecord(void) EDITSECT;
static void SetupRecordData(void) EDITSECT;


static void
EditViewCustomDraw(void * table, Int16 row, Int16 col, RectangleType *b)
{
    UInt16 fieldNum = TblGetRowID(table, row);
    UInt16 fieldType = CurrentSource->schema.fields[fieldNum].type;
    char str[max(longDateStrLength, timeStringLength) + 1];

    switch (fieldType) {
    case FIELD_TYPE_DATE:
	DateToAscii(fields[fieldNum].datetime.month,
		    fields[fieldNum].datetime.day,
		    fields[fieldNum].datetime.year,
		    PrefGetPreference(prefLongDateFormat),
		    str);
	break;

    case FIELD_TYPE_TIME:
	TimeToAscii(fields[fieldNum].datetime.hour,
		    fields[fieldNum].datetime.minute,
		    PrefGetPreference(prefTimeFormat),
		    str);
	break;
    }

    MyWinDrawTruncChars(str, StrLen(str), 
			b->topLeft.x, b->topLeft.y, b->extent.x);
}

static Err
EditViewLoadRecord(void * table, Int16 row, Int16 column,
		   Boolean editing, MemHandle * textH, Int16 * textOffset,
		   Int16 * textAllocSize, FieldPtr fld)
{
    UInt16 fieldNum;
    Char * fieldP;
    FieldAttrType attr;

    fieldNum = (UInt16) TblGetRowID(table, row);
    ErrFatalDisplayIf(fieldNum >= CurrentSource->schema.numFields,
		      "out of range field #");
    fieldP = MemHandleLock(fields[fieldNum].h);
    *textH = fields[fieldNum].h;
    *textOffset = 0;
    *textAllocSize = StrLen(fieldP) + 1;
    MemPtrUnlock(fieldP);

    if (CurrentSource->schema.fields[fieldNum].type == FIELD_TYPE_INTEGER
	&& fld) {
	FldSetMaxChars(fld, 32);
	FldGetAttributes(fld, &attr);
	attr.singleLine = true;
	attr.dynamicSize = false;
	FldSetAttributes(fld, &attr);
    }

    return (0);
}

static Boolean
EditViewSaveRecord(void * table, Int16 row, Int16 column)
{
    FieldPtr fld;

    fld = TblGetCurrentField(table);
    if (fld && FldDirty(fld))
	IsDirty = true;

    return false;
}

static void
EditViewUpdateScrollers(FormPtr frm, UInt16 bottomField, Boolean lastItemClipped)
{
    UInt16 upIndex;
    UInt16 downIndex;
    Boolean scrollableUp;
    Boolean scrollableDown;
      
    /* If the first field displayed is not the fist field in the
     * record, enable the up scroller.
     */
    scrollableUp = TopVisibleField > 0;

    /* If the last field displayed is not the last field in the
     * record, enable the down scroller.
     */
    scrollableDown = (lastItemClipped ||
		      (bottomField < CurrentSource->schema.numFields - 1));
    
    /* Update the scroll button. */
    upIndex = FrmGetObjectIndex(frm, ctlID_EditView_UpButton);
    downIndex = FrmGetObjectIndex(frm, ctlID_EditView_DownButton);
    FrmUpdateScrollers (frm, upIndex, downIndex, scrollableUp, scrollableDown);
}

static UInt16
EditViewGetFieldHeight(TablePtr table, UInt16 fieldNum, UInt16 columnWidth,
		       UInt16 maxHeight)
{
    UInt16 height, lineHeight;
    Char * str;

    switch (CurrentSource->schema.fields[fieldNum].type) {
    case FIELD_TYPE_STRING:
	str = MemHandleLock(fields[fieldNum].h);
	height = FldCalcFieldHeight(str, columnWidth);
	lineHeight = FntLineHeight ();
	height = min (height, (maxHeight / lineHeight));
	height *= lineHeight;
	MemHandleUnlock(fields[fieldNum].h);
	break;

    case FIELD_TYPE_BOOLEAN:
    case FIELD_TYPE_INTEGER:
    case FIELD_TYPE_DATE:
    case FIELD_TYPE_TIME:
	height = FntLineHeight();
	break;

    default:
	height = FntLineHeight();
	break;
    }

    return height;
}

static void
EditViewInitTableRow(TablePtr table, UInt16 row, UInt16 fieldNum,
		     short rowHeight)
{
    /* Make the row usable and selectable. */
    TblSetRowUsable(table, row, true);
    TblSetRowSelectable(table, row, true);

    /* Set the height of the row to the height of the desc. */
    TblSetRowHeight(table, row, rowHeight);

    /* Store the record number as the row id. */
    TblSetRowID(table, row, (UInt16) fieldNum);
    ErrFatalDisplayIf(fieldNum >= CurrentSource->schema.numFields,
		      "out-of-range field #");

    /* Mark the row invalid so that it will draw when we call the 
     * draw routine.
     */
    TblMarkRowInvalid(table, row);

    /* Setup the field label column. */
    TblSetItemStyle(table, row, 0, labelTableItem);
    TblSetItemPtr(table, row, 0, CurrentSource->schema.fields[fieldNum].name);

    /* Setup the field data column. */
    switch (CurrentSource->schema.fields[fieldNum].type) {
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_INTEGER:
	TblSetItemStyle(table, row, 1, textTableItem);
	break;

    case FIELD_TYPE_BOOLEAN:
	TblSetItemStyle(table, row, 1, checkboxTableItem);
	if (fields[fieldNum].b)
	    TblSetItemInt(table, row, 1, 1);
	else
	    TblSetItemInt(table, row, 1, 0);
	break;

    case FIELD_TYPE_DATE:
    case FIELD_TYPE_TIME:
	TblSetItemStyle(table, row, 1, customTableItem);
	break;

    default:
	ErrDisplay("field type not supported");
	break;
    }
}

static void
EditViewLoadTable(TablePtr table)
{
    UInt16 row;
    UInt16 numRows;
    UInt16 fieldIndex, lastFieldIndex;
    UInt16 lineHeight,dataHeight,tableHeight;
    UInt16 columnWidth;
    UInt16 pos, oldPos;
    UInt16 height, oldHeight;
    Boolean rowUsable;
    Boolean rowsInserted = false;
    Boolean lastItemClipped;
    RectangleType r;
	
    TblGetBounds (table, &r);
    tableHeight = r.extent.y;
    columnWidth = TblGetColumnWidth(table, 1);

    /* If we currently have a selected record, make sure that it is not
     * above the first visible record.
     */
    if (CurrentField != noFieldIndex) {
	if (CurrentField < TopVisibleField)
	    TopVisibleField = CurrentField;
    }

    row = 0;
    dataHeight = 0;
    oldPos = pos = 0;
    fieldIndex = TopVisibleField;
    lastFieldIndex = fieldIndex;

    /* Load records into the table. */
    while (fieldIndex < CurrentSource->schema.numFields) {		
	/* Compute the height of the field's text string. */
	height = EditViewGetFieldHeight(table, fieldIndex, columnWidth,
					tableHeight);
	
	/* Is there enough room for at least one line of the the field. */
	lineHeight = FntLineHeight();
	if (tableHeight >= dataHeight + lineHeight) {
	    rowUsable = TblRowUsable(table, row);

	    /* Get the height of the current row. */
	    if (rowUsable)
		oldHeight = TblGetRowHeight(table, row);
	    else
		oldHeight = 0;
	    
	    /* If the field is not already being displayed in the current 
	     * row, load the field into the table.
	     */
	    if ((! rowUsable) ||
		(TblGetRowID (table, row) != fieldIndex)) {
		EditViewInitTableRow(table, row, fieldIndex, height);
	    }
			
	    /* If the height or the position of the item has changed
	     * draw the item.
	     */
	    else if (height != oldHeight) {
		TblSetRowHeight(table, row, height);
		TblMarkRowInvalid (table, row);
	    } else if (pos != oldPos) {
		TblMarkRowInvalid (table, row);
	    }

	    pos += height;
	    oldPos += oldHeight;
	    lastFieldIndex = fieldIndex;
	    fieldIndex++;
	    row++;
	}

	dataHeight += height;

	/* Is the table full? */
	if (dataHeight >= tableHeight) {
	    /* If we have a currently selected field, make sure that it is
	     * not below the last visible field.  If the currently selected 
	     * field is the last visible record, make sure the whole field 
	     * is visible.
	     */
	    if (CurrentField == noFieldIndex)
		break;
	    
	    /* Above last visible? */
	    else if (CurrentField < fieldIndex)
		break;
	    
	    /* Last visible? */
	    else if (fieldIndex == lastFieldIndex) {
		if ((fieldIndex == TopVisibleField) || (dataHeight == tableHeight))
		    break;
	    }	

	    /* Remove the top item from the table and reload the table
	     * again.
	     */
	    TopVisibleField++;
	    fieldIndex = TopVisibleField;


	    // Below last visible.
	    //			else
	    //				TopVisibleField = CurrentField;
				
	    row = 0;
	    dataHeight = 0;
	    oldPos = pos = 0;
	}
    }


    /* Hide the item that don't have any data. */
    numRows = TblGetNumberOfRows (table);
    while (row < numRows) {		
	TblSetRowUsable (table, row, false);
	row++;
    }
		
    /* If the table is not full and the first visible field is not the
     * first field in the record, displays enough fields to fill out
     * the table by adding fields to the top of the table.
     */
    while (dataHeight < tableHeight) {
	fieldIndex = TopVisibleField;
	if (fieldIndex == 0) break;
	fieldIndex--;
	
	/* Compute the height of the field. */
	height = EditViewGetFieldHeight(table, fieldIndex,
					columnWidth, tableHeight);
			
	/* If adding the item to the table will overflow the height of
	 * the table, don't add the item.
	 */
	if (dataHeight + height > tableHeight)
	    break;
		
	/* Insert a row before the first row. */
	TblInsertRow(table, 0);

	EditViewInitTableRow(table, 0, fieldIndex, height);
		
	TopVisibleField = fieldIndex;
	
	rowsInserted = true;
	
	dataHeight += height;
    }
    
    /* If rows were inserted to full out the page, invalidate the
     * whole table, it all needs to be redrawn.
     */
    if (rowsInserted)
	TblMarkTableInvalid(table);

    /* If the height of the data in the table is greater than the
     * height of the table, then the bottom of the last row is clip
     * and the table is scrollable.
     */
    lastItemClipped = (dataHeight > tableHeight);

    /* Update the scroll arrows. */
    EditViewUpdateScrollers(FrmGetActiveForm(),
			    lastFieldIndex, lastItemClipped);
}

static void
EditViewInitTable(TablePtr table)
{
    UInt16 labelWidth, width, numRows, i;

    /* Initialize the table rows to a none default. */
    numRows = TblGetNumberOfRows(table);
    for (i = 0; i < numRows; i++) {
	TblSetItemStyle(table, i, 0, labelTableItem);
	TblSetRowUsable(table, i, false);
    }

    /* Enable the label and data columns. */
    TblSetColumnUsable(table, 0, true);
    TblSetColumnUsable(table, 1, true);

    /* Figure out the width of the label column. */
    labelWidth = FntCharsWidth(CurrentSource->schema.fields[0].name,
			       StrLen(CurrentSource->schema.fields[0].name))
	+ FntCharWidth(':');
    for (i = 1; i < CurrentSource->schema.numFields; i++) {
	width = FntCharsWidth(CurrentSource->schema.fields[i].name,
			      StrLen(CurrentSource->schema.fields[i].name))
	    + FntCharWidth(':');
	if (width > labelWidth)
	    labelWidth = width;
    }
    if (labelWidth > 80)
	labelWidth = 80;
    TblSetColumnWidth(table, 0, labelWidth);
    TblSetColumnWidth(table, 1, 160 - labelWidth);

    TblSetLoadDataProcedure(table, 1, EditViewLoadRecord);
    TblSetSaveDataProcedure(table, 1, EditViewSaveRecord);
    TblSetCustomDrawProcedure(table, 1, EditViewCustomDraw);

    EditViewLoadTable(table);
}

static void
EditViewResizeDescription(EventPtr event)
{
    UInt16 pos = 0;
    UInt16 row;
    UInt16 column;
    UInt16 lastRow;
    UInt16 fieldIndex = 0;
    UInt16 lastFieldIndex;
    UInt16 topFieldIndex;
    FieldPtr fld;
    TablePtr table;
    Boolean restoreFocus = false;
    Boolean lastItemClipped;
    RectangleType itemR;
    RectangleType tableR;
    RectangleType fieldR;


    // Get the current height of the field;
    fld = event->data.fldHeightChanged.pField;
    FldGetBounds (fld, &fieldR);

    // Have the table object resize the field and move the items below
    // the field up or down.
    table = GetObjectPtr(FrmGetActiveForm(), ctlID_EditView_Table);
    TblHandleEvent (table, event);

    // If the field's height has expanded , we're done.
    if (event->data.fldHeightChanged.newHeight >= fieldR.extent.y) {
	topFieldIndex = TblGetRowID (table, 0);
	if (topFieldIndex != TopVisibleField)
	    TopVisibleField = topFieldIndex;
	else {
	    // Since the table has expanded we may be able to scroll
	    // when before we might not have.
	    lastRow = TblGetLastUsableRow (table);
	    TblGetBounds (table, &tableR);
	    TblGetItemBounds (table, lastRow, 1, &itemR);
	    lastItemClipped = (itemR.topLeft.y + itemR.extent.y > 
			       tableR.topLeft.y + tableR.extent.y);
	    lastFieldIndex = TblGetRowID (table, lastRow);
		
	    EditViewUpdateScrollers (FrmGetActiveForm (), lastFieldIndex, 
				     lastItemClipped);
	    
	    return;
	}
    }

    // If the field's height has contracted and the field edit field
    // is not visible then the table may be scrolled.  Release the 
    // focus,  which will force the saving of the field we are editing.
    else if (TblGetRowID (table, 0) != 0) {
	TblGetSelection (table, &row, &column);
	fieldIndex = TblGetRowID (table, row);
	
	fld = TblGetCurrentField (table);
	pos = FldGetInsPtPosition (fld);
	TblReleaseFocus (table);
	
	restoreFocus = true;
    }

    // Add items to the table to fill in the space made available by the 
    // shorting the field.
    EditViewLoadTable(table);
    TblRedrawTable (table);
    
    // Restore the insertion point position.
    if (restoreFocus) {
	TblFindRowID (table, fieldIndex, &row);
	TblGrabFocus (table, row, column);
	FldSetInsPtPosition (fld, pos);
	FldGrabFocus (fld);
    }
}

#if 0
static void
EditViewHandleSelectField(UInt16 row, const UInt8 column)
{
    TablePtr table;
    FieldPtr fld;
    UInt16 fieldNum;
    Char buf[16];

    StrIToA(buf, column);
    FrmCustomAlert(alertID_Debug, "row = ", buf, " ");

    table = GetObjectPtr(FrmGetActiveForm(), ctlID_EditView_Table);
    fieldNum = TblGetRowID(table, row);

    if (fieldNum != CurrentField) {
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	    FrmCustomAlert(alertID_Debug, "focus released", " ", " ");
	}
	TblUnhighlightSelection(table);
	TblGrabFocus(table, row, 1);
	fld = TblGetCurrentField(table);
	if (fld) {
	    FldGrabFocus(fld);
	    FldMakeFullyVisible(fld);
	}
	FrmCustomAlert(alertID_Debug, "focus grabbed", " ", " ");
	CurrentField = fieldNum;
    }
}
#endif

static void
EditViewScroll(WinDirectionType direction)
{
    UInt16 row, height, fieldIndex, columnWidth, tableHeight;
    TablePtr       table;
    RectangleType   r;

    table = GetObjectPtr(FrmGetActiveForm(), ctlID_EditView_Table);
    TblReleaseFocus(table);

    /* Get the height of the table and the width of the description
     * column.
     */
    TblGetBounds (table, &r);
    tableHeight = r.extent.y;
    height = 0;
    columnWidth = TblGetColumnWidth (table, 1);

    /* Scroll the table down. */
    if (direction == winDown) {
	// Get the index of the last visible field, this will become 
	// the index of the top visible field, unless it occupies the 
	// whole screeen, in which case the next field will be the
	// top field.
      
	row = TblGetLastUsableRow (table);
	fieldIndex = TblGetRowID (table, row);
      
	// If the last visible field is also the first visible field
	// then it occupies the whole screeen.
	if (row == 0)
	    fieldIndex = min(CurrentSource->schema.numFields-1, fieldIndex+1);
    }
    // Scroll the table up.
    else {
	// Scan the fields before the first visible field to determine 
	// how many fields we need to scroll.  Since the heights of the 
	// fields vary,  we sum the height of the records until we get
	// a screen full.

	fieldIndex = TblGetRowID (table, 0);
	ErrFatalDisplayIf(fieldIndex >= CurrentSource->schema.numFields,
			  "Invalid field Index");
	if (fieldIndex == 0)
	    return;
         
	height = TblGetRowHeight (table, 0);
	if (height >= tableHeight)
	    height = 0;                     

	while (height < tableHeight && fieldIndex > 0) {
	    char * str;

	    switch (CurrentSource->schema.fields[fieldIndex].type) {
	    case FIELD_TYPE_STRING:
		str = MemHandleLock(fields[fieldIndex].h);
		height += FldCalcFieldHeight(str, columnWidth)
		    * FntLineHeight();
		MemPtrUnlock(str);
		break;
	    case FIELD_TYPE_BOOLEAN:
	    case FIELD_TYPE_INTEGER:
	    default:
		height += FntLineHeight();
		break;
	    }
	    if ((height <= tableHeight)
		|| (fieldIndex == TblGetRowID (table, 0)))
		fieldIndex--;
	}
    }

    TblMarkTableInvalid (table);
    CurrentField = noFieldIndex;
    TopVisibleField = fieldIndex;

    TblUnhighlightSelection (table);
    EditViewLoadTable(table);

    TblRedrawTable (table);
}

/* Returns true if the data in the record is valid. */
static Boolean
ValidateData(void)
{
    Char * s;
    UInt16 i;

    for (i = 0; i < CurrentSource->schema.numFields; i++) {
	switch (CurrentSource->schema.fields[i].type) {
	case FIELD_TYPE_INTEGER:
	    s = MemHandleLock(fields[i].h);
	    if (! IsNumber(s)) {
		MemPtrUnlock(s);
		return false;
	    }
	    MemPtrUnlock(s);
	    break;

	default:
	    /* No checks for other data types. */
	    break;
	}
    }

    return true;
}

static char *
EditView_GetString(void * data, UInt16 fieldNum)
{
    char ** ptrs = data;
    return ptrs[fieldNum];
}

static Int32
EditView_GetInteger(void * data, UInt16 fieldNum)
{
    char ** ptrs = data;
    Int32 value;

    value = String2Long(ptrs[fieldNum]);
    return value;
}

static Boolean
EditView_GetBoolean(void * data, UInt16 fieldNum)
{
    return fields[fieldNum].b;
}

static void
EditView_GetDate(void * data, UInt16 fieldNum,
		 UInt16* year, UInt8* month, UInt8* day)
{
    *month = (fields[fieldNum].datetime.month & 0xFF);
    *day = (fields[fieldNum].datetime.day & 0xFF);
    *year = (fields[fieldNum].datetime.year & 0xFFFF);
}

static void
EditView_GetTime(void * data, UInt16 fieldNum, UInt8* hour, UInt8* minute)
{
    *hour = (fields[fieldNum].datetime.hour & 0xFF);
    *minute = (fields[fieldNum].datetime.minute & 0xFF);
}

static DataSourceGetter EditViewGetter = {
    GetString: EditView_GetString,
    GetInteger: EditView_GetInteger,
    GetBoolean: EditView_GetBoolean,
    GetDate: EditView_GetDate,
    GetTime: EditView_GetTime
};

static void
SaveRecord(void)
{
    char* ptrs[CurrentSource->schema.numFields];
    UInt16 recNum;
    UInt16 i;
    Err err;

    /* Only save the record if the record is dirty. */
    if (!IsDirty) return;
    
    /* Lock all of the handles for strings. */
    for (i = 0; i < CurrentSource->schema.numFields; i++) {
	switch (CurrentSource->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	case FIELD_TYPE_INTEGER:
	    ptrs[i] = MemHandleLock(fields[i].h);
	    break;

	default:
	    ptrs[i] = 0;
	}
    }

    /* Tell the data source layer to add/modify this record. */
    recNum = IsNewRecord ? dmMaxRecordIndex : CurrentRecord;
    err = CurrentSource->ops.UpdateRecord(CurrentSource, recNum, false,
					  &EditViewGetter, &ptrs);
    if (err != 0) {
	FrmAlert(DeviceFullAlert);
    }

    /* Unlock all of the handles locked in the previous step. */
    for (i = 0; i < CurrentSource->schema.numFields; i++) {
	switch (CurrentSource->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	case FIELD_TYPE_INTEGER:
	    MemHandleUnlock(fields[i].h);
	    break;
	default:
	    break;
	}
    }
}

static void
SetupRecordData(void)
{
    DataSourceGetter getter;
    void * rec_data;
    Int32 value;
    Char *s, *p;
    UInt16 i;

    /* Allocate the global space for this record. */
    fields = MemPtrNew(CurrentSource->schema.numFields
		       * sizeof(EditViewFieldData));

    /* For new records, just blank out the entire setup. */
    if (IsNewRecord) {
	for (i = 0; i < CurrentSource->schema.numFields; i++) {
	    switch (CurrentSource->schema.fields[i].type) {
	    case FIELD_TYPE_STRING:
		fields[i].h = MemHandleNew(1);
		s = MemHandleLock(fields[i].h);
		s[0] = '\0';
		MemPtrUnlock(s);
		break;

	    case FIELD_TYPE_INTEGER:
		fields[i].h = MemHandleNew(1);
		s = MemHandleLock(fields[i].h);
		s[0] = '\0';
		MemPtrUnlock(s);
		break;

	    case FIELD_TYPE_BOOLEAN:
		fields[i].b = false;
		break;

	    case FIELD_TYPE_DATE:
	    case FIELD_TYPE_TIME:
		TimSecondsToDateTime(TimGetSeconds(), &fields[i].datetime);
		break;

	    default:
		ErrDisplay("unsupported field type");
		break;
	    }
	}

	/* Do not fall into the normal code path. */
	return;
    }

    /* Retrieve a copy of the record. */
    CurrentSource->ops.LockRecord(CurrentSource, CurrentRecord, false,
				  &getter, &rec_data);

    /* Copy the record data into place. */
    for (i = 0; i < CurrentSource->schema.numFields; i++) {
	switch (CurrentSource->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	    p = getter.GetString(rec_data, i);
	    fields[i].h = MemHandleNew(StrLen(p) + 1);
	    s = MemHandleLock(fields[i].h);
	    StrCopy(s, p);
	    MemPtrUnlock(s);
	    break;

	case FIELD_TYPE_BOOLEAN:
	    fields[i].b = getter.GetBoolean(rec_data, i);
	    break;

	case FIELD_TYPE_INTEGER:
	    value = getter.GetInteger(rec_data, i);
	    fields[i].h = MemHandleNew(32);
	    s = MemHandleLock(fields[i].h);
	    MemSet(s, 32, 0);
	    StrIToA(s, value);
	    MemPtrUnlock(s);
	    break;

	case FIELD_TYPE_DATE:
	    {
		UInt16 year;
		UInt8 month, day;

		getter.GetDate(rec_data, i, &year, &month, &day);
		fields[i].datetime.month = month;
		fields[i].datetime.day   = day;
		fields[i].datetime.year  = year;
	    }
	    break;

	case FIELD_TYPE_TIME:
	    {
		UInt8 hour, minute;

		getter.GetTime(rec_data, i, &hour, &minute);
		fields[i].datetime.hour = hour;
		fields[i].datetime.minute = minute;
	    }
	    break;

	default:
	    break;
	}
    }

    /* Unlock the record. */
    CurrentSource->ops.UnlockRecord(CurrentSource, CurrentRecord, false,
				    rec_data);
}

static void
FreeRecordData(void)
{
    UInt16 i;

    for (i = 0; i < CurrentSource->schema.numFields; ++i) {
	switch (CurrentSource->schema.fields[i].type) {
	case FIELD_TYPE_STRING:
	case FIELD_TYPE_INTEGER:
	    MemHandleFree(fields[i].h);
	    break;

	default:
	    /* Nothing to free for other field types. */
	    break;
	}
    }

    MemPtrFree(fields);
    fields = 0;
}

Boolean
EditViewHandleEvent(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    UInt16 fieldNum;
    FieldPtr fld;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_EditView_Table);

	/* Initialize the globals we use. */
	IsDirty = false;   /* Data is not dirty initially. */
	TopVisibleField = 0;
	CurrentField = noFieldIndex;

	SetupRecordData();
	EditViewInitTable(table);

	/* Disable the cancel button in read-only mode. */
#define HideObject(f,id) FrmHideObject((f),FrmGetObjectIndex((f),(id)))
	if (CurrentSource->readonly) {
	    HideObject(form, ctlID_EditView_CancelButton);
	}
	if (IsNewRecord || CurrentSource->readonly) {
	    HideObject(form, ctlID_EditView_DeleteButton);
	}
#undef HideObject

	FrmDrawForm(form);
	return true;

    case frmCloseEvent:
	/* Make sure table no longer has focus. */
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_EditView_Table);
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	}

	/* Free the record data. */
	FreeRecordData();

	return false;

    case frmSaveEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_EditView_Table);
	if (TblGetCurrentField(table)) {
	    TblReleaseFocus(table);
	}
	if (! CurrentSource->readonly && ValidateData())
	    SaveRecord();
	return true;
	
    case tblSelectEvent:
	table = event->data.tblSelect.pTable;
	fieldNum = TblGetRowID(table, event->data.tblSelect.row);
	if (event->data.tblSelect.column == 0) {
	    switch (CurrentSource->schema.fields[fieldNum].type) {
	    case FIELD_TYPE_STRING:
	    case FIELD_TYPE_INTEGER:
		/* A tap on the label edits the field. */
		if (TblGetCurrentField(event->data.tblSelect.pTable)) {
		    TblReleaseFocus(event->data.tblSelect.pTable);
		}
		TblUnhighlightSelection(event->data.tblSelect.pTable);
		TblGrabFocus(event->data.tblSelect.pTable,
			     event->data.tblSelect.row, 1);
		fld = TblGetCurrentField(event->data.tblSelect.pTable);
		if (fld) {
		    FldGrabFocus(fld);
		    FldMakeFullyVisible(fld);
		}
		return true;

	    default:
		break;
	    }
	} else if (event->data.tblSelect.column == 1) {
	    switch (CurrentSource->schema.fields[fieldNum].type) {
	    case FIELD_TYPE_BOOLEAN:
		if (TblGetItemInt(table,
				  event->data.tblSelect.row,
				  event->data.tblSelect.column))
		    fields[fieldNum].b = true;
		else
		    fields[fieldNum].b = false;
		IsDirty = true;
		break;

	    case FIELD_TYPE_DATE:
		{
		    Int16 month, day, year;

		    month = fields[fieldNum].datetime.month;
		    day = fields[fieldNum].datetime.day;
		    year = fields[fieldNum].datetime.year;
		    if (SelectDay(selectDayByDay,
				  &month, &day, &year,
				  "Edit Date")) {
			fields[fieldNum].datetime.month = month;
			fields[fieldNum].datetime.day = day;
			fields[fieldNum].datetime.year = year;
			TblMarkRowInvalid(table, event->data.tblSelect.row);
			TblRedrawTable(table);
			IsDirty = true;
		    }
		}
		break;

	    case FIELD_TYPE_TIME:
		{
		    Boolean selected = false;

		    if (FeatureSet31) {
			Int16 hour, minute;

			hour = fields[fieldNum].datetime.hour;
			minute = fields[fieldNum].datetime.minute;
			if (SelectOneTime(&hour, &minute, "Edit Time")) {
			    fields[fieldNum].datetime.hour = hour;
			    fields[fieldNum].datetime.minute = minute;
			    selected = true;
			}
		    } else {
			TimeType start, finish;

			start.hours = fields[fieldNum].datetime.hour;
			start.minutes = fields[fieldNum].datetime.minute;
			finish.hours = noTime;
			finish.minutes = noTime;

			if (SelectTimeV33(&start, &finish, false,
					  "Edit Time", 0)) {
			    fields[fieldNum].datetime.hour = start.hours;
			    fields[fieldNum].datetime.minute = start.minutes;
			    selected = true;
			}
		    }

		    if (selected) {
			TblMarkRowInvalid(table, event->data.tblSelect.row);
			TblRedrawTable(table);
			IsDirty = true;	
		    }
		}
		break;

	    default:
		break;
	    }
	}
	return false;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_EditView_DoneButton:
	    if (! CurrentSource->readonly) {
		if (!ValidateData()) {
		    FrmAlert(alertID_ValidationFailure);
		    return true;
		}
		SaveRecord();
	    }
	    /* fall-through */

	case ctlID_EditView_CancelButton:
	    FrmGotoForm(formID_ListView);
	    return true;

	case ctlID_EditView_DeleteButton:
	    if (!IsNewRecord && FrmAlert(alertID_ConfirmDelete) == 0) {
		CurrentSource->ops.DeleteRecord(CurrentSource, CurrentRecord);
		CurrentRecord = dmMaxRecordIndex;
		FrmGotoForm(formID_ListView);
	    }
	    return true;
	}
	break;

    case fldHeightChangedEvent:
	EditViewResizeDescription(event);
	return true;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_EditView_UpButton:
	    EditViewScroll(winUp);
	    break;

	case ctlID_EditView_DownButton:
	    EditViewScroll(winDown);
	    break;
	}
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_PublishLink:
	    {
		Char name[dmDBNameLength];
		LocalID dbID;
		UInt16 cardNo;
		LinkInfoPtr link;
		
		/* Only handle if data source is backed by a PDB. */
		if (CurrentSource->ops.GetPDB) {
		    CurrentSource->ops.GetPDB(CurrentSource, &cardNo, &dbID);
		    DmDatabaseInfo(cardNo, dbID, name, 0,0,0,0,0,0,0,0,0,0);
		    link = (LinkInfoPtr) LinkSimpleNew(CurrentSource->db,
						       CurrentRecord, name);
		    if (! LinkPublish(link)) {
			FrmAlert(alertID_NoLinkMaster);
		    }

		    return true;
		}
	    }
	    break;
	    
	default:
	    return HandleCommonMenuEvent(event->data.menu.itemID);
	}
	break;

    default:
	break;

    }

    return false;
}
